/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/bitops.h>
#include <linux/uaccess.h>
#include <mach/hardware.h>
#include <asm/arch-umvp/a1_sys.h>

#define A1_WDT_CNT_STS	0x00                                                                                                                                                                                                           
#define A1_WDT_RRELOAD	0x04
#define A1_WDT_RESTART	0x08
#define A1_WDT_CTRL		0x0C
#define A1_WDT_TIMEOUT	0x10
#define A1_WDT_CLR		0x14

#define WDT_READ_REG(r)        (*((volatile unsigned int *) (IO_ADDRESS(UMVP_WDT_BASE) + r)))
#define WDT_WRITE_REG(r,v)     (*((volatile unsigned int *) (IO_ADDRESS(UMVP_WDT_BASE) + r)) = ((unsigned int)   (v)))

#define WATCHDOG_TIMEOUT 30		/* 30 sec default timeout , max 38 at 112.5MHz*/

static int nowayout = WATCHDOG_NOWAYOUT;
static int timeout = WATCHDOG_TIMEOUT;

#define SRC_PCLK	0
#define SRC_EXTCLK	1
static int extclk = SRC_PCLK;	/* default is pclk */

static unsigned long wdt_status;
static unsigned long boot_status;
static DEFINE_SPINLOCK(wdt_lock);

/* ticks per second */
static unsigned long wdt_tick_rate = 0;
#define	WDT_IN_USE		0
#define	WDT_OK_TO_CLOSE		1

#define TO_RESET_EN	(0x1UL << 1);
#define TO_INT_EN	(0x1UL << 2);
#define TO_EXT_EN	(0x1UL << 3);

static void a1_wdt_ctrl(void)
{
	unsigned long ctrl;

	WDT_WRITE_REG (A1_WDT_RESTART, 0x4755);	/* set counter restart */

	ctrl = WDT_READ_REG (A1_WDT_CTRL);
	ctrl |= 1;				/* enable WDT */
	WDT_WRITE_REG (A1_WDT_CTRL, ctrl);
}

static void wdt_keepalive(void)
{
	spin_lock(&wdt_lock);
	WDT_WRITE_REG (A1_WDT_RRELOAD,  timeout * wdt_tick_rate);	/* update reload register */
	a1_wdt_ctrl ();
	spin_unlock(&wdt_lock);
}

static void wdt_disable(void)
{
	unsigned long ctrl;

	spin_lock(&wdt_lock);
	ctrl = WDT_READ_REG (A1_WDT_CTRL);
	ctrl &= ~1UL;
	WDT_WRITE_REG (A1_WDT_CTRL, ctrl);
	spin_unlock(&wdt_lock);
}

static int wdt_open(struct inode *inode, struct file *file)
{

	if (test_and_set_bit(WDT_IN_USE, &wdt_status))
		return -EBUSY;

	clear_bit(WDT_OK_TO_CLOSE, &wdt_status);
	wdt_keepalive();
	return nonseekable_open(inode, file);
}

static int wdt_set_heartbeat(int t)
{
	unsigned long timeout_max;

	timeout_max = 0xFFFFFFFF / wdt_tick_rate;
	if (t < 1 || t > timeout_max) {
		printk(KERN_INFO " Watch Dog max timeout = %d\n", timeout_max);
		return -EINVAL;
	}
	timeout = t;
	return 0;
}

static ssize_t
wdt_write(struct file *file, const char *data, size_t len, loff_t *ppos)
{
	if (len) {
		if (!nowayout) {
			size_t i;

			clear_bit(WDT_OK_TO_CLOSE, &wdt_status);

			for (i = 0; i != len; i++) {
				char c;
				if (get_user(c, data + i))
					return -EFAULT;
				if (c == 'V')
					set_bit(WDT_OK_TO_CLOSE, &wdt_status);
			}
		}
		wdt_keepalive();
	}
	return len;
}

static struct watchdog_info ident = {
	.options = WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT |
							WDIOF_MAGICCLOSE,
	.identity	= "ACTi A1 Watchdog",
};


static long wdt_ioctl(struct file *file, unsigned int cmd,
							unsigned long arg)
{
	int new_timeout;

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		return copy_to_user((struct watchdog_info *)arg, &ident,
				   sizeof(ident)) ? -EFAULT : 0;

	case WDIOC_GETSTATUS:
		return  put_user(0, (int *)arg);

	case WDIOC_GETBOOTSTATUS:
		return put_user(boot_status, (int *)arg);

	case WDIOC_KEEPALIVE:
		wdt_keepalive();
		return 0;

	case WDIOC_SETTIMEOUT:
		if (get_user(new_timeout, (int *)arg))
			return -EFAULT;
		if (wdt_set_heartbeat(new_timeout))
			return -EINVAL;
		wdt_keepalive();
		/* Fall through */
	case WDIOC_GETTIMEOUT:
		return put_user(timeout, (int *)arg);
	default:
		return -ENOTTY;
	}
}

static int wdt_release(struct inode *inode, struct file *file)
{
	if (test_bit(WDT_OK_TO_CLOSE, &wdt_status))
		wdt_disable();
	else
		printk(KERN_CRIT "WATCHDOG: Device closed unexpectedly - "
					"timer will not stop\n");
	clear_bit(WDT_IN_USE, &wdt_status);
	clear_bit(WDT_OK_TO_CLOSE, &wdt_status);

	return 0;
}

/*
 *	Notifier for system down
 */

static int wdt_notify_sys(struct notifier_block *this, unsigned long code,
	void *unused)
{
	if (code == SYS_DOWN || code == SYS_HALT)
		wdt_disable();	/* Turn the WDT off */

	return NOTIFY_DONE;
}

static const struct file_operations a1_wdt_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.write		= wdt_write,
	.open		= wdt_open,
	.release	= wdt_release,
	.unlocked_ioctl	= wdt_ioctl,
};

static struct miscdevice a1_wdt_miscdev = {
	.minor		= WATCHDOG_MINOR,
	.name		= "watchdog",
	.fops		= &a1_wdt_fops,
};

/*
 *	The WDT needs to learn about soft shutdowns in order to
 *	turn the timebomb registers off.
 */

static struct notifier_block wdt_notifier = {
	.notifier_call = wdt_notify_sys,
};

static int __init wdt_init(void)
{
	int ret;
	unsigned long ctrl;

	spin_lock_init(&wdt_lock);

	ctrl =  WDT_READ_REG (A1_WDT_CTRL);	
	/* set clksrc */
	if (extclk == 0)
		ctrl &= ~(1UL << 4);
	else
		ctrl |= (1UL << 4);

	ctrl |= TO_RESET_EN;	/* set timeout action */
	WDT_WRITE_REG (A1_WDT_CTRL, ctrl);	

	/* let boot status to recorad ctrl value */
	/* I have no idea what will boot_status be */
	boot_status = ctrl;

	ret = register_reboot_notifier(&wdt_notifier);
	if (ret != 0) {
		printk(KERN_ERR
			"cannot register reboot notifier (err=%d)\n", ret);
		goto unreg_reboot;
	}

	ret = misc_register(&a1_wdt_miscdev);
	if (ret == 0)
		printk(KERN_INFO "ACTi Watchdog Timer: timeout %d sec\n",
			timeout);

	/* Get current APB clock setting as WDT tick rate */
	wdt_tick_rate = PLL_GetAPBClock(A1_SYS_CLOCK_SRC);
	return ret;

unreg_reboot:
	unregister_reboot_notifier(&wdt_notifier);
	return ret;
}

static void __exit wdt_exit(void)
{
	misc_deregister(&a1_wdt_miscdev);
	unregister_reboot_notifier(&wdt_notifier);
}


module_init(wdt_init);
module_exit(wdt_exit);

MODULE_DESCRIPTION("ACTi A1 SOC Watchdog");

module_param(timeout, int, 0);
MODULE_PARM_DESC(timeout, "Watchdog timeout in seconds (default 30s)");

module_param(nowayout, int, 0);
MODULE_PARM_DESC(nowayout, "Watchdog cannot be stopped once started");

module_param(extclk, int, 0);
MODULE_PARM_DESC(extclk, "Watchdog takes external clock source , default pclk");

MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);

