/*
 * EHCI HCD (Host Controller Driver) PCI Bus Glue.
 *
 * Copyright (c) 2000-2004 by David Brownell
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern void usb_disconnect(struct usb_device **);

#ifndef CONFIG_USB
#error "This file is PCI bus glue.  CONFIG_PCI must be defined."
#endif

//#define printk printk
//#define printk(msg...)

irqreturn_t common_interrupt_handler (struct usb_hcd *hcd);


//#ifdef CONFIG_DEBUG_FS

#define MAX_OUTPUT	(64 * 1024)

struct ehci_debug {
	int size;
	char *data;
};


/* called after powerup, by probe or system-pm "wakeup" */
//static int ehci_pci_reinit(struct ehci_hcd *ehci, struct pci_dev *pdev)
static int ehci_ahb_reinit(struct ehci_hcd *ehci)
{


	ehci_port_power(ehci, 0);

	return 0;
}

/* called during probe() after chip reset completes */
//static int ehci_pci_setup(struct usb_hcd *hcd)
static int ehci_ahb_setup(struct usb_hcd *hcd)
{
	struct ehci_hcd		*ehci = hcd_to_ehci(hcd);
//	struct pci_dev		*pdev = to_pci_dev(hcd->self.controller);
//	u32			temp;
	int			retval;

//	printk("ehci_ahb_setup() enter\n");
	
	ehci->caps = hcd->regs;
	ehci->regs = hcd->regs + HC_LENGTH(readl(&ehci->caps->hc_capbase));
	dbg_hcs_params(ehci, "reset");
	dbg_hcc_params(ehci, "reset");

	/* cache this readonly data; minimize chip reads */
	ehci->hcs_params = readl(&ehci->caps->hcs_params);	

        ehci_base1 = (unsigned long *) ehci->regs;

	retval = ehci_halt(ehci);
	
	
	if (retval)
		return retval;

	
	/* data structure init */
	retval = ehci_init(hcd);
	
	
	if (retval)
		return retval;


	if (ehci_is_TDI(ehci))
		ehci_reset(ehci);


#ifdef	CONFIG_USB_SUSPEND
	/* REVISIT: the controller works fine for wakeup iff the root hub
	 * itself is "globally" suspended, but usbcore currently doesn't
	 * understand such things.
	 *
	 * System suspend currently expects to be able to suspend the entire
	 * device tree, device-at-a-time.  If we failed selective suspend
	 * reports, system suspend would fail; so the root hub code must claim
	 * success.  That's lying to usbcore, and it matters for for runtime
	 * PM scenarios with selective suspend and remote wakeup...
	 */
	if (ehci->no_selective_suspend && device_can_wakeup(&pdev->dev))
		ehci_warn(ehci, "selective suspend/wakeup unavailable\n");
#endif

	retval = ehci_ahb_reinit(ehci);
	
//	printk("ehci_ahb_setup() exit\n");
	
	return retval;
}

/*-------------------------------------------------------------------------*/

#ifdef	CONFIG_PM

/* suspend/resume, section 4.3 */

/* These routines rely on the PCI bus glue
 * to handle powerdown and wakeup, and currently also on
 * transceivers that don't need any software attention to set up
 * the right sort of wakeup.
 * Also they depend on separate root hub suspend/resume.
 */

//static int ehci_pci_suspend(struct usb_hcd *hcd, pm_message_t message)
static int ehci_ahb_suspend(struct usb_hcd *hcd, pm_message_t message)
{
	struct ehci_hcd		*ehci = hcd_to_ehci(hcd);
	unsigned long		flags;
	int			rc = 0;

	if (time_before(jiffies, ehci->next_statechange))
		msleep(10);

	/* Root hub was already suspended. Disable irq emission and
	 * mark HW unaccessible, bail out if RH has been resumed. Use
	 * the spinlock to properly synchronize with possible pending
	 * RH suspend or resume activity.
	 *
	 * This is still racy as hcd->state is manipulated outside of
	 * any locks =P But that will be a different fix.
	 */
	spin_lock_irqsave (&ehci->lock, flags);
	if (hcd->state != HC_STATE_SUSPENDED) {
		rc = -EINVAL;
		goto bail;
	}
	writel (0, &ehci->regs->intr_enable);
	(void)readl(&ehci->regs->intr_enable);

	clear_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
 bail:
	spin_unlock_irqrestore (&ehci->lock, flags);

	// could save FLADJ in case of Vaux power loss
	// ... we'd only use it to handle clock skew

	return rc;
}

//static int ehci_pci_resume(struct usb_hcd *hcd)
static int ehci_ahb_resume(struct usb_hcd *hcd)
{
	struct ehci_hcd		*ehci = hcd_to_ehci(hcd);
	unsigned		port;
	//struct pci_dev		*pdev = to_pci_dev(hcd->self.controller);
	int			retval = -EINVAL;

	// maybe restore FLADJ

	if (time_before(jiffies, ehci->next_statechange))
		msleep(100);

	/* Mark hardware accessible again as we are out of D3 state by now */
	set_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);

	/* If CF is clear, we lost PCI Vaux power and need to restart.  */
	if (readl(&ehci->regs->configured_flag) != FLAG_CF)
		goto restart;

	/* If any port is suspended (or owned by the companion),
	 * we know we can/must resume the HC (and mustn't reset it).
	 * We just defer that to the root hub code.
	 */

	for (port = 1; port > 0; ) {		
		u32	status;
		port--;
		status = readl(&ehci->regs->port_status [port]);
		if (!(status & PORT_POWER))
			continue;
		if (status & (PORT_SUSPEND | PORT_RESUME | PORT_OWNER)) {
			usb_hcd_resume_root_hub(hcd);
			return 0;
		}
	}

restart:
	ehci_dbg(ehci, "lost power, restarting\n");
	usb_root_hub_lost_power(hcd->self.root_hub);

	/* Else reset, to cope with power loss or flush-to-storage
	 * style "resume" having let BIOS kick in during reboot.
	 */
	(void) ehci_halt(ehci);
	(void) ehci_reset(ehci);
	(void) ehci_ahb_reinit(ehci); //, pdev);

	/* emptying the schedule aborts any urbs */
	spin_lock_irq(&ehci->lock);
	if (ehci->reclaim)
		ehci->reclaim_ready = 1;
	ehci_work(ehci);
	spin_unlock_irq(&ehci->lock);

	// restart; khubd will disconnect devices
	retval = ehci_run(hcd);

	// here we "know" root ports should always stay powered 
	ehci_port_power(ehci, 1);

	return retval;
}
#endif

static const struct hc_driver ehci_driver = {
	.description =		hcd_name,
	.product_desc =		"EHCI Host Controller",
	.hcd_priv_size =	sizeof(struct ehci_hcd),

	/*
	 * generic hardware linkage
	 */
//	.irq =			ehci_irq,
	.irq =			common_interrupt_handler,
	.flags =		HCD_MEMORY | HCD_USB2,

	/*
	 * basic lifecycle operations
	 */
	.reset =		ehci_ahb_setup,
	.start =		ehci_run,
#ifdef	CONFIG_PM
	.pci_suspend =	ehci_ahb_suspend,
	.pci_resume =	ehci_ahb_resume,
#endif
	.stop =			ehci_stop,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue =		ehci_urb_enqueue,
	.urb_dequeue =		ehci_urb_dequeue,
	.endpoint_disable =	ehci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number =	ehci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data =	ehci_hub_status_data,
	.hub_control =		ehci_hub_control,
	.bus_suspend =		ehci_bus_suspend,
	.bus_resume =		ehci_bus_resume,
	.relinquish_port =	ehci_relinquish_port,
	.port_handed_over =	ehci_port_handed_over,
};




/*-------------------------------------------------------------------------*/

void ahb_ehci_module_cleanup (void)
{
    struct hc_driver     *driver;
    struct usb_hcd        *hcd;
    struct usb_device    *hub;
    driver = (struct hc_driver *)&ehci_driver;


    printk("ahb_ehci_module_cleanup()\n");

   
    hcd = otg->otg_ehci; 

    if (!hcd)
        return;

    if (in_interrupt ()) BUG ();


    
    usb_disconnect (&hub);
    hcd->driver->stop (hcd);


    free_irq (otg->irq, hcd);

}

///////////////////////////////////////////////////////////////////////////
/**
 * usb_hcd_pci_probe - initialize PCI-based HCDs
 * @dev: USB Host Controller being probed
 * @id: pci hotplug id connecting controller to HCD framework
 * Context: !in_interrupt()
 *
 * Allocates basic PCI resources for this USB host controller, and
 * then invokes the start() method for the HCD associated with it
 * through the hotplug entry's driver_data.
 *
 * Store this function in the HCD's struct pci_driver as probe().
 */

//static struct dentry *ehci_debugfs_root;
static struct device dev;   


int ahb_ehci_module_init (void)
{
	struct hc_driver	*driver;
	struct usb_hcd		*hcd;
	int			retval;				

	//printk("ahb_ehci_module_init() enter\n");
	
	
	retval = -ENODEV;
	
	driver = (struct hc_driver *)&ehci_driver;
	if (!(driver))
	{
	    printk("ahb_ehci_module_init:driver pointer is null\n");
	    return -EINVAL;
	}
    
	if (usb_disabled())
		return -ENODEV;

	device_initialize(&dev);
    kobject_set_name(&dev.kobj, "guc_ehci");
	dev.dma_mask = (u64 *)kzalloc(sizeof(u64), GFP_KERNEL);
    kobject_add(&dev.kobj, dev.kobj.parent, "guc_ehci");
	kobject_uevent(&dev.kobj, KOBJ_ADD);
	
	//printk("dev.kobj.name(%x)\n", dev.kobj.name);
	//if (dev.kobj.name)
	//	printk("dev.kobj.name:%s\n", dev.kobj.name);
        
	
	hcd = usb_create_hcd (driver, &dev, ehci_name);
	if (!hcd) {
		retval = -ENOMEM;
		goto err1;
	}

    *(hcd->self.controller->dma_mask) = (u64)0xffffffffUL;
    hcd->self.controller->coherent_dma_mask = 0xffffffffUL;

	//printk("ahb_ehci_module_init: controller %p, co_dmask %lx\n", hcd->self.controller, dev.coherent_dma_mask);


    hcd->driver = driver;
    otg->otg_ehci = hcd;

    Portbase =(unsigned long *) ioremap_nocache (EHC_AHB_BASE_ADDR, 4096);
    hcd->regs = (void *)Portbase;

    printk("Ehci Base address is %x \n",(unsigned int)Portbase);

    writel(0x00080000,hcd->regs);
 
  
    retval = usb_add_hcd (hcd, otg->irq, IRQF_DISABLED | IRQF_SHARED);
    if (retval != 0)
		goto err1;
	return retval;


 err1:
	//dev_err (&dev->dev, "init %s fail, %d\n", ehci_name, retval);
	return retval;
} 



