/*
 *  linux/drivers/char/umvp_serial_16550.c
 *  modified from linux/drivers/serial/amba.c
 *
 *
 *  Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

//#include <linux/config.h>
#include <linux/module.h>
#include <linux/tty.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/serial_core.h>
#include <linux/uaccess.h>

#include <asm/io.h>
#include <asm/irq.h>

/* **** add two define in ioctls.h  ******
#define TIOCGRS485      0x542E
#define TIOCSRS485      0x542F
*/
#include <asm/ioctls.h>

#include <mach/umvp_serial.h>
#include <mach/hardware.h>

/*********** below : system integration dependent ***********/
    /*
     * The following macroes are system dependent.
     * Those who use this driver should provide the macro
     * values to satisfy their own systems. The default macro
     * values reflect the system conditions on GUC's UMVP boards.
     */
/* --------------------------------------------------------------------------------------- */
/*
 * SYS_UART_NR:
 *    define the number of UARTs instanced in your system
 *    (This driver could support up to 2 UARTs.)
 */
#define SYS_UART_NR     2
#define RS485_PORT_ID   1
/*
 * SYS_APB_CLOCK_RATE:
 *    define clock rate of the APB bus where the UART is
 */
#define SYS_APB_CLOCK_RATE      UMVP2500_APBCLOCK
/*
 * SYS_UART_BAUD_RATE:
 *    define the baud rate of the UART transmission
 */
#define SYS_UART_BAUD_RATE  CONFIG_SERIAL_UMVP_CONSOLE_BAUD
/*
 * SYS_UART_CONSOLE:
 *    defined while use UART as console
 */
#define SYS_UART_CONSOLE 1

/* --------------------------------------------------------------------------------------- */
/*
 * SYS_UART0_BASE_PA:
 *    define the 0th uart's physical address in your system
 */
#define SYS_UART0_BASE_PA   UMVP_UART0_BASE

/*
 * SYS_UART0_BASE_VA:
 *    define the 0th uart's virtual address in your system
 */
#define SYS_UART0_BASE_VA   (IO_ADDRESS(UMVP_UART0_BASE))

/*
 * SYS_UART0_IRQ_NR:
 *    define the 0th uart's interrupt number in your system
 */
#define SYS_UART0_IRQ_NR    IRQ_UARTINT0

/* --------------------------------------------------------------------------------------- */
/* RS485 is used UART4, so we disable UART1 and enable uart4 */
/* **********************************************************************
    RS485 use :
    1. UART4_RX  -> X_UART4_SIN  -> V24 -> X_GPIO3_23 pin
    2. UART4_TX  -> X_UART4_SOUT -> U22 -> X_GPIO3_24 pin
    3. UART4_RTS -> X_UART4_RTSn -> U23 -> X_GPIO3_25 pin
    4. The muti-function pin default on UART4_RX, UART4_TX but
       UART4_RTS be switched to gpio3_25
   *********************************************************************** */
#define USED_UART1  0

#if USED_UART1
    /*
     * SYS_UART1_BASE_PA:
     *    define the 1st uart's physical address in your system
     */
    #define SYS_UART1_BASE_PA   UMVP_UART1_BASE

    /*
     * SYS_UART1_BASE_VA:
     *    define the 1st uart's virtual address in your system
     */
    #define SYS_UART1_BASE_VA   (IO_ADDRESS(UMVP_UART1_BASE))

    /*
     * SYS_UART1_IRQ_NR:
     *    define the 4st uart's interrupt number in your system
     */
    #define SYS_UART1_IRQ_NR    IRQ_UARTINT1

#else
    /* Define UART4  address in system */
    #define SYS_UART4_BASE_PA   UMVP_UART4_BASE
    #define SYS_UART4_BASE_VA   (IO_ADDRESS(UMVP_UART4_BASE))
    #define SYS_UART4_IRQ_NR    INT_UARTINT4      /* IRQ_UARTINT4 */
#endif

/* --------------------------------------------------------------------------------------- */
/*
 * SYS_UART_INTR_SENSE(irqnr):
 *    Please implement how to set the sensitivity of the uart
 *    interrupt on your system. The sensitivity of this IP is
 *    "high level" trigger.
 */
#define SYS_UART_INTR_SENSE(irqnr)                      \
{                                                       \
    vic_set_intr_trigger(irqnr, vicc_level_activeHigh); \
}

/*********** above : system integration dependent ***********/

#define UART_TYPE_STRING            "UMVP 16550"

#define UMVP_SERIAL_CONSOLE_NAME    "ttyS"
#define UMVP_SERIAL_TTY_NAME        "ttyAM"
#define UMVP_SERIAL_MAJOR           204
#define UMVP_SERIAL_MINOR           16

#define UMVP_SERIAL_ISR_PASS_LIMIT  256

/*
 * Access macros for the UARTs
 */
#define UART_GET_CHAR(p)    readl((p)->membase + UART_RBR)
#define UART_PUT_CHAR(p, v) writel((v), (p)->membase + UART_THR)
#define UART_GET_DLL(p)     readl((p)->membase + UART_DLL)
#define UART_PUT_DLL(p, v)  writel((v), (p)->membase + UART_DLL)
#define UART_GET_DLH(p)     readl((p)->membase + UART_DLH)
#define UART_PUT_DLH(p, v)  writel((v), (p)->membase + UART_DLH)
#define UART_GET_IER(p)     readl((p)->membase + UART_IER)
#define UART_PUT_IER(p, v)  writel((v), (p)->membase + UART_IER)
#define UART_GET_IIR(p)     readl((p)->membase + UART_IIR)
#define UART_GET_FCR(p)     readl((p)->membase + UART_FCR)
#define UART_PUT_FCR(p, v)  writel((v), (p)->membase + UART_FCR)
#define UART_GET_LCR(p)     readl((p)->membase + UART_LCR)
#define UART_PUT_LCR(p, v)  writel((v), (p)->membase + UART_LCR)
#define UART_GET_LSR(p)     readl((p)->membase + UART_LSR)

#define UART_DUMMY_RSR_RX   256
#define UART_PORT_SIZE      64


/* --------------------------------------------------------------------------------------- */
/*
 * We wrap our port structure around the generic uart_port.
 */
struct umvp_uart_port_S {
    struct uart_port    port;
    struct serial_rs485 rs485;      /* rs485 settings */
};
static struct umvp_uart_port_S umvp_uart_ports[SYS_UART_NR];


static inline struct umvp_uart_port_S * to_umvp_uart_port(struct uart_port *port)
{
    return container_of(port, struct umvp_uart_port_S, port);
}

static void umvp_uart_start_rx(struct uart_port *port);
static void umvp_uart_stop_rx(struct uart_port *port);


/***************************************************************
   1. halSysSetMultiFunctionalPin(HAL_MFP_GPIO_AP):
      This function had switched X_GPIO3_25 pin to gpio
   2. set X_GPIO3_25 direction to output(1).
 ****************************************************************/

/* config gpio as RTS, see umvp_gpio.h */
#define GPIO3_BASE_ADDR         IO_ADDRESS(UMVP_GPIO3_BASE)
#define GPIO3_REG(offset)       (*(volatile __u32 *)(GPIO3_BASE_ADDR+(offset)))

/* GPIO register offset */
#define GPIODATA                0x0000 /* The data register */
#define GPIODIR                 0x0004 /* Data direction control register */
#define GPIOMASK                0x0028 /* Data read/write mask register */

static void a1_rs485_config_rts_gpio(void)
{
    u_int gpio3_b25 = 0x00;

    gpio3_b25 = GPIO3_REG(GPIODIR);
    gpio3_b25 = (gpio3_b25 & ~0x02000000) | 0x02000000;
    GPIO3_REG(GPIODIR) = gpio3_b25;
}
static void a1_rs485_rts(int value)
{
    u_int gpio3_b25 = 0x00;

    /* set mask bit */
    gpio3_b25 = GPIO3_REG(GPIOMASK);
    gpio3_b25 = (gpio3_b25 & ~0x02000000) | 0x02000000;
    GPIO3_REG(GPIOMASK) = gpio3_b25;

    /* write data bit*/
    gpio3_b25 = GPIO3_REG(GPIODATA);
    if (value) gpio3_b25 = (gpio3_b25 & ~0x02000000) | 0x02000000; 
    else       gpio3_b25 = (gpio3_b25 & ~0x02000000) | 0x00000000; 
    GPIO3_REG(GPIODATA)  = gpio3_b25;
}
/* --------------------------------------------------------------------------------------- */

/* Enable or disable the rs485 support */
void umvp_uart_config_rs485(struct uart_port *port, struct serial_rs485 *rs485conf)
{
    struct umvp_uart_port_S *p =  to_umvp_uart_port(port);
    unsigned int cr = UART_GET_IER(port);

    spin_lock(&port->lock);
    /* Disable tx interrupts */
    cr &= ~UART_IER_ETEI;
    UART_PUT_IER(port, cr);
    p->rs485 = *rs485conf;
    if (rs485conf->flags & SER_RS485_ENABLED) {
        printk("Setting ttyAM%d to RS485 and config gpio3 bit25 to input.\n", port->line);
        /* config gpio3 bit 25 to input */
        a1_rs485_config_rts_gpio();
        /* set the gpio3_b25 low to enable the rx path of 485 controller  */
        a1_rs485_rts(0);
    }
    /* Enable tx interrupts */
    cr |= UART_IER_ETEI;
    UART_PUT_IER(port, cr);
    spin_unlock(&port->lock);
}

static int umvp_uart_is_rs458_enable(struct uart_port *port)
{
    struct umvp_uart_port_S *p =  to_umvp_uart_port(port);
    int rc = -1;

    if (port->line != RS485_PORT_ID) return -1;
    if (p->rs485.flags & SER_RS485_ENABLED)
        rc = 0;
    else
        rc = -1;
  //printk(KERN_INFO "[%s] rc(%d) flag(0x%x)\n", __func__, rc, p->rs485.flags);
    return rc;
}

static void umvp_uart_rs485_delay_rts_send(struct uart_port *port)
{
    unsigned int status;

    if (umvp_uart_is_rs458_enable(port) < 0) return;

    while (1) {
        status = UART_GET_LSR(port);
        if (status & UART_LSR_TEMT) break;  /* wait until Transmitter Empty */
    }
}

/* --------------------------------------------------------------------------------------- */

static void umvp_uart_stop_tx(struct uart_port *port)
{
    unsigned int cr;

    cr = UART_GET_IER(port);
    cr &= ~UART_IER_ETEI;
    UART_PUT_IER(port, cr);

    if (umvp_uart_is_rs458_enable(port) < 0) return;

    /* rs485 is enable:
     * set the gpio3_b25 low to enable the rx path of 485 controller
     */
    a1_rs485_rts(0);
    umvp_uart_start_rx(port);
   // printk(KERN_INFO "[RS485:%s]\n", __func__);
}

static void umvp_uart_start_tx(struct uart_port *port)
{
    unsigned int cr;

    cr = UART_GET_IER(port);
    cr |= UART_IER_ETEI;
    UART_PUT_IER(port, cr);

    if (port->line != RS485_PORT_ID) return;
    if (umvp_uart_is_rs458_enable(port) < 0) return;
    /* rs485 is enable:
     * set the gpio3_b25 Height to enable the tx path of 485 controller
     */
    a1_rs485_rts(1);
    umvp_uart_stop_rx(port);
   // printk(KERN_INFO "[RS485:%s]\n", __func__);
}

static void umvp_uart_stop_rx(struct uart_port *port)
{
    unsigned int cr;

    cr = UART_GET_IER(port);
    cr &= ~UART_IER_ERDI;
    UART_PUT_IER(port, cr);
}

static void umvp_uart_start_rx(struct uart_port *port)
{
    unsigned int cr;

    cr = UART_GET_IER(port);
    cr |= UART_IER_ERDI;
    UART_PUT_IER(port, cr);
}


static void umvp_uart_enable_ms(struct uart_port *port)
{
}

static void umvp_uart_rx_chars(struct uart_port *port)
{
    struct tty_struct *tty = port->info->port.tty;
    unsigned int status, ch, flag, lsr, max_count = 256;

    status = UART_GET_LSR(port);
    while ((status & UART_LSR_DR) && max_count--) {

        ch = UART_GET_CHAR(port);
        flag = TTY_NORMAL;
        port->icount.rx++;

        /* ACTi handling Magic Key.
         * Since A1's LSR has bug, we use '!' char as alternative.
         */
        if (port->line == 0)    /* If it is not console port, don't check Sysrq key */
        if (unlikely (ch == '!')) {
            if (!port->sysrq) {
                port->sysrq = jiffies + HZ*5;
                goto ignore_char;
            } else {
                port->sysrq = 0;
            }
        }
        /*
         * Note that the error handling code is
         * out of the main execution path
         */
        lsr = UART_GET_LSR(port);
        if (unlikely(lsr & UART_LSR_ANY)) {
            if (lsr & UART_LSR_BE) {
                lsr &= ~(UART_LSR_FE | UART_LSR_PE);
                port->icount.brk++;
                if (uart_handle_break(port))
                    goto ignore_char;
            } else if (lsr & UART_LSR_PE)
                port->icount.parity++;
            else if (lsr & UART_LSR_FE)
                port->icount.frame++;
            if (lsr & UART_LSR_OE)
                port->icount.overrun++;

            lsr &= port->read_status_mask;

            if (lsr & UART_LSR_BE)
                flag = TTY_BREAK;
            else if (lsr & UART_LSR_PE)
                flag = TTY_PARITY;
            else if (lsr & UART_LSR_FE)
                flag = TTY_FRAME;
        }

        if (uart_handle_sysrq_char(port, ch & 255))
            goto ignore_char;

        uart_insert_char(port, lsr, UART_LSR_OE, ch, flag);

    ignore_char:
        status = UART_GET_LSR(port);
    }
    tty_flip_buffer_push(tty);
    return;
}

static void umvp_uart_tx_chars(struct uart_port *port)
{
    struct circ_buf *xmit = &port->info->xmit;
    int count;

    if (port->x_char) {
        UART_PUT_CHAR(port, port->x_char);
        port->icount.tx++;
        port->x_char = 0;
        return;
    }
    if (uart_circ_empty(xmit) || uart_tx_stopped(port)) {
        umvp_uart_stop_tx(port);
        return;
    }

   count = port->fifosize >> 1;
    do {
        UART_PUT_CHAR(port, xmit->buf[xmit->tail]);
        xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
        port->icount.tx++;
        if (uart_circ_empty(xmit))
            break;
    } while (--count > 0);

    umvp_uart_rs485_delay_rts_send(port);

    if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
        uart_write_wakeup(port);

    if (uart_circ_empty(xmit))
        umvp_uart_stop_tx(port);
}

static irqreturn_t umvp_uart_int(int irq, void *dev_id)
{
    struct uart_port *port = dev_id;
    unsigned int     status, iir, pass_counter = UMVP_SERIAL_ISR_PASS_LIMIT;

    spin_lock(&port->lock);
    status = UART_GET_LSR(port);
    do {
        if (status & UART_LSR_DR)
            umvp_uart_rx_chars(port);

        if (status & UART_LSR_TEMT)
        {
            umvp_uart_tx_chars(port);
            iir = UART_GET_IIR(port);
            if((iir&0x0f) == 0xc)
            {
                //Ted found if the UART cable is not connected, the "character timeout" interrupt will occur.
                //And the interrupt can be released by reading one character from RX.
                UART_GET_CHAR(port);
            }
        }

        if (pass_counter-- == 0)
            break;
        status = UART_GET_LSR(port);
    } while (status & (UART_LSR_TEMT|UART_LSR_DR));

    spin_unlock(&port->lock);
    return IRQ_HANDLED;
}

static unsigned int umvp_uart_tx_empty(struct uart_port *port)
{
    return UART_GET_LSR(port) & UART_LSR_TEMT ? TIOCSER_TEMT : 0;
}

static unsigned int umvp_uart_get_mctrl(struct uart_port *port)
{
    return 0;
}

static void umvp_uart_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
}

static void umvp_uart_break_ctl(struct uart_port *port, int break_state)
{
    unsigned long flags;
    unsigned int lcr;

    spin_lock_irqsave(&port->lock, flags);
    lcr = UART_GET_LCR(port);
    if (break_state == -1)
        lcr |= UART_LCR_BRK;
    else
        lcr &= ~UART_LCR_BRK;
    UART_PUT_LCR(port, lcr);
    spin_unlock_irqrestore(&port->lock, flags);
}

static int umvp_uart_startup(struct uart_port *port)
{
    struct serial_rs485 rs485conf;
    int retval;

    /*
     * Allocate the IRQ
     */
    SYS_UART_INTR_SENSE(port->irq);

    retval = request_irq(port->irq, umvp_uart_int, IRQF_SHARED, "umvp_serial", port);
    if (retval) {
        return retval;
    }

    /* config ttyAM1 to rs485 */
    if (RS485_PORT_ID == port->line) {
        rs485conf.flags  = SER_RS485_ENABLED;
        umvp_uart_config_rs485(port, &rs485conf);
    }

    /*
     * Finally, enable interrupts
     */
    UART_PUT_IER(port, UART_IER_ERDI);

    return 0;
}

static void umvp_uart_shutdown(struct uart_port *port)
{
    /*
     * Free the interrupt
     */
    free_irq(port->irq, port);

    /*
     * disable all interrupts, disable the port
     */
    UART_PUT_IER(port, 0);

    /* disable break condition and fifos */
    UART_PUT_LCR(port, UART_GET_LCR(port)&(~UART_LCR_BRK));
    UART_PUT_FCR(port, UART_GET_FCR(port)&(~UART_FCR_FIFOE));
}

extern unsigned int PLL_GetAPBClock(unsigned int Fin);

static void umvp_uart_set_termios(struct uart_port *port, struct ktermios *termios,
                                  struct ktermios *old)
{
    unsigned int lcr, fcr = 0;
    unsigned long flags;
    unsigned int baud, quot;

    //ted added for run-time to calculate APB clock
    port->uartclk = PLL_GetAPBClock(A1_SYS_CLOCK_SRC);
    /*
     * Ask the core to calculate the divisor for us.
     */
    baud = uart_get_baud_rate(port, termios, old, 0, port->uartclk/16);

    /*R:20140407: RS485 in line1 need to change baud rate on the fly */
    #if 0
//  quot = port->uartclk / (16 * baud);
    quot = port->uartclk / (16 * CONFIG_SERIAL_UMVP_CONSOLE_BAUD);
    #else
        if (baud == 14400) {
           if (tty_termios_baud_rate(termios) == 38400)
                baud = 38400;
        }
        if(port->line > 0)
           quot = port->uartclk / (16 * baud);
        else
           quot = port->uartclk / (16 * CONFIG_SERIAL_UMVP_CONSOLE_BAUD);

        printk(KERN_INFO "[umvp_serial] line(%d)quot(%d) baud(%d)\n", port->line, quot, baud);
    #endif


    switch (termios->c_cflag & CSIZE) {
    case CS5:
        lcr = UART_LCR_WLEN_5;
        break;
    case CS6:
        lcr = UART_LCR_WLEN_6;
        break;
    case CS7:
        lcr = UART_LCR_WLEN_7;
        break;
    default: // CS8
        lcr = UART_LCR_WLEN_8;
        break;
    }
    if (termios->c_cflag & CSTOPB)
        lcr |= UART_LCR_STOP;          /* CSTOPB: Set two stop bits, rather than one.*/
    if (termios->c_cflag & PARENB) {
        lcr |= UART_LCR_PEN;
        if (!(termios->c_cflag & PARODD))
            lcr |= UART_LCR_EPS;
    }
    if (port->fifosize > 1)
        fcr |= (UART_FCR_XMITR|UART_FCR_RCVRR|UART_FCR_FIFOE);


    spin_lock_irqsave(&port->lock, flags);

    /*
     * Update the per-port timeout.
     */
    uart_update_timeout(port, termios->c_cflag, baud);

    port->read_status_mask = UART_LSR_OE;
    if (termios->c_iflag & INPCK)
        port->read_status_mask |= UART_LSR_FE | UART_LSR_PE;
    if (termios->c_iflag & (BRKINT | PARMRK))
        port->read_status_mask |= UART_LSR_BE;

    /*
     * Characters to ignore
     */
    port->ignore_status_mask = 0;
    if (termios->c_iflag & IGNPAR)
        port->ignore_status_mask |= UART_LSR_FE | UART_LSR_PE;

    if (termios->c_iflag & IGNBRK) {
        port->ignore_status_mask |= UART_LSR_BE;
        /*
         * If we're ignoring parity and break indicators,
         * ignore overruns too (for real raw support).
         */
        if (termios->c_iflag & IGNPAR)
            port->ignore_status_mask |= UART_LSR_OE;
    }

    /*
     * Ignore all characters if CREAD is not set.
     */
    if ((termios->c_cflag & CREAD) == 0)
        port->ignore_status_mask |= UART_DUMMY_RSR_RX;


    /* Set baud rate */
    UART_PUT_LCR(port, UART_LCR_DLAB);  /* enable Divisor Latach Address Bit */
    UART_PUT_DLH(port, ((quot >> 8) & 0xFF));
    UART_PUT_DLL(port, (quot & 0xff));

    UART_PUT_FCR(port, fcr);
    UART_PUT_LCR(port, lcr);

    spin_unlock_irqrestore(&port->lock, flags);
}

static const char *umvp_uart_type(struct uart_port *port)
{
    return port->type == PORT_UMVP ? UART_TYPE_STRING : NULL;
}

/*
 * Release the memory region(s) being used by 'port'
 */
static void umvp_uart_release_port(struct uart_port *port)
{
    release_mem_region(port->mapbase, UART_PORT_SIZE);
}

/*
 * Request the memory region(s) being used by 'port'
 */
static int umvp_uart_request_port(struct uart_port *port)
{
    return request_mem_region(port->mapbase, UART_PORT_SIZE, "serial_umvp")
            != NULL ? 0 : -EBUSY;
}

/*
 * Configure/autoconfigure the port.
 */
static void umvp_uart_config_port(struct uart_port *port, int flags)
{
    if (flags & UART_CONFIG_TYPE) {
        port->type = PORT_UMVP;
        umvp_uart_request_port(port);
    }
}

/*
 * verify the new serial_struct (for TIOCSSERIAL).
 */
static int umvp_uart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
    int ret = 0;
    if (ser->type != PORT_UNKNOWN && ser->type != PORT_UMVP)
        ret = -EINVAL;
    if (ser->irq < 0 || ser->irq >= NR_IRQS)
        ret = -EINVAL;
    if (ser->baud_base < 9600)
        ret = -EINVAL;
    return ret;
}

static int umvp_uart_ioctl(struct uart_port *port, unsigned int cmd, unsigned long arg)
{
    struct serial_rs485 rs485conf;

    switch (cmd) {
    case TIOCSRS485:
        if (copy_from_user(&rs485conf, (struct serial_rs485 *) arg, sizeof(rs485conf)))
            return -EFAULT;

        if (port->line == RS485_PORT_ID){ //R:20131227
        	umvp_uart_config_rs485(port, &rs485conf);
        }
        break;

    case TIOCGRS485:
      //if (port->line >= SYS_UART_NR) return -EFAULT;
        if (port->line != RS485_PORT_ID)return -EFAULT; //R:20131227

        if (copy_to_user((struct serial_rs485 *) arg,
                        &(to_umvp_uart_port(port)->rs485),
                          sizeof(rs485conf)))
             return -EFAULT;
        break;

    default:
        return -ENOIOCTLCMD;
    }
    return 0;
}

static struct uart_ops umvp_uart_ops = {
    .tx_empty       = umvp_uart_tx_empty,
    .set_mctrl      = umvp_uart_set_mctrl,
    .get_mctrl      = umvp_uart_get_mctrl,
    .stop_tx        = umvp_uart_stop_tx,
    .start_tx       = umvp_uart_start_tx,
    .stop_rx        = umvp_uart_stop_rx,
    .enable_ms      = umvp_uart_enable_ms,
    .break_ctl      = umvp_uart_break_ctl,
    .startup        = umvp_uart_startup,
    .shutdown       = umvp_uart_shutdown,
    .set_termios    = umvp_uart_set_termios,
    .type           = umvp_uart_type,
    .release_port   = umvp_uart_release_port,
    .request_port   = umvp_uart_request_port,
    .config_port    = umvp_uart_config_port,
    .verify_port    = umvp_uart_verify_port,
    .ioctl          = umvp_uart_ioctl,
};

static struct umvp_uart_port_S umvp_uart_ports[SYS_UART_NR] = {
    {
        .port   = {
            .membase    = (void *) SYS_UART0_BASE_VA,
            .mapbase    = SYS_UART0_BASE_PA,
            .iotype     = SERIAL_IO_MEM,
            .irq        = SYS_UART0_IRQ_NR,
            .uartclk    = SYS_APB_CLOCK_RATE,
            .fifosize   = 16,
            .ops        = &umvp_uart_ops,
            .flags      = ASYNC_BOOT_AUTOCONF,
            .line       = 0,
        },
    },
#if SYS_UART_NR > 1
    #if USED_UART1
    {
        .port   = {
            .membase    = (void *) SYS_UART1_BASE_VA,
            .mapbase    = SYS_UART1_BASE_PA,
            .iotype     = SERIAL_IO_MEM,
            .irq        = SYS_UART1_IRQ_NR,
            .uartclk    = SYS_APB_CLOCK_RATE,
            .fifosize   = 16,
            .ops        = &umvp_uart_ops,
            .flags      = ASYNC_BOOT_AUTOCONF,
            .line       = 1,
        },
    }
    #else
    {
        .port   = {
            .membase    = (void *) SYS_UART4_BASE_VA,
            .mapbase    = SYS_UART4_BASE_PA,
            .iotype     = SERIAL_IO_MEM,
            .irq        = SYS_UART4_IRQ_NR,
            .uartclk    = SYS_APB_CLOCK_RATE,
            .fifosize   = 16,
            .ops        = &umvp_uart_ops,
            .flags      = ASYNC_BOOT_AUTOCONF,
            .line       = 1,
        },
    }
    #endif

#endif
};

#ifdef SYS_UART_CONSOLE

static void umvp_uart_console_write(struct console *co, const char *s, unsigned int count)
{
    struct uart_port *port = &umvp_uart_ports[co->index].port;
    unsigned int status, old_ier;
    int i;

    /*
     *  First save the IER then disable the interrupts
     */
    old_ier = UART_GET_IER(port);
    UART_PUT_IER(port, 0);

    /*
     *  Now, do each character
     */
    for (i = 0; i < count; i++) {
        do {
            status = UART_GET_LSR(port);
        } while (!(status & UART_LSR_THRE));
        UART_PUT_CHAR(port, s[i]);
        if (s[i] == '\n') {
            do {
                status = UART_GET_LSR(port);
            } while (!(status & UART_LSR_THRE));
            UART_PUT_CHAR(port, '\r');
        }
    }

    /*
     *  Finally, wait for transmitter to become empty
     *  and restore the IER
     */
    do {
        status = UART_GET_LSR(port);
    } while (!(status & UART_LSR_TEMT));

    UART_PUT_IER(port, old_ier);
}

static void __init
umvp_uart_console_get_options(struct uart_port *port, int *baud, int *parity, int *bits)
{
    if (UART_GET_IER(port) & UART_IER_ERDI) {
        unsigned int lcr, quot;
        lcr = UART_GET_LCR(port);

        *parity = 'n';
        if (lcr & UART_LCR_PEN) {
            if (lcr & UART_LCR_EPS)
                *parity = 'e';
            else
                *parity = 'o';
        }

        switch (lcr & UART_LCR_WLEN_MASK) {
        case UART_LCR_WLEN_8:
        default:
            *bits = 8;
            break;
        case UART_LCR_WLEN_7:
            *bits = 7;
            break;
        case UART_LCR_WLEN_6:
            *bits = 6;
            break;
        case UART_LCR_WLEN_5:
            *bits = 5;
            break;
        }

        UART_PUT_LCR(port, UART_LCR_DLAB);  /* enable Divisor Latach Address Bit */
        quot = UART_GET_DLL(port) | (UART_GET_DLH(port) << 8);
        UART_PUT_LCR(port, lcr);
    }
}

static int __init umvp_uart_console_setup(struct console *co, char *options)
{
    struct uart_port *port;
    int baud = SYS_UART_BAUD_RATE;
    int bits = 8;
    int parity = 'n';
    int flow = 'n';

    /*
     * Check whether an invalid uart number has been specified, and
     * if so, search for the first available port that does have
     * console support.
     */
    if (co->index >= SYS_UART_NR)
        co->index = 0;

    port = &umvp_uart_ports[co->index].port;

    if (options)
        uart_parse_options(options, &baud, &parity, &bits, &flow);
    else
        umvp_uart_console_get_options(port, &baud, &parity, &bits);

    return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct uart_driver umvp_uart_driver;

static struct console umvp_console = {
    .name       = UMVP_SERIAL_CONSOLE_NAME,
    .write      = umvp_uart_console_write,
    .device     = uart_console_device,
    .setup      = umvp_uart_console_setup,
    .flags      = CON_PRINTBUFFER,
    .index      = -1,
    .data       = &umvp_uart_driver,
};

int __init umvp_uart_console_init(void)
{
    register_console(&umvp_console);
    return 0;
}

console_initcall(umvp_uart_console_init);

#define UMVP_CONSOLE    &umvp_console
#else
#define UMVP_CONSOLE    NULL
#endif


static struct uart_driver umvp_uart_driver = {
    .owner          = THIS_MODULE,
    .major          = UMVP_SERIAL_MAJOR,
    .minor          = UMVP_SERIAL_MINOR,
    .driver_name    = UMVP_SERIAL_TTY_NAME,
    .dev_name       = UMVP_SERIAL_TTY_NAME,
    .nr             = SYS_UART_NR,
    .cons           = UMVP_CONSOLE,
};

static int __init umvp_uart_init(void)
{
    int ret;

    ret = uart_register_driver(&umvp_uart_driver);
    if (ret == 0) {
        int i;

        for (i = 0; i < SYS_UART_NR; i++)
            uart_add_one_port(&umvp_uart_driver, &umvp_uart_ports[i].port);
    }
    return ret;
}

static void __exit umvp_uart_exit(void)
{
    int i;

    for (i = 0; i < SYS_UART_NR; i++)
        uart_remove_one_port(&umvp_uart_driver, &umvp_uart_ports[i].port);

    uart_unregister_driver(&umvp_uart_driver);
}

module_init(umvp_uart_init);
module_exit(umvp_uart_exit);

MODULE_AUTHOR("Global UniChip Corp.");
MODULE_DESCRIPTION("UMVP 16550 serial driver");
MODULE_LICENSE("GPL");

