//  ***************************************************************************
// 
//   Copyright (c) 2009  Evatronix SA 
//
//  ***************************************************************************
//
//   Please review the terms of the license agreement before using     
//   this file. If you are not an authorized user, please destroy this 
//   source code file and notify Evatronix SA immediately that you     
//   inadvertently received an unauthorized copy.                      
//
//  ***************************************************************************
/// @file           nf_memory.h
/// @brief          NAND Flash memory driver function declarations.
/// @version        $Revision: 1.3 $
/// @author         Piotr Sroka
/// @date           $Date: 2010-03-25 13:46:21 $
//  ***************************************************************************

#ifndef NF_MEMORY_H
#define NF_MEMORY_H


#include "nf_types.h"

/******************************************************************************/
/** @name Memory work mode. */
/******************************************************************************/
/** @{ */
/** synchronous work mode */
#define MEMORY_SYNCHRONOUS_MODE         0x01
/** asynchronous work mode */
#define MEMORY_ASYNCHRONOUS_MODE        0x02
/** unknown memory mode */
#define MEMORY_UNKNOWN_MODE             0x03
/** @} */
/******************************************************************************/


/******************************************************************************/
/** @name Memory feature definitions. */
/******************************************************************************/
/** @{ */
#define FEATURE_TIMING_MODE_0                       0x0
#define FEATURE_TIMING_MODE_1                       0x1
#define FEATURE_TIMING_MODE_2                       0x2
#define FEATURE_TIMING_MODE_3                       0x3
#define FEATURE_TIMING_MODE_4                       0x4
#define FEATURE_DATA_INTERFACE_SYNCHRONOUS      (1 << 4)
#define FEATURE_DATA_INTERFACE_ASYNCHRONOUS     (0 << 4)

#define FEATURE_ADDRESS_TIMING  0x1
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name Memory statuses. */
/******************************************************************************/
/** @{ */
/** This bit is set if the most recently finished on the selected LUN operation failed */
#define NF_MEMORY_STATUS_FAIL       (0x1uL << 0)
/** This bit is set if the previous operation on the selected LUN failed */
#define NF_MEMORY_STATUS_FAILC      (0x1uL << 1)
/** This bit is set if array of selected LUN is ready */
#define NF_MEMORY_STATUS_ARDY       (0x1uL << 5)
/** This bit is set if I/O of selected LUN is ready */
#define NF_MEMORY_STATUS_RDY        (0x1uL << 6)
/** This bit is clear if memory is write protected */
#define NF_MEMORY_STATUS_WP         (0x1uL << 7)
/** @} */
/******************************************************************************/

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_read_status(nf_memory_t *memory, uint8_t *status)
 * @brief       Function read status of memory.
 * @param       memory nand flash memory object
 * @param       status status of memory
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_read_status(nf_memory_t *memory, uint8_t *status);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_reset(nf_memory_t *memory)
 * @brief       Function executes reset command on memory
 * @param       memory nand flash memory object
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_reset(nf_memory_t *memory);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_synch_reset(nf_memory_t *memory)
 * @brief       Function executes synchronous reset command on memory.
 * @param       memory nand flash memory object
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_synch_reset(nf_memory_t *memory);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_block_erase(nf_memory_t *memory, 
 *                                          uint32_t block_number)
 * @brief       Function erases block of memory
 * @param       memory nand flash memory object
 * @param       block_number number of block to erase
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_block_erase(nf_memory_t *memory, uint32_t block_number);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_program(nf_memory_t *memory, 
 *                                          uint32_t page_number, 
 *                                          uint16_t page_offset, 
 *                                          uint8_t *buffer, 
 *                                          uint16_t buffer_size)
 * @brief       Function programes one memory page
 * @param       memory nand flash memory object
 * @param       page_number number of page to program
 * @param       page_offset page offset to start program
 * @param       buffer buffer with data to program
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_page_program(nf_memory_t *memory, uint32_t page_number,
                            uint16_t page_offset, uint8_t *buffer, 
                            uint16_t buffer_size);

uint8_t nf_mem_page_empty(nf_memory_t *memory, uint32_t page_number);


/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read(nf_memory_t *memory, 
 *                                          uint32_t page_number, 
 *                                          uint16_t page_offset, 
 *                                          uint8_t *buffer, 
 *                                          uint16_t buffer_size)
 * @brief       Function reads one memory page
 * @param       memory nand flash memory object
 * @param       page_number number of page to read
 * @param       page_offset page offset to start read
 * @param       buffer buffer for read data
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_page_read(nf_memory_t *memory, uint32_t page_number, 
                         uint16_t page_offset, uint8_t *buffer, 
                         uint16_t buffer_size);

uint8_t nf_mem_page_spare(nf_memory_t *memory, uint32_t page_number,                                                                                                                                         
                         uint16_t page_offset, uint8_t *buffer,
                         uint16_t buffer_size);


/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_read_id(nf_memory_t *memory, uint8_t id[5])
 * @brief       Function reads memory ID
 * @param       memory nand flash memory object
 * @param       id buffer for memory ID
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_read_id(nf_memory_t *memory, uint8_t id[5]);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_read_parameter_page(nf_memory_t *memory, 
 *                                                  uint8_t *buffer)
 * @brief       Function reads memory parameter page
 * @param       memory nand flash memory object
 * @param       buffer buffer for memory parameter page. Buffer size should 
 *                  be at least 512 bytes.                          
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_read_parameter_page(nf_memory_t *memory, uint8_t *buffer);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read_multiplane(nf_memory_t *memory, 
 uint32_t page_number)
 * @brief       Function executes read multiplane command on memory
 * @param       memory nand flash memory object
 * @param       page_number number of page to read multiplane command                          
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_page_read_multiplane(nf_memory_t *memory, 
                                    uint32_t page_number);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_select_cache_register(nf_memory_t *memory, 
 *                                                   uint32_t page_number, 
 *                                                   uint16_t page_offset, 
 *                                                   uint8_t *buffer, 
 *                                                   uint16_t buffer_size)
 * @brief       Function executes select cache register command. It is use to
 *                  enable data output on the adresed LUN and cache register
 *                  at the specified collumn address.
 * @param       memory nand flash memory object
 * @param       page_number number of page to read                          
 * @param       page_offset page offset to read                         
 * @param       buffer buffer for read data                        
 * @param       buffer_size size of buffer in bytes                        
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_select_cache_register(nf_memory_t *memory, 
                                     uint32_t page_number, uint16_t page_offset, 
                                     uint8_t *buffer, uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_set_work_mode(nf_memory_t *memory, 
 *                                              uint8_t work_mode, 
 *                                              uint8_t timing_mode)
 * @brief       Function sets memory work mode.
 * @param       memory nand flash memory object
 * @param       work_mode work mode to set. It can be MEMORY_SYNCHRONOUS_MODE 
 *                  or MEMORY_ASYNCHRONOUS_MODE
 * @param       timing_mode it defines NAND flash memory interface frequency
 *                  for more information look to your memory documentation.
 *                  Default value is 0.                 
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_set_work_mode(nf_memory_t *memory, uint8_t work_mode, 
                             uint8_t timing_mode);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_get_features(nf_memory_t *memory, 
 *                                          uint8_t feature_address, 
 *                                          uint32_t *buffer)
 * @brief       Function reads subfeature parameters (P1-P4) from the
 *                  specified feature address. 
 * @param       memory nand flash memory object
 * @param       feature_address address of feature to read
 * @param       buffer buffer for read subfeature
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_get_features(nf_memory_t *memory, 
                            uint8_t feature_address, 
                            uint32_t *buffer);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_set_features(nf_memory_t *memory, 
 *                                          uint8_t feature_address, 
 *                                          uint32_t *buffer)
 * @brief       Function writes subfeature parameters (P1-P4) to the
 *                  specified feature address. To enable or disable 
 *                  target-specific features.
 * @param       memory nand flash memory object
 * @param       feature_address address of feature to write
 * @param       buffer buffer with feature to write
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_set_features(nf_memory_t *memory, 
                            uint8_t feature_address, 
                            uint32_t *buffer);


/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_program_cache(nf_memory_t *memory, 
 *                                                uint32_t page_number,
 *                                                uint16_t page_offset, 
 *                                                uint8_t *buffer, 
 *                                                uint16_t buffer_size)
 * @brief       Function programes one of memory page in cache mode
 * @param       memory nand flash memory object
 * @param       page_number number of page to program
 * @param       page_offset page offset to start program
 * @param       buffer buffer with data to program
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_program_cache(nf_memory_t *memory, uint32_t page_number, 
                                  uint16_t page_offset, uint8_t *buffer, 
                                  uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_program1(nf_memory_t *memory, 
 *                                           uint32_t page_number, 
 *                                           uint16_t page_offset, 
 *                                           uint8_t *buffer, 
 *                                           uint16_t buffer_size)
 * @brief       Function writes data to Flash memory page. Data are writing 
 *                  to flash memory data register. To program memory with data 
 *                  from this regsiter the nf_mem_write_page function 
 *                  must be called. ECC code is not program with data, 
 *                  regardless of whether ECC is enabled or not.
 * @param       memory nand flash memory object
 * @param       page_number number of page to program
 * @param       page_offset page offset to start program
 * @param       buffer buffer with data to program
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_program1(nf_memory_t *memory, uint32_t page_number, 
                             uint16_t page_offset, uint8_t *buffer, 
                             uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_random_program(nf_memory_t *memory, 
 *                                                  uint16_t page_offset, 
 *                                                  uint8_t *buffer, 
 *                                                  uint16_t buffer_size)
 * @brief       Function writes random data to Flash memory page. Function can be 
 *                  call only after the nf_mem_page_program1 function. 
 *                  Data are writing to flash memory using data register. 
 *                  To program memory with data from this register 
 *                  the nf_mem_write_page function must be called.
 * @param       memory nand flash memory object
 * @param       page_offset page offset to start program
 * @param       buffer buffer with data to program
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_random_program(nf_memory_t *memory, 
                                   uint16_t page_offset, uint8_t *buffer, 
                                   uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_write_page(nf_memory_t *memory)
 * @brief       Function starts program of flash memory by sending
 *                  write command to this memory. Function should be used 
 *                  after one of the next functions: 
 *                  nf_mem_page_program1, nf_mem_page_random_program.
 * @param       memory nand flash memory object
 * @param       page_offset page offset to start program
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_write_page(nf_memory_t *memory);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read_cache_first(nf_memory_t *memory, 
 *                                                   uint32_t page_number, 
 *                                                   uint16_t page_offset)
 * @brief       Function starts read cache mode.
 * @param       memory nand flash memory object
 * @param       page_number number of page to read
 * @param       page_offset page offset to start read
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_read_cache_first(nf_memory_t *memory, uint32_t page_number, 
                                     uint16_t page_offset);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read_cache(nf_memory_t *memory, 
 *                                              uint8_t *buffer, 
 *                                              uint16_t buffer_size)
 * @brief       Function reads one memory page in cache mode.
 * @param       memory nand flash memory object
 * @param       buffer buffer for read data                        
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_read_cache(nf_memory_t *memory, uint8_t *buffer, 
                               uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read_cache(nf_memory_t *memory, 
 *                                              uint8_t *buffer, 
 *                                              uint16_t buffer_size)
 * @brief       Function reads  memory page in cache mode 
 *                  and finishes cache mode
 * @param       memory nand flash memory object
 * @param       buffer buffer for read data                        
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_read_cache_last(nf_memory_t *memory, uint8_t *buffer, 
                                    uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_random_read(nf_memory_t *memory, 
 *                                              uint16_t page_offset, 
 *                                              uint8_t *buffer, 
 *                                              uint16_t buffer_size)
 * @brief       Function reads random data from Flash memory page. Function 
 *                  can be called only after the nf_mem_page_read function.
 * @param       memory nand flash memory object
 * @param       page_offset page offset to start read
 * @param       buffer buffer for read data                        
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_page_random_read(nf_memory_t *memory, uint16_t page_offset, 
                                uint8_t *buffer, uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_internal_datamove(nf_memory_t *memory, 
 *                                               uint32_t src_page_number, 
 *                                               uint32_t dest_page_number, 
 *                                               uint8_t *buffer, 
 *                                               uint16_t buffer_size)
 * @brief       This function transfers data from the src_page_number 
 *                  to dest_page_number. And aditionally read data are placed
 *                  to buffer.
 * @param       memory nand flash memory object
 * @param       src_page_number source page number to copy 
 * @param       dest_page_number destination page number to copy
 * @param       buffer buffer for read data                        
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_internal_datamove(nf_memory_t *memory, uint32_t src_page_number, 
                                 uint32_t dest_page_number, 
                                 uint8_t *buffer, uint16_t buffer_size);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_select_target(nf_memory_t *memory)
 * @brief       Function sets chip enable signal of memory 
 *                  and apply new controller settings 
 *                  like page size, block size and spare area size.
 * @param       memory nand flash memory object
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number
 */
/******************************************************************************/
uint8_t nf_mem_select_target(nf_memory_t *memory);

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_page_read1(nf_memory_t *memory, 
 *                                        uint32_t page_number, 
 *                                        uint16_t page_offset, uint8_t *buffer, 
 *                                        uint16_t buffer_size)
 * @brief       Function reads one memory page without checking ECC, 
 *                  regardless of whether ECC is enabled or not
 * @param       memory nand flash memory object
 * @param       page_number number of page to read
 * @param       page_offset page offset to start read
 * @param       buffer buffer for read data
 * @param       buffer_size size of buffer in bytes
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_page_read1(nf_memory_t *memory, uint32_t page_number, 
                          uint16_t page_offset, uint8_t *buffer, 
                          uint16_t buffer_size);


/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_mem_is_factory_bad_block(nf_memory_t *memory, 
 *                                                  uint16_t block_number, 
 *                                                  uint8_t *is_bad)
 * @brief       Function checks if block is checked as bad by manufacturer.
 * @param       memory nand flash memory object
 * @param       block_number block number to check
 * @param       is_bad by this parameter function returns block status.
 *                  1 - means bad
 *                  0 - means good
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
 */
/******************************************************************************/
uint8_t nf_mem_is_factory_bad_block(nf_memory_t *memory, uint16_t block_number, 
                                    uint8_t *is_bad);

#endif
