//  ***************************************************************************
// 
//   Copyright (c) 2009  Evatronix SA 
//
//  ***************************************************************************
//
//   Please review the terms of the license agreement before using     
//   this file. If you are not an authorized user, please destroy this 
//   source code file and notify Evatronix SA immediately that you     
//   inadvertently received an unauthorized copy.                      
//
//  ***************************************************************************
/// @file           nf_dma.h
/// @brief          NAND Flash controller DMA module driver.
/// @version        $Revision: 1.1 $
/// @author         Piotr Sroka
/// @date           $Date: 2009-12-23 08:01:04 $
//  ***************************************************************************

#ifndef NF_DMA_H
#define NF_DMA_H

#include "nf_types.h"

/******************************************************************************/
/** @name DMA descriptor type definitions. */
/******************************************************************************/
/** @{ */
/** Execute current descriptor and go to next line*/
#define NF_DMA_DESCRIPTOR_TYPE_SIMPLE       0x0
/** Execute current descriptor and end transfer */
#define NF_DMA_DESCRIPTOR_TYPE_END          0x1
/** Execute current descriptor and link another descriptor */
#define NF_DMA_DESCRIPTOR_TYPE_LINK         0x2
/** @} */
/******************************************************************************/

/******************************************************************************/
/*!  							   
 * @fn          uint8_t nf_dma_prepare_transfer(nf_controller_t *pcontroller, 
 *                                              uint32_t buffer_address, 
 *                                              uint32_t size, uint8_t direction)
 * @brief 	Function prepares DMA to data transfer
 * @param	pcontroller nand flash controller object
 * @param	buffer_address address of buffer either with data to transfer 
 *                  or with descriptors if scatter gather mode enabled
 * @param	size number of bytes to transfer                          
 * @param	direction data transfer direction. 
 *                  They are defined here @ref directions
 * @return	Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_dma_prepare_transfer(nf_controller_t *pcontroller, 
                                uint32_t buffer_address, 
                                uint32_t size, uint8_t direction);

/******************************************************************************/
/*!  							   
 * @fn          uint8_t nf_dma_finish_transfer(nf_controller_t *pcontroller)
 * @brief 	Function finishes DMA transfer
 * @param	pcontroller nand flash controller object
 * @return	Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_dma_finish_transfer(nf_controller_t *pcontroller);

/******************************************************************************/
/*!  							   
 * @fn          uint8_t 
 *              nf_dma_create_descriptors(uint32_t block_address[], 
 *                                        uint16_t block_length[],
 *                                        uint8_t descr_count, 
 *                                        uint32_t next_descriptor_address,
 *                                        uint32_t *descriptors)
 * @brief 	Function creates table of DMA descriptors
 * @param	block_address table of buffer addresses from/to 
 *                  data will be transfered by DMA module
 * @param	block_length table of buffers size
 * @param	descr_count number of descriptors to create
 * @param	next_descriptor_address address of next descriptors table. 
 *                  If no more descriptors it should be null.
 * @param	descriptors memory where created descriptors will be placed
 * @return	Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_dma_create_descriptors(uint32_t block_address[], 
                                  uint16_t block_length[],
                                  uint8_t descr_count, 
                                  uint32_t next_descriptor_address,
                                  uint32_t *descriptors);

#endif
