/*
 * i2c-algo-guc.c i2c driver algorithm for GUC I2C adapter
 *
 * Copyright (C) 2009 Peter Lee @ GUC co.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-guc.h>

#include <asm/io.h>

#include <asm/arch-umvp/a1_sys.h>


#include <mach/umvp_i2c.h>
#include <mach/i2c-algo-guc.h>

#define DB_PNK	printk
#define DEF_TIMEOUT	1000

#define I2C0_CLOCK	0x18
#define I2C1_CLOCK	0x18
#define I2C2_CLOCK	0x18

#define guc_set(guc_adap, ctrl, val) \
	guc_adap->gucset(guc_adap->data, ctrl, val)
#define guc_get(guc_adap, ctrl)	 \
	guc_adap->gucget(guc_adap->data, ctrl)
#if 0
#define guc_getclock(guc_adap)	 \
	guc_adap->guc_getclock(guc_adap->data)
#endif

#define i2c_outb(guc_adap, val)	 \
	guc_adap->gucset(guc_adap->data, 0, val)
#define i2c_inb(guc_adap)		 \
	guc_adap->gucget(guc_adap->data, 0)

inline static void i2c_start(struct i2c_algo_guc_data *guc_adap)
{
//	DB_PNK("S ");
	guc_set(guc_adap, 1, I2C_GUC_START);
}

inline static void i2c_stop(struct i2c_algo_guc_data *guc_adap)
{
//	DB_PNK(" P");
	guc_set(guc_adap, 1, I2C_GUC_STOP);
}

#if 1
static int wait_for_ack(struct i2c_algo_guc_data *guc_adap)
{
    int timeout = DEF_TIMEOUT * 2;
	int status;
	
	status = guc_get(guc_adap, 1);
#if 1
	while(timeout-- && (status&I2C_GUC_ACK)) {
		//udelay(100);
		udelay(50);
		status = guc_get(guc_adap, 1);
	}
	
	if (timeout <= 0) {
		//printk("Timeout waiting for I2C slave ACK\n");
		return (1);
	}
#endif
	return (0);
}
#endif

static int wait_for_int(struct i2c_algo_guc_data *guc_adap, int *status)
{
    int timeout = DEF_TIMEOUT * 2;
	*status = guc_get(guc_adap, 1);
#if 1
	while (timeout-- && !(*status & I2C_GUC_INT)) {
//		i2c->guc_waitforint(i2c->data);
		//udelay(100);
		udelay(50);
		*status = guc_get(guc_adap, 1);
	}
	if (timeout <= 0) {
		//printk("wait_for_int timeout\n");
		return (1);
	}

#endif
	return (0);
}

static int umvp_sendbytes(struct i2c_adapter *i2c_adap,
			 const char *buf, int count, int last)
{
	struct i2c_algo_guc_data *guc_adap = i2c_adap->algo_data;
	int wrcount, status, timeout;

	for (wrcount=0; wrcount<count; ++wrcount) {
//		DB_PNK(KERN_INFO "writing %2.2X\n", buf[wrcount]&0xff);
		i2c_outb(guc_adap, buf[wrcount]);
		/*Set IF=0*/
		status = guc_get(guc_adap, 1);
		status &= (~I2C_GUC_INT);
		guc_set(guc_adap, 1, status);	
		/*Polling IF=1: write success*/
		timeout = wait_for_int(guc_adap, &status);
		if (timeout) {
			printk(KERN_INFO "umvp_sendbytes timeout:IF\n");
			i2c_stop(guc_adap);
			return -EREMOTEIO;
		}
#if 1
		timeout = wait_for_ack(guc_adap);
		if (timeout) {
			printk(KERN_INFO "umvp_sendbytes timeout:ACK\n");
			i2c_stop(guc_adap);
			return -EREMOTEIO;
		}
#endif
	}
	if (last) {
//		printk("send the last byte\n");
		i2c_stop(guc_adap);
	} else {
		/*repeat start action*/
	//	i2c_start(guc_adap);
	}

	return (wrcount);
}

static int umvp_readbytes(struct i2c_adapter *i2c_adap,
			 char *buf, int count, int last)
{
	int i, status;
	struct i2c_algo_guc_data *guc_adap = i2c_adap->algo_data;
	
	for (i=0; i<count; i++) {
		status = guc_get(guc_adap, 1);
		status &= ~I2C_GUC_INT;
		if (i == count-1) {
			status |= I2C_GUC_ACK;
//			printk("write no_ack\n");
		} else {
			status &= ~I2C_GUC_ACK;
		}
		guc_set(guc_adap, 1, status);
		//printk("umvp_readbytes: count=%d, i=%d\n", count, i);

#if 1
		if (last && wait_for_int(guc_adap, &status)) {
			//printk("umvp_i2c_readbytes: timeout\n");
			i2c_stop(guc_adap);
			return -EREMOTEIO;
		}
#endif
		if (i == count-1) {
			if (last) {
			//	printk("read the last byte\n");
				i2c_stop(guc_adap);
			} else {/*start repeat action*/
				i2c_start(guc_adap);
			}
		}
		buf[i] = i2c_inb(guc_adap);
	}
	return (i);
}

static int umvp_doAddress(struct i2c_algo_guc_data *i2c,
				struct i2c_msg *msg)
{
	unsigned short	flags = msg->flags;
	unsigned char	addr;

//	printk(KERN_INFO "umvp_doAddress\n");
	addr = msg->addr << 1;
	if (flags & I2C_M_RD)
		addr |= 1; /*Read operation*/
	if (flags & I2C_M_REV_DIR_ADDR)
		addr ^= 1;
	i2c_outb(i2c, addr);
	
	return 0;
}

static int umvp_xfer(struct i2c_adapter *i2c_adap,
			struct i2c_msg *msgs,
			int num )
{
	struct i2c_algo_guc_data *guc_adap = i2c_adap->algo_data;
	struct i2c_msg *pmsg=NULL;
	int i;
	int ret=0, timeout, status;

#if 0 // betta mark
	timeout = wait_for_ack(guc_adap);
	if (timeout) {
		//DB_PNK("Timeout for umvp_xfer\n");
		i = -EIO;
		goto out;
	}
#endif

	for (i=0; ret>=0 && i<num; i++) {
		pmsg = &msgs[i];
		ret = umvp_doAddress(guc_adap, pmsg);
		if (i == 0) {
			i2c_start(guc_adap);
			timeout = wait_for_int(guc_adap, &status);
			if (timeout) {
				if (pmsg){
					printk("umvp_xfer: timeout:WAIT_INT_50u(0x%x)\n",(pmsg->addr << 1));
				}	
				goto out;
			}
			#if 1
			timeout = wait_for_ack(guc_adap);
			if (timeout) {
				if (pmsg){
					printk("umvp_xfer: No such i2c device:WAIT_ACK_50u(0x%x)\n",(pmsg->addr << 1));
				}	
				goto out;
			}
			#endif
		}
	}

//	printk("umvp_xfer: num=%d \n", num);
	/*Read*/
	if (pmsg->flags & I2C_M_RD) {
		/*read bytes into buffer*/
	  //ret = umvp_readbytes(i2c_adap, pmsg->buf, pmsg->len, (i+1==num));
		ret = umvp_readbytes(i2c_adap, pmsg->buf, pmsg->len, (i==num));
		if (ret != pmsg->len) {
			//DB_PNK(KERN_INFO "fail: read ret=%d, pmsg->len=%d\n",	ret, pmsg->len);
		} else {
			//DB_PNK(KERN_INFO "read: %d bytes\n", ret);
		}
	} else {/*Write*/
	  //ret = umvp_sendbytes(i2c_adap, pmsg->buf, pmsg->len, (i+1==num));
		ret = umvp_sendbytes(i2c_adap, pmsg->buf, pmsg->len, (i==num));
		#if 0
			if (ret != pmsg->len) {
				//DB_PNK(KERN_INFO "fail:write ret=%d, pmsg->len=%d\n",	ret, pmsg->len);
			} else {
				//DB_PNK(KERN_INFO "write: %d bytes\n", ret);
				//DB_PNK(KERN_INFO "contents:[0x%08X]\n", *(unsigned int *)(IO_ADDRESS(UMVP_I2C1_BASE)+UMVP_I2C1_DATR));
			}
		#endif
	}
//	i2c_stop(adap)
	return ret;
out:
	return -1;
	//return i;
}

static u32 umvp_func(struct i2c_adapter *i2c_adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL |
		I2C_FUNC_PROTOCOL_MANGLING;
}

static const struct i2c_algorithm guc_algo = {
	.master_xfer	= umvp_xfer,
	.functionality	= umvp_func,
};

#if 0 //betta
static int guc_init_umvp(struct i2c_algo_guc_data *adap)
{
	int ret=0;
	writel(guc_getclock(adap)&0x00FF, IO_ADDRESS(UMVP_I2C1_BASE)+UMVP_I2C1_PSLLR);
	writel(guc_getclock(adap)&0xFF00, IO_ADDRESS(UMVP_I2C1_BASE)+UMVP_I2C1_PSLHR);
	printk(KERN_INFO "init clock\n");
	writel(I2C_GUC_EN, IO_ADDRESS(UMVP_I2C1_BASE)+UMVP_I2C1_ENR);
	printk(KERN_INFO "enable umvp-i2c1\n");

	return ret;
}
#endif

static unsigned short i2c_umvp_getclock(int bus_id)
{
	//return (clock);
#if 0	
	if (bus_id == 0) return I2C0_CLOCK;
	else if (bus_id == 1) return I2C1_CLOCK;
	else if(bus_id == 2)	return I2C2_CLOCK;
	else return -1;
#else
	/*
		The relationship between the PCLK and I2C_clk is shown as
		the following equivalence.
		
		I2C_clk = nPCLK/(4*(nScale+1))
		
		In order to meet the I2C's requirement, we need to restrict
		that I2C_clk could not be greater than 400k. Here, restrict
		the I2C_clk not greater than 10k.

		Hence,

		I2C_clk = nPCLK/(4*(nScale+1)) < 10k

		==> nPCLK/10k < 4(nScale+1)

		==> nScale > nPCLK/40k - 1

		we choose: nScale = nPCLK/40k;
	*/
	
	unsigned short nScale;
//Ted	nScale = (unsigned short) (UMVP2500_APBCLOCK/(400000));
//	nScale = (unsigned short) (PLL_GetAPBClock(A1_SYS_CLOCK_SRC)/(400000));

/* set I2C_clk = 200k */
	nScale = (unsigned short) (PLL_GetAPBClock(A1_SYS_CLOCK_SRC)/(200000*4));
	printk("i2c_umvp_getclock(): nScale=%d, APB=%d\n", nScale, PLL_GetAPBClock(A1_SYS_CLOCK_SRC));
//

	return nScale; 	
#endif
}

int guc_init_umvp(void)
{
	int bus_id, ret=0;
	u32 UMVP_I2C_BASE;
	u32 address;
	unsigned int mfs4;
	for (bus_id=0; bus_id<3; bus_id++)
	{
		if (bus_id == 0) UMVP_I2C_BASE = UMVP_I2C0_BASE;
		else if (bus_id == 1) UMVP_I2C_BASE = UMVP_I2C1_BASE;
		else if(bus_id == 2)	UMVP_I2C_BASE = UMVP_I2C2_BASE;
		else return -1;

		writel(i2c_umvp_getclock(bus_id) & 0x00FF, IO_ADDRESS(UMVP_I2C_BASE)+UMVP_I2C_PSLLR);
		writel((i2c_umvp_getclock(bus_id)& 0xFF00)>>8 , IO_ADDRESS(UMVP_I2C_BASE)+UMVP_I2C_PSLHR);
		writel(I2C_GUC_EN, IO_ADDRESS(UMVP_I2C_BASE)+UMVP_I2C_ENR);
		printk(KERN_INFO "enable umvp-i2c %d\n", bus_id);
	}
	//RTC device(on i2c bus 0) will be detected twice if threre are any device on i2c bus 1 or 2,
	//so we switch i2c bus 1 and bus 2 to GPIO funtion during boot time.
	//ISP.ko will switch them back
	address = IO_ADDRESS(UMVP_SYSTEM_BASE) + 0x1C;
	mfs4 =  readl(address);
	mfs4 = (mfs4 & 0xF3FF3FFF)| 0x0C00C000 ;
	writel (mfs4,address);
	
	return ret;
}
EXPORT_SYMBOL(guc_init_umvp);

int i2c_umvp_add_bus(struct i2c_adapter *adap)
{
	//#struct i2c_algo_guc_data *guc_adap = adap->algo_data;
	int rval;

	adap->algo = &guc_algo;
	adap->timeout = 100;
#if 0 //Ted
	rval = guc_init_umvp();
	if (rval)
		return rval;
#endif
	rval = i2c_add_adapter(adap);
	
	return rval;
}
EXPORT_SYMBOL(i2c_umvp_add_bus);

MODULE_AUTHOR("Peter Lee <hugo.lee@globalunichip.com>");
MODULE_DESCRIPTION("GUC UMVP I2C-Bus algorithm");
MODULE_LICENSE("GPL");
