/*
  libsmtp is a library to send mail via SMTP
    These are the utility data functions.
   
Copyright ?2001 Kevin Read <obsidian@berlios.de>

This software is available under the GNU Lesser Public License as described
in the COPYING file.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Kevin Read <obsidian@berlios.de>
Thu Aug 16 2001 */

#include <glib.h>

#include "../config.h"

#include "libsmtp.h"


/* This function returns a pointer to an allocated libsmtp_session_struct
   All GStrings are initialized. */

struct libsmtp_session_struct * libsmtp_session_initialize (void)
{
  struct libsmtp_session_struct *libsmtp_session;
  /* We use calloc here to clear the memory. GLists are initialized when
     they point to NULL, so it must be cleared. */

  libsmtp_session = (struct libsmtp_session_struct *)calloc (1, sizeof(struct libsmtp_session_struct));
  
  if (libsmtp_session == NULL)
    return NULL;
  
  /* The GStrings must be initialized */
  libsmtp_session->From = g_string_new (NULL);
  libsmtp_session->Subject = g_string_new (NULL);
  libsmtp_session->LastResponse = g_string_new (NULL);
  libsmtp_session->username = g_string_new (NULL);
  libsmtp_session->password = g_string_new (NULL);
  libsmtp_session->pop3server = g_string_new (NULL);
  libsmtp_session->auth_type = NO_AUTH;
#ifdef MATRIXSSL
  libsmtp_session->sessionId=NULL;
  libsmtp_session->cipherSuite=0x0000;
  libsmtp_session->conn=NULL;
  #endif
  libsmtp_session->startTLS=0;
  #ifdef WITH_MIME
    libsmtp_session->Parts = NULL;
    libsmtp_session->NumParts = 0;
  #endif
  return libsmtp_session;
}

/* This frees the specified libsmtp_session_struct and all accompanying
   GStrings and GLists */

int libsmtp_free (struct libsmtp_session_struct *libsmtp_session)
{
  unsigned int i;

  /* Lets see if we gotta close the socket */
  
  if (libsmtp_session->socket) {
    close (libsmtp_session->socket);
    libsmtp_session->socket=0;
  }
  
  if(libsmtp_session->startTLS==1){
#ifdef MATRIXSSL
	printf("smtp do shutdown\n");
	  matrixSslFreeSessionId(libsmtp_session->sessionId);
	  if((libsmtp_session->conn)&&(libsmtp_session->conn->ssl))
		  sslFreeConnection(&libsmtp_session->conn);
	  matrixSslFreeKeys(libsmtp_session->keys);
	  matrixSslClose();
#endif
#ifdef OPENSSL
	printf("smtp do shutdown\n");
	SSL_shutdown(libsmtp_session->ssl);
	SSL_free(libsmtp_session->ssl);
	SSL_CTX_free(libsmtp_session->ctx);
#endif

  }
  /* All GStrings and GLists must be freed */
  g_list_free (libsmtp_session->To);
  g_list_free (libsmtp_session->CC);
  g_list_free (libsmtp_session->BCC);
  
  /* Now we free all elements of the Lists we allocated with strdup */
  libsmtp_session->ToResponse = g_list_first (libsmtp_session->ToResponse);
  for (i=0; i<g_list_length (libsmtp_session->ToResponse); i++)
    free (g_list_nth_data (libsmtp_session->ToResponse, i));
  
  libsmtp_session->CCResponse = g_list_first (libsmtp_session->CCResponse);
  for (i=0; i<g_list_length (libsmtp_session->CCResponse); i++)
    free (g_list_nth_data (libsmtp_session->CCResponse, i));

  libsmtp_session->BCCResponse = g_list_first (libsmtp_session->BCCResponse);
  for (i=0; i<g_list_length (libsmtp_session->BCCResponse); i++)
    free (g_list_nth_data (libsmtp_session->BCCResponse, i));
  
  g_list_free (libsmtp_session->ToResponse);
  g_list_free (libsmtp_session->CCResponse);
  g_list_free (libsmtp_session->BCCResponse);
  
  if (libsmtp_session->Parts) 
	  g_node_destroy(libsmtp_session->Parts);

  g_string_free (libsmtp_session->From,1);
  g_string_free (libsmtp_session->Subject,1);
  g_string_free (libsmtp_session->LastResponse,1);
  g_string_free (libsmtp_session->username, 1);
  g_string_free (libsmtp_session->password, 1);
  g_string_free (libsmtp_session->pop3server, 1);
  
  /* Ok, lets free the malloced session struct */
  free (libsmtp_session);

  return 0;
}

/* This function sets the environment for the session. At the moment it
   just sets subject and sender address. SSL and auth stuff should be set
   here in the future. */

int libsmtp_set_environment (char *libsmtp_int_From, char *libsmtp_int_Subject,
      unsigned int libsmtp_int_flags, struct libsmtp_session_struct *libsmtp_session)
{
  if (libsmtp_int_From == NULL || !strlen (libsmtp_int_From) || libsmtp_int_Subject == NULL) {
    libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
    return LIBSMTP_BADARGS;
  }
  
  g_string_assign (libsmtp_session->From, libsmtp_int_From);
  g_string_assign (libsmtp_session->Subject, libsmtp_int_Subject);
//  printf("from <%s>\n", libsmtp_session->From->str);
//  printf("subject <%s>\n", libsmtp_session->Subject->str);

  return LIBSMTP_NOERR;
}

int libsmtp_set_authorization(unsigned int Auth_type, char *auth_username, char *auth_password,\
							  unsigned short pop_port, char *pop_server, struct libsmtp_session_struct *libsmtp_session)
{
	libsmtp_session->auth_type = Auth_type;
	if (pop_server == NULL || !strlen (pop_server)) {
		libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
		return LIBSMTP_BADARGS;
	}
	g_string_assign(libsmtp_session->pop3server, pop_server);

	if (Auth_type) {
		if (Auth_type == POP_RELAY) {
			libsmtp_session->pop3port = pop_port;
		}
		if (auth_username == NULL || !strlen (auth_username) || auth_password == NULL) {
			libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
			return LIBSMTP_BADARGS;
		}
	  g_string_assign(libsmtp_session->username, auth_username);
	  g_string_assign(libsmtp_session->password, auth_password);
//  printf("username <%s>\n", libsmtp_session->username->str);
//  printf("password <%s>\n", libsmtp_session->password->str);
  }

  return LIBSMTP_NOERR;
}

int libsmtp_add_recipient (int libsmtp_int_rec_type, char *libsmtp_int_address,
      struct libsmtp_session_struct *libsmtp_session)
{
  /* Do we need a copy? */
  char *libsmtp_int_copy;
  /* Lets just check that rec_type isn't an invalid value */
  if ((libsmtp_int_rec_type < 0) || (libsmtp_int_rec_type > LIBSMTP_REC_MAX))
  {
    libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
    return LIBSMTP_BADARGS;
  }

  /* Zero length string as argument? */
  if (!strlen (libsmtp_int_address))
  {
    libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
    return LIBSMTP_BADARGS;
  }
  
  switch (libsmtp_int_rec_type)
  {
    case (LIBSMTP_REC_TO):
      libsmtp_int_copy = (char *)strdup (libsmtp_int_address);
      libsmtp_session->To = g_list_append (libsmtp_session->To, libsmtp_int_copy);
      break;

    case (LIBSMTP_REC_CC):
      libsmtp_session->CC = g_list_append (libsmtp_session->CC, libsmtp_int_copy);
      break;

    case (LIBSMTP_REC_BCC):
      libsmtp_session->BCC = g_list_append (libsmtp_session->BCC, libsmtp_int_copy);
      break;

    default:
      /* Lets just check that rec_type isn't an invalid value */
      libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
      return LIBSMTP_BADARGS;
      break;
  }
  
  return LIBSMTP_NOERR;
}
