#include <stdlib.h>
#include "global.h"
#include "md5.h"

#include "digestmd5.h"

static char *skip_lws (char *s)
{
    if(!s) return NULL;
    
    /* skipping spaces: */
    while (s[0] == ' ' || s[0] == HT || s[0] == CR || s[0] == LF) {
	if (s[0]=='\0') break;
	s++;
    }  
    
    return s;
}

static char *skip_token (char *s, int caseinsensitive)
{
    if(!s) return NULL;
    
    while (s[0]>SP) {
	if (s[0]==DEL || s[0]=='(' || s[0]==')' || s[0]=='<' || s[0]=='>' ||
	    s[0]=='@' || s[0]==',' || s[0]==';' || s[0]==':' || s[0]=='\\' ||
	    s[0]=='\'' || s[0]=='/' || s[0]=='[' || s[0]==']' || s[0]== '?' ||
	    s[0]=='=' || s[0]== '{' || s[0]== '}') {
	    if (caseinsensitive == 1) {
		if (!isupper((unsigned char) s[0]))
		    break;
	    } else {
		break;
	    }
	}
	s++;
    }  
    return s;
}

#define CHAR64(c)  (((c) < 0 || (c) > 127) ? -1 : index_64[(c)])

static char basis_64[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/???????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????????";

int sasl_encode64(const char *_in, unsigned inlen,
		  char *_out, unsigned outmax, unsigned *outlen)
{
    const unsigned char *in = (const unsigned char *)_in;
    unsigned char *out = (unsigned char *)_out;
    unsigned char oval;
    char *blah;
    unsigned olen;

    /* check params */
    if ((inlen >0) && (in == NULL)) return SASL_FAIL;
    
    /* Will it fit? */
    olen = (inlen + 2) / 3 * 4;
    if (outlen)
      *outlen = olen;
    if (outmax < olen)
      return SASL_FAIL;

    /* Do the work... */
    blah=(char *) out;
    while (inlen >= 3) {
      /* user provided max buffer size; make sure we don't go over it */
        *out++ = basis_64[in[0] >> 2];
        *out++ = basis_64[((in[0] << 4) & 0x30) | (in[1] >> 4)];
        *out++ = basis_64[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
        *out++ = basis_64[in[2] & 0x3f];
        in += 3;
        inlen -= 3;
    }
    if (inlen > 0) {
      /* user provided max buffer size; make sure we don't go over it */
        *out++ = basis_64[in[0] >> 2];
        oval = (in[0] << 4) & 0x30;
        if (inlen > 1) oval |= in[1] >> 4;
        *out++ = basis_64[oval];
        *out++ = (inlen < 2) ? '=' : basis_64[(in[1] << 2) & 0x3c];
        *out++ = '=';
    }

    if (olen < outmax)
      *out = '\0';
    
    return SASL_OK;
}

/* NULL - error (unbalanced quotes), 
   otherwise pointer to the first character after value */
static char *unquote (char *qstr)
{
    char *endvalue;
    int   escaped = 0;
    char *outptr;
    
    if(!qstr) return NULL;
    
    if (qstr[0] == '"') {
	qstr++;
	outptr = qstr;
	
	for (endvalue = qstr; endvalue[0] != '\0'; endvalue++, outptr++) {
	    if (escaped) {
		outptr[0] = endvalue[0];
		escaped = 0;
	    }
	    else if (endvalue[0] == '\\') {
		escaped = 1;
		outptr--; /* Will be incremented at the end of the loop */
	    }
	    else if (endvalue[0] == '"') {
		break;
	    }      
	    else {
		outptr[0] = endvalue[0];      
	    }
	}
	
	if (endvalue[0] != '"') {
	    return NULL;
	}
	
	while (outptr <= endvalue) {
	    outptr[0] = '\0';
	    outptr++;
	}
	endvalue++;
    }
    else { /* not qouted value (token) */
	endvalue = skip_token(qstr,0);
    };
    
    return endvalue;  
} 

static void get_pair(char **in, char **name, char **value)
{
    char  *endpair;
    /* int    inQuotes; */
    char  *curp = *in;
    *name = NULL;
    *value = NULL;
    
    if (curp == NULL) return;
    if (curp[0] == '\0') return;
    
    /* skipping spaces: */
    curp = skip_lws(curp);
    
    *name = curp;
    
    curp = skip_token(curp,1);
    
    /* strip wierd chars */
    if (curp[0] != '=' && curp[0] != '\0') {
	*curp++ = '\0';
    };
    
    curp = skip_lws(curp);
    
    if (curp[0] != '=') { /* No '=' sign */ 
	*name = NULL;
	return;
    }
    
    curp[0] = '\0';
    curp++;
    
    curp = skip_lws(curp);  
    
    *value = (curp[0] == '"') ? curp+1 : curp;
    
    endpair = unquote (curp);
    if (endpair == NULL) { /* Unbalanced quotes */ 
	*name = NULL;
	return;
    }
    if (endpair[0] != ',') {
	if (endpair[0]!='\0') {
	    *endpair++ = '\0'; 
	}
    }
    
    endpair = skip_lws(endpair);
    
    /* syntax check: MUST be '\0' or ',' */  
    if (endpair[0] == ',') {
	endpair[0] = '\0';
	endpair++; /* skipping <,> */
    } else if (endpair[0] != '\0') { 
	*name = NULL;
	return;
    }
    
    *in = endpair;
}

void sasl_rand (char *buf, unsigned len)
{
    unsigned int lup;
    /* check params */
    if (!buf) return;

	srandom(time(NULL));
    for (lup=0;lup<len;lup++) {
		buf[lup] = (char) (random() >> 8);
	}
}

/* copy a string */
int _plug_strdup(char *in, char **out, int *outlen)
{
  size_t len = strlen(in);

  if(!in || !out) {
      return SASL_FAIL;
  }

  *out = malloc(len + 1);
  if (!*out) {
      return SASL_FAIL;
  }

  strcpy((char *) *out, in);

  if (outlen)
      *outlen = len;

  return SASL_OK;
}

/* Basically a conditional call to realloc(), if we need more */
int _plug_buf_alloc(char **rwbuf,
		    int *curlen, unsigned int newlen) 
{
    if(!rwbuf || !curlen) {
		return SASL_FAIL;
    }

    if(!(*rwbuf)) {
		*rwbuf = malloc(newlen);
		if (*rwbuf == NULL) {
			*curlen = 0;
			return SASL_FAIL;
		}
		*curlen = newlen;
    } else if(*rwbuf && *curlen < newlen) {
		size_t needed = 2*(*curlen);
	
		while(needed < newlen)
			needed *= 2;
	
		*rwbuf = realloc(*rwbuf, needed);
		if (*rwbuf == NULL) {
			*curlen = 0;
			return SASL_FAIL;
		}
		*curlen = needed;
    } 

    return SASL_OK;
}

static int add_to_challenge(char **str, unsigned *buflen, unsigned *curlen,
			    char *name,
			    unsigned char *value,
			    bool need_quotes)
{
    int             namesize = strlen(name);
    int             valuesize = strlen((char *) value);
    int             ret;
    
    ret = _plug_buf_alloc(str, buflen,
			  *curlen + 1 + namesize + 2 + valuesize + 2);
    if(ret != SASL_OK) return ret;
    
    *curlen = *curlen + 1 + namesize + 2 + valuesize + 2;
    
    strcat(*str, ",");
    strcat(*str, name);
    
    if (need_quotes) {
		strcat(*str, "=\"");
		strcat(*str, (char *) value);	/* XXX. What about quoting??? */
		strcat(*str, "\"");
    } else {
		strcat(*str, "=");
		strcat(*str, (char *) value);
    }
    
    return SASL_OK;
}

static unsigned char *create_nonce()
{
    unsigned char  *base64buf;
    int             base64len;
    
    char           *ret = (char *) malloc(NONCE_SIZE);
    if (ret == NULL)
		return NULL;
    
    sasl_rand((char *) ret, NONCE_SIZE);
    
    /* base 64 encode it so it has valid chars */
    base64len = (NONCE_SIZE * 4 / 3) + (NONCE_SIZE % 3 ? 4 : 0);
    
    base64buf = (unsigned char *) malloc(base64len + 1);
    if (base64buf == NULL) {
		return NULL;
    }
    
    /*
     * Returns SASL_OK on success, SASL_BUFOVER if result won't fit
     */
	sasl_encode64(ret, NONCE_SIZE, (char *) base64buf, base64len, NULL);
	free(ret);
    return base64buf;
}

static bool UTF8_In_8859_1(const unsigned char *base, int len)
{
    const unsigned char *scan, *end;
    
    end = base + len;
    for (scan = base; scan < end; ++scan) {
		if (*scan > 0xC3)
			break;			/* abort if outside 8859-1 */
		if (*scan >= 0xC0 && *scan <= 0xC3) {
			if (++scan == end || *scan < 0x80 || *scan > 0xBF)
			break;
		}
    }
    
    /* if scan >= end, then this is a 8859-1 string. */
    return (scan >= end);
}

/*
 * if the string is entirely in the 8859-1 subset of UTF-8, then translate to
 * 8859-1 prior to MD5
 */
void MD5_UTF8_8859_1(MD5_CTX * ctx,
		     bool In_ISO_8859_1,
		     const unsigned char *base,
		     int len)
{
    const unsigned char *scan, *end;
    unsigned char   cbuf;
    
    end = base + len;
    
    /* if we found a character outside 8859-1, don't alter string */
    if (!In_ISO_8859_1) {
		MD5Update(ctx, (unsigned char *)base, len);
		return;
    }
    /* convert to 8859-1 prior to applying hash */
    do {
		for (scan = base; scan < end && *scan < 0xC0; ++scan);
		if (scan != base)
			MD5Update(ctx, (unsigned char *)base, scan - base);
		if (scan + 1 >= end)
			break;
		cbuf = ((scan[0] & 0x3) << 6) | (scan[1] & 0x3f);
		MD5Update(ctx, &cbuf, 1);
		base = scan + 2;
    } while (base < end);
}

static const unsigned char *COLON = ":";

static void DigestCalcSecret(unsigned char *pszUserName,
			     unsigned char *pszRealm,
			     unsigned char *Password,
			     int PasswordLen,
			     HASH HA1)
{
    bool            In_8859_1;
    
    MD5_CTX         Md5Ctx;
    
    /* Chris Newman clarified that the following text in DIGEST-MD5 spec
       is bogus: "if name and password are both in ISO 8859-1 charset"
       We shoud use code example instead */
    
    MD5Init(&Md5Ctx);
    /* We have to convert UTF-8 to ISO-8859-1 if possible */
    In_8859_1 = UTF8_In_8859_1(pszUserName, strlen((char *) pszUserName));
    MD5_UTF8_8859_1(&Md5Ctx, In_8859_1,
		    pszUserName, strlen((char *) pszUserName));

	MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    
    if (pszRealm != NULL && pszRealm[0] != '\0') {
		/* a NULL realm is equivalent to the empty string */
		MD5Update(&Md5Ctx, pszRealm, strlen((char *) pszRealm));
    }      
    
    MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    
    /* We have to convert UTF-8 to ISO-8859-1 if possible */
    In_8859_1 = UTF8_In_8859_1(Password, PasswordLen);
    MD5_UTF8_8859_1(&Md5Ctx, In_8859_1, Password, PasswordLen);
    
    MD5Final(HA1, &Md5Ctx);
}

static void CvtHex(HASH Bin, HASHHEX Hex)
{
    unsigned short  i;
    unsigned char   j;
    
    for (i = 0; i < HASHLEN; i++) {
	j = (Bin[i] >> 4) & 0xf;
	if (j <= 9)
	    Hex[i * 2] = (j + '0');
	else
	    Hex[i * 2] = (j + 'a' - 10);
	j = Bin[i] & 0xf;
	if (j <= 9)
	    Hex[i * 2 + 1] = (j + '0');
	else
	    Hex[i * 2 + 1] = (j + 'a' - 10);
    }
    Hex[HASHHEXLEN] = '\0';
}

static void
DigestCalcHA1(context_t * text,
	      unsigned char *pszUserName,
	      unsigned char *pszRealm,
	      unsigned char * pszPassword,
	      unsigned char *pszAuthorization_id,
	      unsigned char *pszNonce,
	      unsigned char *pszCNonce,
	      HASHHEX SessionKey)
{
    MD5_CTX         Md5Ctx;
    HASH            HA1;
    
    DigestCalcSecret(pszUserName,
		     pszRealm,
		     pszPassword,
		     strlen(pszPassword),
		     HA1);
    /* calculate the session key */
    MD5Init(&Md5Ctx);
    MD5Update(&Md5Ctx, HA1, HASHLEN);
    MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    MD5Update(&Md5Ctx, pszNonce, strlen((char *) pszNonce));
    MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    MD5Update(&Md5Ctx, pszCNonce, strlen((char *) pszCNonce));
    if (pszAuthorization_id != NULL) {
		MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
		MD5Update(&Md5Ctx, pszAuthorization_id, 
				 strlen((char *) pszAuthorization_id));
    }
    MD5Final(HA1, &Md5Ctx);
    
    CvtHex(HA1, SessionKey);
    
    /* xxx rc-* use different n */
    
    /* save HA1 because we'll need it for the privacy and integrity keys */
    memcpy(text->HA1, HA1, sizeof(HASH));
    
}

/*
 * calculate request-digest/response-digest as per HTTP Digest spec
 */
void
DigestCalcResponse(HASHHEX HA1,	/* H(A1) */
		   unsigned char *pszNonce,	/* nonce from server */
		   unsigned int pszNonceCount,	/* 8 hex digits */
		   unsigned char *pszCNonce,	/* client nonce */
		   unsigned char *pszQop,	/* qop-value: "", "auth",
						 * "auth-int" */
		   unsigned char *pszDigestUri,	/* requested URL */
		   unsigned char *pszMethod,
		   HASHHEX HEntity,	/* H(entity body) if qop="auth-int" */
		   HASHHEX Response	/* request-digest or response-digest */
    )
{
    MD5_CTX         Md5Ctx;
    HASH            HA2;
    HASH            RespHash;
    HASHHEX         HA2Hex;
    char ncvalue[10];
    
    /* calculate H(A2) */
    MD5Init(&Md5Ctx);
    
    if (pszMethod != NULL) {
		MD5Update(&Md5Ctx, pszMethod, strlen((char *) pszMethod));
    }
    MD5Update(&Md5Ctx, (unsigned char *) COLON, 1);
    
    /* utils->MD5Update(&Md5Ctx, (unsigned char *) "AUTHENTICATE:", 13); */
    MD5Update(&Md5Ctx, pszDigestUri, strlen((char *) pszDigestUri));
    if (strcasecmp((char *) pszQop, "auth") != 0) {
		/* append ":00000000000000000000000000000000" */
		MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
		MD5Update(&Md5Ctx, HEntity, HASHHEXLEN);
    }
    MD5Final(HA2, &Md5Ctx);
    CvtHex(HA2, HA2Hex);
    
    /* calculate response */
    MD5Init(&Md5Ctx);
    MD5Update(&Md5Ctx, HA1, HASHHEXLEN);
    MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    MD5Update(&Md5Ctx, pszNonce, strlen((char *) pszNonce));
    MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    if (*pszQop) {
		sprintf(ncvalue, "%08x", pszNonceCount);
		MD5Update(&Md5Ctx, ncvalue, strlen(ncvalue));
		MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
		MD5Update(&Md5Ctx, pszCNonce, strlen((char *) pszCNonce));
		MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
		MD5Update(&Md5Ctx, pszQop, strlen((char *) pszQop));
		MD5Update(&Md5Ctx, (unsigned char *)COLON, 1);
    }
    MD5Update(&Md5Ctx, HA2Hex, HASHHEXLEN);
    MD5Final(RespHash, &Md5Ctx);
    CvtHex(RespHash, Response);
}


static char *calculate_response(context_t * text,
				unsigned char *username,
				unsigned char *realm,
				unsigned char *nonce,
				unsigned int ncvalue,
				unsigned char *cnonce,
				char *qop,
				unsigned char *digesturi,
				char * passwd,
				unsigned char *authorization_id,
				char **response_value)
{
    HASHHEX         SessionKey;
    HASHHEX         HEntity = "00000000000000000000000000000000";
    HASHHEX         Response;
    char           *result;
    
    /* Verifing that all parameters was defined */
    if(!username || !cnonce || !nonce || !ncvalue || !digesturi || !passwd) {
		return NULL;
    }
    
    if (realm == NULL) {
	/* a NULL realm is equivalent to the empty string */
		realm = (unsigned char *) "";
    }
    
    if (qop == NULL) {
	/* default to a qop of just authentication */
		qop = "auth";
    }

    DigestCalcHA1(text,
		  username,
		  realm,
		  passwd,
		  authorization_id,
		  nonce,
		  cnonce,
		  SessionKey);
    
    DigestCalcResponse(SessionKey,/* H(A1) */
		       nonce,	/* nonce from server */
		       ncvalue,	/* 8 hex digits */
		       cnonce,	/* client nonce */
		       (unsigned char *) qop,	/* qop-value: "", "auth",
						 * "auth-int" */
		       digesturi,	/* requested URL */
		       (unsigned char *) "AUTHENTICATE",
		       HEntity,	/* H(entity body) if qop="auth-int" */
		       Response	/* request-digest or response-digest */
	);
    
    result = malloc(HASHHEXLEN + 1);
    memcpy(result, Response, HASHHEXLEN);
    result[HASHHEXLEN] = 0;

//	printf("result = %s\n", result);
    
    if (response_value != NULL) {
		DigestCalcResponse(SessionKey,	/* H(A1) */
				   nonce,	/* nonce from server */
				   ncvalue,	/* 8 hex digits */
				   cnonce,	/* client nonce */
				   (unsigned char *) qop,	/* qop-value: "", "auth",
								 * "auth-int" */
				   (unsigned char *) digesturi,	/* requested URL */
				   NULL,
				   HEntity,	/* H(entity body) if qop="auth-int" */
				   Response	/* request-digest or response-digest */
			);
		
		*response_value = malloc(HASHHEXLEN + 1);
		if (*response_value == NULL)
			return NULL;
		
		memcpy(*response_value, Response, HASHHEXLEN);
		(*response_value)[HASHHEXLEN] = 0;
//		printf("*response_value = %s\n", *response_value);
		
    }
    
    return result;
}

static int
make_client_response(context_t *text, char *userid, char *password)
{
    char *qop = NULL;
    unsigned nbits = 0;
    unsigned char  *digesturi = NULL;
    bool            IsUTF8 = 1;
    char           ncvalue[10];
    char           maxbufstr[64];
    char           *response = NULL;
    unsigned        resplen = 0;
    int result;
	char *service = "smtp";

    switch (text->protection) {
//    case DIGEST_PRIVACY:
//		qop = "auth-conf";
//		oparams->encode = &digestmd5_privacy_encode; 
//		oparams->decode = &digestmd5_privacy_decode;
//		oparams->mech_ssf = ctext->cipher->ssf;
//	
//		nbits = ctext->cipher->n;
//		text->cipher_enc = ctext->cipher->cipher_enc;
//		text->cipher_dec = ctext->cipher->cipher_dec;
//		text->cipher_free = ctext->cipher->cipher_free;
//		text->cipher_init = ctext->cipher->cipher_init;
//		break;
//    case DIGEST_INTEGRITY:
//		qop = "auth-int";
//		oparams->encode = &digestmd5_integrity_encode;
//		oparams->decode = &digestmd5_integrity_decode;
//		oparams->mech_ssf = 1;
//		break;
    case DIGEST_NOLAYER:
    default:
		qop = "auth";
//		oparams->encode = NULL;
//		oparams->decode = NULL;
//		oparams->mech_ssf = 0;
    }

    digesturi = malloc(strlen(service) + 1 + strlen(text->serverFQDN) + 1 + 1);
    if (digesturi == NULL) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    };
    
    /* allocated exactly this. safe */
    strcpy((char *) digesturi, service);
    strcat((char *) digesturi, "/");
    strcat((char *) digesturi, text->serverFQDN);

    /* response */
    response = calculate_response(text,
			   (char *) userid,
			   (unsigned char *) text->realm,
			   text->nonce,
			   text->nonce_count,
			   text->cnonce,
			   qop,
			   digesturi,
			   password,
			   NULL,	
			   &text->response_value);
//	printf("response %s\n", response);
    
    resplen = strlen(userid) + strlen("username") + 5;
    result = _plug_buf_alloc(&text->out_buf, &text->out_buf_len, resplen);
    if (result != SASL_OK) 
		goto FreeAllocatedMem;

    sprintf(text->out_buf, "username=\"%s\"", userid);
    
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
		"realm", (unsigned char *) text->realm, 1) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
#if 0
    if (strcmp(oparams->user, oparams->authid)) {
		if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "authzid", (char *) oparams->user, 1) != SASL_OK) {
			result = SASL_FAIL;
			goto FreeAllocatedMem;
		}
    }
#endif
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "nonce", text->nonce, 1) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "cnonce", text->cnonce, 1) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    snprintf(ncvalue, sizeof(ncvalue), "%08x", text->nonce_count);
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "nc", (unsigned char *) ncvalue, 0) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "qop", (unsigned char *) qop, 0) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }

#if 0
    if (ctext->cipher != NULL) {
		if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
					 "cipher", (unsigned char *) ctext->cipher->name, 1) != SASL_OK) {
			result = SASL_FAIL;
			goto FreeAllocatedMem;
		}
    }
#endif

    if (text->protection & DIGEST_INTEGRITY) {
		snprintf(maxbufstr, sizeof(maxbufstr), "%d", text->server_maxbuf);
		if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
					 "maxbuf", (unsigned char *) maxbufstr, 0) != SASL_OK) {
			printf("internal error: add_to_challenge maxbuf failed\n");
			goto FreeAllocatedMem;
		}
    }

	if (IsUTF8) {
		if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
					 "charset", (unsigned char *) "utf-8", 0) != SASL_OK) {
			result = SASL_FAIL;
			goto FreeAllocatedMem;
		}
    }
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "digest-uri", digesturi, 1) != SASL_OK) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    if (add_to_challenge(&text->out_buf, &text->out_buf_len, &resplen,
			 "response", (unsigned char *) response, 0) != SASL_OK) {
	
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    
    /* self check */
    if (strlen(text->out_buf) > 2048) {
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }

    /* set oparams */
//    oparams->maxoutbuf = ctext->server_maxbuf;
//    if(oparams->mech_ssf > 1) {
//		/* MAC block (privacy) */
//		oparams->maxoutbuf -= 25;
//    } else if(oparams->mech_ssf == 1) {
//		/* MAC block (integrity) */
//		oparams->maxoutbuf -= 16;
//    }
    
//    if (oparams->mech_ssf > 0) {
//		char enckey[16];
//		char deckey[16];
//		
//		create_layer_keys(text, params->utils, text->HA1, nbits,
//				  enckey, deckey);
//		
//		/* initialize cipher if need be */
//		if (text->cipher_init)
//			text->cipher_init(text, enckey, deckey);		       
//	}
    
	result = SASL_OK;

 FreeAllocatedMem:
	if (digesturi) free(digesturi);
	if (response) free(response);

    return result;
}

static int parse_server_challenge(context_t *text, 
				  const char *serverin, unsigned serverinlen, 
				  char ***outrealms, int *noutrealm)
{
    int result = SASL_OK;
    char *in_start = NULL;
    char *in = NULL;
    char **realms = NULL;
    int nrealm = 0;
    int protection = 0;
    int ciphers = 0;
    int maxbuf_count = 0;
    int algorithm_count = 0;
	int server_maxbuf;
	char *name, *value;

    if (!serverin || !serverinlen){
		return -1;
    }

    in_start = in = malloc(serverinlen + 1);
    if (in == NULL) return SASL_FAIL;
    
    memcpy(in, serverin, serverinlen);
    in[serverinlen] = 0;
    
    server_maxbuf = 65536; /* Default value for maxbuf */

    /* create a new cnonce */
    text->cnonce = create_nonce();
    if (text->cnonce == NULL) {
		printf("failed to create cnonce\n");
		result = -1;
		goto FreeAllocatedMem;
    }
    /* parse the challenge */
    while (in[0] != '\0') {
		
		get_pair(&in, &name, &value);

		//printf("%s %s\n", name, value);
		/* if parse error */
		if (name == NULL) {
			result = SASL_FAIL;
			goto FreeAllocatedMem;
		}
		
		if (strcasecmp(name, "realm") == 0) {
			nrealm++;
			
			if(!realms)
				realms = malloc(sizeof(char *) * (nrealm + 1));
			else
				realms = realloc(realms, 
							sizeof(char *) * (nrealm + 1));
			if (realms == NULL) {
				result = SASL_FAIL;
				goto FreeAllocatedMem;
			}
			_plug_strdup(value, &realms[nrealm-1], NULL);
			realms[nrealm] = NULL;
		} else if (strcasecmp(name, "nonce") == 0) {
			_plug_strdup(value, (char **) &text->nonce, NULL);
			text->nonce_count = 1;
		} else if (strcasecmp(name, "qop") == 0) {
			while (value && *value) {
				char *comma = strchr(value, ',');
				if (comma != NULL) {
					*comma++ = '\0';
				}
				//now we only support auth...
				if (strcasecmp(value, "auth") == 0) {
					protection |= DIGEST_NOLAYER;
				//} else if (strcasecmp(value, "auth-conf") == 0) {
				//	protection |= DIGEST_PRIVACY;
				//} else if (strcasecmp(value, "auth-int") == 0) {
				//	protection |= DIGEST_INTEGRITY;
				} else {
					printf("Server supports unknown layer: %s\n",value);
				}
				value = comma;
			}
			
			if (protection == 0) {
				result = SASL_FAIL;
				printf("Server doesn't support known qop level\n");
				goto FreeAllocatedMem;
			}
	#if 0
		} else if (strcasecmp(name, "cipher") == 0) {
			while (value && *value) {
				char *comma = strchr(value, ',');
				struct digest_cipher *cipher = available_ciphers;
				
				if (comma != NULL) {
					*comma++ = '\0';
				}
			
				/* do we support this cipher? */
				while (cipher->name) {
					if (!strcasecmp(value, cipher->name)) break;
					cipher++;
				}
				if (cipher->name) {
					ciphers |= cipher->flag;
				} else {
					printf("Server supports unknown cipher: %s\n",value);
				}
				
				value = comma;
			}
	#endif
	#if 0
		} else if (strcasecmp(name, "stale") == 0 && ctext->password) {
			/* clear any cached password */
			if (ctext->free_password)
			_plug_free_secret(params->utils, &ctext->password);
			ctext->password = NULL;
	#endif 
		} else if (strcasecmp(name, "maxbuf") == 0) {
			/* maxbuf A number indicating the size of the largest
			 * buffer the server is able to receive when using
			 * "auth-int". If this directive is missing, the default
			 * value is 65536. This directive may appear at most once;
			 * if multiple instances are present, the client should
			 * abort the authentication exchange.  
			 */
			maxbuf_count++;
			
			if (maxbuf_count != 1) {
				result = SASL_FAIL;
				printf("At least two maxbuf directives found. Authentication aborted\n");
				goto FreeAllocatedMem;
			} else if (sscanf(value, "%u", &text->server_maxbuf) != 1) {
				result = SASL_FAIL;
				printf("Invalid maxbuf parameter received from server\n");
				goto FreeAllocatedMem;
			} else {
				if (text->server_maxbuf<=16) {
					result = SASL_FAIL;
					printf("Invalid maxbuf parameter received from server (too small: %s)\n", value);
					goto FreeAllocatedMem;
				}
			}
		} else if (strcasecmp(name, "charset") == 0) {
			if (strcasecmp(value, "utf-8") != 0) {
				result = SASL_FAIL;
				printf("Charset must be UTF-8\n");
				goto FreeAllocatedMem;
			}// else {
			//	IsUTF8 = TRUE;
			//}
		} else if (strcasecmp(name,"algorithm")==0) {
			if (strcasecmp(value, "md5-sess") != 0) {
				printf("'algorithm' isn't 'md5-sess'\n");
				result = SASL_FAIL;
				goto FreeAllocatedMem;
			}
			
			algorithm_count++;
			if (algorithm_count > 1) {
				printf("Must see 'algorithm' only once\n");
				result = SASL_FAIL;
				goto FreeAllocatedMem;
			}
		} else {
			printf("DIGEST-MD5 unrecognized pair %s/%s: ignoring\n",
					   name, value);
		}
    }

    if (algorithm_count != 1) {
		printf("Must see 'algorithm' once. Didn't see at all\n");
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }
    /* make sure we have everything we require */
    if (text->nonce == NULL) {
		printf("Don't have nonce.\n");
		result = SASL_FAIL;
		goto FreeAllocatedMem;
    }

#if 0
    /* get requested ssf */
    external = params->external_ssf;
    
    /* what do we _need_?  how much is too much? */
    if (params->props.maxbufsize == 0) {
		musthave = 0;
		limit = 0;
    } else {
		if (params->props.max_ssf > external) {
			limit = params->props.max_ssf - external;
		} else {
			limit = 0;
		}
		if (params->props.min_ssf > external) {
			musthave = params->props.min_ssf - external;
		} else {
			musthave = 0;
		}
    }

    /* we now go searching for an option that gives us at least "musthave"
       and at most "limit" bits of ssf. */
    if ((limit > 1) && (protection & DIGEST_PRIVACY)) {
		struct digest_cipher *cipher;
		
		/* let's find an encryption scheme that we like */
		cipher = available_ciphers;
		while (cipher->name) {
			/* examine each cipher we support, see if it meets our security
			   requirements, and see if the server supports it.
			   choose the best one of these */
			if ((limit >= cipher->ssf) && (musthave <= cipher->ssf) &&
			(ciphers & cipher->flag) &&
			(!ctext->cipher || (cipher->ssf > ctext->cipher->ssf))) {
			ctext->cipher = cipher;
			}
			cipher++;
		}
	
		if (ctext->cipher) {
			/* we found a cipher we like */
			ctext->protection = DIGEST_PRIVACY;
		} else {
			/* we didn't find any ciphers we like */
			printf("No good privacy layers\n");
		}
    }
    
    if (ctext->cipher == NULL) {
	/* we failed to find an encryption layer we liked;
	   can we use integrity or nothing? */
	
		if ((limit >= 1) && (musthave <= 1) 
			&& (protection & DIGEST_INTEGRITY)) {
			/* integrity */
			text->protection = DIGEST_INTEGRITY;
		} else if (musthave <= 0) {
			/* no layer */
			text->protection = DIGEST_NOLAYER;
	
			/* See if server supports not having a layer */
			if ((protection & DIGEST_NOLAYER) != DIGEST_NOLAYER) {
				printf("Server doesn't support \"no layer\"\n");
				result = SASL_FAIL;
				goto FreeAllocatedMem;
			}
		} else {
			printf("Can't find an acceptable layer\n");
			result = SASL_FAIL;
			goto FreeAllocatedMem;
		}
    }
#endif
	*outrealms = realms;
	*noutrealm = nrealm;

FreeAllocatedMem:
    if (in_start) free(in_start);

    if (result != SASL_OK && realms) {
		int lup;
		
		/* need to free all the realms */
		for (lup = 0;lup < nrealm; lup++)
			free(realms[lup]);
		
		free(realms);
    }

    return result;
}

int digestmd5_client_mech_step2(context_t *text, 
	char *challenge, int challen, char *userid, char *password, 
		char **clientout, int *clientoutlen)
{
    int result = SASL_FAIL;
    char **realms = NULL;
    int nrealm = 0;

    /* don't bother parsing the challenge more than once */
	result = parse_server_challenge(text, challenge, challen,
					&realms, &nrealm);
	if (result != SASL_OK) goto FreeAllocatedMem;

	if (nrealm == 1) {
		/* only one choice! */
		text->realm = realms[0];
		/* free realms */
		free(realms);
		realms = NULL;
	}

//	printf("text->nonce %s\n", text->nonce);

	/*
     * (username | realm | nonce | cnonce | nonce-count | qop | digest-uri |
     * response | maxbuf | charset | auth-param )
     */
    result = make_client_response(text, userid, password);
    if (result != SASL_OK) goto FreeAllocatedMem;

    *clientoutlen = strlen(text->out_buf);
    *clientout = text->out_buf;
    
    result = SASL_CONTINUE;

  FreeAllocatedMem:
    if (realms) {
		int lup;
		
		/* need to free all the realms */
		for (lup = 0;lup < nrealm; lup++) 
			free(realms[lup]);
		free(realms);
    }

    return result;
}

int
digestmd5_client_mech_step3(context_t *text, const char *serverin,unsigned serverinlen)
{
    char           *in = NULL;
    char           *in_start;
    int result = SASL_FAIL;
    
    printf("DIGEST-MD5 client step 3\n");

    /* Verify that server is really what he claims to be */
    in_start = in = malloc(serverinlen + 1);
    if (in == NULL) return SASL_FAIL;

    memcpy(in, serverin, serverinlen);
    in[serverinlen] = 0;
    
    /* parse the response */
    while (in[0] != '\0') {
		char *name, *value;
		get_pair(&in, &name, &value);
		
		if (name == NULL) {
			printf("DIGEST-MD5 Received Garbage\n");
			break;
		}
		
		if (strcasecmp(name, "rspauth") == 0) {
			if (strcmp(text->response_value, value) != 0) {
				printf("DIGEST-MD5: This server wants us to believe that he knows shared secret\n");
				result = SASL_FAIL;
			} else {
				result = SASL_OK;
			}
			break;
		} else {
			printf("DIGEST-MD5 unrecognized pair %s/%s: ignoring\n",
					   name, value);
		}
    }
    
    free(in_start);

    return result;
}

void digestmd5_free(context_t *text)
{
	if (text->cnonce)
		free(text->cnonce);
	if (text->nonce) 
		free(text->nonce);
	if (text->response_value) 
		free(text->response_value);
	if (text->realm) 
		free(text->realm);
	if (text->out_buf)
		free(text->out_buf);
	text->cnonce = NULL;
	text->nonce = NULL;
	text->response_value = NULL;
	text->realm = NULL;
	text->out_buf = NULL;
}
