/* ######################################################################################
 * OnvifUrl.c
 *    send the loopback URL commands to streaming server to set or get camera's
 * configurations.
 *    transfer settings between onvif domain and URL command domain.
 * ###################################################################################### */
#include "onvif.h"

#define ONVIF_URL_DEBUG         0

extern tOnvifConf gOnvifConf;

/* ######################################################################################
 * Static Functions
 * ###################################################################################### */
static int OnvifTcpSockRead(int sock, char *buf, int max, int sec) {
	int len = 0;
	int n;
	struct timeval tout;
	fd_set RxFd;

	FD_ZERO(&RxFd);
	FD_SET(sock, &RxFd);
	tout.tv_sec  = sec;
	tout.tv_usec = 0;
	n = select(sock+1, &RxFd, NULL, NULL, &tout);
	if(n > 0) { /* data might be in skb */
		if(FD_ISSET(sock, &RxFd)) {
			len = read(sock, buf, max);
			if(len > 0) return len;
		}
		return -1;
	} else if(n == 0) { /* time out */
		return 0;
	} else { /* select error */
		return -1;
	}
}
static int OnvifGetHttpMsg(int sock, char *buf, int pos, int max) {
    int i = 0;
    int len = 0;
    int start = pos;
    
    for(i = 0 ; i < 5 ; i ++) {
        len = OnvifTcpSockRead(sock, &buf[start], (max-start-1), 3);
        if(len > 0) {
            start += len;
        } else if(len == 0) {
            if(start > 0) break; /* revceive done */
        } else if(len < 0) {
            break; /* receive done */
        }
    }
    if(start == 0) {
        fprintf(stderr, "%s: no http msg received\n", __func__);
        return -1;
    }
    return start;
}
static int OnvifHttpHeaderGet(int sock, char *buf, int max) {
	int i   = 0;
	int pos = 0;
	int len = 0;
	char *p;

	for(i = 0 ; i < 3 ; i ++) {
		len = OnvifTcpSockRead(sock, &buf[pos], (max-pos-1), 5);
		if(len > 0) {
			pos += len;
			buf[pos] = '\0';
			if(pos > 5) {
				p = strstr(buf, "\r\n\r\n");
				if(p) { /* found the terminator, get the http return code */
                    p += 4;
					if(strstr(buf, "200 OK")) {
                        pos = sprintf(buf, "%s", p); /* remove http header */
                        len = OnvifGetHttpMsg(sock, &buf[pos], pos, (max-pos-1)); 
                        if(len > 0) return len;
					}
					/* not found. got the error return */
					L1("error. 200 OK not found (%d)\n", pos);
                    fprintf(stderr, "%s", buf);
					return -1;
				} else {
					if(pos < (max-pos-1)) continue;
                    L1("error, not found http terminator (%d)\n", pos);
                    fprintf(stderr, "%s", buf);
					return -1;
				}
			} else { /* read again */
				continue;
			}
		} else if(len == 0) { /* timeout, keep reading */
			continue;
		} else { /* socket error */
            fprintf(stderr, "[%s]:error, read socket\n", __func__);
			return -1;
		}
	}
	/* not found http header */
	fprintf(stderr, "[%s]:error. no http header found %d\n", __func__, len); 
	return -1;
}
static int OnvifUrlReplyRead(int sock, tOnvifUrl *url) {
	int len = 0;
	int pos = 0;
    char buf[512];

	if((url->reply == NULL) || (url->maxReplyLen == 0)) {
		/* no expect for return message */
        #if ONVIF_URL_DEBUG
		printf("[%s]: no expected return message for cmd %s\n", __func__, url->cmd);
        #endif
        do {
			len = OnvifTcpSockRead(sock, buf, (sizeof(buf)-1), 1);
		} while(len > 0);
		return 0;
	}
	/* read the reply message form streaming engine */
	pos = OnvifHttpHeaderGet(sock, url->reply, url->maxReplyLen);
	if(pos > 0) { /* read the rest of return message if has */
		do {
			len = OnvifTcpSockRead(sock, &url->reply[pos], (url->maxReplyLen-pos-1), 5);
			if(len > 0) {
				pos += len;
				url->reply[pos] = '\0';
			}
		} while((len > 0) && (pos < url->maxReplyLen));
		return pos;
	}
	return -1;
}
/* ######################################################################################
 * Share APIs
 * ###################################################################################### */
int OnvifGetValueFmUrlReply(char *retMsg, char *key, char *value, int maxLen) {
    char *p = NULL;
    char *e = NULL;
    int  len = 0;
    
    p = strstr(retMsg, key);
    if(p) {
        p += strlen(key);
        if(*p == '=') p ++;
        if(*p == '\'') p ++;
        e = strchr(p, '\n');
        if(e == NULL) {
            e = strchr(p, '\0');
            if(e == NULL) {
                L1("error, not found terminator for %s from %s\n", key, retMsg);
                return -1;
            }
        }
        if(*(e-1) == '\'') e--;
        len = e-p;
        if(len < maxLen) {
            if(len > 0) memcpy(value, p, len);
            value[len] = '\0';
            return len;
        }
        L1("error, get %s setting. %d/%d\n", key, len, maxLen);
    } else L1("error, not found %s in %s\n", key, retMsg);
    return -1;
}
void OnvifUrlCmdInit(tOnvifUrl *url, char *cgi) {

	snprintf(url->cgi, 11, "%s", cgi);
	url->len = 0;
	url->maxReplyLen = 0;
	url->reply = NULL;
}
int OnvifSendUrlCmd(tOnvifUrl *url, tOnvifConf *conf) {
	int sock = 0;
	int len  = 0;
	int sockOpEnabled = 1;
	char buf[1024];
	struct timeval tv;
	struct sockaddr_in addr;

	/* create a TCP socket to connect to the http server now */
	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if(sock > 0) {
		(void)SockOptReuseAddrSet(sock, sockOpEnabled);
		tv.tv_sec  = 5; /* five seconds to connect to streaming server */
		tv.tv_usec = 0;
		(void)setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof (tv));
		(void)setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char*)&sockOpEnabled, sizeof(int));
		memset((unsigned char *)&addr, 0, sizeof(struct sockaddr_in));
		addr.sin_family      = AF_INET;
		addr.sin_addr.s_addr = inet_addr("127.0.0.1");
		addr.sin_port        = htons(conf->HttpPort);
		if(connect(sock, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) == 0) {
			/* connected, send the URL command */
			len = snprintf( buf, (sizeof(buf)-1),
							"GET /cgi-bin/%s?USER=%s&PWD=%s&%s HTTP/1.1\r\n"
							"HOST: 127.0.0.1\r\n\r\n",
							url->cgi, conf->root, conf->passwd, url->cmd);
			if(send(sock, (char *)buf, len, 0) == len) {
				/* get the reply message */
				len = OnvifUrlReplyRead(sock, url);
				close(sock);
				if(len >= 0) return len;
                fprintf (stderr, "%s:error. get reply len=%d\n", __func__, len);
				return -1;
			}
			fprintf (stderr, "%s:error. send %s (%s)\n", __func__, buf, strerror(errno));
		} else {
			fprintf (stderr, "%s:error. connect 127.0.0.1:%d(%s)\n", __func__, conf->HttpPort, strerror(errno));
		}
		close(sock);
	} else {
		fprintf (stderr, "%s:error. create TCP socket.(%s)\n", __func__, strerror(errno));
	}
	return -1;
}
int OnvifSingleUrlCmdSend(char *cgi, char *cmd, char *reply, int replyLen) {
	tOnvifUrl url;

	snprintf(url.cgi, (URLCGI_LEN-1), "%s", cgi);
	url.len = snprintf(url.cmd, (URLCMD_LEN-1), "%s", cmd);
	url.maxReplyLen = replyLen;
	url.reply = reply;
	return OnvifSendUrlCmd(&url, &gOnvifConf);
}
int OnvifGetDIsFmUrl(int *dioStatus) {
    char retMsg[256];
    char value[16];

    if(OnvifSingleUrlCmdSend("encoder", "DIO_STATUS", retMsg, (sizeof(retMsg)-1)) > 0) {
        if(OnvifGetValueFmUrlReply(retMsg, "DIO_STATUS=", value, (sizeof(value)-1)) > 0) {
            *dioStatus = (int)strtoul(value, NULL, 0);
            return 0;
        }
    }
    return -1;
}
int OnvifPTVelocityFmUrl(float velocity) {
    int speed = 0;
    
    /* the speed in pan and tilt in URL is in the range of 1~5. But, it is in -1.0 ~ 1.0 in the onvif
     * assume the range of velocity was validated. However, I set the max speed if the valicity is not
     * in valide range */
    if (velocity == 0) return 0;
    if (velocity > 0) {
        speed =  (int)(1 + (4 * velocity));
        if(speed <= 5) return speed;
        return 5;
    }
    speed = (int)(-1 + (4 * velocity));
    if(speed >= -5) return speed;
    return -5;
}
int OnvifZoomVelocityFmUrl(float velocity) {
    int speed = 0;
    
    /* the speed in zoom in URL is in the range of 2~7. But, it is in -1.0 ~ 1.0 in the onvif
     * assume the range of velocity was validated. However, I set the max speed if the valicity is not
     * in valide range */
    if (velocity == 0) return 0;
    if (velocity > 0) {
        speed =  (int)(2 + (5 * velocity));
        if(speed <= 7) return speed;
        return 5;
    }
    speed = (int)(-2 + (5 * velocity));
    if(speed >= -7) return speed;
    return -7;
}
