
/*******************************************************************
 *
 * File: wireless_sm.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wireless_sm.c,v 1.24 2006/06/13 21:16:27 chessing Exp $
 * $Date: 2006/06/13 21:16:27 $
 * $Log: wireless_sm.c,v $
 * Revision 1.24  2006/06/13 21:16:27  chessing
 * Updates to the new EAP-TNC code.
 *
 * Revision 1.23  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.22  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.21  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.20  2006/05/23 23:38:39  chessing
 * Patch to allow the madwifi driver to associate and authenticate using the regular wireless extensions.  Should this start working, we are dumpping the madwifi specific driver crap! ;)
 *
 * Revision 1.19  2006/05/23 23:07:00  chessing
 * Cleaned up the previous patch for timer_tick.  (There was a cleaner way of doing it.)  Fixed up some stuff that was causing issues getting associated/dynamic WEP authenticated with the madwifi driver.  Madwifi seems to be happily working (with dynamic WEP) now.
 *
 * Revision 1.18  2006/05/17 02:40:06  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.17  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.16  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.15  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.14  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.13  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.12  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.11  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.10  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.9  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.8  2005/10/13 18:57:28  chessing
 * Removed some leftover debug strings.  Cleaned up some compiler warnings. Fixed an association bug in the madwifi driver.
 *
 * Revision 1.7  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.6  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.5  2005/09/17 18:38:17  chessing
 * Some updates to make the madwifi driver work with dynamic WEP again.
 *
 * Revision 1.4  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.3  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.2  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.1  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <string.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wireless_sm.h"
#include "eapol.h"
#include "event_core.h"
#include "cardif/cardif.h"
#include "config_ssid.h"
#include "timer.h"
#include "statemachine.h"
#include "eap.h"
#include "cardif/linux/cardif_linux_wext.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

struct wireless_state *wireless_sm = NULL;

void set_static_wep_keys(struct interface_data *ctx,
                         struct config_static_wep *wepdata);

// This is in mschapv2.c.  We need it for the static WEP piece.
extern void process_hex(char *, int, char *);

/********************************************************************
 *
 * Return a pointer to the wireless state machine for external functions
 * that may want to monkey with it.  (Keep in mind that certain types of
 * monkeying will result in bad stuff.  So avoid monkeying where possible!)
 *
 ********************************************************************/
struct wireless_state *get_wireless_sm()
{
  return wireless_sm;
}

/********************************************************************
 *
 * Return the current state that we are in.
 *
 ********************************************************************/
int wireless_sm_get_state()
{
  xsup_assert((wireless_sm == NULL), "wireless_sm == NULL", TRUE);

  return wireless_sm->state;
}

/********************************************************************
 *
 * Initialize our wireless statemachine.  We should always start in
 * ACTIVE_SCAN mode, since we won't know anything about the SSID we
 * are attempting to connect to.
 *
 ********************************************************************/
void wireless_sm_init(int zeros_on_roam, struct interface_data *ctx)
{
   struct config_network *network_data;

   if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
     return;

   debug_printf(DEBUG_STATE, "Init wireless state machine.\n");
   wireless_sm = (struct wireless_state *)malloc(sizeof(struct wireless_state));
   if (!wireless_sm)
     {
       debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for the wireless"
		    " state machine!\n");
       _exit(1);
     }
   
   bzero(wireless_sm, sizeof(struct wireless_state));
   
   if (cardif_get_if_state(ctx) == FALSE)
     {
       // Our interface is currently down.  So go to PORT_DOWN mode.
       debug_printf(DEBUG_NORMAL, "PORT_DOWN from %s!\n", __FUNCTION__);
       wireless_sm_change_state(PORT_DOWN, ctx);
     } else {
       wireless_sm_change_state(ACTIVE_SCAN, ctx);
     }
   
   if (zeros_on_roam == TRUE) 
     {
       SET_FLAG(wireless_sm->flags, ZEROSONROAM);
     } else {
       UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
     }
   
   // Probably don't need to do this here, but do it anyway for good 
   // measure, unless we determine it causes a problem for some cards.
   network_data = config_get_network_config();
   if ((network_data == NULL) || (network_data->initial_wep == NULL))
     {
       cardif_clear_keys(ctx);
     }
   else
     {
       set_static_wep_keys(ctx, network_data->initial_wep);
     }
}


/**
 * Sets the WEP keys according to the config_static_wep structure.
 * This is used with the initial_wep setting and with static_wep.
 *
 * @todo It may be a good idea to clear the keys not being set
 *
 * @param ctx     The interface data
 * @param wepdata WEP keys
 */
void set_static_wep_keys(struct interface_data *ctx,
                         struct config_static_wep *wepdata)
{
   int keyidx, klen, t;
   char key[26];

   if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
     return;

   if (!xsup_assert((wepdata != NULL), "wepdata != NULL", FALSE))
     return;

   for (keyidx = 1; keyidx < 5; keyidx++)
   {
      if (wepdata->key[keyidx] != NULL)
      {
         klen = strlen((char *)wepdata->key[keyidx]);
         if (((klen/2) == 5) || ((klen/2) == 13))
         {
            // We have a valid length key.  So, convert it, and set it.
            process_hex((char *)wepdata->key[keyidx], klen, key);

            // Calculate the proper key index.
            t = keyidx-1;

            if (keyidx == wepdata->tx_key)
            {
               debug_printf(DEBUG_INT, "%s: Setting TX key! [%d]\n",
                            __func__, wepdata->tx_key);
               t |= 0x80;
               cardif_set_wep_key(ctx, (uint8_t *)key, (klen/2), t);
            }
            else
            {
               cardif_set_wep_key(ctx, (uint8_t *)key, (klen/2), t);
            }
         }
      }
   }
}


/********************************************************************
 *
 * Deinit our wireless statemachine.  Currently does nothing, but is 
 * included to maintain the init/active/deinit model.
 *
 ********************************************************************/
void wireless_sm_deinit()
{
  if (wireless_sm)
    {
      free(wireless_sm);
      wireless_sm = NULL;
    }
}

/********************************************************************
 *
 * Print out the text value for the state that we have been passed.
 *
 ********************************************************************/
void wireless_sm_disp_state(int debug_level, int showstate)
{
  switch (showstate)
    {
    case UNASSOCIATED:
      debug_printf_nl(debug_level, "UNASSOCIATED");
      break;

    case ASSOCIATED:
      debug_printf_nl(debug_level, "ASSOCIATED");
      break;

    case ACTIVE_SCAN:
      debug_printf_nl(debug_level, "ACTIVE_SCAN");
      break;

    case ASSOCIATING:
      debug_printf_nl(debug_level, "ASSOCIATING");
      break;

    case ASSOCIATION_TIMEOUT_S:
      debug_printf_nl(debug_level, "ASSOCIATION_TIMEOUT");
      break;

    case STATIC_ASSOCIATION:
      debug_printf_nl(debug_level, "STATIC_ASSOCIATION");
      break;

    case PORT_DOWN:
      debug_printf_nl(debug_level, "PORT_DOWN");
      break;

    case NO_ENC_ASSOCIATION:
      debug_printf_nl(debug_level, "NO_ENC_ASSOCIATION");
      break;

    case AUTH_RESTART:
      debug_printf_nl(debug_level, "AUTH_RESTART");
      break;

    case AUTH_STOPPED:
      debug_printf_nl(debug_level, "AUTH_STOPPED");
      break;

    default:
      debug_printf_nl(DEBUG_INT, "UNKNOWN!!!!!");
      break;
    }
}

/********************************************************************
 *
 * Display information about our state change.
 *
 ********************************************************************/
void wireless_sm_disp_state_change(int debug_level, int newstate)
{
  wireless_sm_disp_state(debug_level, wireless_sm->state);
  debug_printf_nl(debug_level, " -> ");
  wireless_sm_disp_state(debug_level, newstate);
  debug_printf_nl(debug_level, "\n");
}

/********************************************************************
 *
 * Change to unassociated state.
 *
 ********************************************************************/
void wireless_sm_change_to_unassociated(struct interface_data *ctx)
{
  struct config_network *network_data;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm->associated = 2;   // Change to something that will keep us from
                                 // triggering this change over and over.

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  debug_printf(DEBUG_NORMAL, "The wireless card is not associated to an AP.\n");

  if (wireless_sm->state != ASSOCIATED)
    {
      // Only change to unassociated state if we are already in associated
      // state.
      return;
    }

  // We are allowed to switch from any state to unassociated state.
  wireless_sm_disp_state_change(DEBUG_STATE, UNASSOCIATED);

  eap_clear_active_method(network_data->activemethod);

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = UNASSOCIATED;

  // Then, switch to active scan state to see if we can associate again.
  wireless_sm_change_state(ACTIVE_SCAN, ctx);
}

/********************************************************************
 *
 * Change to ACTIVE_SCAN state.
 *
 ********************************************************************/
void wireless_sm_change_to_active_scan(struct interface_data *ctx)
{
  //  uint8_t abil;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ACTIVE_SCAN);

  debug_printf(DEBUG_NORMAL, "Scanning for wireless networks.  Please wait."
	       "\n");
  /*
    // XXX This code is broken!
  if ((wireless_sm->state == ASSOCIATING) ||
      (wireless_sm->state == ASSOCIATED))
    {
      abil = config_ssid_get_ssid_abilities();

      // We only flip the zeros on roam bit if we are doing WEP.
      if ((!(abil & WPA_IE)) && (!(abil & RSN_IE)))
	{
	  // We switched to active scan from associating, so flip the
	  // zeros on roam bit.
	  if (TEST_FLAG(wireless_sm->flags, ZEROSONROAM))
	    {
	      debug_printf(DEBUG_INT, "Next association will attempt to use no"
			   " encryption.\n");
	      UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
	    } else {
	      debug_printf(DEBUG_INT, "Next association will attempt to use all"
			   " zeros.\n");
	      SET_FLAG(wireless_sm->flags, ZEROSONROAM);
	    }
	}
    }
  */

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = ACTIVE_SCAN;
  wireless_sm->initialize = FALSE;
  wireless_sm->ssid_change = FALSE;

  // Do any setup that is needed to enter the new state.
  statemachine_reinit(ctx);
  cardif_do_wireless_scan(ctx, FALSE);
}

/********************************************************************
 *
 * Change to ASSOCIATED state.
 *
 ********************************************************************/
void wireless_sm_change_to_associated(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm->associated = 2;        // Change to something that will keep
                                      // us from triggering this state over
                                      // and over.

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATED);
  
  debug_printf(DEBUG_NORMAL, "Associated.\n  - Starting "
	       "authentication/encryption process.\n");

  // We are associated, so clear the timer.
  timer_cancel(ASSOCIATION_TIMER);

  if (wireless_sm->state == ACTIVE_SCAN)
    {
      // We jumpped here from active scan state.  We probably don't have 
      // useful information in our ssid configuration.  So, create it
      // now.
      debug_printf(DEBUG_STATE, "Jumpped directly to associated state.\n"
		   "Getting SSID information.\n");

      config_ssid_set_active_ssid(ctx->cur_essid);
    }

  // Update our state variables to indicate what state we are in now.
  wireless_sm->state = ASSOCIATED;
}


/********************************************************************
 *
 * We have a static key for this SSID. So set it.
 *
 ********************************************************************/
void wireless_sm_change_to_static_association(struct interface_data *ctx)
{
  struct config_network *netdata;
  struct config_static_wep *wepdata;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm_disp_state_change(DEBUG_STATE, STATIC_ASSOCIATION);

  debug_printf(DEBUG_NORMAL, "Associating to network '%s' with static WEP"
	       " keys.\n", ctx->cur_essid);

  debug_printf(DEBUG_INT, "We appear to be using static WEP.\n");

  netdata = config_get_network_config();
  if (!netdata)
    {
      debug_printf(DEBUG_NORMAL, "No valid network config! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  if (!netdata->methods)
    {
      debug_printf(DEBUG_NORMAL, "No valid method config! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  wireless_sm->state = STATIC_ASSOCIATION;

  wepdata = (struct config_static_wep *)netdata->methods->method_data;

  set_static_wep_keys(ctx, wepdata);
}

/********************************************************************
 *
 * We have no key, so turn off encryption.
 *
 ********************************************************************/
void wireless_sm_change_to_no_enc_association(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm_disp_state_change(DEBUG_STATE, NO_ENC_ASSOCIATION);

  debug_printf(DEBUG_NORMAL, "No configuration is available for this network."
	       "\n");

  debug_printf(DEBUG_INT, "We have no configuration, clearing encryption.\n");
  if (ctx) cardif_enc_disable(ctx);
  wireless_sm->state = NO_ENC_ASSOCIATION;
}

/********************************************************************
 *
 * If we get a timeout trying to associate.
 *
 ********************************************************************/
void wireless_sm_association_timeout(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // Cancel our timer, so we don't keep running the timeout function.
  timer_cancel(ASSOCIATION_TIMER);

  // Our association timer expired, so change to ASSOCIATION_TIMEOUT_S
  // state.
  wireless_sm_change_state(ASSOCIATION_TIMEOUT_S, ctx);
}

/********************************************************************
 *
 * Change state to ASSOCIATING.
 *
 ********************************************************************/
void wireless_sm_change_to_associating(struct interface_data *ctx)
{
  struct config_network *netdata;
  struct config_globals *globals;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // Cancel the scan timer that is running.
  timer_cancel(RESCAN_TIMER);

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATING);

  debug_printf(DEBUG_NORMAL, "Attempting to associate.\n");

  netdata = config_get_network_config();

  if (!netdata)
    {
      debug_printf(DEBUG_INT, "No valid network data!! (%s)\n",
		   __FUNCTION__);
      return;
    }

  // Clear the replay counter.
  bzero(ctx->statemachine->replay_counter, 8);

  /*
  if ((wireless_sm->state != ASSOCIATING) && 
      (wireless_sm->state != ACTIVE_SCAN))
    {
      // We have attempted to switch to associating mode from an invalid
      // state.
      debug_printf(DEBUG_NORMAL, "Attempted to change to associating state "
		   "from an invalid state.  Forcing a reinit of wireless sm "
		   "in an attempt to recover.\n");
      wireless_sm->initialize = TRUE;
      return;
      }*/

  wireless_sm->state = ASSOCIATING;

  globals = config_get_globals();
  if (globals)
    {
      if (timer_check_existing(ASSOCIATION_TIMER))
	{
	  timer_reset_timer_count(ASSOCIATION_TIMER, globals->assoc_timeout);
	} else {
	  timer_add_timer(ASSOCIATION_TIMER, globals->assoc_timeout, NULL,
			  &wireless_sm_association_timeout);
	}
    } else {
      debug_printf(DEBUG_NORMAL, "Couldn't read global variable information!"
		   " Association timer will not function correctly!\n");
    }

  statemachine_reinit(ctx);

  if (!(netdata->initial_wep == NULL))
    {
      set_static_wep_keys(ctx, netdata->initial_wep);
    }
				  
  if (config_ssid_using_wep())
    {
      // Do any setup that is needed to enter the new state.
      cardif_wep_associate(ctx, (wireless_sm->flags & ZEROSONROAM));
      debug_printf(DEBUG_NORMAL, "Listed SSID is %s\n", ctx->cur_essid);
    }  else {
      cardif_reassociate(ctx, 1);
    }

  /*
  if (cardif_GetSSID(ctx, temp_ssid) != XENONE)
    {
      cardif_reassociate(ctx, 1);
    } else {
      if (strcmp(temp_ssid, ctx->cur_essid) != 0)
	{
	  cardif_reassociate(ctx, 1);
	}
    }
  */
}

/********************************************************************
 *
 * If we timed out trying to associate, then do this, and switch to
 * active scan mode.
 *
 ********************************************************************/
void wireless_sm_change_to_association_timeout(struct interface_data *ctx)
{
  uint8_t abilities;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, ASSOCIATION_TIMEOUT_S);

  debug_printf(DEBUG_NORMAL, "The association attempt timed out.\n");

  abilities = config_ssid_get_ssid_abilities();

  // If we are doing WPA or RSN, then we shouldn't mess with the zeros on
  // roam flag.
  if ((!(abilities & WPA_IE)) && (!(abilities & RSN_IE)))
    {
      if (TEST_FLAG(wireless_sm->flags, ZEROSONROAM))
	{
	  UNSET_FLAG(wireless_sm->flags, ZEROSONROAM);
	} else {
	  SET_FLAG(wireless_sm->flags, ZEROSONROAM);
	}
    }

  wireless_sm->state = ASSOCIATION_TIMEOUT_S;
  wireless_sm_change_state(ACTIVE_SCAN, ctx);
}

/********************************************************************
 *
 * If the port is down, change to the PORT_DOWN state.
 *
 ********************************************************************/
void wireless_sm_change_to_port_down(struct interface_data *ctx)
{
  struct config_network *network_data;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  debug_printf(DEBUG_NORMAL, "Your interface is down.\n");

  // Make sure that last state we were in has a valid transition to this
  // state.
  wireless_sm_disp_state_change(DEBUG_STATE, PORT_DOWN);  
  wireless_sm->state = PORT_DOWN;

  network_data = config_get_network_config();

  /* If we have network configuration data, then call a clear on the
   * active EAP method in use. */
  if (network_data)
    {
      eap_cleanup(&network_data->activemethod);
    }

  cardif_clear_keys(ctx);
  statemachine_reinit(ctx);
}

// Forward declaration needed for next function.
void wireless_sm_do_associated(struct interface_data *);

/********************************************************************
 *
 * Send a logoff, wipe keys, clear IEs, and send a disassociate.
 *
 ********************************************************************/
void wireless_sm_clear_interface(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // Send logoff
  ctx->statemachine->userLogoff = TRUE;
  wireless_sm_do_associated(ctx);        // To get the SM to send the logoff.

  // Send a disassociate
  cardif_disassociate(ctx, DISASSOC_UNSPECIFIED);

  // Clear IEs
  cardif_disable_wpa_state(ctx);

  // Wipe keys
  cardif_delete_key(ctx, 0, 1);
  cardif_delete_key(ctx, 0, 0);
  cardif_delete_key(ctx, 1, 0);
  cardif_delete_key(ctx, 2, 0);
  cardif_delete_key(ctx, 3, 0);
  cardif_enc_disable(ctx);
}

/********************************************************************
 *
 * We *ONLY* change to restart state when instructed to by events outside
 * of the main Xsupplicant program.  (i.e. A GUI interface.)
 *
 * In restart state, we send a logoff, and disassociate, clear any keys,
 * and wipe any IEs.  Then, we reinit the state machines, and switch back 
 * to ACTIVE_SCAN state.  This should effectively restart the authentication.
 *
 ********************************************************************/
void wireless_sm_change_to_auth_restart(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm_clear_interface(ctx);
  statemachine_change_state(ctx, DISCONNECTED);
}

/********************************************************************
 *
 * We *ONLY* change to stopped state when instructed to by events outside
 * of the main Xsupplicant program.  (i.e. A GUI interface.)
 *
 * In stopped mode, we send a logoff, and disassociate, clear any keys,
 * and delete any IEs in use.  Then, each pass through the state, we flush
 * any buffers that might contain information that would be queued.  (Basically
 * we ignore everything.)
 *
 ********************************************************************/
void wireless_sm_change_to_auth_stopped(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm_clear_interface(ctx);
  statemachine_change_state(ctx, S_FORCE_UNAUTH);
}

/********************************************************************
 *
 * Check for global events that would signal a state change.
 *
 ********************************************************************/
void wireless_sm_check_globals(struct interface_data *ctx)
{
  struct config_network *netdata;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  netdata = config_get_network_config();

  if (!netdata)
    {
      debug_printf(DEBUG_INT, "No valid network data!! (%s)\n",
		   __FUNCTION__);
      return;
    } else {
      if ((netdata->methods == NULL) &&
	  (wireless_sm->state != NO_ENC_ASSOCIATION))
	{
	  ctx->statemachine->portControl = FORCE_AUTHORIZED;
	  statemachine_change_state(ctx, S_FORCE_AUTH);
	  wireless_sm_change_state(NO_ENC_ASSOCIATION, ctx);
	  return;
	}
      
      if ((netdata->methods != NULL) &&
	  (netdata->methods->method_num == STATIC_WEP_METHOD) &&
	  (wireless_sm->state != STATIC_ASSOCIATION))
	{
	  ctx->statemachine->portControl = FORCE_AUTHORIZED;
	  statemachine_change_state(ctx, S_FORCE_AUTH);
	  wireless_sm_change_state(STATIC_ASSOCIATION, ctx);
	  return;
	}

    }

  if ((wireless_sm->port_active == TRUE) && (!cardif_get_if_state(ctx)))
    {
      // Our interface is down.
      wireless_sm->port_active = FALSE;
      wireless_sm_change_state(PORT_DOWN, ctx);
    } else {
      wireless_sm->port_active = TRUE;
    }

  if ((wireless_sm->initialize) || (wireless_sm->ssid_change))
    {
      debug_printf(DEBUG_INT, "Initialize : %d    SSID Change : %d\n",
		   wireless_sm->initialize, wireless_sm->ssid_change);

      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    }

  if (wireless_sm->associated == TRUE)
    {
      wireless_sm_change_state(ASSOCIATED, ctx);
    }

  if (wireless_sm->associated == FALSE)
    {
      wireless_sm_change_state(UNASSOCIATED, ctx);
    }

}

/********************************************************************
 *
 * Change state from where we are, to a new state.  The individual state
 * change handlers *MUST* make sure they are changing from a valid state
 * in to the new state.
 *
 ********************************************************************/
void wireless_sm_change_state(int newstate, struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // If we aren't wireless, then ignore state change requests.
  if (!TEST_FLAG(ctx->flags, IS_WIRELESS)) return;

  switch (newstate)
    {
    case UNASSOCIATED:
      wireless_sm_change_to_unassociated(ctx);
      break;

    case ASSOCIATED:
      wireless_sm_change_to_associated(ctx);
      break;

    case ACTIVE_SCAN:
      wireless_sm_change_to_active_scan(ctx);
      break;

    case ASSOCIATING:
      wireless_sm_change_to_associating(ctx);
      break;

    case ASSOCIATION_TIMEOUT_S:
      wireless_sm_change_to_association_timeout(ctx);
      break;

    case STATIC_ASSOCIATION:
      wireless_sm_change_to_static_association(ctx);
      break;

    case PORT_DOWN:
      wireless_sm_change_to_port_down(ctx);
      break;

    case NO_ENC_ASSOCIATION:
      wireless_sm_change_to_no_enc_association(ctx);
      break;
      
    case AUTH_RESTART:
      wireless_sm_change_to_auth_restart(ctx);
      break;

    case AUTH_STOPPED:
      wireless_sm_change_to_auth_stopped(ctx);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Attempt to change to invalid state in "
		   "%s(), %s, at line %d.\n", __FUNCTION__, __FILE__,
		   __LINE__);
      debug_printf(DEBUG_NORMAL, "Changing to ACTIVE_SCAN to attempt to "
		   "recover.\n");
      wireless_sm_change_to_active_scan(ctx);
      break;
    }
}

/*********************************************************************
 *
 * Handle an event while in port down state.
 *
 *********************************************************************/
void wireless_sm_do_port_down(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  wireless_sm->port_active = cardif_get_if_state(ctx);

  if (wireless_sm->port_active == TRUE)
    {
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    }
}

/*********************************************************************
 *
 * Handle an event while we are in unassociated mode.
 *
 *********************************************************************/
void wireless_sm_do_unassociated(struct interface_data *ctx)
{
  struct config_globals *globals = NULL;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  globals = config_get_globals();

  xsup_assert((globals != NULL), "globals != NULL", TRUE);

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ASSOC_AUTO))
    {
      // We are set to auto associate.  So, switch to active scan mode.
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
    } else {
      // Otherwise, we have nothing to do, so take a short nap.  (To avoid
      // hammering the processor.

      if (wireless_sm->associated)
	{
	  wireless_sm_change_state(ASSOCIATED, ctx);
	} 
    }
}

/*********************************************************************
 *
 * Handle an event while we are in associated mode.
 *
 *********************************************************************/
void wireless_sm_do_associated(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // If we are associated, then go ahead and run the EAPOL state machine.
  statemachine_run(ctx);
}

/**********************************************************************
 *
 * The rescan timer expired.  Clear it, and start another scan.
 *
 **********************************************************************/
void wireless_sm_clear_rescan_timer(struct interface_data *ctx)
{
  // First, cancel the existing timer.
  timer_cancel(RESCAN_TIMER);

  // Then, change states to ACTIVE_SCAN.
  wireless_sm_change_state(ACTIVE_SCAN, ctx);
}

/**********************************************************************
 *
 * Initiate a rescan timeout.
 *
 **********************************************************************/
void wireless_sm_set_rescan_timer()
{
  struct config_globals *globals;

  globals = config_get_globals();
  if (globals)
    {
      // This function will get called every second when we don't have scan data.
      //  We should only instate a new timer if there isn't an existing one.
      if (!timer_check_existing(RESCAN_TIMER))
	{
	  // We need to set up a new rescan timer.
	  timer_add_timer(RESCAN_TIMER, globals->active_timeout, NULL,
			  &wireless_sm_clear_rescan_timer);
	}
    } 
  else
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set rescan timer!  Xsupplicant "
		   "may have problems associating.\n");
    }
}

/*********************************************************************
 *
 * Handle an event while we are in active scan mode.
 *
 *********************************************************************/
void wireless_sm_do_active_scan(struct interface_data *ctx)
{
  char *newssid;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  if (!TEST_FLAG(ctx->flags, SCANNING))
    {
      //      config_ssid_dump();

      // We aren't scanning, so see if we have a valid SSID we can attach
      // to.
      newssid = config_ssid_get_desired_ssid(ctx);
      if (newssid != NULL)
	{
	  debug_printf(DEBUG_STATE, "Switching to Associating mode to"
		       " connect to %s.\n", newssid);

	  // Clear out the SSID we are currently connected to.
	  if (ctx->cur_essid != NULL)
	    {
	      free(ctx->cur_essid);
	      ctx->cur_essid = NULL;
	    }

	  ctx->cur_essid = strdup(newssid);

	  config_build(ctx, ctx->cur_essid);

	  wireless_sm_change_state(ASSOCIATING, ctx);
	  return;
	} else {
	  // If we didn't find anything, sleep for a few seconds before we try
	  // again. 
	  wireless_sm_set_rescan_timer();
	  //	  timer_sleep(15);

	  //	  cardif_do_wireless_scan(ctx, FALSE);
	}
    } 
}

/*********************************************************************
 *
 * Handle an event while we are in associating mode.
 *
 *********************************************************************/
void wireless_sm_do_associating(struct interface_data *ctx)
{
  // Nothing to do here, but wait. ;)
}

/*********************************************************************
 *
 * Handle events while we are in static association mode.
 *
 *********************************************************************/
void wireless_sm_do_static_association(struct interface_data *ctx)
{
  // Nothing to do here.
}

/*********************************************************************
 *
 * Handle events while we are in no encryption association mode.
 *
 *********************************************************************/
void wireless_sm_do_no_enc_association(struct interface_data *ctx)
{
  // Nothing to do here.
}

/*********************************************************************
 *
 * Handle events while we are in STOPPED state.
 *
 *********************************************************************/
void wireless_sm_do_auth_stopped(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  // We basically do the same thing as when we are associated.  It is
  // just that the EAPOL state machine is jammed in to a force unauthenticated
  // state.
  wireless_sm_do_associated(ctx);
}

/*********************************************************************
 *
 * Handle events while in AUTH_RESTART state.
 *
 *********************************************************************/
void wireless_sm_do_auth_restart(struct interface_data *ctx)
{
  debug_printf(DEBUG_NORMAL, "You should't get here!!  Fix the code!\n");
  debug_printf(DEBUG_NORMAL, "File : %s   Function : %s  Line : %d\n",
	       __FILE__, __FUNCTION__, __LINE__);
  exit(255);
}

/*********************************************************************
 *
 * Actually call routines that act on the state that we are currently
 * in.
 *
 *********************************************************************/
void wireless_sm_do_state(struct interface_data *ctx)
{
  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  if (!wireless_sm)
    {
      debug_printf(DEBUG_NORMAL, "No valid wireless state machine context!\n");

      // We can't continue!
      _exit(2);
      return;
    }

  event_core(ctx);

  wireless_sm_check_globals(ctx);

  switch (wireless_sm->state)
    {
    case UNASSOCIATED:
      wireless_sm_do_unassociated(ctx);
      break;

    case ASSOCIATED:
      wireless_sm_do_associated(ctx);
      break;

    case ACTIVE_SCAN:
      wireless_sm_do_active_scan(ctx);
      break;

    case ASSOCIATING:
      wireless_sm_do_associating(ctx);
      break;

    case ASSOCIATION_TIMEOUT_S:
      // The association timeout state simply changes the configuration
      // needed to attempt to associate, and then changes to ACTIVE_SCAN
      // state.
      break;

    case STATIC_ASSOCIATION:
      wireless_sm_do_static_association(ctx);
      break;

    case PORT_DOWN:
      wireless_sm_do_port_down(ctx);
      break;

    case NO_ENC_ASSOCIATION:
      wireless_sm_do_no_enc_association(ctx);
      break;

    case AUTH_RESTART:
      wireless_sm_do_auth_restart(ctx);
      break;

    case AUTH_STOPPED:
      wireless_sm_do_auth_stopped(ctx);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown state %d!\n", wireless_sm->state);
      debug_printf(DEBUG_NORMAL, "Switching to ACTIVE_SCAN state to attempt"
		   " to recover.\n");
      wireless_sm_change_state(ACTIVE_SCAN, ctx);
      break;
    }

  if (ctx->tick) timer_tick(ctx);
}
