/*******************************************************************
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: ipc_callout.c
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: ipc_callout.c,v 1.34 2006/10/03 03:02:31 chessing Exp $
 * $Date: 2006/10/03 03:02:31 $
 * $Log: ipc_callout.c,v $
 * Revision 1.34  2006/10/03 03:02:31  chessing
 * Fix a potential stack smash attack.
 *
 * Revision 1.33  2006/06/09 23:25:44  chessing
 * More GUI work.
 *
 * Revision 1.32  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.31  2006/05/17 02:40:06  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.30  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.29  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.28  2006/04/17 03:56:24  chessing
 * Added some support to enable/disable TNC support both via the configuration file, and via IPC.
 *
 * Revision 1.27  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.26  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/
#include <netinet/in.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_ipc.h"
#include "ipc_callout.h"
#include "wireless_sm.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

/*******************************************************************
 *
 * Respond to a "ping" from a connected client.
 *
 *******************************************************************/
void ipc_callout_ping_resp(struct interface_data *ctx, char *resbuf,
			   int *resbufptr)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)resbuf;

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = PING;
  cmd->getset = IPC_SET;
  cmd->len = 0;
  
  (*resbufptr) += sizeof(struct ipc_cmd);
}

/*******************************************************************
 *
 * Fill in the next command record with the authentication state of the
 * selected interface.
 *
 *******************************************************************/
void ipc_callout_auth_state(struct interface_data *thisint, int *bufptr,
			    char *buffer, int bufsize, char *resbuf, 
			    int *resbufptr)
{
  int assertfail = FALSE;
  struct ipc_cmd *cmd;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    {
      assertfail = TRUE;
    } 
  else
    {
      if (!xsup_assert((thisint->statemachine != NULL),
		       "thisint->statemachine != NULL", FALSE))
	assertfail = TRUE;
    }

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbuf != NULL), "resbuf != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbufptr != NULL), "resbufptr != NULL", FALSE))
    assertfail = TRUE;

  if (assertfail == TRUE)
    {
      (*bufptr) = bufsize;
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC version request!  (Version %d)"
		   "\n", cmd->version);
      return;
    }

  if (cmd->attribute != AUTH_STATE) 
    {
      debug_printf(DEBUG_NORMAL, "Incorrect call to ipc_callout_auth_state!"
		   "\n");
      *bufptr = bufsize;
      return;
    }

  if (cmd->getset != IPC_GET)
    {
      debug_printf(DEBUG_NORMAL, "Attempt to SET authentication state!? "
		   "Skipping!\n");
      *bufptr += (sizeof(struct ipc_cmd)+cmd->len);
      return;
    }

  if (cmd->len != 0)
    {
      debug_printf(DEBUG_NORMAL, "Invalid length!  This will be the last"
		   " request we answer! (In this packet.)\n");
      *bufptr = bufsize;
      return;
    }
  *bufptr+=sizeof(struct ipc_cmd);

  // Build the actual answer.
  cmd = (struct ipc_cmd *)&resbuf[*resbufptr];
  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_RESPONSE;
  cmd->len = 1;
  *resbufptr+=sizeof(struct ipc_cmd);

  resbuf[*resbufptr] = thisint->statemachine->curState;
  *resbufptr+=1;
}

/******************************************************************
 *
 * Build a message to be sent.  This should be used *ONLY* as a call
 * internal to the ipc_callout.c file!
 *
 ******************************************************************/
void ipc_callout_build_msg(struct interface_data *thisint, int *bufptr,
			   char *buffer, int bufsize, int msgtype, 
			   char *message)
{
  struct ipc_cmd *cmd;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    return;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    return;

  if (!xsup_assert((message != NULL), "message != NULL", FALSE))
    return;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = msgtype;
  cmd->len = strlen(message);

  *bufptr += sizeof(struct ipc_cmd);

  strcpy((char *)&buffer[*bufptr], message);

  *bufptr += strlen(message);
}

/****************************************************************
 *
 * Send an error message to a client.
 *
 ****************************************************************/
void ipc_callout_send_error(struct interface_data *thisint, int *bufptr,
			    char *buffer, int bufsize, char *message)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    return;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    return;

  if (!xsup_assert((message != NULL), "message != NULL", FALSE))
    return;

  ipc_callout_build_msg(thisint, bufptr, buffer, bufsize, ERROR_MSG, message);
}

/****************************************************************
 *
 * Send a notification to the client.
 *
 ****************************************************************/
void ipc_callout_send_notify(struct interface_data *thisint, int *bufptr,
			     char *buffer, int bufsize, char *message)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    return;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    return;

  if (!xsup_assert((message != NULL), "message != NULL", FALSE))
    return;

  ipc_callout_build_msg(thisint, bufptr, buffer, bufsize, NOTIFY, message);
}

/***************************************************************
 *
 * Set a temporary password.  This password will be used by the first EAP
 * method that needs it.  Once it has been used, the EAP method should
 * bzero, and free the memory, in order to keep the password from sitting
 * in memory too long.
 *
 ***************************************************************/
void ipc_callout_set_password(struct interface_data *thisint, int *bufptr,
			      char *buffer, int bufsize, char *resbuf, 
			      int *resbufptr)
{
  int assertfail = FALSE;
  struct ipc_cmd *cmd;

  if (!xsup_assert((thisint != NULL), "thisint == NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((bufptr != NULL), "bufptr == NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbuf != NULL), "resbuf == NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbufptr != NULL), "resbufptr == NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((((*bufptr) >= 0) && ((*bufptr < bufsize))), 
		   "(bufptr >= 0) && (bufptr < bufsize)", FALSE))
    assertfail = TRUE;

  if (assertfail == TRUE)
    {
      (*bufptr) = bufsize;
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC control version requested! ("
		   "Version : %d)\n", cmd->version);
      return;
    }

  if (cmd->attribute != TEMPPASSWORD)
    {
      debug_printf(DEBUG_NORMAL, "Invalid call to %s!\n", __FUNCTION__);
      *bufptr = bufsize;
      return;
    }

  if (cmd->getset != IPC_SET)
    {
      debug_printf(DEBUG_NORMAL, "Cannot GET a temp password value!\n");
      *bufptr = bufsize;
      return;
    }

  // If we already have a temp password, we need to get rid of it.
  if (thisint->tempPassword != NULL)
    {
      free(thisint->tempPassword);
      thisint->tempPassword = NULL;
    }
  
  thisint->tempPassword = (char *)malloc(cmd->len+1);
  if (thisint->tempPassword == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for temporary "
		   "password!\n");
      return;
    }

  bzero(thisint->tempPassword, cmd->len+1);
  *bufptr += sizeof(struct ipc_cmd);
  strcpy(thisint->tempPassword, (char *)&buffer[*bufptr]);

  *bufptr += strlen(thisint->tempPassword);

  cmd = (struct ipc_cmd *)&resbuf[*resbufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = TEMPPASSWORD;
  cmd->getset = IPC_RESPONSE;
  cmd->len = 1;
  
  *resbufptr += sizeof(struct ipc_cmd);

  resbuf[*resbufptr] = ACK;
  *resbufptr += 1;
}

/*****************************************************************
 *
 * Request/change the state for the interface state machine.
 *
 *****************************************************************/
void ipc_callout_set_auth_ctrl(struct interface_data *thisint, int *bufptr,
			       char *buffer, int bufsize, char *resbuf,
			       int *resbufptr)
{
  int assertfail = FALSE;
  struct ipc_cmd *cmd, *ocmd;

  // The following assertions are not fatal.  They will simply cause
  // us to ignore this request.
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE)) 
    assertfail = TRUE;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE)) 
    assertfail = TRUE;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbuf != NULL), "resbuf != NULL", FALSE)) 
    assertfail = TRUE;

  if (!xsup_assert((resbufptr != NULL), "resbufptr != NULL", FALSE)) 
    assertfail = TRUE;

  if (assertfail)
    {
      (*bufptr) = bufsize;
      return;
    }

  // Now that we know we should have valid data, let's get on with it!

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC control version requested! ("
		   "Version : %d)\n", cmd->version);
      *bufptr += bufsize;
      return;
    }

  if (cmd->attribute != AUTH_CTRL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid call to %s in %s at %d.\n",
		   __FUNCTION__, __FILE__, __LINE__);
      *bufptr += cmd->len;
      return;
    }

  // Start to build the answer.
  ocmd = (struct ipc_cmd *)&resbuf[*resbufptr];
  ocmd->version = IPC_VERSION_NUM;
  ocmd->attribute = AUTH_CTRL;

  if (cmd->getset == IPC_SET)
    {
      // Do our IPC set functions.
      if (cmd->len != 1)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid length for IPC command in file "
		       "%s, function %s(), line %d\n", __FILE__, __FUNCTION__,
		       __LINE__);
	  *bufptr = bufsize;
	  return;
	}

      // Move to the point that we have new information.
      *bufptr += sizeof(struct ipc_cmd);

      ocmd->getset = IPC_RESPONSE;
      ocmd->len = 1;
      *resbufptr += sizeof(struct ipc_cmd);

      switch (buffer[*bufptr])
	{
	case AUTH_STATE_STOPPED:
	  wireless_sm_change_state(AUTH_STOPPED, thisint);
	  resbuf[*resbufptr] = AUTH_STATE_STOPPED;
	  (*resbufptr) ++;
	  break;

	case AUTH_STATE_RESTART:
	case AUTH_STATE_START:
	  wireless_sm_change_state(ACTIVE_SCAN, thisint);
	  resbuf[*resbufptr] = AUTH_STATE_RUNNING;
	  (*resbufptr) ++;
	  break;

	default:
	  debug_printf(DEBUG_NORMAL, "Invalid request type in file %s, "
		       "function %s(), line %d.\n", __FILE__, __FUNCTION__,
		       __LINE__);
	  break;
	}
      (*bufptr) ++;
      return;
    }

  if (cmd->getset == IPC_GET)
    {
      // Return the current state.

      if (cmd->len != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid size for AUTH_CTRL request in "
		       "file %s, function %s(), line %d.\n", __FILE__,
		       __FUNCTION__, __LINE__);
	  *bufptr = bufsize;
	}

      // Move to the point that we have the new state.
      *bufptr += sizeof(struct ipc_cmd);
      
      ocmd->getset = IPC_RESPONSE;
      ocmd->len = 1;

      *resbufptr += sizeof(struct ipc_cmd);

      switch (wireless_sm_get_state())
	{
	case AUTH_STOPPED:
	  resbuf[*resbufptr] = AUTH_STATE_STOPPED;
	  (*resbufptr) ++;
	  break;

	case AUTH_RESTART:
	  resbuf[*resbufptr] = AUTH_STATE_RESTART;
	  (*resbufptr) ++;
	  break;

	default:
	  // It is impossible to have a "START" since it is basically the
	  // same as "RUNNING".
	  resbuf[*resbufptr] = AUTH_STATE_RUNNING;
	  (*resbufptr) ++;
	  break;
	}
      return;
    }

  // Otherwise...
  *bufptr += (sizeof(struct ipc_cmd) + cmd->len);
}

/*****************************************************************
 *
 * Change the state of the use of TNC for authentication.
 *
 *****************************************************************/
void ipc_callout_set_tnc(struct interface_data *thisint, int *bufptr,
			 char *buffer, int bufsize, char *resbuf,
			 int *resbufptr)
{
  struct ipc_cmd *cmd, *ocmd;
  struct config_network *net;
  int assertfail = FALSE;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbuf != NULL), "resbuf != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((resbufptr != NULL), "resbufptr != NULL", FALSE))
    assertfail = TRUE;

  if (assertfail)
    {
      (*bufptr) = bufsize;
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM)
    {
      debug_printf(DEBUG_NORMAL, "Invalid IPC control version requested! ("
                   "Version : %d)\n", cmd->version);
      *bufptr = bufsize;
      return;
    }

  if (cmd->attribute != TNCENABLE)
    {
      debug_printf(DEBUG_NORMAL, "Invalid call to %s!\n", __FUNCTION__);
      *bufptr = bufsize;
      return;
    }

  net = config_get_network_config();
  if (!net)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get current network config!\n");
      *bufptr = bufsize;
      return;
    }

  // Start to build the answer.
  ocmd = (struct ipc_cmd *)&resbuf[*resbufptr];
  ocmd->version = IPC_VERSION_NUM;
  ocmd->attribute = TNCENABLE;

  if (cmd->getset == IPC_SET)
    {
      // Do our IPC set functions.

      if (cmd->len != 1)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid length for IPC command!\n");
	  *bufptr = bufsize;
	  return;
	}

      // Move to the point that we have the new state.
      *bufptr += sizeof(struct ipc_cmd);

      ocmd->getset = IPC_RESPONSE;
      ocmd->len = 1;
      *resbufptr += sizeof(struct ipc_cmd);

      if (buffer[*bufptr] != 0)
	{
	  // Enable TNC
	  resbuf[*resbufptr] = 1;
	  SET_FLAG(net->flags, CONFIG_NET_USE_TNC);
	  debug_printf(DEBUG_NORMAL, "TNC support will now be used. (If "
		       "it was compiled in!)\n");
	}
      else
	{
	  // Disable TNC
	  resbuf[*resbufptr] = 0;
	  UNSET_FLAG(net->flags, CONFIG_NET_USE_TNC);
	  debug_printf(DEBUG_NORMAL, "TNC support will no longer be used. "
		       "(Even if it was compiled in!)\n");
	}
      (*bufptr) ++;
      (*resbufptr) ++;
      return;
    }

  if (cmd->getset == IPC_GET)
    {
      // Return the current state.

      if (cmd->len != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid size for \"Get TNC State\""
		       " command!\n");
	  *bufptr = bufsize;
	  return;
	}

      // Move to the point that we have the new state.
      *bufptr += sizeof(struct ipc_cmd);

      ocmd->getset = IPC_RESPONSE;
      ocmd->len = 1;

      *resbufptr += sizeof(struct ipc_cmd);

      if (TEST_FLAG(net->flags, CONFIG_NET_USE_TNC))
	{
	  debug_printf(DEBUG_INT, "Returning that TNC is enabled.\n");
	  resbuf[*resbufptr] = 1;
	  (*resbufptr) ++;
	} 
      else
	{
	  debug_printf(DEBUG_INT, "Returning that TNC is disabled.\n");
	  resbuf[*resbufptr] = 0;
	  (*resbufptr) ++;
	}
      return;
    }

  // Otherwise...
  *bufptr += (sizeof(struct ipc_cmd)+cmd->len);
}

/***********************************************************************
 *
 * Ask any attached clients for a password.  In this message, we will
 * also pass the EAP type that is requesting the password, and any
 * challenge string that the EAP type may need.
 *
 ***********************************************************************/
void ipc_callout_request_password(int *bufptr, char *buffer, int bufsize, 
				  char *eapname, char *challenge)
{
  struct ipc_cmd *cmd;
  int assertfail = FALSE;

  if (!xsup_assert((bufptr != NULL), "bufptr != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((buffer != NULL), "buffer != NULL", FALSE))
    assertfail = TRUE;

  if (!xsup_assert((eapname != NULL), "eapname != NULL", FALSE))
    assertfail = TRUE;

  if (assertfail == TRUE)
    {
      (*bufptr) = bufsize;
      return;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = PASSWORD;
  cmd->getset = IPC_GET;
  if (challenge != NULL)
    {
      cmd->len = strlen(eapname)+strlen(challenge)+2;  // The string, with a NULL.
    } else {
      cmd->len = strlen(eapname)+1;
    }

  if (((cmd->len) + sizeof(struct ipc_cmd)) > bufsize)
    {
      debug_printf(DEBUG_NORMAL, "Buffer not large enough to hold response!\n");
      return;
    }

  *bufptr += sizeof(struct ipc_cmd);

  bzero((char *)&buffer[*bufptr],cmd->len);
  strcpy((char *)&buffer[*bufptr], eapname);
  *bufptr += (strlen(eapname)+1);

  if (challenge != NULL)
    {
      strcpy((char *)&buffer[*bufptr], challenge);
      *bufptr += (strlen(challenge)+1);
    }
}
