/*******************************************************************
 * Handle keying for WPA keys.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eapol_key_type254.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapol_key_type254.c,v 1.65 2006/06/01 22:49:49 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:49 $
 * $Log: eapol_key_type254.c,v $
 * Revision 1.65  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.64  2006/05/29 04:17:57  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.63  2006/05/28 21:24:04  chessing
 * Small fix to madwifi driver for some other changes that were made in the code.
 *
 * Revision 1.62  2006/05/26 23:21:12  chessing
 * Fixed some memory leaks.
 *
 * Revision 1.61  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.60  2006/05/25 05:20:44  chessing
 * Lots more WPA/WPA2 fixes.
 *
 * Revision 1.57  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.56  2006/05/17 03:56:09  chessing
 * Updates to WPA2.
 *
 * Revision 1.55  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.54  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.53  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.52  2006/04/06 06:39:47  chessing
 * A few fixes for bugs found at Interop's Hot Stage.  We now work with madwifi-ng.
 *
 * Revision 1.51  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.50  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.49  2006/01/23 07:13:49  chessing
 * A couple of fixups to the WPA and WPA2 code.
 *
 * Revision 1.48  2006/01/15 06:56:29  chessing
 * Some fixes to WPA/WPA2.  WPA is working correctly now.  WPA2 needs a little work to correctly identify key frames that are for the GTK.
 *
 * Revision 1.47  2006/01/12 17:54:02  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.46  2006/01/03 04:02:35  chessing
 * Added the ability to store the PEAP password in a hashed format.  (Basically, an MS-CHAPv1 hash.)  Also added an 'ntpwdhash' program to the tools directory that will convert a cleartext password in to a hash that can be copied to the configuration file.
 *
 * Revision 1.45  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.44  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.43  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.42  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.41  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 ******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "profile.h"
#include "xsupconfig.h"
#include "key_statemachine.h"
#include "eapol_key_type254.h"
#include "eapol.h"
#include "frame_structs.h"
#include "wpa.h"
#include "wpa_common.h"
#include "psk.h"
#include "mic.h"
#include "snmp.h"
#include "config_ssid.h"
#include "cardif/cardif.h"
#include "eap_types/mschapv2/mschapv2.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

uint8_t group_key_ver = 0, pairwise_key_ver = 0;

/**************************************************************
 *
 * Given a frame, parse all of the data that is contained in it, and
 * provide a human readable output that is useful for debugging.
 *
 **************************************************************/
void eapol_key_type254_dump(char *framedata)
{
  uint16_t value16=0;
  int need_comma = 0;
  struct wpa_key_packet *keydata;

  if (!xsup_assert((framedata != NULL), "framedata != NULL", FALSE))
    return;

  keydata = (struct wpa_key_packet *)&framedata[OFFSET_TO_EAPOL+4];

  debug_printf(DEBUG_INT, "Key Descriptor      = %d\n", keydata->key_descriptor);
  memcpy(&value16, keydata->key_information, 2);
  debug_printf(DEBUG_INT, "Key Information     = %04X  (Flags : ", ntohs(value16));
  if (ntohs(value16) & WPA_PAIRWISE_KEY)
    {
      debug_printf_nl(DEBUG_INT, "Pairwise Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_INSTALL_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Install Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_KEY_ACK_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Key Ack");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_KEY_MIC_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "MIC");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_SECURE_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Secure");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_ERROR_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Error");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA_REQUEST_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Request");
      need_comma = 1;
    }

  debug_printf_nl(DEBUG_INT, ")\n");

  switch (ntohs(value16) & WPA_KEYTYPE_MASK)
    {
    case 1:
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-MD5 for MIC and RC4 for encryption.\n");
      break;

    case 2:
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-SHA1-128 for MIC and AES for encryption.\n");
      break;
    }

  debug_printf(DEBUG_INT, "Key Length          = %d\n", 
	       ntohs(keydata->key_length));

  debug_printf(DEBUG_INT, "Key Replay Counter  = ");  
  debug_hex_printf(DEBUG_INT, keydata->key_replay_counter, 8);

  debug_printf(DEBUG_INT, "Key NONCE           = ");
  debug_hex_printf(DEBUG_INT, keydata->key_nonce, 32);

  debug_printf(DEBUG_INT, "Key IV              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_iv, 16);
  
  debug_printf(DEBUG_INT, "Key RSC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_rsc, 8);
  
  debug_printf(DEBUG_INT, "Key ID              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_id, 8);

  debug_printf(DEBUG_INT, "Key MIC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_mic, 16);

  value16 = ntohs(keydata->key_material_len);
  debug_printf(DEBUG_INT, "Key Material Length = %d\n", value16);

  if (value16 > 0)
    {
      debug_printf(DEBUG_INT, "Key Data : (%d)\n", value16);
      debug_hex_dump(DEBUG_INT, keydata->keydata, value16);
    }
}

/*******************************************************
 *
 * Generate the pre-Temporal key. (PTK) Using the authenticator, and 
 * supplicant nonces.  (Anonce, and Snonce.)  The PTK is used for keying
 * when we are ready.
 *
 *******************************************************/
char *eapol_key_type254_gen_ptk(struct interface_data *intdata, char *Anonce)
{
  char prfdata[76];  // 6*2 (MAC addrs) + 32*2 (nonces)
  char *retval;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return NULL;

  if (!xsup_assert((Anonce != NULL), "Anonce != NULL", FALSE))
    return NULL;

  bzero((char *)&prfdata, 76);

  debug_printf(DEBUG_INT, "Dest MAC: ");
  debug_hex_printf(DEBUG_INT, intdata->dest_mac, 6);

  if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac, 6) < 0)
    {
      memcpy((char *)&prfdata[0], (char *)&intdata->source_mac, 6);
      memcpy((char *)&prfdata[6], (char *)&intdata->dest_mac, 6);
    } else if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac,
		      6) > 0)
      {
	memcpy((char *)&prfdata[0], (char *)&intdata->dest_mac, 6);
	memcpy((char *)&prfdata[6], (char *)&intdata->source_mac, 6);
      } else {
	debug_printf(DEBUG_NORMAL, "Source and Destination MAC addresses "
		     "match!  The PTK won't be valid!\n");
	return NULL;
      }

  if (memcmp(intdata->statemachine->SNonce, Anonce, 32) < 0)
    {
      memcpy((char *)&prfdata[12], intdata->statemachine->SNonce, 32);
      memcpy((char *)&prfdata[44], Anonce, 32);
    } else if (memcmp(intdata->statemachine->SNonce, Anonce, 32) > 0)
      {
	memcpy((char *)&prfdata[12], Anonce, 32);
	memcpy((char *)&prfdata[44], intdata->statemachine->SNonce, 32);
      } else {
	debug_printf(DEBUG_NORMAL, "ANonce and SNonce match!  The PTK won't"
		     " be valid!\n");
	return NULL;
      }
  
  retval = (char *)malloc(80);
  if (retval == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for retval in %s "
		   "at %d!\n", __FUNCTION__, __LINE__);
      return NULL;
    }

  debug_printf(DEBUG_INT, "PMK : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) intdata->statemachine->PMK, 32);
  wpa_PRF((uint8_t *) intdata->statemachine->PMK, 32, (uint8_t *) "Pairwise key "
	  "expansion", 22, (uint8_t *)&prfdata, 76, (uint8_t *) retval, 64);

  debug_printf(DEBUG_INT, "PTK : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) retval, 64);

  return retval;
}

/*****************************************************************
 *
 * When a MIC failure occures, we need to send the AP a request for
 * a new key. (Reference 802.11i-D3.0.pdf page 43, line 8)
 *
 *****************************************************************/
void eapol_key_type254_request_new_key(struct interface_data *intdata, 
				       char unicast)
{
  struct wpa_key_packet *outkeydata;
  uint16_t value16, keyindex, len;
  int eapolver;
  char key[16];
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  value16 = (WPA_REQUEST_FLAG | WPA_ERROR_FLAG);

  if (unicast == 1)
    {
      // Set the key flags to indicate this is a pairwise key, with an
      // index of 0.
      keyindex = (WPA_PAIRWISE_KEY | pairwise_key_ver);
    } else {
      // Set the key flags to indicate this is a group key. We don't know the
      // index, so for now we will use 1.
      keyindex = ((1 << 4) | group_key_ver);
    }

  value16 = (value16 | keyindex);

  memcpy(outkeydata->key_information, &value16, 2);

  // Build the response.
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 
  
  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available!  We will not be "
		   "able to request a new key!\n");
      return;
    }

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
}

/****************************************************************
 *
 * When we have completed the PTK piece, and the pairwise key has been
 * applied to the interface, we need to get the group key.  The authenticator
 * will send us a group key that is encrypted.  We should decrypt it, apply
 * it to our interface, and send the authenticator a message to let it know
 * that we have a group key.
 *
 ****************************************************************/
void eapol_key_type254_do_gtk(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t value16, keyflags, version, keyindex, len;
  int eapolver;
  unsigned char *keydata = NULL;
  char key[16], rc4_ek[32];
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // First, make sure that the inkeydata replay counter is higher than
  // the last counter we saw.
  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) &&
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      debug_printf(DEBUG_INT, "Recieved counter : ");
      debug_hex_printf(DEBUG_INT, inkeydata->key_replay_counter, 8);
      debug_printf(DEBUG_INT, "Our counter : ");
      debug_hex_printf(DEBUG_INT, intdata->statemachine->replay_counter, 8);
      intdata->recv_size = 0;
      return;
    }

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;
  group_key_ver = version;
  keyindex = ((value16 & WPA_KEY_INDEX) >> 4);

  // Verify that our flags are correct.  (We don't check for the install flag,
  // for now.  Since the WPA spec doesn't explicitly require that the install
  // flag be set.)
  if (!((value16 & WPA_KEY_ACK_FLAG) &&
	(value16 & WPA_KEY_MIC_FLAG) && (value16 & WPA_SECURE_FLAG)))
    {
      debug_printf(DEBUG_NORMAL, "Invalid flags in GTK message 1!\n");
      return;
    }

  value16 = ntohs(inkeydata->key_material_len);

  keydata = (unsigned char *)malloc(value16+8);
  if (keydata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for key data!\n");
      return;
    }

  memcpy(keydata, 
	 &intdata->recvframe[OFFSET_TO_EAPOL+4+sizeof(struct wpa_key_packet)],
	 value16);

  debug_printf(DEBUG_INT, "Setting GTK! (Version : %d  Length : %d  Slot : %d)"
	       "\n", version, value16, keyindex);

  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available!  You will probably"
		   " not be able to pass data.\n");
      return;
    }

  switch (version)
    {
    case 1:
      // Decrypt the GTK.
      bzero(rc4_ek, 32);
      memcpy(rc4_ek, inkeydata->key_iv, 16);
      memcpy(&rc4_ek[16], &intdata->statemachine->PTK[16], 16);
      rc4_skip((uint8_t *) rc4_ek, 32, 256, keydata, value16);
      
      debug_printf_nl(DEBUG_INT, "GTK (%d) : ", value16);
      debug_hex_printf(DEBUG_INT, keydata, value16);


      wpa_common_set_key(intdata, NULL, keyindex, FALSE, (char *)keydata,
			 value16);
      break;

    case 2:
      // First, decrypt the GTK
      bzero(key, 16);
      aes_unwrap((uint8_t *) &intdata->statemachine->PTK[16], (value16-8)/8, 
		 keydata, (uint8_t *) key);

      
      wpa_common_set_key(intdata, NULL, keyindex, FALSE, (char *)keydata,
			 (value16 -8));
      break;
    }

  debug_printf(DEBUG_NORMAL, "Set new group WPA key.\n");

  // Build the response.
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  value16 = ((version | (keyindex << 4)) | WPA_KEY_MIC_FLAG | WPA_SECURE_FLAG);
  value16 = htons(value16);
  memcpy(&outkeydata->key_information, &value16, 2);

  outkeydata->key_length = inkeydata->key_length;
  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 
  
  memcpy(&key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type254_dump((char *) intdata->sendframe);

  if (network_data->methods->method_num == WPA_PSK)
    {
      // If we are using PSK, and we made it here, then we are in 
      // AUTHENTICATED state.
      intdata->statemachine->eapSuccess = TRUE;
    }

  // Drop unencrypted frames.
  cardif_drop_unencrypted(intdata, 1);

  if (keydata)
    {
      free(keydata);
      keydata = NULL;
    }
}

/***************************************************************
 *
 * Handle the first packet in the four-way handshake.
 *
 ***************************************************************/
void eapol_key_type254_do_type1(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t keyflags, len, value16;
  int i, version, eapolver, ielen;
  char key[16], wpa_ie[26];
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // First, make sure that the inkeydata replay counter is higher than
  // the last counter we saw.
  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) &&
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      intdata->recv_size = 0;
      return;
    }

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa_key_packet));

  // XXX Need to do this better.  Tie it in with Nonce code from SIM/AKA.
  for (i=0;i<32;i++)
    {
      outkeydata->key_nonce[i] = rand();
    }
  intdata->statemachine->SNonce = (char *)malloc(32);
  if (intdata->statemachine->SNonce == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for SNonce in "
		   "%s at %d!\n", __FUNCTION__, __LINE__);
      return;
    }
  memcpy(intdata->statemachine->SNonce, (char *)&outkeydata->key_nonce[0],
	 32);

  // Calculate the PTK.
  if (intdata->statemachine->PTK != NULL)
    {
      free(intdata->statemachine->PTK);
      intdata->statemachine->PTK = NULL;
    }
  intdata->statemachine->PTK = eapol_key_type254_gen_ptk(intdata,
							 (char *)&inkeydata->key_nonce);

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;
  pairwise_key_ver = version;
  keyflags |= (WPA_PAIRWISE_KEY | WPA_KEY_MIC_FLAG);
  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  cardif_get_wpa_ie(intdata, wpa_ie, &ielen);

  memcpy(&intdata->sendframe[OFFSET_TO_EAPOL+4+sizeof(struct wpa_key_packet)], 
	 &wpa_ie, ielen);
  value16 = ielen;
  value16 = htons(value16);
  intdata->send_size += ielen;

  outkeydata->key_material_len = value16;

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe);

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type254_dump((char *) intdata->sendframe);
}

/********************************************************
 *
 * Compare the IE that we got from a key packet to the IE that we got from
 * the AP, to see if they match.
 *
 ********************************************************/
char eapol_key_type254_cmp_ie(struct interface_data *intdata, uint8_t *wpaie, 
			      char wpaielen)
{
  uint8_t *apie, apielen;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((wpaie != NULL), "wpaie != NULL", FALSE))
    return XEMALLOC;

  debug_printf(DEBUG_INT, "WPA IE from Key Packet (%d) : ", wpaielen);
  debug_hex_printf(DEBUG_INT, wpaie, wpaielen);

  config_ssid_get_wpa_ie(&apie, &apielen);
  debug_printf(DEBUG_INT, "WPA IE from AP Scan (%d)    : ", apielen);
  debug_hex_printf(DEBUG_INT, apie, apielen);

  if (wpaielen != apielen)
    {
      debug_printf(DEBUG_NORMAL, "IE from the AP and IE from the key messages"
		   " are different lengths!\n");
      free(apie);
      return -1;
    }

  if (memcmp(wpaie, apie, apielen) != 0)
    {
      debug_printf(DEBUG_NORMAL, "IE from the AP and IE from the key messages"
		   " do not match!\n");
      free(apie);
      return -1;
    }

  return XENONE;
}

/********************************************************
 *
 * Handle the third packet in the 4 way handshake.  We should be able to
 * generate the pairwise key at this point.
 *
 ********************************************************/
void eapol_key_type254_do_type3(struct interface_data *intdata)
{
  struct wpa_key_packet *inkeydata, *outkeydata;
  uint16_t keyflags, len, value16, keyindex;
  int eapolver, version;
  char key[32];
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  bzero(key, 32);

  inkeydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // First, make sure that the inkeydata replay counter is higher than
  // the last counter we saw.
  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) &&
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      intdata->recv_size = 0;
      return;
    }

  // Update our replay counter.
  memcpy(intdata->statemachine->replay_counter, inkeydata->key_replay_counter,
	 8);

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL], 
	sizeof(struct wpa_key_packet)+28);

  outkeydata->key_descriptor = WPA_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);
  
  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  version = keyflags;  

  keyflags = 0;
  keyflags = (value16 & WPA_KEYTYPE_MASK);
  keyflags |= (WPA_PAIRWISE_KEY | WPA_KEY_MIC_FLAG);
  keyindex = ((value16 & WPA_KEY_INDEX) >> 4);

  // If the authenticator sets the secure flag, we need to do the same.
  if (value16 & WPA_SECURE_FLAG) keyflags |= WPA_SECURE_FLAG;
  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);
  memcpy(&outkeydata->key_nonce, intdata->statemachine->SNonce, 32);

  memcpy(outkeydata->key_nonce, intdata->statemachine->SNonce,
	 32);

  eapolver = eapol_get_eapol_ver();
  
  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 

  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available!  You will probably"
		   " not get valid keys!  (And traffic will probably not "
		   "flow correctly.)\n");
      return;
    }

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);  

  // Dump what we built.
  eapol_key_type254_dump((char *) intdata->sendframe);

  if (eapol_key_type254_cmp_ie(intdata, inkeydata->keydata, 
			       ntohs(inkeydata->key_material_len)) != XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Error comparing IEs.  Possible attack in "
		   "progress!  Disconnecting.\n");
      cardif_disassociate(intdata, DISASSOC_INVALID_IE);
      return;
    }

  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available.  We will not be "
		   "able to get valid keys.  (And traffic will not flow "
		   "properly.\n");
      return;
    }

  // Get TK1
  value16 = ntohs(inkeydata->key_length);
  memcpy(key, (char *)&intdata->statemachine->PTK[32], value16);
  
  debug_printf(DEBUG_INT, "TK1 : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) key, value16);
  
  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
  intdata->send_size = 0;

  debug_printf(DEBUG_INT, "Setting PTK1! (Index : %d Length : %d)\n", 
	       keyindex, value16);

  switch (version) 
    {
    case 1:
      wpa_common_set_key(intdata, intdata->dest_mac, keyindex, TRUE,
			 key, value16);
      break;

    case 2:
      wpa_common_set_key(intdata, intdata->dest_mac, keyindex, TRUE,
			 key, value16);
      break;
    }
  debug_printf(DEBUG_NORMAL, "Set new pairwise WPA key.\n");
}

void eapol_key_type254_determine_key(struct interface_data *intdata)
{
  struct wpa_key_packet *keydata;
  int keyflags, eapolver;
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  keydata = (struct wpa_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];  
  memcpy(&keyflags, keydata->key_information, 2);

  keyflags = ntohs(keyflags);

  if (keyflags & WPA_KEY_MIC_FLAG)
    {
      if (mic_wpa_validate((char *) intdata->recvframe, intdata->recv_size, 
			   intdata->statemachine->PTK, 16) == FALSE)
	{
	  intdata->statemachine->MICVerified = TRUE;
	  intdata->statemachine->IntegrityFailed = TRUE;
	  debug_printf(DEBUG_INT, "MIC failure!\n");
       	  return;   // Silently discard.
	} else {
	  intdata->statemachine->IntegrityFailed = FALSE;
	  intdata->statemachine->MICVerified = TRUE;
	}
    }

  if ((keyflags & WPA_PAIRWISE_KEY) && (keyflags & WPA_KEY_ACK_FLAG) &&
      (keyflags & WPA_KEY_MIC_FLAG) && (keyflags & WPA_INSTALL_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #3 (response) :\n");
      eapol_key_type254_do_type3(intdata);
    } else if ((keyflags & WPA_PAIRWISE_KEY) && (keyflags & WPA_KEY_ACK_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #1 (response) :\n");
      eapol_key_type254_do_type1(intdata);
    } else if ((keyflags & WPA_KEY_MIC_FLAG) && (keyflags & WPA_PAIRWISE_KEY))
    {
      debug_printf(DEBUG_NORMAL, "Got Key Packet #2 or #4!  (This shouldn't happen!)\n");
    } else if (!(keyflags & WPA_PAIRWISE_KEY))
      {
	// We have a group key packet.
	eapol_key_type254_do_gtk(intdata);
      }

  if (intdata->recv_size > 0)
    {
      eapolver = eapol_get_eapol_ver();

      eapol_build_header(EAPOL_KEY, (intdata->recv_size-OFFSET_TO_EAPOL-4), 
			 eapolver, (char *) intdata->recvframe);
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }      
}

/*************************************************************
 *
 * Process a WPA frame that we get from the authenticator.
 *
 *************************************************************/
void eapol_key_type254_process(struct interface_data *intdata)
{
  struct config_wpa_psk *psk;
  uint8_t *inframe;
  int insize;
  char tpmk[254];
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inframe = intdata->recvframe;
  insize = intdata->recv_size;

  debug_printf(DEBUG_INT, "Processing WPA key message!\n");

  eapol_key_type254_dump((char *)inframe);

  if (intdata->statemachine->PMK == NULL)
    {
      if (network_data->methods->method_num == WPA_PSK)
	{
	  psk = (struct config_wpa_psk *)network_data->methods->method_data;
	  if (psk->key)
	    {
	      if (intdata->cur_essid == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Unknown SSID, checking!\n");
		  intdata->cur_essid = (char *)malloc(99);
		  if (intdata->cur_essid == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! "
				   "(%s:%d)\n", __FUNCTION__, __LINE__);
		      return;
		    }

		  memset(intdata->cur_essid, 0x00, 99);

		  if (cardif_GetSSID(intdata, intdata->cur_essid) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
		      free(intdata->cur_essid);
		      return;
		    }
		}

	      // We have an ASCII password, so calculate it.
	      if (psk_wpa_pbkdf2(psk->key, (uint8_t *) intdata->cur_essid, 
				 strlen(intdata->cur_essid), (uint8_t *)&tpmk) 
		  == TRUE)
		{
		  if (intdata->statemachine->PMK)
		    {
		      free(intdata->statemachine->PMK);
		      intdata->statemachine->PMK = NULL;
		    }

		  intdata->statemachine->PMK = (char *)malloc(32);
		  if (intdata->statemachine->PMK == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for"
				   " intdata->statemachine->PMK in %s:%d!\n",
				   __FUNCTION__, __LINE__);
		      return;
		    }
		  
		  memcpy(intdata->statemachine->PMK, (char *)&tpmk, 32);
		}
	    } else {
	      // We have a hex key, we need to convert it from ASCII to real
	      // hex.
	      if (strlen(psk->hex_key) != 64)
		{
		  debug_printf(DEBUG_NORMAL, "Invalid HEX key defined for "
			       "WPA-PSK!\n");
		  return;
		}
	      process_hex(psk->hex_key, strlen(psk->hex_key), (char *)&tpmk);

	      if (intdata->statemachine->PMK)
		{
		  free(intdata->statemachine->PMK);
		  intdata->statemachine->PMK = NULL;
		}

	      intdata->statemachine->PMK = (char *)malloc(32);
	      if (intdata->statemachine->PMK == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for "
			       "intdata->statemachine->PMK in %s:%d!\n",
			       __FUNCTION__, __LINE__);
		  return;
		}
	    }
	} else {
	  debug_printf(DEBUG_NORMAL, "There is no PMK available!  WPA cannot"
		       " continue!\n");
	  return;
	}
    } 

  eapol_key_type254_determine_key(intdata);

  if (intdata->send_size > 0)
    {
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }

}
