/*******************************************************************
 * Linux wireless extensions interface.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux_wext.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux_wext.c,v 1.91 2006/10/08 03:42:53 chessing Exp $
 * $Date: 2006/10/08 03:42:53 $
 * $Log: cardif_linux_wext.c,v $
 * Revision 1.91  2006/10/08 03:42:53  chessing
 * Last batch of changes before 1.2.8.
 *
 * Revision 1.90  2006/10/05 22:33:28  chessing
 * Proprietary Cisco LEAP functionality is now available to drivers that support it.
 *
 * Revision 1.89  2006/10/05 22:23:50  chessing
 * Added new association option to the config file, and attempt to associate using methods other than open system.
 *
 * Revision 1.88  2006/09/27 19:37:19  chessing
 * Fixed a potential problem in there was a partial configuration for a network.
 *
 * Revision 1.87  2006/09/10 04:10:39  chessing
 * Defined __user so that it doesn't cause compile problems anymore.
 *
 * Revision 1.86  2006/08/28 04:31:54  chessing
 * A couple of fixes.
 *
 * Revision 1.85  2006/08/25 23:37:18  chessing
 * Numerous patches that have come in over the last month or two.
 *
 * Revision 1.84  2006/06/23 00:45:19  chessing
 * Completed basic implementation of operstate support.  Probably need to consider extending it to be used to populate the up/down state of the interface in the statemachine.
 *
 * Revision 1.83  2006/06/22 23:24:21  chessing
 * Added more operstate support.  It now works with the dhcpclient from http://www.flamewarmaster.de/software/dhcpclient/
 *
 * Revision 1.82  2006/06/20 18:17:18  chessing
 * Fix some endianness problems with WPA and WPA2 IE parsing.
 *
 * Revision 1.81  2006/06/02 21:56:49  chessing
 * One last small fix.
 *
 * Revision 1.80  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.79  2006/05/30 16:56:31  chessing
 * More patches from Carsten Grohmann and a #if that was missing from the 1.2.5 release.
 *
 * Revision 1.78  2006/05/29 04:17:58  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.77  2006/05/29 02:21:41  chessing
 * Madwifi now works with WPA/WPA2 and the wext driver.
 *
 * Revision 1.76  2006/05/28 22:32:23  chessing
 * More fixes to make madwifi behave itself.
 *
 * Revision 1.75  2006/05/28 21:24:04  chessing
 * Small fix to madwifi driver for some other changes that were made in the code.
 *
 * Revision 1.74  2006/05/27 23:12:31  chessing
 * More bug fixes that surround the madwifi driver.
 *
 * Revision 1.73  2006/05/26 23:21:12  chessing
 * Fixed some memory leaks.
 *
 * Revision 1.72  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.71  2006/05/24 04:36:30  chessing
 * More work on the GUI interface.  (We can now save.)  Fixed xsupconfwrite so that it doesn't write the firmware roaming option twice.
 *
 * Revision 1.70  2006/05/23 23:38:39  chessing
 * Patch to allow the madwifi driver to associate and authenticate using the regular wireless extensions.  Should this start working, we are dumpping the madwifi specific driver crap! ;)
 *
 * Revision 1.69  2006/05/23 23:07:01  chessing
 * Cleaned up the previous patch for timer_tick.  (There was a cleaner way of doing it.)  Fixed up some stuff that was causing issues getting associated/dynamic WEP authenticated with the madwifi driver.  Madwifi seems to be happily working (with dynamic WEP) now.
 *
 * Revision 1.68  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.67  2006/05/22 04:02:09  chessing
 * Some small changes/fixes.
 *
 * Revision 1.66  2006/05/17 22:18:10  chessing
 * A couple of small changes to Xsupplicant, and some major changes to the GUI configuration/monitor tool.
 *
 * Revision 1.65  2006/05/17 04:14:07  chessing
 * Fixed a scanning problem with older versions of wireless extensions.
 *
 * Revision 1.64  2006/05/17 03:56:09  chessing
 * Updates to WPA2.
 *
 * Revision 1.63  2006/05/17 02:40:07  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.62  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.61  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.60  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.59  2006/04/25 01:17:43  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.58  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.57  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.56  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.55  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.54  2006/01/14 06:22:10  chessing
 * Fixed WPA functionality for cards that don't use IWAUTH to set up the WPA IE.
 *
 * Revision 1.53  2006/01/12 17:54:03  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.52  2005/12/25 04:54:11  chessing
 * Fixup to code that added wext support needed for using ndiswrapper.
 *
 * Revision 1.51  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.50  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.49  2005/12/18 07:44:12  chessing
 * Added the ability to associate using IWAUTH options instead of setting a full IE.  This allows NDISwrapper to work with vanilla wireless extensions.  For some reason, we can't parse IWEVASSOCREQIE/IWEVASSOCRESPIE since we see the length as 0, even though iwevent sees the correct IE information. :-/  Need to figure out why.
 *
 * Revision 1.48  2005/12/03 22:18:51  chessing
 * Added an include to fix some problems when compiling with WE19.
 *
 * Revision 1.47  2005/11/14 17:27:39  chessing
 * Removed the driver specific code for hostap, ipw, and ndiswrapper, since they all support WE18 now.
 *
 * Revision 1.46  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.45  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.44  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.43  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.42  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.41  2005/09/17 18:38:17  chessing
 * Some updates to make the madwifi driver work with dynamic WEP again.
 *
 * Revision 1.40  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.39  2005/09/08 16:27:02  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.38  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.37  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.36  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

// Use kernel headers.
#define HEADERS_KERNEL

#ifndef __user
#define __user
#endif

#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/compiler.h>
#include <linux/wireless.h>
#include <iwlib.h>
#include <linux/if_packet.h>
#include <linux/netlink.h>

#include "profile.h"
#include "xsupconfig.h"
#include "config_ssid.h"
#include "xsup_common.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "wpa.h"
#include "wpa2.h"
#include "wpa_common.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "wireless_sm.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"
#include "timer.h"
#include "wpa.h"
#include "wpa2.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

// Old versions of wireless.h may not have this defined.
#ifndef IW_ENCODE_TEMP
#define IW_ENCODE_TEMP            0x0400
#endif

/**************************************************************
 *
 * Cancel our scan timer, if we get here before something else cancels
 * it for us. ;)
 *
 **************************************************************/
void cardif_linux_wext_cancel_scantimer(struct interface_data *ctx)
{
  timer_cancel(SCANCHECK_TIMER);
}

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_linux_wext_scan(struct interface_data *thisint, char passive)
{
  struct lin_sock_data *sockData;
  struct iwreq iwr;
  //  char all4s[6]={0x44, 0x44, 0x44, 0x44, 0x44, 0x44};
#if WIRELESS_EXT > 17
  struct iw_scan_req iwsr;
#endif
  struct config_globals *globals;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEGENERROR;

  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      debug_printf(DEBUG_INT, "%s is not a wireless interface!\n", 
		   thisint->intName);
      return XENOWIRELESS;
    }

  memset(&iwr, 0x00, sizeof(iwr));

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  cardif_linux_wext_wpa_state(thisint, 1);

  debug_printf(DEBUG_INT, "Issuing %s scan request for interface %s!\n",
	       passive ? "passive":"active", thisint->intName);

#if WIRELESS_EXT > 17  
  // Build our extended scan structure.
  memset(&iwsr, 0x00, sizeof(iwsr));

  if (passive)
    {
      iwsr.scan_type = IW_SCAN_TYPE_PASSIVE;
      //      iwsr.scan_type = IW_SCAN_TYPE_ACTIVE;

      // If we are doing a passive scan, then we only care about other APs
      // that are on this SSID.  Otherwise, we might end up picking an SSID
      // later that isn't in the same layer2/3 space.
      //      iwr.u.data.flags = IW_SCAN_THIS_ESSID | IW_SCAN_ALL_FREQ | 
      //	IW_SCAN_THIS_MODE | IW_SCAN_ALL_RATE;

      //      iwr.u.data.flags = IW_SCAN_DEFAULT;
      iwr.u.data.flags = IW_SCAN_THIS_ESSID;
    }
  else
    {
      // Some cards won't do a full scan if they are associated.
      //      cardif_linux_wext_set_bssid(thisint, all4s);

      iwsr.scan_type = IW_SCAN_TYPE_ACTIVE;
      iwr.u.data.flags = IW_SCAN_DEFAULT;
    }

  // We aren't looking for a specific BSSID.
  memset(iwsr.bssid.sa_data, 0xff, 6);

  iwr.u.data.length = sizeof(iwsr);
  iwr.u.data.pointer = (caddr_t) &iwsr;

#else
  iwr.u.data.length = 0;
  iwr.u.data.pointer = NULL;
  iwr.u.data.flags = IW_SCAN_DEFAULT;
#endif
  
  strcpy((char *)&iwr.ifr_name, thisint->intName);

  if (ioctl(sockData->sockInt, SIOCSIWSCAN, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error with SCAN ioctl!  (Perhaps your card "
		   "doesn't support scanning, or isn't up?)\n");

      return -1;
    }

  SET_FLAG(thisint->flags, SCANNING);

  globals = config_get_globals();
  
  if (!globals)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get a handle to the scan timeout "
		   "variable!  (Perhaps the configuration isn't "
		   "initalized?)\n");
      debug_printf(DEBUG_NORMAL, "Scanning will commence, but will only be "
		   "successful on cards that send scan complete events.\n");
      return XENONE;
    }

  timer_add_timer(SCANCHECK_TIMER, globals->assoc_timeout, 
		  cardif_linux_rtnetlink_scancheck, 
		  cardif_linux_wext_cancel_scantimer);

  return XENONE;
}

/*************************************************************
 *
 * Set all of the keys to 0s.
 *
 *************************************************************/
void cardif_linux_wext_set_zero_keys(struct interface_data *thisint)
{
  char zerokey[13];
  char keylen = 13;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  debug_printf(DEBUG_INT, "Setting keys to zeros!\n");

  memset(zerokey, 0x00, 13);

  // We set the key index to 0x80, to force key 0 to be set to all 0s,
  // and to have key 0 be set as the default transmit key.
  cardif_set_wep_key(thisint, (uint8_t *)&zerokey, keylen, 0x80);
  cardif_set_wep_key(thisint, (uint8_t *)&zerokey, keylen, 0x01);
  cardif_set_wep_key(thisint, (uint8_t *)&zerokey, keylen, 0x02);
  cardif_set_wep_key(thisint, (uint8_t *)&zerokey, keylen, 0x03);
}

/**************************************************************
 *
 * If we have detected, or forced this interface to reset keys, then
 * we need to reset them.  Otherwise, we will just ignore the fact that
 * we changed APs, and return.
 *
 **************************************************************/
void cardif_linux_wext_zero_keys(struct interface_data *thisint)
{
  struct config_network *network_data;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the "
		   "key!  Roaming may not work!!!\n");
      return;
    }

  if (TEST_FLAG(thisint->flags, ROAMED))
    {
      return;
    }

  SET_FLAG(thisint->flags, ROAMED);

  cardif_linux_wext_set_zero_keys(thisint);
  cardif_linux_wext_enc_open(thisint);
}

/**************************************************************
 *
 * Disable encryption on the wireless card.  This is used in cases
 * where we roam to a different AP and the card needs to have WEP
 * disabled.
 *
 **************************************************************/
int cardif_linux_wext_enc_disable(struct interface_data *thisint)
{
  int rc = 0;
  struct iwreq wrq;
  struct config_network *network_data;
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return XEBADCONFIG;

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the"
		   " key!  Roaming may not work!!!\n");
      return -1;
    }

  memset((struct iwreq *)&wrq, 0x00, sizeof(struct iwreq));

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  Strncpy(wrq.ifr_name, thisint->intName, sizeof(wrq.ifr_name));

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_DISABLED;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(sockData->sockInt, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption disabled!\n");
    }
 
  return rc;
}

/********************************************************************
 *
 * Create the WPA2 Information Element.
 *
 ********************************************************************/
int cardif_linux_wext_get_wpa2_ie(struct interface_data *thisint, 
				  char *iedata, int *ielen)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ielen != NULL), "ielen != NULL", FALSE))
    return XEMALLOC;

#if WIRELESS_EXT > 17

  // Should we use capabilities here?
  wpa2_gen_ie(thisint, iedata, ielen);

  debug_printf(DEBUG_INT, "Setting WPA2 IE : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *)iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA2 isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

/****************************************************************************
 *
 *  Generate the WPA1 Information Element
 *
 ****************************************************************************/
int cardif_linux_wext_get_wpa_ie(struct interface_data *thisint, 
				 char *iedata, int *ielen)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ielen != NULL), "ielen != NULL", FALSE))
    return XEMALLOC;

#if WIRELESS_EXT > 17

  wpa_gen_ie(thisint, iedata);
  *ielen = 24;

  debug_printf(DEBUG_INT, "Setting WPA IE : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *)iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");
#else
  debug_printf(DEBUG_NORMAL, "WPA isn't implemented in this version of the "
	       "wireless extensions!  Please upgrade to the latest version "
	       "of wireless extensions, or specify the driver to use with the"
	       " -D option!\n");

  iedata = NULL;
  *ielen = 0;
#endif
  return XENONE;
}

/**************************************************************
 *
 * Set encryption to open on the wireless card.
 *
 **************************************************************/
int cardif_linux_wext_enc_open(struct interface_data *thisint)
{
  int rc = 0;
  struct iwreq wrq;
  struct config_network *network_data;
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return XEMALLOC;

  if (network_data->wireless_ctrl == CTL_NO) 
    {
      debug_printf(DEBUG_INT, "Config file has instructed us not to reset the "
		   "key!  Roaming may not work!!!\n");
      return -1;
    }

  memset((struct iwreq *)&wrq, 0x00, sizeof(struct iwreq));

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  Strncpy(wrq.ifr_name, thisint->intName, sizeof(wrq.ifr_name));

  // We got some data, so see if we have encryption or not.
  wrq.u.encoding.flags = IW_ENCODE_OPEN;
  wrq.u.encoding.length = 0;
  wrq.u.encoding.pointer = (caddr_t)NULL;

  rc = ioctl(sockData->sockInt, SIOCSIWENCODE, &wrq);
  if (rc < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't disable encryption!\n");
    } else {
      debug_printf(DEBUG_INT, "Encryption set to Open!\n");
    }
 
  return rc;
}


/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_linux_wext_set_WEP_key(struct interface_data *thisint, uint8_t *key, 
				  int keylen, int index)
{
  int rc = 0;
  int settx = 0;
  struct iwreq wrq;
  struct config_network *network_data;
  struct lin_sock_data *sockData;
  char seq[6] = {0x00,0x00,0x00,0x00,0x00,0x00};
  char addr[6] = {0xff,0xff,0xff,0xff,0xff,0xff};

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (index & 0x80) settx = 1;

#if WIRELESS_EXT > 17
  rc = cardif_linux_wext_set_key_ext(thisint, IW_ENCODE_ALG_WEP, addr,
				     (index & 0x7f), settx, seq, 6, key,
				     keylen);


  if (rc == XENONE) return rc;
#endif

  debug_printf(DEBUG_INT, "Couldn't use extended key calls to set keys. \n");
  debug_printf(DEBUG_INT, "Trying old method.\n");

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return XEMALLOC;

  memset(&wrq, 0x00, sizeof(wrq));

  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      if ((cardif_int_is_wireless(thisint->intName) != TRUE) ||
	  (network_data->type == WIRED) ||
	  (network_data->wireless_ctrl == CTL_NO))
	{
	  debug_printf(DEBUG_NORMAL, "Interface isn't wireless, but an attempt"
		       " to set a key was made!\n");
	  return XENOWIRELESS;
	} else {
	  thisint->flags |= IS_WIRELESS;
	}
    }

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  Strncpy(wrq.ifr_name, thisint->intName, sizeof(wrq.ifr_name));

  wrq.u.data.flags = ((index & 0x7f)+1);

  if (TEST_FLAG(thisint->flags, DONT_USE_TEMP))
    wrq.u.data.flags |= IW_ENCODE_OPEN;
  else
    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

  wrq.u.data.length = keylen;
  wrq.u.data.pointer = (caddr_t)key;

  if ((rc = ioctl(sockData->sockInt, SIOCSIWENCODE, &wrq)) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Failed to set WEP key [%d], error %d : %s\n",
		   (index & 0x7f) + 1, errno, strerror(errno));

      rc = XENOKEYSUPPORT;
    } else {
      debug_printf(DEBUG_INT, "Successfully set WEP key [%d]\n",
		   (index & 0x7f)+1);

      if (index & 0x80)
	{
	  // This is a unicast key, use it for transmissions.
	  Strncpy(wrq.ifr_name, thisint->intName, sizeof(wrq.ifr_name));

	  wrq.u.data.flags = (((index & 0x7f) + 1) & IW_ENCODE_INDEX) | IW_ENCODE_NOKEY;
	  if (TEST_FLAG(thisint->flags, DONT_USE_TEMP))
	    wrq.u.data.flags |= IW_ENCODE_OPEN;
	  else
	    wrq.u.data.flags |= IW_ENCODE_OPEN | IW_ENCODE_TEMP;

	  wrq.u.data.length = 0;
	  wrq.u.data.pointer = (caddr_t)NULL;

	  if (ioctl(sockData->sockInt, SIOCSIWENCODE, &wrq) < 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed to set the WEP transmit key ID [%d]\n", (index & 0x7f)+1);
	      rc = XENOKEYSUPPORT;
	    } else {
	      debug_printf(DEBUG_INT, "Successfully set the WEP transmit key [%d]\n", (index & 0x7f)+1);
	    }
	}  
    }
 
  return rc;
}

/**********************************************************
 *
 * Set the SSID of the wireless card.
 *
 **********************************************************/
int cardif_linux_wext_set_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEGENERROR;

  if (!xsup_assert((ssid_name != NULL), "ssid_name != NULL", FALSE))
    return XEGENERROR;

  memset(&iwr, 0x00, sizeof(iwr));

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  if (sockData->sockInt <= 0)
    return XENOSOCK;

  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      if (!TEST_FLAG(thisint->flags, WAS_DOWN))
	return XENOWIRELESS;

      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  SET_FLAG(thisint->flags, IS_WIRELESS);
	} else {
	  UNSET_FLAG(thisint->flags, IS_WIRELESS);
	}

      if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	{
	  UNSET_FLAG(thisint->flags, WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  Strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  memset(newssid, 0x00, 100);
  strcpy(newssid, ssid_name);

  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = strlen(newssid);

  // Starting in WE 21, we don't NULL terminate SSIDs we set.
  if (cardif_linux_rtnetlink_get_we_ver(thisint) < 21)
    iwr.u.essid.length++;

  iwr.u.essid.flags = 1;

  if (ioctl(sockData->sockInt, SIOCSIWESSID, &iwr) < 0) return XENOWIRELESS;

  // Allow us to correlate SSID set events.
  SET_FLAG(thisint->flags, SSID_SET);

  debug_printf(DEBUG_INT, "Requested SSID be set to '%s'\n", newssid);

  UNSET_FLAG(thisint->flags, WAS_DOWN);

  return XENONE;
}

/******************************************
 *
 * Set the Broadcast SSID (MAC address) of the AP we are connected to.
 *
 ******************************************/
int cardif_linux_wext_set_bssid(struct interface_data *intdata, uint8_t *bssid)
{
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEGENERROR;

  if (!xsup_assert((bssid != NULL), "bssid != NULL", FALSE))
    return XEGENERROR;

  debug_printf(DEBUG_INT, "Setting BSSID : ");
  debug_hex_printf(DEBUG_INT, bssid, 6);

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  memset(&wrq, 0x00, sizeof(wrq));

  Strncpy((char *)&wrq.ifr_name, intdata->intName, sizeof(wrq.ifr_name));
  memcpy(&wrq.u.ap_addr.sa_data, bssid, 6);
  wrq.u.ap_addr.sa_family = ARPHRD_ETHER;

  if (ioctl(sockData->sockInt, SIOCSIWAP, &wrq) < 0)
    {
      // If we couldn't set the BSSID, it isn't the end of the world.  The
      // driver just may not need it.
      debug_printf(DEBUG_NORMAL, "Error setting BSSID!  We may not associate/"
      		   "authenticate correctly!\n");
      return XESOCKOP;
    }

  return XENONE;
}


/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_linux_wext_get_bssid(struct interface_data *thisint, 
				char *bssid_dest)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((bssid_dest != NULL), "bssid_dest != NULL", FALSE))
    return XEMALLOC;

  // If we are a wired interface, don't bother.
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS)) return XENONE;

  sockData = thisint->sockData;
  
  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  memset(&iwr, 0x00, sizeof(iwr));

  // Specify the interface name we are asking about.
  Strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  if (ioctl(sockData->sockInt, SIOCGIWAP, &iwr) < 0) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get MAC address for AP!\n");
      return XENOWIRELESS;
    }

  memcpy(bssid_dest, iwr.u.ap_addr.sa_data, 6);
  return XENONE;
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_linux_wext_get_ssid(struct interface_data *thisint, char *ssid_name)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  char newssid[100];

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ssid_name != NULL), "ssid_name != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  memset(&iwr, 0x00, sizeof(iwr));

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);
 
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      // We want to verify that the interface is in fact, not wireless, and
      // not that we are in a situation where the interface has just been 
      // down.
      debug_printf(DEBUG_NORMAL, "This interface isn't wireless!\n");
      return XENOWIRELESS;
    } 

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  SET_FLAG(thisint->flags, IS_WIRELESS);
	} else {
	  UNSET_FLAG(thisint->flags, IS_WIRELESS);
	}

      if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	{
	  UNSET_FLAG(thisint->flags, WAS_DOWN);
	}
    }

  // Specify the interface name we are asking about.
  Strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  memset(newssid, 0x00, IW_ESSID_MAX_SIZE+1);
  iwr.u.essid.pointer = (caddr_t) newssid;
  iwr.u.essid.length = 100;
  iwr.u.essid.flags = 0;

  if (ioctl(sockData->sockInt, SIOCGIWESSID, &iwr) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
      debug_printf(DEBUG_NORMAL, "Error (%d) : %s\n", errno, strerror(errno));
      return XENOWIRELESS;
    }

  UNSET_FLAG(thisint->flags, WAS_DOWN);

  Strncpy(ssid_name, newssid, iwr.u.essid.length);

  return XENONE;
}

/***********************************************************************
 *
 *  This function is called when we roam, or disassociate.  It should
 *  reset the card to a state where it can associate with a new AP.
 *
 ***********************************************************************/
int cardif_linux_wext_wep_associate(struct interface_data *intdata, 
				    int zero_keys)
{
  uint8_t *bssid;
  struct config_globals *globals;
  struct config_network *netdata;
  uint32_t alg;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  cardif_linux_wext_wpa_state(intdata, 0);

#if WIRELESS_EXT > 17
  // Determine the type of association we want, and set it up if we are using
  // the proper versions of WEXT.

  netdata = config_get_network_config();
  if (netdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't determine network configuration "
		   "information.  You may not be able to associate correctly!\n");
      return XEBADCONFIG;
    }

  switch (netdata->assoc_type)
    {
    case ASSOC_OPEN:
      alg = IW_AUTH_ALG_OPEN_SYSTEM;
      break;

    case ASSOC_SHARED:
      alg = IW_AUTH_ALG_SHARED_KEY;
      break;

    case ASSOC_LEAP:
      alg = IW_AUTH_ALG_LEAP;
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown 802.11 authetication alg.  Defaulting "
		   "to Open System.\n");
      alg = IW_AUTH_ALG_OPEN_SYSTEM;
      break;
    }

  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_80211_AUTH_ALG,
                                   alg, "802.11 auth. alg to open") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set 802.11 auth. alg.\n");
    }
#endif

  cardif_linux_wext_set_ssid(intdata, intdata->cur_essid);

  if (zero_keys == 0)
    {
      debug_printf(DEBUG_INT, "WEP: turning encryption off.\n");
      return cardif_linux_wext_enc_disable(intdata);
    } else if (zero_keys == 1)
      {
      debug_printf(DEBUG_INT, "WEP: zero keys.\n");
	cardif_linux_wext_zero_keys(intdata);
	return XENONE;
      } else
	{
	  debug_printf(DEBUG_NORMAL, "Invalid association value.\n");
	}

  bssid = config_ssid_get_mac();
  if (bssid != NULL)
    {
      debug_printf(DEBUG_INT, "Dest. BSSID : ");
      debug_hex_printf(DEBUG_INT, bssid, 6);
    }

  globals = config_get_globals();

  if ((!globals) || (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM)))
    {
      cardif_linux_wext_set_bssid(intdata, bssid);
    }

  return XENOTHING_TO_DO;
}

#if WIRELESS_EXT > 17
int cardif_linux_wext_mlme(struct interface_data *thisint, uint16_t mlme_type,
			   uint16_t mlme_reason)
{
  struct iwreq iwr;
  struct lin_sock_data *sockData;
  struct iw_mlme iwm;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  // If we get here, and isWireless == FALSE, then we need to double
  // check that our interface is really not wireless.
  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
    {
      if (cardif_int_is_wireless(thisint->intName) == TRUE)
	{
	  SET_FLAG(thisint->flags, IS_WIRELESS);
	} else {
	  UNSET_FLAG(thisint->flags, IS_WIRELESS);
	}

      if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	{
	  UNSET_FLAG(thisint->flags, WAS_DOWN);
	}
    }  

  memset(&iwr, 0, sizeof(iwr));
  // Specify the interface name we are asking about.
  Strncpy(iwr.ifr_name, thisint->intName, sizeof(iwr.ifr_name));

  memset(&iwm, 0, sizeof(iwm));
  
  // Set up our MLME struct.
  iwm.cmd = mlme_type;
  iwm.reason_code = mlme_reason;
  iwm.addr.sa_family = ARPHRD_ETHER;

  // Need to specify the MAC address that we want to do this MLME for.
  memcpy(iwm.addr.sa_data, thisint->source_mac, 6);
  iwr.u.data.pointer = (caddr_t)&iwm;
  iwr.u.data.length = sizeof(iwm);

  if (ioctl(sockData->sockInt, SIOCSIWMLME, &iwr))
    {
      debug_printf(DEBUG_NORMAL, "Couldn't issue MLME request!\n");
    }
  
  return XENONE;
}
#endif

int cardif_linux_wext_disassociate(struct interface_data *intdata, int reason)
{
#if WIRELESS_EXT > 17
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  cardif_linux_wext_mlme(intdata, IW_MLME_DISASSOC, reason);
  cardif_linux_wext_mlme(intdata, IW_MLME_DEAUTH, reason);
#endif
  return XENONE;
}

int cardif_linux_wext_set_key_ext(struct interface_data *intdata, int alg, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *seq,  int seqlen, char *key, 
				   int keylen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_encode_ext *iwee;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  memset(&wrq, 0x00, sizeof(wrq));
  Strncpy((char *)&wrq.ifr_name, intdata->intName, sizeof(wrq.ifr_name));

  // Allocate enough memory to hold our iw_encode_ext struct, and the
  // key itself.
  iwee = (struct iw_encode_ext *)malloc(sizeof(struct iw_encode_ext) + keylen);

  memset(iwee, 0x00, sizeof(struct iw_encode_ext));

  iwee->alg = alg;
  iwee->ext_flags = keyidx+1;

  if ((seq != NULL) && (seqlen > 0))
    {
      iwee->ext_flags |= IW_ENCODE_EXT_RX_SEQ_VALID;
      memcpy(iwee->rx_seq, seq, seqlen);
    }
  
  if (settx) 
    {
      iwee->ext_flags |= IW_ENCODE_EXT_SET_TX_KEY;
      if ((addr != NULL) && 
	  (memcmp(addr, "\0xff\0xff\0xff\0xff\0xff\0xff", 6) != 0))
	{
	  memcpy(iwee->addr.sa_data, addr, 6);
	} else {
	  memcpy(iwee->addr.sa_data, intdata->dest_mac, 6);
	}
    } else {
      iwee->ext_flags |= IW_ENCODE_EXT_GROUP_KEY;
      memset(iwee->addr.sa_data, 0xff, 6);
    }
  iwee->addr.sa_family = ARPHRD_ETHER;

  iwee->key_len = keylen;
  memcpy(iwee->key, key, keylen);

  if (key)
    {
      debug_printf(DEBUG_INT, "Key Index : %d   Length : %d\n", keyidx, keylen);
      debug_printf(DEBUG_INT, "Destination MAC : ");
      debug_hex_printf(DEBUG_INT, iwee->addr.sa_data, 6);

      debug_printf(DEBUG_INT, "Setting key : ");
      debug_hex_printf(DEBUG_INT, iwee->key, keylen);
    }

  wrq.u.encoding.pointer = (caddr_t)iwee;
  wrq.u.encoding.flags = (keyidx + 1);
  wrq.u.encoding.length = sizeof(struct iw_encode_ext) + keylen;

  if (ioctl(sockData->sockInt, SIOCSIWENCODEEXT, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting key!! (IOCTL "
		   "failure.)\n");
      debug_printf(DEBUG_NORMAL, "Error %d : %s\n", errno, strerror(errno));
    }

  if (iwee)
    {
      free(iwee);
      iwee = NULL;
    }

#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif

  return XENONE;
}


int cardif_linux_wext_set_tkip_key(struct interface_data *intdata, 
				   unsigned char *addr, int keyidx, int settx, 
				   char *key, int keylen)
{
#if WIRELESS_EXT > 17
  char seq[6] = {0x00,0x00,0x00,0x00,0x00,0x00};

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  // IEEE 802.11i section 8.3.2.6 says we should set the TSC to 1.
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_TKIP, addr,
				       keyidx, settx, seq, 6, key, keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_set_ccmp_key(struct interface_data *intdata,
				   unsigned char *addr, int keyidx, int settx,
				   char *key, int keylen)
{
#if WIRELESS_EXT > 17
  char seq[6] = {0x00,0x00,0x00,0x00,0x00,0x00};

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  // According to 802.11i, section 8.3.3.4.3e says we should set the PN to
  // 0 when a CCMP key is set. 
  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_CCMP, addr,
				       keyidx, settx, seq, 6, key,
				       keylen);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}

int cardif_linux_wext_wpa(struct interface_data *intdata, char state)
{
#if WIRELESS_EXT > 17
  int retval = 0;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  retval = cardif_linux_wext_set_iwauth(intdata, IW_AUTH_WPA_ENABLED,
					state, "change WPA state");

  if (retval == 0)
    {
      retval = cardif_linux_wext_set_iwauth(intdata, 
					    IW_AUTH_TKIP_COUNTERMEASURES,
					    FALSE, "TKIP countermeasures");

      if (retval == 0)
	{
	  retval = cardif_linux_wext_set_iwauth(intdata, 
						IW_AUTH_DROP_UNENCRYPTED,
						TRUE, "drop unencrypted");
	}
    }

  return retval;
#endif
  return XENONE;
}

int cardif_linux_wext_set_wpa_ie(struct interface_data *intdata, 
				 unsigned char *wpaie, unsigned int wpalen)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  memset(&wrq, 0x00, sizeof(wrq));

  Strncpy((char *)&wrq.ifr_name, intdata->intName, sizeof(wrq.ifr_name));

  wrq.u.data.pointer = (caddr_t) wpaie;
  wrq.u.data.length = wpalen;
  wrq.u.data.flags = 0;

  if (ioctl(sockData->sockInt, SIOCSIWGENIE, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting WPA IE!\n");
    } 
#endif
  return XENONE;
}


int cardif_linux_wext_wpa_state(struct interface_data *intdata, char state)
{

  // If we have wireless extensions 18 or higher, we can support WPA/WPA2
  // with standard ioctls.

#if WIRELESS_EXT > 17
  char wpaie[24];

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (state)
    {
      // Enable WPA if the interface doesn't already have it.
      cardif_linux_wext_wpa(intdata, TRUE);
    } else {
      cardif_linux_wext_wpa(intdata, FALSE);

      // Otherwise, make sure we don't have an IE set.
      memset(wpaie, 0x00, sizeof(wpaie));
      if (cardif_linux_wext_set_wpa_ie(intdata, (unsigned char *)wpaie, 0) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't clear WPA IE on device %s!\n",
		       intdata->intName);
	}
    }
#endif

  return XENONE;
}

/*************************************************************************
 *
 * Set values for IWAUTH.
 *
 *************************************************************************/
int cardif_linux_wext_set_iwauth(struct interface_data *intdata,
				 int setting, uint32_t value, 
				 char *setting_name)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  memset(&wrq, 0x00, sizeof(wrq));
  Strncpy((char *)&wrq.ifr_name, intdata->intName, sizeof(wrq.ifr_name));

  wrq.u.param.flags = setting & IW_AUTH_INDEX;
  wrq.u.param.value = value;

  if (ioctl(sockData->sockInt, SIOCSIWAUTH, &wrq) < 0)
    {
      if (errno != ENOTSUP)
	{
	  if (xsup_assert((setting_name != NULL), "setting_name != NULL",
			  FALSE))
	    {
	      debug_printf(DEBUG_NORMAL, "Error changing setting for '%s'! "
			   "It is possible that your driver does not support "
			   "the needed functionality.\n", setting_name);
	      debug_printf(DEBUG_NORMAL, "Error was (%d) : %s\n", errno,
			   strerror(errno));
	    }
	  return -1;
	}
    }
#endif
  return XENONE;
}

int cardif_linux_wext_unencrypted_eapol(struct interface_data *intdata,
					int state)
{
#if WIRELESS_EXT > 17
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  return cardif_linux_wext_set_iwauth(intdata, IW_AUTH_RX_UNENCRYPTED_EAPOL,
				      state, "RX unencrypted EAPoL");
#endif
  return XENONE;
}



/***********************************************************************
 *
 * Convert our cipher designator to something that will be understood by
 * the linux wireless extensions.
 *
 ***********************************************************************/
int cardif_linux_wext_iw_cipher(int cipher)
{
#if WIRELESS_EXT > 17
  switch (cipher)
    {
    case CIPHER_NONE:
      return 0;
      break;

    case CIPHER_WEP40:
      return IW_AUTH_CIPHER_WEP40;
      break;

    case CIPHER_TKIP:
      return IW_AUTH_CIPHER_TKIP;
      break;

    case CIPHER_WRAP:
      debug_printf(DEBUG_NORMAL, "WRAP is not supported!\n");
      return -1;
      break;

    case CIPHER_CCMP:
      return IW_AUTH_CIPHER_CCMP;
      break;
      
    case CIPHER_WEP104:
      return IW_AUTH_CIPHER_WEP104;
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown cipher value of %d!\n", cipher);
      return -1;
      break;
    }
#else
  return -1;
#endif
}

/*********************************************************************
 *
 * Set all of the card settings that are needed in order to complete an
 * association, so that we can begin the authentication.
 *
 *********************************************************************/
void cardif_linux_wext_associate(struct interface_data *intdata)
{
  uint8_t *bssid;
#if WIRELESS_EXT > 17
  int len = 0, akm = 0;
  uint32_t cipher, alg;
  uint8_t wpaie[255];
#endif
  struct config_network *network_data;
  struct config_globals *globals;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  if (network_data->methods == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No EAP methods defined for this network!\n");
      return;
    }

#if WIRELESS_EXT > 17
  // Determine the type of association we want, and set it up if we are using
  // the proper versions of WEXT.

  switch (network_data->assoc_type)
    {
    case ASSOC_OPEN:
      alg = IW_AUTH_ALG_OPEN_SYSTEM;
      break;

    case ASSOC_SHARED:
      alg = IW_AUTH_ALG_SHARED_KEY;
      break;

    case ASSOC_LEAP:
      alg = IW_AUTH_ALG_LEAP;
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown 802.11 authetication alg.  Defaulting "
                   "to Open System.\n");
      alg = IW_AUTH_ALG_OPEN_SYSTEM;
      break;
    }

  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_80211_AUTH_ALG,
                                   alg, "802.11 auth. alg to open") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set 802.11 auth. alg.\n");
    }

  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_DROP_UNENCRYPTED,
                                   TRUE, "drop unencrypted data") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable dropping of unencrypted"
                   " data!\n");
    }
#endif

  if (config_ssid_get_ssid_abilities() & RSN_IE)
    {
#if WIRELESS_EXT > 17
      cardif_linux_wext_get_wpa2_ie(intdata, (char *) wpaie, &len);
      if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_WPA_VERSION,
				       IW_AUTH_WPA_VERSION_WPA2,
				       "WPA version") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set WPA version!\n");
	}
      network_data->wpa_group_crypt = wpa2_get_group_crypt(intdata);
      network_data->wpa_pairwise_crypt = wpa2_get_pairwise_crypt(intdata);
#endif
    } else if (config_ssid_get_ssid_abilities() & WPA_IE)
      {
#if WIRELESS_EXT > 17
	cardif_linux_wext_get_wpa_ie(intdata, (char *) wpaie, &len);
	if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_WPA_VERSION,
					 IW_AUTH_WPA_VERSION_WPA,
					 "WPA2 version") < 0)
	  {
	    debug_printf(DEBUG_NORMAL, "Couldn't set WPA2 version!\n");
	  }
	network_data->wpa_group_crypt = wpa_get_group_crypt(intdata);
	network_data->wpa_pairwise_crypt = wpa_get_pairwise_crypt(intdata);
#endif
      }

#if WIRELESS_EXT > 17
  if (cardif_linux_wext_set_wpa_ie(intdata, wpaie, len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE on device %s!\n",
		   intdata->intName);
    }
  
  // For drivers that require something other than just setting a
  // WPA IE, we will set the components for the IE instead.
  cipher = cardif_linux_wext_iw_cipher(network_data->wpa_pairwise_crypt);
  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_CIPHER_PAIRWISE,
				   cipher, "pairwise cipher") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set pairwise cipher to %d.\n",
		   cipher);
    }
  
  cipher = cardif_linux_wext_iw_cipher(network_data->wpa_group_crypt);
  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_CIPHER_GROUP,
				   cipher, "group cipher") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set group cipher to %d.\n",
		   cipher);
    }
  
  if (network_data->methods->method_num != WPA_PSK)
    {
      akm = IW_AUTH_KEY_MGMT_802_1X;
    } else {
      akm = IW_AUTH_KEY_MGMT_PSK;
    }
  
  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_KEY_MGMT, akm,
				   "Authenticated Key Management Suite") 
      < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set Authenticated Key "
		   "Management Suite.\n");
    }

  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_PRIVACY_INVOKED,
				   TRUE, "Privacy Invoked") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't invoke privacy!\n");
    }

  if (cardif_linux_wext_set_iwauth(intdata, IW_AUTH_RX_UNENCRYPTED_EAPOL,
                                   TRUE, "RX unencrypted EAPoL") < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable RX of unencrypted"
                   " EAPoL.\n");
    }

#endif

  cardif_linux_wext_set_ssid(intdata, intdata->cur_essid);

  bssid = config_ssid_get_mac();
  if (bssid != NULL)
    {
      debug_printf(DEBUG_INT, "Dest. BSSID : ");
      debug_hex_printf(DEBUG_INT, bssid, 6);
    }

  globals = config_get_globals();

  if ((!globals) || (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM)))
    {
      cardif_linux_wext_set_bssid(intdata, bssid);
    }
  
  if ((intdata->wpa_ie == NULL) && (intdata->rsn_ie == NULL))
    {
      // We need to set up the card to allow unencrypted EAPoL frames.
      cardif_linux_wext_unencrypted_eapol(intdata, TRUE);
    }
  
  return;
}

int cardif_linux_wext_countermeasures(struct interface_data *intdata,
				      char endis)
{
#if WIRELESS_EXT > 17
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  return cardif_linux_wext_set_iwauth(intdata, IW_AUTH_TKIP_COUNTERMEASURES, 
				      endis,
				      "enable/disable TKIP countermeasures");
#endif
  return XENONE;
}

int cardif_linux_wext_drop_unencrypted(struct interface_data *intdata,
				       char endis)
{
#if WIRELESS_EXT > 17
  struct lin_sock_data *sockData;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  return cardif_linux_wext_set_iwauth(intdata, IW_AUTH_DROP_UNENCRYPTED,
				      endis, "drop unencrypted frames");
#endif
  return XENONE;
}

void cardif_linux_wext_enc_capabilities(struct interface_data *intdata)
{
#if WIRELESS_EXT > 17
  struct iwreq wrq;
  struct lin_sock_data *sockData;
  struct iw_range *range;
  char buffer[sizeof(struct iw_range) * 2];
  int i;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  sockData = intdata->sockData;

  xsup_assert((sockData != NULL), "sockData != NULL", TRUE);

  intdata->enc_capa = 0;

  memset(&wrq, 0x00, sizeof(wrq));
  Strncpy((char *)&wrq.ifr_name, intdata->intName, sizeof(wrq.ifr_name));

  wrq.u.data.pointer = (caddr_t) buffer;
  wrq.u.data.length = sizeof(buffer);
  wrq.u.data.flags = 0;

  if (!xsup_assert((sockData->sockInt > 0), "sockData->sockInt > 0", FALSE))
    return;

  if (ioctl(sockData->sockInt, SIOCGIWRANGE, &wrq) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get encryption capabilites!\n");
      return;
    }

  // Otherwise, determine what we have.
  range = (struct iw_range *)buffer;

  for (i=0; i<range->num_encoding_sizes; i++)
    {
      if (range->encoding_size[i] == 5) intdata->enc_capa |= DOES_WEP40;
      if (range->encoding_size[i] == 13) intdata->enc_capa |= DOES_WEP104;
    }

  if (range->enc_capa & IW_ENC_CAPA_WPA) intdata->enc_capa |= DOES_WPA;
  if (range->enc_capa & IW_ENC_CAPA_WPA2) intdata->enc_capa |= DOES_WPA2;
  if (range->enc_capa & IW_ENC_CAPA_CIPHER_TKIP)
    intdata->enc_capa |= DOES_TKIP;
  if (range->enc_capa & IW_ENC_CAPA_CIPHER_CCMP)
    intdata->enc_capa |= DOES_CCMP;
#else
  debug_printf(DEBUG_NORMAL, "You need wireless extensions > 17 in order to"
	       " support detection of encryption methods.\n");
  intdata->enc_capa = 0;
#endif
}

int cardif_linux_wext_delete_key(struct interface_data *intdata, int key_idx,
				 int set_tx)
{
#if WIRELESS_EXT > 17
  char seq[6] = {0x00,0x00,0x00,0x00,0x00,0x00};

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  return cardif_linux_wext_set_key_ext(intdata, IW_ENCODE_ALG_NONE, NULL,
				       key_idx, set_tx, seq, 6, NULL, 0);
#else
  debug_printf(DEBUG_NORMAL, "%s : Not supported by WE(%d)!\n", __FUNCTION__,
	       WIRELESS_EXT);
#endif
  return XENONE;
}


struct cardif_funcs cardif_linux_wext_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_linux_wext_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_linux_wext_set_tkip_key,
  .set_ccmp_key = cardif_linux_wext_set_ccmp_key,
  .delete_key = cardif_linux_wext_delete_key,
  .associate = cardif_linux_wext_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_linux_wext_wpa_state,
  .wpa = cardif_linux_wext_wpa,
  .wep_associate = cardif_linux_wext_wep_associate,
  .countermeasures = cardif_linux_wext_countermeasures,
  .drop_unencrypted = cardif_linux_wext_drop_unencrypted,
  .get_wpa_ie = cardif_linux_wext_get_wpa_ie,
  .get_wpa2_ie = cardif_linux_wext_get_wpa2_ie,
  .enc_disable = cardif_linux_wext_enc_disable,
  .enc_capabilities = cardif_linux_wext_enc_capabilities,
  .setbssid = cardif_linux_wext_set_bssid,
  .set_operstate = cardif_linux_rtnetlink_set_operstate,
  .set_linkmode = cardif_linux_rtnetlink_set_linkmode,
};


#endif
