/*******************************************************************
 *
 * Licensed under a dual GPL/BSD license. (See LICENSE file for more info.)
 *
 * File: gui_interface.c
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: xsupgui.c,v 1.4 2006/05/26 22:04:58 chessing Exp $
 * $Date: 2006/05/26 22:04:58 $
 * $Log: xsupgui.c,v $
 * Revision 1.4  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.3  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.2  2006/04/17 03:56:24  chessing
 * Added some support to enable/disable TNC support both via the configuration file, and via IPC.
 *
 * Revision 1.1  2005/10/26 18:51:08  chessing
 * Bring CVS up-to-date.
 *
 * Revision 1.2  2005/10/20 16:50:38  chessing
 * A little bit of cleanup in libxsupgui, and an attempt to make CVS spit things out correctly again.
 *
 * Revision 1.1  2005/10/16 04:31:30  chessing
 * Changed gui_interface.c/h to be a library called libxsupgui, and moved it to a new lib directory.
 *
 *******************************************************************/
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include "xsupgui.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

#define XSUP_SOCKET "/tmp/xsupplicant.sock"
#define DEBUG  1

/*******************************************************************
 *
 * Establish a handler to talk to the supplicant.
 *
 *******************************************************************/
int xsupgui_connect(char *intname)
{
  int sockErr, skfd;
  struct sockaddr_un sa;

  skfd = socket(PF_UNIX, SOCK_STREAM, 0);
  if (skfd < 0)
    {
#if DEBUG
      printf("Error getting socket!\n");
#endif
      return ERR_SOCK;
    }

  strncpy(sa.sun_path, XSUP_SOCKET, sizeof(sa.sun_path)-1);
  strncat(sa.sun_path, ".", 1);
  strcat(sa.sun_path, intname);

  sa.sun_family = AF_LOCAL;
  
  sockErr = connect(skfd, (struct sockaddr *)&sa, sizeof(sa));
  if (sockErr < 0) 
    {
#if DEBUG
      printf("Socket Error : %d -- %s  (%s:%d)\n", errno, strerror(errno),
	     __FUNCTION__, __LINE__);
#endif
      return errno;
    }

  return skfd;
}

/******************************************************************
 *
 * Disconnect from the daemon.
 *
 ******************************************************************/
int xsupgui_disconnect(int skfd)
{
  close(skfd);
  return 0;
}

/******************************************************************
 *
 * See if we have any data.  If we are blocking, we will wait here
 * forever until we get something back!
 *
 ******************************************************************/
int xsupgui_get_packet(int skfd, char *buffer, int *bufsize, int block)
{
  int readStat = -1;
  fd_set tset;
  struct timeval tv;

  FD_ZERO(&tset);
  FD_SET(skfd, &tset);
  tv.tv_sec = 0;
  tv.tv_usec = 0;

  if (block != 1)
    {
      readStat = select(skfd+1, &tset, NULL, NULL, &tv);
      if (readStat <= 0) return ERR_NONE;
    }

  readStat = recv(skfd, buffer, *bufsize, 0);
 
  if (readStat < 0) 
    {
      printf("Socket error : %s\n",strerror(errno));
      *bufsize = 0;
      return ERR_SOCK;
    }

  *bufsize = readStat;

  return ERR_NONE;
}

/******************************************************************
 *
 * Validate, and send a packet.
 *
 ******************************************************************/
int xsupgui_send_packet(int skfd, char *buffer, int bufptr)
{
  int i;

  i=send(skfd, buffer, bufptr, 0);
  if (i < 0)
    {
      printf("Socket error! (%d : %d)\n",i, errno);
      return ERR_SOCK;
    }

  return ERR_NONE;
}

/******************************************************************
 *
 * bufptr should point to the next command record to be processed. We
 * will pull the record out, copy the data for the record in to the
 * *data, and return the value of the record we are using.
 *
 * The caller should check that *bufptr < buffer length.
 *
 ******************************************************************/
int xsupgui_parse_packet(char *buffer, int *bufptr, char *data, 
			 int *datalen)
{
  struct ipc_cmd *cmd;

  if (buffer == NULL)
    {
#if DEBUG
      printf("buffer == NULL!\n");
#endif
      return ERR_PKT_BAD;
    }

  if (*bufptr < 0) 
    {
      *bufptr = 0;
    } 

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  if (cmd->version != IPC_VERSION_NUM) 
    {
#if DEBUG
      printf("IPC version mismatch!\n");
#endif
      return ERR_PKT_BAD;
    }

  /* zero out the buffer, just to be safe. */
  bzero(data, cmd->len+1);

  *bufptr += sizeof(struct ipc_cmd);  /* Skip the command structure. */

  memcpy(data, (char *)&buffer[*bufptr], cmd->len);

  *bufptr += cmd->len;
  *datalen = cmd->len;

  return cmd->attribute;
}

/*****************************************************************
 *
 * Request the authentication state of an interface.
 *
 *****************************************************************/
int xsupgui_get_state(char *buffer, int *bufptr)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];
  
  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_GET;
  cmd->len = 0;
  *bufptr += sizeof(struct ipc_cmd);

  return ERR_NONE;
}

/**********************************************************************
 *
 * Get the state of TNC.
 *
 **********************************************************************/
int xsupgui_get_tnc_state(char *buffer, int *bufptr)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = TNCENABLE;
  cmd->getset = IPC_GET;
  cmd->len = 0;
  *bufptr += sizeof(struct ipc_cmd);

  return ERR_NONE;
}

/****************************************
 *
 * Get the state of authentication.
 *
 ****************************************/
int xsupgui_get_auth_ctrl_state(char *buffer, int *bufptr)
{
  struct ipc_cmd *cmd;

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = AUTH_CTRL;
  cmd->getset = IPC_GET;
  cmd->len = 0;
  *bufptr += sizeof(struct ipc_cmd);

  return ERR_NONE;
}

/****************************************
 *
 * Send in a password set command.
 *
 ****************************************/
int xsupgui_set_password(char *buffer, int *bufptr, char *password)
{
  struct ipc_cmd *cmd;

  if (!password)
    {
      return ERR_BAD_DATA;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = TEMPPASSWORD;
  cmd->getset = IPC_SET;
  cmd->len = strlen(password);
  *bufptr += sizeof(struct ipc_cmd);

  strcpy((char *)&buffer[*bufptr], password);
  *bufptr += strlen(password);
  
  return ERR_NONE;
}

/***********************************************************************
 *
 * Change the state of useage of TNC support.
 *
 ***********************************************************************/
int xsupgui_set_use_tnc(char *buffer, int *bufptr, char enable)
{
  struct ipc_cmd *cmd;

  if ((enable != 0) && (enable != 1))
    {
      return ERR_BAD_DATA;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = TNCENABLE;
  cmd->getset = IPC_SET;
  cmd->len = 1;
  *bufptr += sizeof(struct ipc_cmd);

  buffer[*bufptr] = enable;
  (*bufptr) ++;

  return ERR_NONE;
}

/***********************************************************************
 *
 * Set the authentication control state.
 *
 ***********************************************************************/
int xsupgui_set_auth_ctrl(char *buffer, int *bufptr, char newstate)
{
  struct ipc_cmd *cmd;

  if ((newstate < 1) || (newstate > 3))
    {
      return ERR_BAD_DATA;
    }

  cmd = (struct ipc_cmd *)&buffer[*bufptr];

  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = AUTH_CTRL;
  cmd->getset = IPC_SET;
  cmd->len = 1;
  *bufptr += sizeof(struct ipc_cmd);

  buffer[*bufptr] = newstate;
  (*bufptr) ++;

  return ERR_NONE;
}

/****************************************
 *
 * Test for a PID file, and return an error if something seems to be running.
 *
 ****************************************/
int xsupgui_is_xsup_running(char *intname)
{
  char fullpidfilename[255];
  struct stat mystats;
  int err;

  strncpy(fullpidfilename, XSUP_SOCKET, 255);
  strncat(fullpidfilename, ".", 1);
  strncat(fullpidfilename, intname, 255);

  err = stat(fullpidfilename, &mystats);
  if (err == 0)
    {
      return TRUE;
    }
  return FALSE;
}
