/*******************************************************************
 * Implementation for writing a config from memory to the disk.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsupconfwrite.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsupconfwrite.c,v 1.14 2006/06/05 19:56:42 chessing Exp $
 * $Date: 2006/06/05 19:56:42 $
 * $Log: xsupconfwrite.c,v $
 * Revision 1.14  2006/06/05 19:56:42  chessing
 * Fixed a 16k memory leak in the config parse code.  Cleaned up some of the TLS code.
 *
 * Revision 1.13  2006/05/30 23:40:23  chessing
 * Small patches/additions and addition to the readme file that contains a link to the location of libtnc.
 *
 * Revision 1.12  2006/05/30 16:56:31  chessing
 * More patches from Carsten Grohmann and a #if that was missing from the 1.2.5 release.
 *
 * Revision 1.11  2006/05/30 04:33:19  chessing
 * Some small updates for the GUI code.
 *
 * Revision 1.10  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.9  2006/05/24 04:36:30  chessing
 * More work on the GUI interface.  (We can now save.)  Fixed xsupconfwrite so that it doesn't write the firmware roaming option twice.
 *
 * Revision 1.8  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.7  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.6  2006/04/17 03:56:24  chessing
 * Added some support to enable/disable TNC support both via the configuration file, and via IPC.
 *
 * Revision 1.5  2006/03/08 00:16:04  chessing
 * Fixed EAP hints code to work correctly when the request ID packet is padded out with null bytes.  (Observed in Aruba APs.)  Some changes/fixes for the EAP-AKA module.
 *
 * Revision 1.4  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.3  2006/01/03 04:02:35  chessing
 * Added the ability to store the PEAP password in a hashed format.  (Basically, an MS-CHAPv1 hash.)  Also added an 'ntpwdhash' program to the tools directory that will convert a cleartext password in to a hash that can be copied to the configuration file.
 *
 * Revision 1.2  2005/11/16 02:18:38  chessing
 * Updates to the config writing library.  The config-parser tool has been modified to be able to parse a config, and rewrite it.
 *
 * Revision 1.1  2005/11/14 18:06:55  chessing
 * Added library to take the parsed configuration structure out of memory, and write it to a file.  (This will be useful for GUI interfaces.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>

#include "xsupconfig.h"
#include "xsupconfwrite.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

// Enable this for textual debug output.
//#define DEBUG  1

/*******************************************************************
 *
 * Error result strings for various errors that can occur.
 *
 *******************************************************************/

char *xsupconfwrite_errunknown = "Unknown Error";
char *xsupconfwrite_errnone = "There was no error.";
char *xsupconfwrite_errcheckerrno = "Please check errno for the proper error result.";
char *xsupconfwrite_errgenioerr = "An I/O error occurred.  This may be because the requested file cannot be written, or no longer exists.";
char *xsupconfwrite_errinvalid_dest = "An invalid value was set for the 'destination' in the config file.";
char *xsupconfwrite_errbadnetlist = "You must have a value set for 'network_list'.  (If you are unsure, set this to 'all'.)";
char *xsupconfwrite_errnofacility = "Your log file is set to syslog, but you have not defined a logging facility.";
char *xsupconfwrite_errnofilehandle = "There is no available file handle to write your new config!";
char *xsupconfwrite_errnoglobaldata = "There is no available global configuration data to write!";
char *xsupconfwrite_errnonetworkdata = "There is no available network configuration data to write!";
char *xsupconfwrite_errnonetworkname = "There is no network name available in the network configuration data.";
char *xsupconfwrite_errinvalidcrypto = "Attempted to write an invalid crypto type value.";
char *xsupconfwrite_errnoeapdata = "There was no EAP data available to write.";
char *xsupconfwrite_errbadttlsphase2 = "Invalid TTLS phase 2 configuration.";
char *xsupconfwrite_errnophase2eapdata = "Invalid PEAP phase 2 data.";
char *xsupconfwrite_errinvalidinnereap = "Invalid EAP method requested for PEAP phase 2.";

/*******************************************************************
 *
 * Create a config file, and return an error if we get one.
 *
 *******************************************************************/
int xsupconfwrite_create_config_file(char *filename, FILE **desc)
{
#ifdef DEBUG
  printf("Opening file %s\n", filename);
#endif
  (*desc) = fopen(filename, "w");
  if (!(*desc)) return XSUPCONFWRITE_CHECKERRNO;

  return XSUPCONFWRITE_ERRNONE;
}

/********************************************************************
 *
 * Write a silly little header that indicates that this is an Xsupplicant
 * config file.
 *
 ********************************************************************/
int xsupconfwrite_write_header(FILE *fileid)
{
  if (!fileid)
    {
#ifdef DEBUG
      printf("Invalid file handle! In %s():%d!\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (fprintf(fileid, "# Xsupplicant configuration file for Xsupplicant %s (or"
      " later)\n\n\n", VERSION) < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/********************************************************************
 *
 * Write a string list as a set of comma seperated strings.
 *
 ********************************************************************/
int xsupconfwrite_write_stringlist(FILE *fileid, 
				   struct config_string_list *list)
{
  int comma = 0;
  struct config_string_list *cur;

  cur = list;

  while (cur != NULL)
    {
      if (comma) 
	{
	  if (fprintf(fileid, ",") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "%s", cur->name) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
      cur = cur->next;
    }

  if (fprintf(fileid, "\n\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/********************************************************************
 *
 * This function will write appropriate values in the global config section
 * of the file.
 *
 ********************************************************************/
int xsupconfwrite_write_globals(FILE *fileid, struct config_globals *globals)
{
  int err = 0;

  if (!fileid) 
    {
#ifdef DEBUG
      printf("No valid file handle found at %s():%d!\n", __FUNCTION__,
	     __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (!globals)
    {
#ifdef DEBUG
      printf("No valid data to write to config file at %s():%d!\n", 
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOGLOBALDATA;
    }

  if (fprintf(fileid, "# --- Global Variables ---\n\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (!globals->allowed_nets) 
    {
      if (fprintf(fileid, "network_list = all\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    } else {
      if (fprintf(fileid, "network_list = ") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      
      err = xsupconfwrite_write_stringlist(fileid, globals->allowed_nets);
      if (err < 0) return err;
    }

  if (fprintf(fileid, "default_netname = %s\n\n", globals->default_net) < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (globals->destination != DEST_AUTO)
    {
      if (fprintf(fileid, "destination = ") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      // We will never see DEST_AUTO, so ignore that case.
      switch (globals->destination)
	{
	case DEST_BSSID:
	  if (fprintf(fileid, "BSSID\n\n") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  break;

	case DEST_MULTICAST:
	  if (fprintf(fileid, "multicast\n\n") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  break;

	case DEST_SOURCE:
	  if (fprintf(fileid, "source\n\n") < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  break;

	default:
	  return XSUPCONFWRITE_INVALID_DEST;
	}
    }

  if (globals->logfile)
    {
      if (fprintf(fileid, "logfile = %s\n\n", globals->logfile) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      
      if (strncmp(globals->logfile, "syslog", 6) == 0)
	{
	  // We are logging to syslog, so we need to write a facility
	  // value.
	  
	  if (!globals->log_facility) return XSUPCONFWRITE_NOFACILITY;
	  
	  if (fprintf(fileid, "log_facility = %s\n\n", globals->log_facility) < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}
    }

  if (globals->ipc_group_name)
    {
      if (fprintf(fileid, "ipc_group = %s\n\n", globals->ipc_group_name) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_AUTH_PER))
    {
      if (fprintf(fileid, "auth_period = %d\n\n", globals->auth_period) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_HELD_PER))
    {
      if (fprintf(fileid, "held_period = %d\n\n", globals->held_period) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_MAX_STARTS))
    {
      if (fprintf(fileid, "max_starts = %d\n\n", globals->max_starts) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
    {
      if (fprintf(fileid, "friendly_warnings = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_PASSIVE_SCAN))
    {
      if (fprintf(fileid, "passive_scanning = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (globals->passive_timeout != 0)
    {
      if (fprintf(fileid, "passive_timer = %d\n\n", globals->passive_timeout) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }
  
  if (globals->active_timeout != RESCAN_TIMEOUT)
    {
      if (fprintf(fileid, "scan_timeout = %d\n\n", globals->active_timeout) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to the file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM))
    {
      if (fprintf(fileid, "roaming = firmware\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_ALLMULTI))
    {
      if (fprintf(fileid, "allmulti = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_ASSOC_AUTO))
    {
      if (fprintf(fileid, "association = manual\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_EAP_HINTS))
    {
      if (fprintf(fileid, "use_eap_hints = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (globals->assoc_timeout != ASSOCIATION_TIMEOUT)
    {
      if (fprintf(fileid, "association_timeout = %d\n\n", 
		  globals->assoc_timeout) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  return XSUPCONFWRITE_ERRNONE;
}

/**************************************************************************
 *
 * If 'comma' is >=1 then write a comma, and a space.  Since this call should
 * only come deep inside of another function, we don't check to be sure that
 * fileid is valid.  (Why would you write a comma to a file that you hadn't
 * written something to already? ;)
 *
 *************************************************************************/
int xsupconfwrite_write_comma(FILE *fileid, int comma)
{
  if (comma >= 1)
    {
      return fprintf(fileid, ", ");
    }

  return 0;
}

/****************************************************************************
 *
 * Write out a list of EAP types that we are going to allow.
 *
 ****************************************************************************/
int xsupconfwrite_write_allowed_eaps(FILE *fileid, struct config_network *cur)
{
  int comma = 0, retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Invalid file handle at %s():%d!\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (!cur)
    {
#ifdef DEBUG
      printf("Invalid network configuration structure at %s():%d!\n",
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NONETWORKDATA;
    }

  if ((cur->flags & CONFIG_NET_ALLOW_ALL) == CONFIG_NET_ALLOW_ALL)
    {
      if (fprintf(fileid, "\tallow_types = all\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	} else {
	  // We can return here, because if we are supporting ALL, then there
	  // is nothing more to write.
	  return XSUPCONFWRITE_ERRNONE;
	}
    }

  if ((cur->flags & CONFIG_NET_ALLOW_ALL) > 0)
    {
      // Otherwise, we need to write every specific type we support.
      if (fprintf(fileid, "\tallow_types = ") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_TLS))
	{
	  if (fprintf(fileid, "eap-tls") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_MD5))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "eap-md5") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_TTLS))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;

	  if (fprintf(fileid, "eap-ttls") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_LEAP))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;

	  if (fprintf(fileid, "eap-leap") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_MSCV2))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;

	  if (fprintf(fileid, "eap-mschapv2") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_PEAP))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "eap-peap") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_SIM))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;

	  if (fprintf(fileid, "eap-sim") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_GTC))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "eap-gtc") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_OTP))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "eap-otp") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}

      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_AKA))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "eap-aka") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (TEST_FLAG(cur->flags, CONFIG_NET_ALLOW_WPA_PSK))
	{
	  retval = xsupconfwrite_write_comma(fileid, comma);
	  if (retval < 0) return retval;
      
	  if (fprintf(fileid, "wpa-psk") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	  
	  comma = 1;
	}
      
      if (fprintf(fileid, "\n\n") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the text representation of the crypto method we want to use.
 *
 *******************************************************************/
int xsupconfwrite_write_crypt(FILE *fileid, int crypt)
{
  switch (crypt)
    {
    case CRYPT_WEP40:
      if (fprintf(fileid, "WEP40") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CRYPT_TKIP:
      if (fprintf(fileid, "TKIP") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CRYPT_WRAP:
      // Nobody uses this one, but we put it in here for kicks anyway. ;)
      if (fprintf(fileid, "WRAP") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CRYPT_CCMP:
      if (fprintf(fileid, "CCMP") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CRYPT_WEP104:
      if (fprintf(fileid, "WEP104") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    default:
      return XSUPCONFWRITE_INVALIDCRYPTO;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for the smartcard part of EAP-TLS
 * if it is needed.
 *
 *******************************************************************/
int xsupconfwrite_write_tls_smartcard(FILE *fileid, 
				      struct config_eap_tls *eaptls)
{
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (!eaptls)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  // This is icky, but we need to determine if any of the smartcard values
  // contain valid data before we write anything to the file.
  if ((eaptls->sc.opensc_so_path) && ((eaptls->sc.engine_id) || 
				      (eaptls->sc.key_id)))
    {
      // We have valid data.
      if (fprintf(fileid, "\n\t\tsmartcard\n") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      if (fprintf(fileid, "\t\t{\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      if (eaptls->sc.opensc_so_path)
	{
	  if (fprintf(fileid, "\t\t\topensc_so_path = \"%s\"\n",
		      eaptls->sc.opensc_so_path) < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (eaptls->sc.engine_id)
	{
	  if (fprintf(fileid, "\t\t\tengine_id = %s\n",
		      eaptls->sc.engine_id) < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (eaptls->sc.key_id)
	{
	  if (fprintf(fileid, "\t\t\tkey_id = %s\n",
		      eaptls->sc.key_id) < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\t}\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  return XSUPCONFWRITE_ERRNONE;
}

/********************************************************************
 *
 * Write the configuration information for EAP-MD5
 *
 ********************************************************************/
int xsupconfwrite_eap_md5(FILE *fileid, struct config_eap_method *cur, 
			  int phase)
{
  struct config_eap_md5 *eapmd5;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapmd5 = (struct config_eap_md5 *)cur->method_data;

  if (!eapmd5)
    {
#ifdef DEBUG
      printf("The EAP-MD5 structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\teap-md5\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapmd5->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapmd5->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapmd5->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapmd5->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/********************************************************************
 *
 * Write the configuration information for LEAP
 *
 ********************************************************************/
int xsupconfwrite_eap_leap(FILE *fileid, struct config_eap_method *cur,
			   int phase)
{
  struct config_eap_leap *eapleap;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapleap = (struct config_eap_leap *)cur->method_data;

  if (!eapleap)
    {
#ifdef DEBUG
      printf("The EAP-LEAP structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\teap-leap\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapleap->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapleap->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapleap->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapleap->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/********************************************************************
 *
 * Write the configuration information for EAP-MSCHAPv2
 *
 ********************************************************************/
int xsupconfwrite_eap_mschapv2(FILE *fileid, struct config_eap_method *cur,
			       int phase)
{
  struct config_eap_mschapv2 *eapmschapv2;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapmschapv2 = (struct config_eap_mschapv2 *)cur->method_data;

  if (!eapmschapv2)
    {
#ifdef DEBUG
      printf("The EAP-MSCHAPv2 structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\teap-mschapv2\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapmschapv2->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapmschapv2->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapmschapv2->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapmschapv2->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapmschapv2->nthash)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tntpwdhash = \"%s\"\n", eapmschapv2->nthash) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/********************************************************************
 *
 * Write the configuration information for EAP-OTP/GTC
 *
 ********************************************************************/
int xsupconfwrite_eap_otp(FILE *fileid, struct config_eap_method *cur,
			  int phase, int otp)
{
  struct config_eap_otp *eapotp;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapotp = (struct config_eap_otp *)cur->method_data;

  if (!eapotp)
    {
#ifdef DEBUG
      printf("The EAP-OTP/GTC structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (otp)
    {
      if (fprintf(fileid, "\teap-otp\n") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    } else {
      if (fprintf(fileid, "\teap-gtc\n") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  /*
    // XXX Usernames aren't allowed (yet), so enable this code later.
  if (eapotp->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) return XSUPCONFWRITE_GENIOERR;
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapotp->username) < 0)
	return XSUPCONFWRITE_GENIOERR;
    }
  */

  if (eapotp->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapotp->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/********************************************************************
 *
 * Write the configuration information for EAP-SIM
 *
 ********************************************************************/
int xsupconfwrite_eap_sim(FILE *fileid, struct config_eap_method *cur,
			  int phase)
{
  struct config_eap_sim *eapsim;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapsim = (struct config_eap_sim *)cur->method_data;

  if (!eapsim)
    {
#ifdef DEBUG
      printf("The EAP-SIM structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\teap-sim\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapsim->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapsim->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapsim->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapsim->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapsim->auto_realm == TRUE)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tauto_realm = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    } 

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/********************************************************************
 *
 * Write the configuration information for EAP-AKA
 *
 ********************************************************************/
int xsupconfwrite_eap_aka(FILE *fileid, struct config_eap_method *cur,
			  int phase)
{
  struct config_eap_aka *eapaka;
  
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
  
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }


  eapaka = (struct config_eap_aka *)cur->method_data;

  if (!eapaka)
    {
#ifdef DEBUG
      printf("The EAP-AKA structure is NULL!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\teap-aka\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapaka->username)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tusername = \"%s\"\n", eapaka->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapaka->password)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tpassword = \"%s\"\n", eapaka->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapaka->auto_realm == TRUE)
    {
      if (phase ==2)
	{
	  if (fprintf(fileid, "\t") < 0) 
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}

      if (fprintf(fileid, "\t\tauto_realm = true\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    } 

  if (phase ==2)
    {
      if (fprintf(fileid, "\t") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
  
  return XSUPCONFWRITE_ERRNONE;  
}

/*******************************************************************
 *
 * Write phase 2 data for PAP
 *
 *******************************************************************/
int xsupconfwrite_ttls_phase2_pap(FILE *fileid, void *cur)
{
  struct config_pap *pap;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_BADTTLSPHASE2),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_BADTTLSPHASE2;
    }

  pap = (struct config_pap *)cur;

  if (!pap) return XSUPCONFWRITE_BADTTLSPHASE2;

  if (fprintf(fileid, "\t\tpap\n\t\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
 
  if (pap->username)
    {
      if (fprintf(fileid, "\t\t\tusername = \"%s\"\n", pap->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (pap->password)
    {
      if (fprintf(fileid, "\t\t\tpassword = \"%s\"\n", pap->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t\t}\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/******************************************************************
 *
 * Write phase 2 data for CHAP.
 *
 ******************************************************************/
int xsupconfwrite_ttls_phase2_chap(FILE *fileid, void *cur)
{
  struct config_chap *chap;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_BADTTLSPHASE2),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_BADTTLSPHASE2;
    }

  chap = (struct config_chap *)cur;

  if (!chap) return XSUPCONFWRITE_BADTTLSPHASE2;

  if (fprintf(fileid, "\t\tchap\n\t\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
 
  if (chap->username)
    {
      if (fprintf(fileid, "\t\t\tusername = \"%s\"\n", chap->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (chap->password)
    {
      if (fprintf(fileid, "\t\t\tpassword = \"%s\"\n", chap->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t\t}\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/******************************************************************
 *
 * Write phase 2 data for MS-CHAP
 *
 ******************************************************************/
int xsupconfwrite_ttls_phase2_mschap(FILE *fileid, void *cur)
{
  struct config_mschap *mschap;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_BADTTLSPHASE2),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_BADTTLSPHASE2;
    }

  mschap = (struct config_mschap *)cur;

  if (!mschap) return XSUPCONFWRITE_BADTTLSPHASE2;

  if (fprintf(fileid, "\t\tmschap\n\t\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
 
  if (mschap->username)
    {
      if (fprintf(fileid, "\t\t\tusername = \"%s\"\n", mschap->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (mschap->password)
    {
      if (fprintf(fileid, "\t\t\tpassword = \"%s\"\n", mschap->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t\t}\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*****************************************************************
 *
 * Write phase 2 data for MS-CHAPv2
 *
 *****************************************************************/
int xsupconfwrite_ttls_phase2_mschapv2(FILE *fileid, void *cur)
{
  struct config_mschapv2 *mschapv2;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_BADTTLSPHASE2),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_BADTTLSPHASE2;
    }
  mschapv2 = (struct config_mschapv2 *)cur;

  if (!mschapv2) return XSUPCONFWRITE_BADTTLSPHASE2;

  if (fprintf(fileid, "\t\tmschapv2\n\t\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }
 
  if (mschapv2->username)
    {
      if (fprintf(fileid, "\t\t\tusername = \"%s\"\n", mschapv2->username) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (mschapv2->password)
    {
      if (fprintf(fileid, "\t\t\tpassword = \"%s\"\n", mschapv2->password) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t\t}\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the phase 2 configuration for EAP-TTLS
 *
 *******************************************************************/
int xsupconfwrite_eap_ttls_phase2(FILE *fileid, struct config_eap_ttls *cur)
{
  struct config_ttls_phase2 *p2type;
  struct config_eap_method data;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  switch (cur->phase2_type)
    {
    case TTLS_PHASE2_PAP:
      if (fprintf(fileid, "\t\tphase2_type = pap\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case TTLS_PHASE2_CHAP:
      if (fprintf(fileid, "\t\tphase2_type = chap\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case TTLS_PHASE2_MSCHAP:
      if (fprintf(fileid, "\t\tphase2_type = mschap\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case TTLS_PHASE2_MSCHAPV2:
      if (fprintf(fileid, "\t\tphase2_type = mschapv2\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case TTLS_PHASE2_EAP_MD5:
      if (fprintf(fileid, "\t\tphase2_type = eap_md5\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case TTLS_PHASE2_UNDEFINED:
    default:
#ifdef DEBUG
      printf("Either the TTLS phase 2 method was undefined, or is something "
	     "that is not allowed!\n");
#endif
      return XSUPCONFWRITE_BADTTLSPHASE2;
      break;
    }

  p2type = cur->phase2;

  while (p2type)
    {
      switch (p2type->phase2_type)
	{
	case TTLS_PHASE2_PAP:
	  retval = xsupconfwrite_ttls_phase2_pap(fileid, p2type->phase2_data);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case TTLS_PHASE2_CHAP:
	  retval = xsupconfwrite_ttls_phase2_chap(fileid, p2type->phase2_data);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case TTLS_PHASE2_MSCHAP:
	  retval = xsupconfwrite_ttls_phase2_mschap(fileid, 
						    p2type->phase2_data);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case TTLS_PHASE2_MSCHAPV2:
	  retval = xsupconfwrite_ttls_phase2_mschapv2(fileid,
						      p2type->phase2_data);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case TTLS_PHASE2_EAP_MD5:
	  data.method_data = p2type->phase2_data;
	  retval = xsupconfwrite_eap_md5(fileid, &data, 2);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	default:
#ifdef DEBUG
	  printf("%s\n", xsupconfwrite_strerr(XSUPCONFWRITE_BADTTLSPHASE2));
#endif
	  return XSUPCONFWRITE_BADTTLSPHASE2;
	  break;
	}

      p2type = p2type->next;
    }
  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for EAP-TTLS.
 *
 *******************************************************************/
int xsupconfwrite_eap_ttls(FILE *fileid, struct config_eap_method *cur)
{
  struct config_eap_ttls *eapttls;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  eapttls = (struct config_eap_ttls *)cur->method_data;

  if (!eapttls)
    {
#ifdef DEBUG
      printf("Error : The EAP-TTLS structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\teap-ttls\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eapttls->user_cert)
    {
      if (fprintf(fileid, "\t\tuser_cert = \"%s\"\n", eapttls->user_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->root_cert)
    {
      if (fprintf(fileid, "\t\troot_cert = \"%s\"\n", eapttls->root_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->root_dir)
    {
      if (fprintf(fileid, "\t\troot_dir = \"%s\"\n", eapttls->root_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->crl_dir)
    {
      if (fprintf(fileid, "\t\tcrl_dir = %s\n", eapttls->crl_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->user_key)
    {
      if (fprintf(fileid, "\t\tuser_key = \"%s\"\n", eapttls->user_key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->user_key_pass)
    {
      if (fprintf(fileid, "\t\tuser_key_pass = \"%s\"\n", eapttls->user_key_pass) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->session_resume == RES_YES)
    {
      if (fprintf(fileid, "\t\tsession_resume = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  // XXX TODO : Move this to a #defined value in xsupconfig.h
  if ((eapttls->chunk_size != 1398) && (eapttls->chunk_size != 0))
    {
      if (fprintf(fileid, "\t\tchunk_size = %d\n", eapttls->chunk_size) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->random_file)
    {
      if (fprintf(fileid, "\t\trandom_file = \"%s\"\n", eapttls->random_file) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->cncheck)
    {
      if (fprintf(fileid, "\t\tcncheck = %s\n", eapttls->cncheck) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eapttls->cnexact == TRUE)
    {
      if (fprintf(fileid, "\t\tcnexact = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  retval = xsupconfwrite_eap_ttls_phase2(fileid, eapttls);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for EAP-TLS.
 *
 *******************************************************************/
int xsupconfwrite_eap_tls(FILE *fileid, struct config_eap_method *cur)
{
  struct config_eap_tls *eaptls;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  eaptls = (struct config_eap_tls *)cur->method_data;

  if (!eaptls)
    {
#ifdef DEBUG
      printf("Error : The EAP-TLS structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\teap-tls\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eaptls->user_cert)
    {
      if (fprintf(fileid, "\t\tuser_cert = \"%s\"\n", eaptls->user_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->root_cert)
    {
      if (fprintf(fileid, "\t\troot_cert = \"%s\"\n", eaptls->root_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->root_dir)
    {
      if (fprintf(fileid, "\t\troot_dir = \"%s\"\n", eaptls->root_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->crl_dir)
    {
      if (fprintf(fileid, "\t\tcrl_dir = %s\n", eaptls->crl_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->user_key)
    {
      if (fprintf(fileid, "\t\tuser_key = \"%s\"\n", eaptls->user_key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->user_key_pass)
    {
      if (fprintf(fileid, "\t\tuser_key_pass = \"%s\"\n", eaptls->user_key_pass) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->session_resume == RES_YES)
    {
      if (fprintf(fileid, "\t\tsession_resume = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  // XXX TODO : Move this to a #defined value in xsupconfig.h
  if ((eaptls->chunk_size != 1398) && (eaptls->chunk_size != 0))
    {
      if (fprintf(fileid, "\t\tchunk_size = %d\n", eaptls->chunk_size) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eaptls->random_file)
    {
      if (fprintf(fileid, "\t\trandom_file = \"%s\"\n", eaptls->random_file) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  retval = xsupconfwrite_write_tls_smartcard(fileid, eaptls);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for WPA-PSK
 *
 *******************************************************************/
int xsupconfwrite_wpa_psk(FILE *fileid, struct config_eap_method *cur)
{
  struct config_wpa_psk *wpapsk;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  wpapsk = (struct config_wpa_psk *)cur->method_data;

  if (!wpapsk)
    {
#ifdef DEBUG
      printf("Error : The WPA-PSK structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\twpa-psk\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (wpapsk->key)
    {
      if (fprintf(fileid, "\t\tkey = \"%s\"\n", wpapsk->key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (wpapsk->hex_key)
    {
      if (fprintf(fileid, "\t\thex_key = \"%s\"\n", wpapsk->hex_key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/********************************************************************
 *
 * This function is generic to both the static and initial wep methods.
 * Both use the AVPs, but need some slightly different calls for things
 * not included in this function.
 *
 ********************************************************************/
int xsupconfwrite_write_wepkeys(FILE *fileid, 
				struct config_static_wep *staticwep)
{
  int i;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!staticwep)
    {
#ifdef DEBUG
      printf("Error : The WEP structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }
  
  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if ((staticwep->tx_key > 0) && (staticwep->tx_key < 4))
    {
      if (fprintf(fileid, "\t\ttx_key = %d\n", staticwep->tx_key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  for (i=1;i<5;i++)
    {
      if (staticwep->key[i])
	{
	  if (fprintf(fileid, "\t\tkey%d = \"%s\"\n", i, staticwep->key[i]) < 0)
	    {
#ifdef DEBUG
	      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	      return XSUPCONFWRITE_GENIOERR;
	    }
	}
    }

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for static WEP.
 *
 *******************************************************************/
int xsupconfwrite_static_wep(FILE *fileid, struct config_eap_method *cur)
{
  struct config_static_wep *staticwep;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  staticwep = (struct config_static_wep *)cur->method_data;

  if (!staticwep)
    {
#ifdef DEBUG
      printf("Error : The static WEP structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\tstatic-wep\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return xsupconfwrite_write_wepkeys(fileid, staticwep);
}

/*******************************************************************
 *
 * Write the configuration information for initial WEP keys.
 *
 *******************************************************************/
int xsupconfwrite_initial_wep(FILE *fileid, 
			      struct config_static_wep *staticwep)
{
  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!staticwep)
    {
#ifdef DEBUG
      printf("Error : The initial WEP structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\tinitial-wep\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return xsupconfwrite_write_wepkeys(fileid, staticwep);
}

/*******************************************************************
 *
 * Write the phase 2 information for EAP-PEAP.
 *
 *******************************************************************/
int xsupconfwrite_peap_phase2(FILE *fileid, struct config_eap_method *cureap)
{
  struct config_eap_method *cur;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cureap)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  cur = cureap;

  while (cur)
    {
      switch (cur->method_num)
	{
	case EAP_TYPE_MSCHAPV2:
	  retval = xsupconfwrite_eap_mschapv2(fileid, cur, 2);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;
	  
	case EAP_TYPE_MD5:
	  retval = xsupconfwrite_eap_md5(fileid, cur, 2);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case EAP_TYPE_SIM:
	  retval = xsupconfwrite_eap_sim(fileid, cur, 2);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case EAP_TYPE_GTC:
	  retval = xsupconfwrite_eap_otp(fileid, cur, 2, FALSE);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	case EAP_TYPE_OTP:
	  retval = xsupconfwrite_eap_otp(fileid, cur, 2, TRUE);
	  if (retval != XSUPCONFWRITE_ERRNONE) return retval;
	  break;

	default:
#ifdef DEBUG
	  printf("%s\n", xsupconfwrite_strerr(XSUPCONFWRITE_INVALIDINNEREAP));
#endif
	  return XSUPCONFWRITE_INVALIDINNEREAP;
	  break;
	}

      cur = cur->next;
    }
  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the list of allowed phase 2 types.
 *
 *******************************************************************/
int xsupconfwrite_peap_phase2_types(FILE *fileid, struct config_eap_peap *cur)
{
  int comma = 0, retval = 0;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (cur->flags & CONFIG_PEAP_ALLOW_ALL)
    {
      if (fprintf(fileid, "\t\tallow_types = all\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      return XSUPCONFWRITE_ERRNONE;
    }

  if (fprintf(fileid, "\t\tallow_types = ") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (cur->flags & CONFIG_PEAP_ALLOW_MSCV2)
    {
      if (fprintf(fileid, "eap_mschapv2") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
    }

  retval = xsupconfwrite_write_comma(fileid, comma);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (cur->flags & CONFIG_PEAP_ALLOW_MD5)
    {
      if (fprintf(fileid, "eap_md5") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
    }

  retval = xsupconfwrite_write_comma(fileid, comma);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (cur->flags & CONFIG_PEAP_ALLOW_SIM)
    {
      if (fprintf(fileid, "eap_sim") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
    }

  retval = xsupconfwrite_write_comma(fileid, comma);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (cur->flags & CONFIG_PEAP_ALLOW_GTC)
    {
      if (fprintf(fileid, "eap_gtc") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
    }

  retval = xsupconfwrite_write_comma(fileid, comma);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (cur->flags & CONFIG_PEAP_ALLOW_OTP)
    {
      if (fprintf(fileid, "eap_otp") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      comma = 1;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Write the configuration information for EAP-PEAP.
 *
 *******************************************************************/
int xsupconfwrite_eap_peap(FILE *fileid, struct config_eap_method *cur)
{
  struct config_eap_peap *eappeap;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }
 
  if (!cur)
    {
#ifdef DEBUG
      printf("Error : %s, at %s():%d!\n",
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOEAPDATA),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  eappeap = (struct config_eap_peap *)cur->method_data;

  if (!eappeap)
    {
#ifdef DEBUG
      printf("Error : The EAP-PEAP structure was missing useful data!\n");
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  if (fprintf(fileid, "\teap-peap\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (fprintf(fileid, "\t{\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  if (eappeap->identity)
    {
      if (fprintf(fileid, "\t\tidentity = \"%s\"\n", eappeap->identity) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->user_cert)
    {
      if (fprintf(fileid, "\t\tuser_cert = \"%s\"\n", eappeap->user_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->root_cert)
    {
      if (fprintf(fileid, "\t\troot_cert = \"%s\"\n", eappeap->root_cert) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->root_dir)
    {
      if (fprintf(fileid, "\t\troot_dir = \"%s\"\n", eappeap->root_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->crl_dir)
    {
      if (fprintf(fileid, "\t\tcrl_dir = %s\n", eappeap->crl_dir) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->user_key)
    {
      if (fprintf(fileid, "\t\tuser_key = \"%s\"\n", eappeap->user_key) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->user_key_pass)
    {
      if (fprintf(fileid, "\t\tuser_key_pass = \"%s\"\n", 
		  eappeap->user_key_pass) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->session_resume == RES_YES)
    {
      if (fprintf(fileid, "\t\tsession_resume = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  // XXX TODO : Move this to a #defined value in xsupconfig.h
  if ((eappeap->chunk_size != 1398) && (eappeap->chunk_size != 0))
    {
      if (fprintf(fileid, "\t\tchunk_size = %d\n", eappeap->chunk_size) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->random_file)
    {
      if (fprintf(fileid, "\t\trandom_file = \"%s\"\n", eappeap->random_file) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->cncheck)
    {
      if (fprintf(fileid, "\t\tcncheck = %s\n", eappeap->cncheck) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->cnexact == TRUE)
    {
      if (fprintf(fileid, "\t\tcnexact = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->ias_quirk == TRUE)
    {
      if (fprintf(fileid, "\t\tias_quirk = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (eappeap->proper_peapv1 == TRUE)
    {
      if (fprintf(fileid, "\t\tproper_peap_v1_keying = yes\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  retval = xsupconfwrite_peap_phase2_types(fileid, eappeap);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  retval = xsupconfwrite_peap_phase2(fileid, eappeap->phase2);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  if (fprintf(fileid, "\t}\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Loop through all of the available EAP methods, and write their data.
 *
 *******************************************************************/
int xsupconfwrite_write_eap_methods(FILE *fileid, struct config_network *net)
{
  struct config_eap_method *cur = NULL;
  int retval = 0;

  if (!fileid) 
    {
#ifdef DEBUG
      printf("Error : %s @ %s():%d\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NOFILEHANDLE),
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  // It is perfectly valid to have no network information to write here.
  if (!net) 
    {
#ifdef DEBUG
      printf("Warning : %s @ %s():%d (This is probably okay.)\n\n", 
	     xsupconfwrite_strerr(XSUPCONFWRITE_NONETWORKDATA), __FUNCTION__,
	     __LINE__);
#endif
      return XSUPCONFWRITE_ERRNONE;
    }

  cur = net->methods;

  if (!cur)
    {
#ifdef DEBUG
      printf("No EAP methods to write at %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOEAPDATA;
    }

  while (cur)
    {
      switch (cur->method_num)
	{
	case EAP_TYPE_TLS:
	  retval = xsupconfwrite_eap_tls(fileid, cur);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing TLS data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_MD5:
	  retval = xsupconfwrite_eap_md5(fileid, cur, 1);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing MD5 data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_LEAP:
	  retval = xsupconfwrite_eap_leap(fileid, cur, 1);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing LEAP data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_MSCHAPV2:
	  retval = xsupconfwrite_eap_mschapv2(fileid, cur, 1);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing MSCHAPv2 data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_PEAP:
	  retval = xsupconfwrite_eap_peap(fileid, cur);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing PEAP data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_TTLS:
	  retval = xsupconfwrite_eap_ttls(fileid, cur);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing TTLS data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_SIM:
	  retval = xsupconfwrite_eap_sim(fileid, cur, 1);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing SIM data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_GTC: 
	  // OTP and GTC are so similar, they use the same structure.  The
	  // only difference is we need to write a different EAP header to
	  // the config file.  The 4th param says we are doing GTC instead
	  // of OTP.
	  retval = xsupconfwrite_eap_otp(fileid, cur, 1, FALSE);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing GTC data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_OTP:
	  retval = xsupconfwrite_eap_otp(fileid, cur, 1, TRUE);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing OTP data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case EAP_TYPE_AKA:
	  retval = xsupconfwrite_eap_aka(fileid, cur, 1);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing AKA data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case WPA_PSK:
	  retval = xsupconfwrite_wpa_psk(fileid, cur);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing WPA-PSK data @ %s():%d\n", __FUNCTION__,
		     __LINE__);
#endif
	      return retval;
	    }
	  break;

	case STATIC_WEP_METHOD:
	  retval = xsupconfwrite_static_wep(fileid, cur);
	  if (retval != XSUPCONFWRITE_ERRNONE) 
	    {
#ifdef DEBUG
	      printf("Error writing static WEP data @ %s():%d\n",
		     __FUNCTION__, __LINE__);
#endif
	      return retval;
	    }
	  break;

	default:
	  // Ignore it, and move on.
#ifdef DEBUG
	  printf("Invalid EAP method number %d!  No data will be written.\n",
		 cur->method_num);
	  printf("Error at : %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  break;
	}

      cur = cur->next;
    }
  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Actually write the network specific configuration data, and call
 * the needed function to write EAP method specific data.
 *
 *******************************************************************/
int xsupconfwrite_network(FILE *fileid, struct config_network *cur)
{
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("Invalid file handle at %s():%d!\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (!cur)
    {
#ifdef DEBUG
      printf("Invalid network configuration data at %s():%d!\n",
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NONETWORKDATA;
    }

  if (!cur->name)
    {
#ifdef DEBUG
      printf("There was no network name provided at %s():%d!\n",
	     __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NONETWORKNAME;
    }

  // Put all network names in quotes in order to avoid having to scan the
  // string looking for characters that might cause problems.
  if (fprintf(fileid, "\"%s\"\n{\n", cur->name) < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  // *** From here until we close this section, we should tab in everything
  // that we write to the config file.

  if (cur->initial_wep)
    {
      retval = xsupconfwrite_initial_wep(fileid, cur->initial_wep);
      if (retval != XSUPCONFWRITE_ERRNONE) return retval;
    }

  retval = xsupconfwrite_write_allowed_eaps(fileid, cur);
  if (retval != XSUPCONFWRITE_ERRNONE) return retval;

  switch (cur->type)
    {
    case UNSET:
      // Unset is the default, so don't do anything.
      break;

    case WIRED:
      if (fprintf(fileid, "\ttype = wired\n\n") < 0) 
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case WIRELESS:
      if (fprintf(fileid, "\ttype = wireless\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    default:
      // If we have an unknown value, then don't set anything.
#ifdef DEBUG
      printf("Invalid value for interface type at %s():%d!\n",
	     __FUNCTION__, __LINE__);
#endif
      break;
    }

  if (cur->identity)
    {
      if (fprintf(fileid, "\tidentity = \"%s\"\n\n", cur->identity) < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (!TEST_FLAG(cur->flags, CONFIG_NET_USE_TNC))
    {
      if (fprintf(fileid, "\tuse_tnc = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (cur->force_eapol_ver > 0)
    {
      if (fprintf(fileid, "\tforce_eapol_ver = %d\n\n", 
		  cur->force_eapol_ver) < 0)
	{
#ifdef DEBUG
	  printf("Error writing file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  switch (cur->wireless_ctrl)
    {
    case CTL_NO:
      if (fprintf(fileid, "\twireless_control = no\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CTL_YES:
      if (fprintf(fileid, "\twireless_control = yes\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
      break;

    case CTL_UNSET:
      break;

    default:
#ifdef DEBUG
      printf("Invalid value for wireless_control in %s():%d!\n",
	     __FUNCTION__, __LINE__);
#endif
      break;
    }

  if (strncmp((char *)cur->dest_mac, "\0x00\0x00\0x00\0x00\0x00\0x00", 6) != 0)
    {
      if (fprintf(fileid, "\tdest_mac = %02X:%02X:%02X:%02X:%02X:%02X\n\n",
		  cur->dest_mac[0], cur->dest_mac[1], cur->dest_mac[2],
		  cur->dest_mac[3], cur->dest_mac[4], cur->dest_mac[5]) < 0) 
	{
#ifdef DEBUG
	  printf("Couldn't write destination MAC address!\n");
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (cur->wpa_pairwise_crypt > 0)
    {
      if (fprintf(fileid, "\twpa_pairwise_cipher = ") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      retval = xsupconfwrite_write_crypt(fileid, cur->wpa_pairwise_crypt);
      if (retval != XSUPCONFWRITE_ERRNONE) 
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return retval;
	}

      if (fprintf(fileid, "\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (cur->wpa_group_crypt > 0)
    {
      if (fprintf(fileid, "\twpa_group_cipher = ") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}

      retval = xsupconfwrite_write_crypt(fileid, cur->wpa_group_crypt);
      if (retval != XSUPCONFWRITE_ERRNONE) 
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return retval;
	}

      if (fprintf(fileid, "\n\n") < 0)
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  if (cur->priority != DEFAULT_PRIORITY)
    {
      if (fprintf(fileid, "\tpriority = %d\n\n", cur->priority) < 0)
	{
#ifdef DEBUG
	  printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
	  return XSUPCONFWRITE_GENIOERR;
	}
    }

  // Now, write the actual EAP method data.
  retval = xsupconfwrite_write_eap_methods(fileid, cur);
  if ((retval != XSUPCONFWRITE_ERRNONE) && (retval != XSUPCONFWRITE_NOEAPDATA))
    {
#ifdef DEBUG
      printf("Error : %s @ %s():%d\n\n", 
	     xsupconfwrite_strerr(retval), __FUNCTION__, __LINE__);
#endif
      return retval;
    }

  if (fprintf(fileid, "}\n\n\n") < 0) 
    {
#ifdef DEBUG
      printf("Error writing file at %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * Loop through the linked list of eap types, and create the specific 
 * configuration data for each EAP method.  (This may also include some
 * non-EAP methods such as static WEP, and WPA-PSK.)
 *
 *******************************************************************/
int xsupconfwrite_write_eaps(FILE *fileid, struct config_network *networks)
{
  struct config_network *cur;
  int retval;

  if (!fileid)
    {
#ifdef DEBUG
      printf("No valid fileid to write to at %s():%d!\n", __FUNCTION__,
	     __LINE__);
#endif
      return XSUPCONFWRITE_NOFILEHANDLE;
    }

  if (!networks)
    {
#ifdef DEBUG
      printf("No network configuration information to write to our config "
	     "file at %s():%d.\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_NONETWORKDATA;
    }

  if (fprintf(fileid, "# --- Network Definitions ---\n\n") < 0)
    {
#ifdef DEBUG
      printf("Error writing to file @ %s():%d\n", __FUNCTION__, __LINE__);
#endif
      return XSUPCONFWRITE_GENIOERR;
    }

  cur = networks;

  while (cur != NULL)
    {
      retval = xsupconfwrite_network(fileid, cur);
      if (retval != XSUPCONFWRITE_ERRNONE)
	{
#ifdef DEBUG
	  printf("There was an error writing the network configuration to the"
		 " desired file.  Please see error messages above (if any) "
		 "for a better idea of the problem.\n");
	  printf("Error was : %s\n", xsupconfwrite_strerr(retval));
#endif
	  return retval;
	}

      cur = cur->next;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * This function writes the config information out to the config file.
 * The confdata should be the root of the configuration structure that 
 * we want to create a config file for.  destfile is not a required option.
 * If destfile is set to NULL, then we will use the value that is stored
 * in the config_fname variable of the root structure.  (Which should be
 * the path to the config that we originally parsed to get the structure.)
 *
 *******************************************************************/
int xsupconfwrite_write_config(struct config_data *confdata, char *destfile)
{
  int err;
  FILE *fileid;

  if (!destfile)
    {
#ifdef DEBUG
      printf("No destination file found.  Will write to original file.\n");
#endif
      err = xsupconfwrite_create_config_file(confdata->config_fname, &fileid);
    } else {
#ifdef DEBUG
      printf("Writing to %s\n", destfile);
#endif
      err = xsupconfwrite_create_config_file(destfile, &fileid);
    }

  if (err < 0)
    {
#ifdef DEBUG
      printf("Error creating config file! Error was : %s\n", 
	     xsupconfwrite_strerr(err));
#endif
      return err;
    }

  err = xsupconfwrite_write_header(fileid);
  if (err < 0)
    {
#ifdef DEBUG
      printf("Error writing to file!  Error was : %s\n",
	     xsupconfwrite_strerr(err));
#endif
      return err;
    }

  err = xsupconfwrite_write_globals(fileid, confdata->globals);
  if (err < 0)
    {
#ifdef DEBUG
      printf("Error writing to file!  Error was : %s\n",
	     xsupconfwrite_strerr(err));
#endif
      return err;
    }

  err = xsupconfwrite_write_eaps(fileid, confdata->networks);
  if (err < 0)
    {
#ifdef DEBUG
      printf("Error writing file!  Error was : %s\n",
	     xsupconfwrite_strerr(err));
#endif
      return err;
    }

  return XSUPCONFWRITE_ERRNONE;
}

/*******************************************************************
 *
 * If an error is returned, this function can be used to create a text
 * version of the meaning of that error code.
 *
 *******************************************************************/
char *xsupconfwrite_strerr(int err)
{
  switch (err)
    {
    case XSUPCONFWRITE_ERRNONE:
      return xsupconfwrite_errnone;
      break;

    case XSUPCONFWRITE_CHECKERRNO:
      return xsupconfwrite_errcheckerrno;
      break;

    case XSUPCONFWRITE_GENIOERR:
      return xsupconfwrite_errgenioerr;
      break;
      
    case XSUPCONFWRITE_INVALID_DEST:
      return xsupconfwrite_errinvalid_dest;
      break;

    case XSUPCONFWRITE_BADNETLIST:
      return xsupconfwrite_errbadnetlist;
      break;

    case XSUPCONFWRITE_NOFILEHANDLE:
      return xsupconfwrite_errnofilehandle;
      break;

    case XSUPCONFWRITE_NOGLOBALDATA:
      return xsupconfwrite_errnoglobaldata;
      break;

    case XSUPCONFWRITE_NONETWORKDATA:
      return xsupconfwrite_errnonetworkdata;
      break;

    case XSUPCONFWRITE_NONETWORKNAME:
      return xsupconfwrite_errnonetworkname;
      break;

    case XSUPCONFWRITE_INVALIDCRYPTO:
      return xsupconfwrite_errinvalidcrypto;
      break;

    case XSUPCONFWRITE_NOEAPDATA:
      return xsupconfwrite_errnoeapdata;
      break;

    case XSUPCONFWRITE_BADTTLSPHASE2:
      return xsupconfwrite_errbadttlsphase2;
      break;

    case XSUPCONFWRITE_NOPHASE2EAPDATA:
      return xsupconfwrite_errnophase2eapdata;
      break;

    case XSUPCONFWRITE_INVALIDINNEREAP:
      return xsupconfwrite_errinvalidinnereap;
      break;

    default:
      return xsupconfwrite_errunknown;
      break;
    }
}

