/*******************************************************************
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 *******************************************************************/

#ifndef _XSUPCONFIG_H_
#define _XSUPCONFIG_H_
#include <stdint.h>

#ifndef TRUE
#define TRUE          1
#endif

#ifndef FALSE
#define FALSE         0
#endif

#define SET_FLAG(var,flag)    (var |= flag)
#define UNSET_FLAG(var,flag)  (var &= ~flag)
#define TEST_FLAG(var,flag)   (var & flag)

#define DEST_AUTO      0
#define DEST_BSSID     1
#define DEST_MULTICAST 2
#define DEST_SOURCE    3

// The default network priority that is assigned to all networks that
// don't explicitly list a priority.
#define DEFAULT_PRIORITY  0xfe

// if you change this, update the printing and parsing functions
// accordingly
#define CONFIG_MAC_LEN 6

// If a WEP key sits around for more than the timeout below, we should barf
// out a warning letting the user know that their data isn't as secure as
// it could be.
#define STALE_KEY_WARN_TIMEOUT 600   // 10 minutes.

// The amount of time we should wait before we assume an authentication will
// not be able to succeed.
#define AUTHENTICATION_TIMEOUT  30   // 30 seconds.

// The length of time to wait before we assume an assocation failed. (Defined
// by 802.11i section 8.1.3 as dot11RSNAConfigSA-Timeout.)  Default is 60, but
// we may want to provide the ability to change this in the future.
#define ASSOCIATION_TIMEOUT           60    // in seconds.

// The length of time to wait before we attempt another passive scan.
#define PASSIVE_TIMEOUT               300   // in seconds.

// The length of time to wait before we attempt another active scan.
#define RESCAN_TIMEOUT                15    // in seconds.

typedef enum {RES_UNSET, RES_YES, RES_NO} sess_res;

/*** EAP Type IDs (Only for EAP types that are implemented!!) ***/

// Bogus invalid EAP method # that indicates that we are talking about
// static WEP.
#define STATIC_WEP_METHOD   -2
#define WPA_PSK             -1
#define EAP_TYPE_MD5        4
#define EAP_TYPE_OTP        5
#define EAP_TYPE_GTC        6
#define EAP_TYPE_TLS        13
#define EAP_TYPE_LEAP       17
#define EAP_TYPE_SIM        18
#define EAP_TYPE_TTLS       21
#define EAP_TYPE_AKA        23
#define EAP_TYPE_PEAP       25
#define EAP_TYPE_MSCHAPV2   26
#define EAP_TYPE_TNC        38   /* tentative assignment per TNC_IFT_v1_0_r3 */
                                 /* (Section 7.1.3, page 27) */

/*** DEVELOPER CHECKLIST ****/
/* When adding a value to one of these structs you must
    1. update initialize_config_<struct>
    2. update delete_config_<struct>
    3. update dump_config_<struct> if it exists
    4. modify the grammar to account for the new fields (config_grammar.y)
    5. modify the lexicon for the new tokens (config_lexicon.l)
    6. add the code to have it written to a file in libxsupconfwrite.
    7. if it is a required piece to make an authentication work, add it to
        libxsupconfcheck.
*/

struct smartcard
{
  char *engine_id;
  char *opensc_so_path;
  char *key_id;
  //char *cert_id;
  //char *rootcert_id;
};

struct config_static_wep
{
  uint8_t *key[5];        // Index 0 should NEVER be used!
  uint8_t tx_key;
};

struct config_eap_tls 
{
  char * user_cert;
  char *root_cert;
  char *root_dir;
  char *crl_dir;
  char * user_key;
  char * user_key_pass;
  sess_res session_resume;
  int chunk_size;
  char * random_file;
  struct smartcard sc;

};

struct config_eap_md5
{
  char *username;
  char *password;
};

typedef enum {TTLS_PHASE2_UNDEFINED,
	TTLS_PHASE2_PAP,
	TTLS_PHASE2_CHAP,
	TTLS_PHASE2_MSCHAP,
	TTLS_PHASE2_MSCHAPV2,
        TTLS_PHASE2_EAP_MD5 } ttls_phase2_type;

struct config_ttls_phase2  
{ 
  ttls_phase2_type phase2_type;
  void *phase2_data;
  struct config_ttls_phase2 *next;
};

struct config_pap
{
  char *username;
  char *password;
};

struct config_chap
{
  char *username;
  char *password;
};

struct config_mschap
{
  char *username;
  char *password;
};

struct config_mschapv2
{
  char *username;
  char *password;
};

struct config_eap_otp
{
  char *password;
};

// The items in this structure need to match those in config_eap_tls up to 
// the random_file item, or else things may have problems.
struct config_eap_ttls
{
  char * user_cert;
  char *root_cert;
  char *root_dir;
  char *crl_dir;
  char * user_key;
  char * user_key_pass;
  sess_res session_resume;
  int  chunk_size;
  char *random_file;

  char *cncheck;
  int  cnexact;

  ttls_phase2_type phase2_type; //the type to actually do
  struct config_ttls_phase2 *phase2; // all types with info defined  
  struct generic_eap_data *phase2_eap_data;
};

struct config_eap_leap
{
  char *username;
  char *password;
};

struct config_eap_mschapv2
{
  char *username;
  char *password;
  char *nthash;
};

// The items in this structure need to match those in config_eap_tls up to 
// the random_file item, or else things may have problems.
struct config_eap_peap
{
  char * user_cert;
  char *root_cert;
  char *root_dir;
  char *crl_dir;
  char * user_key;
  char * user_key_pass;
  sess_res session_resume;
  int  chunk_size;
  char *random_file;

  char *cncheck;
  char proper_peapv1;
  int cnexact;

  char *identity; // phase2 identity

#define CONFIG_PEAP_ALLOW_MSCV2   0x00000001
#define CONFIG_PEAP_ALLOW_MD5     0x00000002
#define CONFIG_PEAP_ALLOW_SIM     0x00000004
#define CONFIG_PEAP_ALLOW_GTC     0x00000008
#define CONFIG_PEAP_ALLOW_OTP     0x00000010
#define CONFIG_PEAP_ALLOW_ALL (CONFIG_PEAP_ALLOW_MSCV2| CONFIG_PEAP_ALLOW_MD5 \
                              |CONFIG_PEAP_ALLOW_SIM  | CONFIG_PEAP_ALLOW_GTC \
                              |CONFIG_PEAP_ALLOW_OTP  )
  int flags;
  unsigned char ias_quirk;
  struct config_eap_method *phase2; 
};

struct config_eap_sim
{
  char *username;
  char *password;
  int auto_realm;
};

struct config_eap_aka
{
  char *username;
  char *password;
  int auto_realm;
};

struct config_wpa_psk
{
  char *key;
  char *hex_key;
};


/* A generic wrapper struct for above */
struct config_eap_method
{
  int method_num;
  void *method_data; // one of the structs above
  struct config_eap_method *next;
};

struct config_network
{
#define CONFIG_NET_ALLOW_TLS    0x00000001
#define CONFIG_NET_ALLOW_MD5    0x00000002
#define CONFIG_NET_ALLOW_TTLS   0x00000004
#define CONFIG_NET_ALLOW_LEAP   0x00000008
#define CONFIG_NET_ALLOW_MSCV2  0x00000010
#define CONFIG_NET_ALLOW_PEAP   0x00000020
#define CONFIG_NET_ALLOW_SIM    0x00000040
#define CONFIG_NET_ALLOW_GTC    0x00000080
#define CONFIG_NET_ALLOW_OTP    0x00000100
#define CONFIG_NET_ALLOW_AKA    0x00000200
#define CONFIG_NET_ALLOW_WPA_PSK 0x00000400
#define CONFIG_NET_ALLOW_ALL (CONFIG_NET_ALLOW_TLS  | CONFIG_NET_ALLOW_MD5  \
                             |CONFIG_NET_ALLOW_TTLS | CONFIG_NET_ALLOW_LEAP \
                             |CONFIG_NET_ALLOW_MSCV2| CONFIG_NET_ALLOW_PEAP \
                             |CONFIG_NET_ALLOW_SIM  | CONFIG_NET_ALLOW_GTC  \
                             |CONFIG_NET_ALLOW_OTP  | CONFIG_NET_ALLOW_AKA  \
                             |CONFIG_NET_ALLOW_WPA_PSK)
#define CONFIG_NET_PREFER_TLS   0x00001000
#define CONFIG_NET_PREFER_MD5   0x00002000
#define CONFIG_NET_PREFER_TTLS  0x00004000
#define CONFIG_NET_PREFER_LEAP  0x00008000
#define CONFIG_NET_PREFER_MSCV2 0x00010000
#define CONFIG_NET_PREFER_PEAP  0x00020000
#define CONFIG_NET_PREFER_SIM   0x00040000
#define CONFIG_NET_PREFER_GTC   0x00080000
#define CONFIG_NET_PREFER_OTP   0x00100000
#define CONFIG_NET_PREFER_AKA   0x00200000
#define CONFIG_NET_PREFER_WPA_PSK 0x00400000
#define CONFIG_NET_PREFER_ALL (CONFIG_NET_PREFER_TLS  | CONFIG_NET_PREFER_MD5 \
                             |CONFIG_NET_PREFER_TTLS | CONFIG_NET_PREFER_LEAP \
                             |CONFIG_NET_PREFER_MSCV2| CONFIG_NET_PREFER_PEAP \
                             |CONFIG_NET_PREFER_SIM  | CONFIG_NET_PREFER_GTC  \
                             |CONFIG_NET_PREFER_OTP  | CONFIG_NET_PREFER_AKA  \
                             |CONFIG_NET_ALLOW_WPA_PSK)

  // indicates the variable below is set and should be used
#define CONFIG_NET_DEST_MAC     0x01000000

  // indiecates that we should, or shouldn't use TNC
#define CONFIG_NET_USE_TNC      0x02000000

  char *name;
  int flags;
  enum {UNSET, WIRED, WIRELESS} type;
  char *ssid;
  char *identity;
  char force_eapol_ver;

#define ASSOC_OPEN     0
#define ASSOC_SHARED   1
#define ASSOC_LEAP     2

  uint8_t assoc_type;
  enum {CTL_UNSET, CTL_YES, CTL_NO}  wireless_ctrl;

  uint8_t  dest_mac[CONFIG_MAC_LEN];

  // Different types of encryption that are allowed.
#define CRYPT_WEP40       1
#define CRYPT_TKIP        2
#define CRYPT_WRAP        3
#define CRYPT_CCMP        4
#define CRYPT_WEP104      5

  uint8_t wpa_group_crypt;
  uint8_t wpa_pairwise_crypt;

  uint8_t priority;

  // initial set of wep keys to use
  struct config_static_wep *initial_wep;

  // EAP Methods that can be in the config file
  struct config_eap_method *methods; 
 
  // This is used to hook the currently active "phase 1" to.  It shouldn't
  // be given a value when the config is parsed!
  struct generic_eap_data *activemethod;

  struct config_network *next;
};


struct config_string_list 
{
  char *name;
  struct config_string_list *next;
};

struct config_globals
{
  char *default_net;
  struct config_string_list *allowed_nets;  
  char *logfile;
  char *default_int;
  char *log_facility;
  char *ipc_group_name;
  // the following indicate the values below are set and should be used
#define CONFIG_GLOBALS_AUTH_PER             0x00000001
#define CONFIG_GLOBALS_HELD_PER             0x00000002
#define CONFIG_GLOBALS_MAX_STARTS           0x00000004
#define CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS 0x00000008
#define CONFIG_GLOBALS_ALLMULTI             0x00000010
#define CONFIG_GLOBALS_ASSOC_AUTO           0x00000020
#define CONFIG_GLOBALS_FIRMWARE_ROAM        0x00000040
#define CONFIG_GLOBALS_PASSIVE_SCAN         0x00000080
#define CONFIG_GLOBALS_NO_EAP_HINTS         0x00000100
  int flags;
  char destination;
  int auth_period;
  int held_period;
  int max_starts;
  int stale_key_timeout;
  int assoc_timeout;
  int passive_timeout;
  int active_timeout;
};

typedef struct config_globals config_globals;
typedef struct config_network config_network;
typedef struct config_eap_method config_eap_method;

struct config_data
{
  char *config_fname;
  struct config_globals *globals;
  struct config_network *networks;
};



int config_setup(char *);
void config_destroy();
struct config_network *config_get_network_config();
struct config_network *config_get_root_network_config();
void config_set_network_config(struct config_network *);
int config_delete_net(char *);
void config_create_new_config();
void config_terminate();
struct config_globals *config_get_globals();
uint8_t config_get_network_priority(char *);
struct config_network *config_find_network(struct config_network *, char *);

int config_parse();
int config_contains_network(char *);
int config_allows_network(struct config_data *, char *);

struct config_data *config_get_config_info();

// * private functions for config code
void initialize_config_wpa_psk(struct config_wpa_psk **);
void delete_config_wpa_psk(struct config_wpa_psk **);
void dump_config_wpa_psk(struct config_wpa_psk *);

void initialize_config_static_wep(struct config_static_wep **);
void delete_config_static_wep(struct config_static_wep **);
void dump_config_static_wep(struct config_static_wep *, int);

void initialize_config_eap_tls(struct config_eap_tls **);
void delete_config_eap_tls(struct config_eap_tls **);
void dump_config_eap_tls(struct config_eap_tls *);

void initialize_config_eap_md5(struct config_eap_md5 **);
void delete_config_eap_md5(struct config_eap_md5 **);
void dump_config_eap_md5(struct config_eap_md5 *, int);

void initialize_config_pap(struct config_pap **);
void delete_config_pap(struct config_pap **);
void dump_config_pap(struct config_pap *);

void initialize_config_chap(struct config_chap **);
void delete_config_chap(struct config_chap **);
void dump_config_chap(struct config_chap *);

void initialize_config_mschap(struct config_mschap **);
void delete_config_mschap(struct config_mschap **);
void dump_config_mschap(struct config_mschap *);

void initialize_config_mschapv2(struct config_mschapv2 **);
void delete_config_mschapv2(struct config_mschapv2 **);
void dump_config_mschapv2(struct config_mschapv2 *);

void add_config_ttls_phase2(struct config_ttls_phase2 **,
			    ttls_phase2_type, void *);
int  config_ttls_phase2_contains_phase2(struct config_ttls_phase2 *,
					ttls_phase2_type);
void delete_config_ttls_phase2(struct config_ttls_phase2 **);
void dump_config_ttls_phase2(struct config_ttls_phase2 *);

void initialize_config_eap_ttls(struct config_eap_ttls **);
void delete_config_eap_ttls(struct config_eap_ttls **);
void dump_config_eap_ttls(struct config_eap_ttls *);
int  check_config_eap_ttls(struct config_eap_ttls *);

void initialize_config_eap_leap(struct config_eap_leap **);
void delete_config_eap_leap(struct config_eap_leap **);
void dump_config_eap_leap(struct config_eap_leap *);

void initialize_config_eap_mschapv2(struct config_eap_mschapv2 **);
void delete_config_eap_mschapv2(struct config_eap_mschapv2 **);
void dump_config_eap_mschapv2(struct config_eap_mschapv2 *, int);

void initialize_config_eap_peap(struct config_eap_peap **);
void delete_config_eap_peap(struct config_eap_peap **);
void dump_config_eap_peap(struct config_eap_peap *);

void initialize_config_eap_sim(struct config_eap_sim **);
void delete_config_eap_sim(struct config_eap_sim **);
void dump_config_eap_sim(struct config_eap_sim *, int);

void initialize_config_eap_aka(struct config_eap_aka **);
void delete_config_eap_aka(struct config_eap_aka **);
void dump_config_eap_aka(struct config_eap_aka *, int);

void add_config_eap_method(struct config_eap_method **,
			   int, void *);
void delete_config_eap_method(struct config_eap_method **);
int config_eap_method_contains_method(struct config_eap_method *, int);
void dump_config_eap_method(struct config_eap_method *, int);

void initialize_config_network(struct config_network **);
void delete_config_single_network(struct config_network **);
void delete_config_network(struct config_network **);
int  config_network_contains_net(struct config_network *, char *);
void config_network_add_net(struct config_network **, struct config_network *);
void dump_config_network(struct config_network *);

void initialize_config_string_list(struct config_string_list **);
void delete_config_string_list(struct config_string_list **);
int  config_string_list_contains_string(struct config_string_list *, char *);
void config_string_list_add_string(struct config_string_list **, char *);
void dump_config_string_list(struct config_string_list *, char *);

void initialize_config_globals(struct config_globals **);
void delete_config_globals(struct config_globals **);
void dump_config_globals(struct config_globals *);

void initialize_config_data(struct config_data **);
void delete_config_data(struct config_data **);
void dump_config_data(struct config_data *);

#endif
