/*******************************************************************
 * Implementation for parsing configuration file, and storing needed data.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsupconfig.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsupconfig.c,v 1.21 2006/10/05 22:23:49 chessing Exp $
 * $Date: 2006/10/05 22:23:49 $
 * $Log: xsupconfig.c,v $
 * Revision 1.21  2006/10/05 22:23:49  chessing
 * Added new association option to the config file, and attempt to associate using methods other than open system.
 *
 * Revision 1.20  2006/06/13 18:02:56  chessing
 * Fixed a problem that would result in a segfault if a HUP was received.  Created stub functions for implementation of EAP-TNC to support Trusted Network Connect.
 *
 * Revision 1.19  2006/06/05 19:56:42  chessing
 * Fixed a 16k memory leak in the config parse code.  Cleaned up some of the TLS code.
 *
 * Revision 1.18  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.17  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.16  2006/05/31 05:10:51  chessing
 * Beginnings of EAP-TTLS-EAP-MD5, and added internet draft for EAP-TTLS to doc/standards.
 *
 * Revision 1.15  2006/05/30 23:40:23  chessing
 * Small patches/additions and addition to the readme file that contains a link to the location of libtnc.
 *
 * Revision 1.14  2006/05/30 04:33:19  chessing
 * Some small updates for the GUI code.
 *
 * Revision 1.13  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.12  2006/05/18 20:40:19  chessing
 * More work on the GUI interface.  The advanced button in the configuration tab is now complete.
 *
 * Revision 1.11  2006/05/17 22:18:10  chessing
 * A couple of small changes to Xsupplicant, and some major changes to the GUI configuration/monitor tool.
 *
 * Revision 1.10  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.9  2006/04/17 03:56:23  chessing
 * Added some support to enable/disable TNC support both via the configuration file, and via IPC.
 *
 * Revision 1.8  2006/04/06 06:39:47  chessing
 * A few fixes for bugs found at Interop's Hot Stage.  We now work with madwifi-ng.
 *
 * Revision 1.7  2006/03/21 18:22:09  chessing
 * Fixes to EAP-AKA code.  Changed a few defaults to disable Xsupplicant based roaming, and passive scanning for this release.  (It would currently cause more problems than it would solve.
 *
 * Revision 1.6  2006/02/23 22:26:50  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.5  2006/01/03 04:02:35  chessing
 * Added the ability to store the PEAP password in a hashed format.  (Basically, an MS-CHAPv1 hash.)  Also added an 'ntpwdhash' program to the tools directory that will convert a cleartext password in to a hash that can be copied to the configuration file.
 *
 * Revision 1.4  2005/11/16 02:18:37  chessing
 * Updates to the config writing library.  The config-parser tool has been modified to be able to parse a config, and rewrite it.
 *
 * Revision 1.3  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.2  2005/10/17 03:56:52  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.1  2005/10/17 00:15:55  chessing
 * Moved the config parsing routines out to a new library called libxsupconfig.a, so that it will be easier to create third-party tools that can easily parse a config.
 *
 * Revision 1.82  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.81  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.80  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.79  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.78  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.77  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.76  2005/08/21 18:12:12  chessing
 * Added the ability to reload our configuration using a HUP.  Cleaned up a bunch of memory leaks in the config parse code.
 *
 * Revision 1.75  2005/08/20 19:06:52  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.74  2005/08/18 03:19:04  chessing
 * Added the ability to define an SSID with static WEP keys.  When we switch to a network that has this type of configuration we will set the keys, and stop the various association timers.
 *
 * Revision 1.73  2005/08/12 03:34:06  chessing
 * Fix to the TLS implementation, should help avoid some of the weird 'block cipher pad' errors.  Also includes a partial implementation of the ability to use static WEP keys based on the SSID in use.
 *
 * Revision 1.72  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>

#include "xsup_err.h"
#include "xsupconfig.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

struct config_data *config_info = NULL; // the configuration data structure
struct config_network *current_network_data = NULL;  // Point to the config
                                                     // data we currently are
                                                     // using.

extern FILE *yyin;
extern int config_linenum;
extern int yyparse(void);
extern int yycleanup(void);

#define FREE_STRING(x) if (x != NULL) {free(x); x = NULL;}


/****************************************
 *
 * Load all of the configuration information in to memory.  We should
 * set a flag to make sure that we have loaded the config before we
 * call config_build().
 *
 ****************************************/
int config_setup(char *path_to_config)
{
  /* Make sure we got a parameter */
  if (path_to_config == NULL) {
    return XECONFIGFILEFAIL;
  }

  /* check to see if we can really open this file */
  yyin = fopen(path_to_config, "r");
  if (yyin == NULL) {
    return XECONFIGFILEFAIL;
  }

  /* check to see if the configuration is already set */
  if (config_info) {
      return XECONFIGALREADYLOADED;
  }

  /* parse the file */
  if (config_parse() != XENONE) {
    delete_config_data(&config_info);
    return XECONFIGPARSEFAIL;
  }

  // set the file name
  if (config_info)
    config_info->config_fname = strdup(path_to_config);

  return XENONE;
}

/*****************************************************************
 *
 * Return the pointer to config_info.
 *
 *****************************************************************/
struct config_data *config_get_config_info()
{
  return config_info;
}

/*****************************************************************
 *
 *  Create a fresh config in memory with nothing other than a
 *  default network name.
 *
 *  NOTE: THIS FUNCTION SHOULD *NEVER* BE CALLED INSIDE OF XSUPPLICANT!!
 *  IT IS PROVIDED SO THAT GUI INTERFACES CAN HAVE A File|New.. OPTION!
 *
 *****************************************************************/
void config_create_new_config()
{
  initialize_config_data(&config_info);

  config_info->config_fname = NULL;   // We don't know what the file name will be.

  // Populate the globals
  initialize_config_globals(&(config_info->globals));

  // Populate our 'default' network.
  initialize_config_network(&(config_info->networks));

  config_info->globals->default_net = strdup("default");
  config_info->networks->name = strdup("default");
}

/*****************************************************************
 *
 * Given the network name 'matchname', find the network, and return the
 * network priority that was assigned to it.
 *
 *****************************************************************/
uint8_t config_get_network_priority(char *matchname)
{
  struct config_network *cur;

  if (!matchname)
    {
      printf("Looking for NULL network priority!?  (It is likely that the"
	     " scan results were truncated for some reason.)\n");
      return 0xff;
    }

  // Start at the top of the list.
  cur = config_info->networks;

  while ((cur != NULL) && (strcmp(cur->name, matchname) != 0))
    {
      cur = cur->next;
    }

  if (!cur) return 0xff;

  return cur->priority;
}

/***********************************************************
 *
 * Given a network name, find the configuration information in memory.
 *
 ***********************************************************/
struct config_network *config_find_network(struct config_network *nethead, 
					   char *matchname)
{
  struct config_network *cur;

  cur = nethead;

  if ((nethead == NULL) || (matchname == NULL))
    {
      return NULL;
    }

  while ((cur != NULL) && (strcmp(cur->name, matchname) != 0))
    {
      cur = cur->next;
    }
  
  // If we got a match, return it.
  if (cur != NULL)
    {
      return cur;
    }
  
  // Otherwise, look against the essid.
  cur = nethead;

  while (cur != NULL)
  {
    if ((cur->ssid == NULL) || (strcmp(cur->ssid,matchname) != 0)) 
	{
	  cur = cur->next;
	} else {
  	  break;
	}
  }
  
  // Do we have a match on ssid?
  if (cur != NULL)
  {
	return cur;
  }

  return NULL;
}

/****************************************
 *
 * Return our root network data structure.
 *
 ****************************************/
struct config_network *config_get_root_network_config()
{
  return config_info->networks;
}

/****************************************
 *
 * Remove a network from our list.
 *
 ****************************************/
int config_delete_net(char *netname)
{
  struct config_network *cur, *prev;

  if (strcmp(config_info->networks->name, netname) == 0)
    {
      // The first one is the one we want to remove.
      cur = config_info->networks;
      config_info->networks = cur->next;

      // Then, delete everything in that network.
      delete_config_single_network(&cur);

      return XENONE;
    }

  // Otherwise, it will be somewhere else.
  cur = config_info->networks->next;
  prev = config_info->networks;

  while ((cur) && (strcmp(cur->name, netname) != 0))
    {
      cur = cur->next;
      prev = prev->next;
    }

  if ((cur) && (strcmp(cur->name, netname) == 0))
    {
      // We found the network to delete.
      prev->next = cur->next;

      delete_config_single_network(&cur);
      return XENONE;
    }
  
  return XENOTHING_TO_DO;
}

/****************************************
 *
 * Return our current network data structure.
 *
 ****************************************/
struct config_network *config_get_network_config()
{
  return current_network_data;
}

/****************************************
 *
 * Set our current network data structure.
 *
 ****************************************/
void config_set_network_config(struct config_network *newnet)
{
  current_network_data = newnet;
}

/*****************************************************************************
 *
 * Get a pointer to the config global information.  This pointer should
 * *NEVER* be freed by any callers, or things will break!!!!!!
 *
 *****************************************************************************/
struct config_globals *config_get_globals()
{
  if (config_info == NULL)
    {
      return NULL;
    }

  return config_info->globals;
}

/************************************
 *
 * Clean up any memory that we have used to store the configuration information
 * 
 ************************************/
void config_destroy()
{
  /* close the input file */
  if (yyin)
    fclose(yyin);

  /* see if there really is something to cleanup */
  delete_config_data(&config_info);
}

/************************************
 *
 * Temporary test function for parsing
 *
 ************************************/
int config_parse()
{
  if (yyparse() != XENONE) {
    return XECONFIGPARSEFAIL;
  }

  return XENONE;
}

/*************************************
 *
 * Functions to call when the program terminates.
 *
 *************************************/
void config_terminate()
{
  yycleanup();
}


//****************************************
// CONFIG QUERIES
//****************************************

/******************************************
 *
 * See if the network config is currently in memory
 *
 ******************************************/
int config_contains_network(char *netname) 
{
  if (!config_info || !config_info->networks)
    return FALSE;
  return config_network_contains_net(config_info->networks, netname);
}

/******************************************
 *
 * See if network config is  allowed
 * 
 ******************************************/
int config_allows_network(struct config_data *conf, char *netname)
{
  struct config_string_list *current;
  // make sure we have a config and globals
  if (!conf || !conf->globals)
    return FALSE;

  current = conf->globals->allowed_nets;
  
  // lack of an allowed list means all nets are allowed
  if (current == NULL) 
    return TRUE;

  if (config_string_list_contains_string(current, netname))
    return TRUE;

  return FALSE;
}


//**********************************************
// Private functions for config parsing. Do 
// not call these from outside config code
//**********************************************

  /*********************/
 /* CONFIG_STATIC_WEP */
/*********************/

void delete_config_static_wep(struct config_static_wep **tmp_static_wep)
{
  int i;

  if ((*tmp_static_wep) == NULL)
    return;

  for (i=1;i<5;i++)
    {
      FREE_STRING((*tmp_static_wep)->key[i]);
    }

  free(*tmp_static_wep);
  *tmp_static_wep = NULL;
}

void initialize_config_static_wep(struct config_static_wep **tmp_static_wep)
{
  if (*tmp_static_wep != NULL) {
    delete_config_static_wep(tmp_static_wep);
  }
  *tmp_static_wep = 
    (struct config_static_wep *)malloc(sizeof(struct config_static_wep));
  if (*tmp_static_wep)
    memset(*tmp_static_wep, 0, sizeof(struct config_static_wep));
}

void dump_config_static_wep(struct config_static_wep *tmp_static_wep, 
			    int initial)
{
  int i;

  if (!tmp_static_wep)
    return;

  if (initial)
    {
      printf("\t------------initial wep-------------\n");
    } else {
      printf("\t-------------static-wep-------------\n");
    }
  printf("\t TX Key       : %d\n", tmp_static_wep->tx_key);
  
  for (i=1;i<5;i++)
    {
      if (tmp_static_wep->key[i])
	printf("\t Static Key %d : %s\n", i, tmp_static_wep->key[i]);
    }
  printf("\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_WPA_PSK  */
/*******************/

void delete_config_wpa_psk(struct config_wpa_psk **tmp_wpa_psk)
{
  if (*tmp_wpa_psk == NULL)
    return;

  FREE_STRING((*tmp_wpa_psk)->key);
  FREE_STRING((*tmp_wpa_psk)->hex_key);

  free(*tmp_wpa_psk);
  *tmp_wpa_psk = NULL;
}

void initialize_config_wpa_psk(struct config_wpa_psk **tmp_wpa_psk)
{
  if (*tmp_wpa_psk != NULL) {
    delete_config_wpa_psk(tmp_wpa_psk);
  }
  *tmp_wpa_psk = 
    (struct config_wpa_psk *)malloc(sizeof(struct config_wpa_psk));
  if (*tmp_wpa_psk)
    memset(*tmp_wpa_psk, 0, sizeof(struct config_wpa_psk));
}

void dump_config_wpa_psk(struct config_wpa_psk *tmp_wpa_psk)
{
  if (!tmp_wpa_psk)
    return;
  printf("\t---------------wpa-psk--------------\n");
  printf("\t  ASCII key: \"%s\"\n", tmp_wpa_psk->key);
  printf("\t  HEX key: \"%s\"\n", tmp_wpa_psk->hex_key);
  printf("\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_TLS      */
/*******************/

/* take a pointer to a config_eap_tls and cleanly delete the structure
   then make it NULL */
void delete_config_eap_tls(struct config_eap_tls **tmp_tls)
{
  if (*tmp_tls == NULL)
    return;

  FREE_STRING((*tmp_tls)->user_cert);
  FREE_STRING((*tmp_tls)->root_cert);
  FREE_STRING((*tmp_tls)->root_dir);  
  FREE_STRING((*tmp_tls)->crl_dir);  
  FREE_STRING((*tmp_tls)->user_key);
  FREE_STRING((*tmp_tls)->user_key_pass);
  FREE_STRING((*tmp_tls)->random_file);
  FREE_STRING((*tmp_tls)->sc.engine_id);
  FREE_STRING((*tmp_tls)->sc.opensc_so_path);
  FREE_STRING((*tmp_tls)->sc.key_id);
  
  free (*tmp_tls);
  *tmp_tls = NULL;
}

/* take a pointer to a config_eap_tls and put a blank one there */
void initialize_config_eap_tls(struct config_eap_tls **tmp_tls)
{
  if (*tmp_tls != NULL) {
    delete_config_eap_tls(tmp_tls);
  }
  *tmp_tls = 
    (struct config_eap_tls *)malloc(sizeof(struct config_eap_tls));
  if (*tmp_tls)
    {
      memset(*tmp_tls, 0, sizeof(struct config_eap_tls));
      bzero(&(*tmp_tls)->sc, sizeof((*tmp_tls)->sc));
    }
}

void dump_config_eap_tls(struct config_eap_tls *tls)
{
  if (!tls)
    return;
  printf("\t---------------eap-tls--------------\n");
  printf("\t  TLS Cert: \"%s\"\n", tls->user_cert);
  printf("\t  TLS Root Cert: \"%s\"\n", tls->root_cert);
  printf("\t  TLS Root Dir: \"%s\"\n", tls->root_dir);
  printf("\t  TLS CRL Dir: \"%s\"\n", tls->crl_dir);
  printf("\t  TLS Key: \"%s\"\n", tls->user_key);
  printf("\t  TLS Key Pass: \"%s\"\n", tls->user_key_pass);
  printf("\t  TLS Chunk Size: %d\n", tls->chunk_size);
  printf("\t  TLS Random Source: \"%s\"\n", 
	       tls->random_file);
  printf("\t  TLS Session Resumption: ");
  switch (tls->session_resume)
    {
    case RES_UNSET:
      printf("UNSET\n");
      break;
    case RES_YES:
      printf("YES\n");
      break;
    case RES_NO:
      printf("NO\n");
      break;
    }
  printf("\t  Using Smartcard:\n");
  printf("\t\tEngine        : \"%s\"\n", tls->sc.engine_id);
  printf("\t\tOpensc SO_PATH: \"%s\"\n", tls->sc.opensc_so_path);
  printf("\t\tKey ID        : \"%s\"\n", tls->sc.key_id);
  //printf("\t\tCertificate ID: \"%s\"", tls.sc->cert_id);
  //printf("\t\tRoot Cert ID  : \"%s\"", tls.sc->root_id); 
  printf("\t------------------------------------\n");
}


  /*******************/
 /* CONFIG_MD5      */
/*******************/
void delete_config_eap_md5(struct config_eap_md5 **tmp_md5)
{
  if (*tmp_md5 == NULL)
    return;

  FREE_STRING((*tmp_md5)->username);
  FREE_STRING((*tmp_md5)->password);

  free (*tmp_md5);
  *tmp_md5 = NULL;
}

void initialize_config_eap_md5(struct config_eap_md5 **tmp_md5)
{
  if (*tmp_md5 != NULL) {
    delete_config_eap_md5(tmp_md5);
  }
  *tmp_md5 = 
    (struct config_eap_md5 *)malloc(sizeof(struct config_eap_md5));  
  if (*tmp_md5)
    memset(*tmp_md5, 0, sizeof(struct config_eap_md5));
}

void dump_config_eap_md5(struct config_eap_md5 *md5, int level)
{
  if (!md5)
    return;
  if (level == 0) {
    printf("\t---------------eap-md5--------------\n");
    printf("\t  MD5 User: \"%s\"\n", md5->username);
    printf("\t  MD5 Pass: \"%s\"\n", md5->password);
    printf("\t------------------------------------\n");
  }else {
    printf("\t\t^ ^ ^  eap-md5  ^ ^ ^\n");
    printf("\t\t  MD5 User: \"%s\"\n", md5->username);
    printf("\t\t  MD5 Pass: \"%s\"\n", md5->password);
    printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");    
  }
  
}


  /*******************/
 /* CONFIG_TTLS     */
/*******************/

//--------
// PAP
//--------
void delete_config_pap(struct config_pap **tmp_pap)
{
  if (*tmp_pap == NULL)
    return;

  FREE_STRING((*tmp_pap)->username);
  FREE_STRING((*tmp_pap)->password);

  free (*tmp_pap);
  *tmp_pap = NULL;
}

void initialize_config_pap(struct config_pap **tmp_pap)
{
  if (*tmp_pap != NULL) {
    delete_config_pap(tmp_pap);
  }
  *tmp_pap = 
    (struct config_pap *)malloc(sizeof(struct config_pap));  
  if (*tmp_pap)
    memset(*tmp_pap, 0, sizeof(struct config_pap));
}

void dump_config_pap(struct config_pap *pap)
{
  if (!pap)
    return;
  printf("\t\t^ ^ ^  pap  ^ ^ ^\n");
  printf("\t\t  PAP User: \"%s\"\n", pap->username);
  printf("\t\t  PAP Pass: \"%s\"\n", pap->password);
  printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// CHAP
//--------
void delete_config_chap(struct config_chap **tmp_chap)
{
  if (*tmp_chap == NULL)
    return;

  FREE_STRING((*tmp_chap)->username);
  FREE_STRING((*tmp_chap)->password);

  free (*tmp_chap);
  *tmp_chap = NULL;
}

void initialize_config_chap(struct config_chap **tmp_chap)
{
  if (*tmp_chap != NULL) {
    delete_config_chap(tmp_chap);
  }
  *tmp_chap = 
    (struct config_chap *)malloc(sizeof(struct config_chap));  
  if (*tmp_chap)
    memset(*tmp_chap, 0, sizeof(struct config_chap));
}

void dump_config_chap(struct config_chap *chap)
{
  if (!chap)
    return;
  printf("\t\t^ ^ ^  chap  ^ ^ ^\n");
  printf("\t\t  CHAP User: \"%s\"\n", chap->username);
  printf("\t\t  CHAP Pass: \"%s\"\n", chap->password);
  printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// MSCHAP
//--------
void delete_config_mschap(struct config_mschap **tmp_mschap)
{
  if (*tmp_mschap == NULL)
    return;

  FREE_STRING((*tmp_mschap)->username);
  FREE_STRING((*tmp_mschap)->password);

  free (*tmp_mschap);
  *tmp_mschap = NULL;
}

void initialize_config_mschap(struct config_mschap **tmp_mschap)
{
  if (*tmp_mschap != NULL) {
    delete_config_mschap(tmp_mschap);
  }
  *tmp_mschap = 
    (struct config_mschap *)malloc(sizeof(struct config_mschap));  
  if (*tmp_mschap)
    memset(*tmp_mschap, 0, sizeof(struct config_mschap));
}

void dump_config_mschap(struct config_mschap *mschap)
{
  if (!mschap)
    return;
  printf("\t\t^ ^ ^  mschap  ^ ^ ^\n");
  printf("\t\t  MSCHAP User: \"%s\"\n", mschap->username);
  printf("\t\t  MSCHAP Pass: \"%s\"\n", mschap->password);
  printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// MSCHAPV2
//--------
void delete_config_mschapv2(struct config_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 == NULL)
    return;

  FREE_STRING((*tmp_mschapv2)->username);
  FREE_STRING((*tmp_mschapv2)->password);

  free (*tmp_mschapv2);
  *tmp_mschapv2 = NULL;
}

void initialize_config_mschapv2(struct config_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 != NULL) {
    delete_config_mschapv2(tmp_mschapv2);
  }
  *tmp_mschapv2 = 
    (struct config_mschapv2 *)malloc(sizeof(struct config_mschapv2));  
  if (*tmp_mschapv2)
    memset(*tmp_mschapv2, 0, sizeof(struct config_mschapv2));
}

void dump_config_mschapv2(struct config_mschapv2 *mschapv2)
{
  if (!mschapv2)
    return;
  printf("\t\t^ ^ ^  mschapv2  ^ ^ ^\n");
  printf("\t\t  MSCHAPv2 User   : \"%s\"\n", mschapv2->username);
  printf("\t\t  MSCHAPv2 Pass   : \"%s\"\n", mschapv2->password);
  printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}


//-------------
// TTLS_PHASE2
//------------
// Be SURE to call config_ttls_phase2_contains_phase2 BEFORE adding.
// no such check will be done here.
void add_config_ttls_phase2(struct config_ttls_phase2 **phase2,
			   ttls_phase2_type phase2_type, void *phase2_data)
{
  struct config_ttls_phase2 *tmp, *newphase2;

  if (!phase2_data)
    return;

  newphase2 = 
    (struct config_ttls_phase2 *)malloc(sizeof(struct config_ttls_phase2));
  if (newphase2 == NULL)
    return;
  memset(newphase2, 0, sizeof(struct config_ttls_phase2));
  newphase2->phase2_type = phase2_type;
  newphase2->phase2_data = phase2_data;
  
  if (*phase2 == NULL) {
    *phase2 = newphase2;
    return;
  }

  tmp = *phase2;

  while (tmp->next != NULL) {
    tmp = tmp->next;
  }
  tmp->next = newphase2;
}

int config_ttls_phase2_contains_phase2(struct config_ttls_phase2 *phase2,
				      ttls_phase2_type new_type)
{
  struct config_ttls_phase2 *tmp;

  if (!phase2)
    return 0;
  
  tmp = phase2;
  while (tmp) {
    if (tmp->phase2_type == new_type)
      return 1;
    tmp = tmp->next;
  }

  return 0;
}

void delete_config_ttls_phase2 (struct config_ttls_phase2 **phase2)
{
  struct config_ttls_phase2 *temp;

  if (*phase2 == NULL)
    return;
  switch ((*phase2)->phase2_type) {
  case TTLS_PHASE2_PAP:
    delete_config_pap((struct config_pap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_CHAP: 
    delete_config_chap((struct config_chap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAP:
    delete_config_mschap((struct config_mschap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAPV2:
    delete_config_mschapv2((struct config_mschapv2 **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_EAP_MD5:
    delete_config_eap_md5((struct config_eap_md5 **)&(*phase2)->phase2_data);
    break;
  default:
    printf("AAAH! Trying to delete an undefined config"
	   " type.\nNotify developers. Type: 0x%x\n", 
	   (*phase2)->phase2_type);
  }
  temp = (*phase2)->next;
  free((*phase2));

  if (temp)
    delete_config_ttls_phase2(&temp);
}

void dump_config_ttls_phase2(struct config_ttls_phase2 *phase2) {
  if (phase2 == NULL)
    return;
  switch ((phase2)->phase2_type) {
  case TTLS_PHASE2_PAP:
    dump_config_pap((struct config_pap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_CHAP: 
    dump_config_chap((struct config_chap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAP:
    dump_config_mschap((struct config_mschap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAPV2:
    dump_config_mschapv2((struct config_mschapv2 *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_EAP_MD5:
    dump_config_eap_md5((struct config_eap_md5 *)(phase2)->phase2_data, 1);
    break;
  default:
    printf("AAAH! Trying to dump an undefined config"
	   " type.\nNotify developers. Type: 0x%x\n", 
	   (phase2)->phase2_type);
  }
  if ((phase2)->next)
    dump_config_ttls_phase2((phase2)->next);
}

void delete_config_eap_ttls(struct config_eap_ttls **tmp_ttls)
{
  if (*tmp_ttls == NULL)
    return;

  FREE_STRING((*tmp_ttls)->user_cert);
  FREE_STRING((*tmp_ttls)->root_cert);
  FREE_STRING((*tmp_ttls)->root_dir);
  FREE_STRING((*tmp_ttls)->crl_dir);
  FREE_STRING((*tmp_ttls)->user_key);
  FREE_STRING((*tmp_ttls)->user_key_pass);
  FREE_STRING((*tmp_ttls)->random_file);  
  FREE_STRING((*tmp_ttls)->cncheck);
  if ((*tmp_ttls)->phase2) 
    delete_config_ttls_phase2(&(*tmp_ttls)->phase2);

  free (*tmp_ttls);
  *tmp_ttls = NULL;
}

void initialize_config_eap_ttls(struct config_eap_ttls **tmp_ttls)
{
  if (*tmp_ttls != NULL) {
    delete_config_eap_ttls(tmp_ttls);
  }
  *tmp_ttls = 
    (struct config_eap_ttls *)malloc(sizeof(struct config_eap_ttls));  
  if (*tmp_ttls == NULL)
    return;
  memset(*tmp_ttls, 0, sizeof(struct config_eap_ttls));

  (*tmp_ttls)->phase2_type = TTLS_PHASE2_UNDEFINED;
}

void dump_config_eap_ttls(struct config_eap_ttls *ttls)
{
  if (!ttls) {
    return;
  }
  printf("\t---------------eap-ttls--------------\n");
  printf("\t  TTLS Cert: \"%s\"\n", ttls->user_cert);
  printf("\t  TTLS Root Cert: \"%s\"\n", ttls->root_cert);
  printf("\t  TTLS Root Dir: \"%s\"\n", ttls->root_dir);
  printf("\t  TTLS CRL Dir: \"%s\"\n", ttls->crl_dir);
  printf("\t  TTLS Key: \"%s\"\n", ttls->user_key);
  printf("\t  TTLS Key Pass: \"%s\"\n", ttls->user_key_pass);
  printf("\t  TTLS Chunk Size: %d\n", ttls->chunk_size);
  printf("\t  TTLS Random Source: \"%s\"\n", ttls->random_file);
  printf("\t  TTLS CN to Check : \"%s\"\n", ttls->cncheck);
  printf("\t  TTLS Exact CN Match : %s\n", ttls->cnexact ? "yes" : "no"); 
  printf("\t  TTLS Session Resumption: ");
  switch (ttls->session_resume)
    {
    case RES_UNSET:
      printf("UNSET\n");
      break;
    case RES_YES:
      printf("YES\n");
      break;
    case RES_NO:
      printf("NO\n");
      break;
    }
  switch (ttls->phase2_type) {
  case TTLS_PHASE2_PAP:
    printf("\t  TTLS phase2: pap\n");    
    break;
  case TTLS_PHASE2_CHAP:
    printf("\t  TTLS phase2: chap\n");    
    break;
  case TTLS_PHASE2_MSCHAP:
    printf("\t  TTLS phase2: mschap\n");    
    break;
  case TTLS_PHASE2_MSCHAPV2:
    printf("\t  TTLS phase2: mschapv2\n");        
    break;
  case TTLS_PHASE2_EAP_MD5:
    printf("\t  TTLS phase2: eap_md5\n");
    break;
  default:
    printf("\t  TTLS phase2: UNDEFINED\n");    
    break;
  }
  if (ttls->phase2) dump_config_ttls_phase2(ttls->phase2);
  printf("\t------------------------------------\n");
}

int check_config_eap_ttls(struct config_eap_ttls *tmp_ttls)
{
  int errno = 0;
  
  if (tmp_ttls->phase2_type == TTLS_PHASE2_UNDEFINED || !tmp_ttls->phase2) {
    printf("No phase2 defined for ttls\n");
    errno = -1;
  }
  
  if (!config_ttls_phase2_contains_phase2(tmp_ttls->phase2, 
					  tmp_ttls->phase2_type)) {
    printf("Phase2 type chosen, but not defined.\n");
    errno = -1;      
  }
  return errno;
}

  /*******************/
 /* CONFIG_LEAP     */
/*******************/
void delete_config_eap_leap(struct config_eap_leap **tmp_leap)
{
  if (*tmp_leap == NULL)
    return;

  FREE_STRING((*tmp_leap)->username);
  FREE_STRING((*tmp_leap)->password);

  free (*tmp_leap);
  *tmp_leap = NULL;
}

void initialize_config_eap_leap(struct config_eap_leap **tmp_leap)
{
  if (*tmp_leap != NULL) {
    delete_config_eap_leap(tmp_leap);
  }
  *tmp_leap = 
    (struct config_eap_leap *)malloc(sizeof(struct config_eap_leap));  
  if (*tmp_leap)
    memset(*tmp_leap, 0, sizeof(struct config_eap_leap));
}

void dump_config_eap_leap(struct config_eap_leap *leap)
{
  if (!leap)
    return;
  printf("\t---------------eap-leap--------------\n");
  printf("\t  LEAP User: \"%s\"\n", leap->username);
  printf("\t  LEAP Pass: \"%s\"\n", leap->password);
  printf("\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_MSCHAPV2 */
/*******************/
void delete_config_eap_mschapv2(struct config_eap_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 == NULL)
    return;

  FREE_STRING((*tmp_mschapv2)->username);
  FREE_STRING((*tmp_mschapv2)->password);
  FREE_STRING((*tmp_mschapv2)->nthash);

  free (*tmp_mschapv2);
  *tmp_mschapv2 = NULL;
}

void initialize_config_eap_mschapv2(struct config_eap_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 != NULL) {
    delete_config_eap_mschapv2(tmp_mschapv2);
  }
  *tmp_mschapv2 = 
    (struct config_eap_mschapv2 *)malloc(sizeof(struct config_eap_mschapv2));  
  if (*tmp_mschapv2)
    memset(*tmp_mschapv2, 0, sizeof(struct config_eap_mschapv2));
}

void dump_config_eap_mschapv2(struct config_eap_mschapv2 *mschapv2, int level)
{
  if (!mschapv2)
    return;
  if (level == 0) {
    printf("\t---------------eap-mschapv2--------------\n");
    printf("\t  MSCHAPV2 User      : \"%s\"\n", mschapv2->username);
    printf("\t  MSCHAPV2 Pass      : \"%s\"\n", mschapv2->password);
    printf("\t  MSCHAPV2 NtPwd Hash: \"%s\"\n", mschapv2->nthash);
    printf("\t------------------------------------\n");
  }else {
  printf("\t\t^ ^ ^  eap-mschapv2  ^ ^ ^\n");
    printf("\t\t  MSCHAPV2 User      : \"%s\"\n", mschapv2->username);
    printf("\t\t  MSCHAPV2 Pass      : \"%s\"\n", mschapv2->password);
    printf("\t\t  MSCHAPV2 NtPwd Hash: \"%s\"\n", mschapv2->nthash);
  printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }

}


  /*******************/
 /* CONFIG_PEAP     */
/*******************/
void delete_config_eap_peap(struct config_eap_peap **tmp_peap)
{
  if (*tmp_peap == NULL)
    return;

  FREE_STRING((*tmp_peap)->identity);
  FREE_STRING((*tmp_peap)->user_cert);
  FREE_STRING((*tmp_peap)->root_cert);
  FREE_STRING((*tmp_peap)->root_dir);
  FREE_STRING((*tmp_peap)->crl_dir);
  FREE_STRING((*tmp_peap)->user_key)
  FREE_STRING((*tmp_peap)->user_key_pass);
  FREE_STRING((*tmp_peap)->random_file);
  FREE_STRING((*tmp_peap)->cncheck);
  if ((*tmp_peap)->phase2)
    delete_config_eap_method(&(*tmp_peap)->phase2);

  free (*tmp_peap);
  *tmp_peap = NULL;
}

void initialize_config_eap_peap(struct config_eap_peap **tmp_peap)
{
  if (*tmp_peap != NULL) {
    delete_config_eap_peap(tmp_peap);
  }
  *tmp_peap = 
    (struct config_eap_peap *)malloc(sizeof(struct config_eap_peap));  
  if (*tmp_peap)
    {
      memset(*tmp_peap, 0, sizeof(struct config_eap_peap));
    }
}

void dump_config_eap_peap(struct config_eap_peap *peap)
{
  if (!peap)
    return;
  printf("\t---------------eap-peap--------------\n");
  printf("\t  PEAP phase2 identity: \"%s\"\n", peap->identity);
  printf("\t  PEAP Cert: \"%s\"\n", peap->user_cert);
  printf("\t  PEAP Root Cert: \"%s\"\n", peap->root_cert);
  printf("\t  PEAP Root Dir: \"%s\"\n", peap->root_dir);
  printf("\t  PEAP CRL Dir: \"%s\"\n", peap->crl_dir);
  printf("\t  PEAP Key: \"%s\"\n", peap->user_key);
  printf("\t  PEAP Key Pass: \"%s\"\n", peap->user_key_pass);
  printf("\t  PEAP Chunk Size: %d\n", peap->chunk_size);
  printf("\t  PEAP Random Source: \"%s\"\n", peap->random_file);
  printf("\t  PEAP CN to Check : \"%s\"\n", peap->cncheck);
  printf("\t  PEAP Exact CN Match : %s\n", peap->cnexact ? "yes" : "no");  
  printf("\t  PEAP Session Resumption: ");
  switch (peap->session_resume)
    {
    case RES_UNSET:
      printf("UNSET\n");
      break;
    case RES_YES:
      printf("YES\n");
      break;
    case RES_NO:
      printf("NO\n");
      break;
    }
  printf("\t  Broken PEAPv1 Keying : ");
  switch (peap->proper_peapv1)
    {
    case 0:
      printf("NO\n");
      break;
    case 1:
      printf("YES\n");
      break;
    }
  printf("\t  PEAP IAS_QUIRK: \"%s\"\n", peap->ias_quirk ? "Yes" : "No");
    
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_MSCV2))
    printf("\t   Allow Phase 2 Type: MSCHAPv2\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_MD5))
    printf("\t   Allow Phase 2 Type: MD5\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_SIM))
    printf("\t   Allow Phase 2 Type: SIM\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_GTC))
    printf("\t   Allow Phase 2 Type: GTC\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_OTP))
    printf("\t   Allow Phase 2 Type: OTP\n");
  if (peap->phase2) dump_config_eap_method(peap->phase2, 1);
  printf("\t------------------------------------\n");
}


  /*******************/
 /* CONFIG_SIM      */
/*******************/
void delete_config_eap_sim(struct config_eap_sim **tmp_sim)
{
  if (*tmp_sim == NULL)
    return;

  FREE_STRING((*tmp_sim)->username);
  FREE_STRING((*tmp_sim)->password);

  free (*tmp_sim);
  *tmp_sim = NULL;
}

void initialize_config_eap_sim(struct config_eap_sim **tmp_sim)
{
  if (*tmp_sim != NULL) {
    delete_config_eap_sim(tmp_sim);
  }
  *tmp_sim = 
    (struct config_eap_sim *)malloc(sizeof(struct config_eap_sim));  
  if (*tmp_sim)
    memset(*tmp_sim, 0, sizeof(struct config_eap_sim));
}

void dump_config_eap_sim(struct config_eap_sim *sim, int level)
{
  if (!sim)
    return;
  if (level == 0) {
    printf("\t---------------eap-sim--------------\n");
    printf("\t  SIM User: \"%s\"\n", sim->username);
    printf("\t  SIM Pass: \"%s\"\n", sim->password);
    printf("\t  SIM Auto Realm: %s\n", sim->auto_realm ? "yes" : "no");  
    printf("\t------------------------------------\n");
  } else {
    printf("\t\t^ ^ ^  eap-sim  ^ ^ ^\n");
    printf("\t\t  SIM User: \"%s\"\n", sim->username);
    printf("\t\t  SIM Pass: \"%s\"\n", sim->password);
    printf("\t\t  SIM Auto Realm: %s\n", sim->auto_realm ? "yes" : "no");  
    printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }
}

  /*******************/
 /* CONFIG_AKA      */
/*******************/
void delete_config_eap_aka(struct config_eap_aka **tmp_aka)
{
  if (*tmp_aka == NULL)
    return;

  FREE_STRING((*tmp_aka)->username);
  FREE_STRING((*tmp_aka)->password);

  free (*tmp_aka);
  *tmp_aka = NULL;
}

void initialize_config_eap_aka(struct config_eap_aka **tmp_aka)
{
  if (*tmp_aka != NULL) {
    delete_config_eap_aka(tmp_aka);
  }
  *tmp_aka = 
    (struct config_eap_aka *)malloc(sizeof(struct config_eap_aka));  
  if (*tmp_aka)
    memset(*tmp_aka, 0, sizeof(struct config_eap_aka));
}

void dump_config_eap_aka(struct config_eap_aka *aka, int level)
{
  if (!aka)
    return;
  if (level == 0) {
    printf("\t---------------eap-aka--------------\n");
    printf("\t  AKA User: \"%s\"\n", aka->username);
    printf("\t  AKA Pass: \"%s\"\n", aka->password);
    printf("\t  AKA Auto Realm: %s\n", aka->auto_realm ? "yes" : "no");  
    printf("\t------------------------------------\n");
  } else {
    printf("\t\t^ ^ ^  eap-aka  ^ ^ ^\n");
    printf("\t\t  AKA User: \"%s\"\n", aka->username);
    printf("\t\t  AKA Pass: \"%s\"\n", aka->password);
    printf("\t\t  AKA Auto Realm: %s\n", aka->auto_realm ? "yes" : "no");  
    printf("\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }
}


  /*********************/
 /* CONFIG_EAP_METHOD */
/*********************/

// Be SURE to call config_eap_method_contains_method BEFORE adding.
// no such check will be done here.
void add_config_eap_method(struct config_eap_method **method,
			   int method_num, void *method_data)
{
  struct config_eap_method *tmp, *newmethod;

  if (!method_data)
    return;

  newmethod = 
    (struct config_eap_method *)malloc(sizeof(struct config_eap_method));
  if (newmethod == NULL)
    return;
  memset(newmethod, 0, sizeof(struct config_eap_method));
  newmethod->method_num = method_num;
  newmethod->method_data = method_data;
  
  if (*method == NULL) {
    *method = newmethod;
    return;
  }

  tmp = *method;

  while (tmp->next != NULL) {
    tmp = tmp->next;
  }
  tmp->next = newmethod;
}

void delete_config_eap_method(struct config_eap_method **method)
{
  struct config_eap_method *temp;

  if (*method == NULL)
    return;
  switch ((*method)->method_num) {
  case STATIC_WEP_METHOD:
    delete_config_static_wep((struct config_static_wep **)&((*method)->method_data));
    break;
  case WPA_PSK:
    delete_config_wpa_psk((struct config_wpa_psk **)&((*method)->method_data));
    break;
  case EAP_TYPE_TLS:
    delete_config_eap_tls((struct config_eap_tls **)&((*method)->method_data));
    break;
  case EAP_TYPE_MD5:
    delete_config_eap_md5((struct config_eap_md5 **)&(*method)->method_data);
    break;
  case EAP_TYPE_PEAP:
    delete_config_eap_peap((struct config_eap_peap **)&(*method)->method_data);
    break;

#ifdef EAP_SIM_ENABLE
  case EAP_TYPE_SIM:
    delete_config_eap_sim((struct config_eap_sim **)&(*method)->method_data);
    break;
  case EAP_TYPE_AKA:
    delete_config_eap_aka((struct config_eap_aka **)&(*method)->method_data);
    break;
#endif

  case EAP_TYPE_TTLS:
    delete_config_eap_ttls((struct config_eap_ttls **)&(*method)->method_data);
    break; 
  case EAP_TYPE_LEAP:
    delete_config_eap_leap((struct config_eap_leap **)&(*method)->method_data);
    break;
  case EAP_TYPE_MSCHAPV2:
    delete_config_eap_mschapv2((struct config_eap_mschapv2 **)&(*method)->method_data);
    break;
    
  default:
    printf("AAAH! Trying to delete an undefined config"
	   " type.\nNotify developers. Type: 0x%x\n", 
	   (*method)->method_num);
  }

  temp = (*method)->next;
  free((*method));

  if (temp)
    delete_config_eap_method(&temp);
  
}

void dump_config_eap_method(struct config_eap_method *method, int dumplevel)
{
  if (method == NULL)
    return;

  switch ((method)->method_num) {
  case STATIC_WEP_METHOD:
    dump_config_static_wep((struct config_static_wep *)((method)->method_data),
			   FALSE);
    break;
  case WPA_PSK:
    dump_config_wpa_psk((struct config_wpa_psk *)((method)->method_data));
    break;
  case EAP_TYPE_TLS:
    dump_config_eap_tls((struct config_eap_tls *)((method)->method_data));
    break;
  case EAP_TYPE_MD5:
    dump_config_eap_md5((struct config_eap_md5 *)(method)->method_data, 
			dumplevel);
    break;
  case EAP_TYPE_PEAP:
    dump_config_eap_peap((struct config_eap_peap *)(method)->method_data);
    break;

#ifdef EAP_SIM_ENABLE
  case EAP_TYPE_SIM:
    dump_config_eap_sim((struct config_eap_sim *)(method)->method_data,
			dumplevel);
    break;
  case EAP_TYPE_AKA:
    dump_config_eap_aka((struct config_eap_aka *)(method)->method_data,
			dumplevel);
    break;
#endif

  case EAP_TYPE_TTLS:
    dump_config_eap_ttls((struct config_eap_ttls *)(method)->method_data);
    break; 
  case EAP_TYPE_LEAP:
    dump_config_eap_leap((struct config_eap_leap *)(method)->method_data);
  case EAP_TYPE_MSCHAPV2:
    dump_config_eap_mschapv2((struct config_eap_mschapv2 *)(method)->method_data,
			     dumplevel);
    break;
    
  default:
    printf("AAAH! Trying to dump an undefined config"
	   " type\n.Notify developers. Type: 0x%x\n", 
	   (method)->method_num);
  }

  dump_config_eap_method(method->next, dumplevel);
}

int config_eap_method_contains_method(struct config_eap_method *method,
				      int new_num)
{
  struct config_eap_method *tmp;

  if (!method)
    return 0;
  
  tmp = method;
  while (tmp) {
    if (tmp->method_num == new_num)
      return 1;
    tmp = tmp->next;
  }

  return 0;
}

  /*******************/
 /* CONFIG_NETWORK  */
/*******************/
void delete_config_single_network(struct config_network **tmp_network)
{
  if (*tmp_network == NULL)
    return;

  FREE_STRING((*tmp_network)->name);
  FREE_STRING((*tmp_network)->ssid);
  FREE_STRING((*tmp_network)->identity);

  if ((*tmp_network)->initial_wep)
    delete_config_static_wep(&(*tmp_network)->initial_wep);

  if ((*tmp_network)->methods)
    delete_config_eap_method(&(*tmp_network)->methods);

  free ((*tmp_network));
  (*tmp_network) = NULL;
}

void delete_config_network(struct config_network **tmp_network)
{
  struct config_network *next, *cur;

  if (*tmp_network == NULL)
    return;

  cur = (*tmp_network);
  next = (*tmp_network)->next;

  while (cur)
    {
      delete_config_single_network(&cur);
      cur = next;
      if (next)
	{
	  next = next->next;
	}
    }

  *tmp_network = NULL;
}

void initialize_config_network(struct config_network **tmp_network)
{
  if (*tmp_network != NULL) {
    delete_config_network(tmp_network);
  }
  *tmp_network = 
    (struct config_network *)malloc(sizeof(struct config_network));  
  if (*tmp_network)
    memset(*tmp_network, 0, sizeof(struct config_network));

  (*tmp_network)->priority = DEFAULT_PRIORITY;
  SET_FLAG((*tmp_network)->flags, CONFIG_NET_USE_TNC);
}

int config_network_contains_net(struct config_network *net, char *netname)
{
  while (net != NULL) {
    if (strcmp(net->name, netname) == 0)
      return TRUE;
    net = net->next;
  }
  return FALSE;
}

void config_network_add_net(struct config_network **list, 
			    struct config_network *toadd)
{
  struct config_network **current = list;
  while (*current != NULL) {
    if (strcmp((*current)->name, toadd->name) == 0) {
      return;
    }
    current = &(*current)->next;
  }
  (*current) = toadd;
}

void dump_config_network_wpa(uint8_t crypt)
{
  switch (crypt)
    {
    case CRYPT_WEP40:
      printf("WEP40\n");
      return;

    case CRYPT_TKIP:
      printf("TKIP\n");
      return;

    case CRYPT_WRAP:
      printf("WRAP\n");
      return;

    case CRYPT_CCMP:
      printf("CCMP\n");
      return;

    case CRYPT_WEP104:
      printf("WEP104\n");
      return;
    
    default:
      printf("NONE\n");
      return;
    }
}

void dump_config_network(struct config_network *net)
{
  if (!net)
    return;
  printf("+-+-+-+-+  Network Name: \"%s\" +-+-+-+-+\n", net->name);
  if (net->initial_wep)
    dump_config_static_wep(net->initial_wep, TRUE);

  if (net->type == UNSET)
    printf("  Type: UNSET\n");
  else if (net->type == WIRED)
    printf("  Type: WIRED\n");
  else
    printf("  Type: WIRELESS\n");

  switch (net->force_eapol_ver)
    {
    case 1:
      printf("  Forced EAPOL Version : 1\n");
      break;
      
    case 2:
      printf("  Forced EAPOL Version : 2\n");
      break;

    default:
      printf("  Forced EAPOL Version : Default\n");
      break;
    }

  switch (net->assoc_type)
    {
    case ASSOC_OPEN:
      printf("  Association Type : Open\n");
      break;

    case ASSOC_SHARED:
      printf("  Association Type : Shared Key\n");
      break;

    case ASSOC_LEAP:
      printf("  Association Type : LEAP\n");
      break;
    }

  if (net->wireless_ctrl == CTL_UNSET)
    printf("  Wireless Control: UNSET\n");
  else if (net->wireless_ctrl == CTL_YES)
    printf("  Wireless Control: YES\n");
  else
    printf("  Wireless Control: NO\n");

  printf("  WPA Group Crypt : ");
  dump_config_network_wpa(net->wpa_group_crypt);
  
  printf("  WPA Pairwise Crypt : ");
  dump_config_network_wpa(net->wpa_pairwise_crypt);

  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_WPA_PSK))
    printf("  Allow Type: WPA-PSK\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_TLS))
    printf("  Allow Type: TLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_MD5))
    printf("  Allow Type: MD5\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_TTLS))
    printf("  Allow Type: TTLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_LEAP))
    printf("  Allow Type: LEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_MSCV2))
    printf("  Allow Type: MSCHAPv2\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_PEAP))
    printf("  Allow Type: PEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_SIM))
    printf("  Allow Type: SIM\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_AKA))
    printf("  Allow Type: AKA\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_GTC))
    printf("  Allow Type: GTC\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_OTP))
    printf("  Allow Type: OTP\n");

  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_WPA_PSK))
    printf("  Prefer Type: WPA-PSK\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_TLS))
    printf("  Prefer Type: TLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_MD5))
    printf("  Prefer Type: MD5\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_TTLS))
    printf("  Prefer Type: TTLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_LEAP))
    printf("  Prefer Type: LEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_MSCV2))
    printf("  Prefer Type: MSCHAPv2\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_PEAP))
    printf("  Prefer Type: PEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_SIM))
    printf("  Prefer Type: SIM\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_AKA))
    printf("  Prefer Type: AKA\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_GTC))
    printf("  Prefer Type: GTC\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_OTP))
    printf("  Prefer Type: OTP\n");

  printf("  SSID: \"%s\"\n", net->ssid);
  printf("  Identity: \"%s\"\n", net->identity);

  if (TEST_FLAG(net->flags, CONFIG_NET_USE_TNC))
    {
      printf("  Use TNC support : Yes\n");
    } 
  else
    {
      printf("  Use TNC support : No\n");
    }

  if (TEST_FLAG(net->flags, CONFIG_NET_DEST_MAC))
    printf("  DEST MAC: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n",
	   net->dest_mac[0], net->dest_mac[1], net->dest_mac[2],
	   net->dest_mac[3], net->dest_mac[4], net->dest_mac[5]);

  dump_config_eap_method(net->methods, 0);

  if (net->next)
    dump_config_network(net->next);
      
}


  /*****************************/
 /* CONFIG_STRING_LIST       */
/*****************************/
void delete_config_string_list(struct config_string_list **tmp_string_list)
{
  if (*tmp_string_list == NULL)
    return;

  free (*tmp_string_list);
  *tmp_string_list = NULL;
}

void initialize_config_string_list(struct config_string_list **tmp_string_list)
{
  if (*tmp_string_list != NULL) {
    delete_config_string_list(tmp_string_list);
  }
  *tmp_string_list = 
    (struct config_string_list *)malloc(sizeof(struct config_string_list));  
  if (*tmp_string_list)
    memset(*tmp_string_list, 0, sizeof(struct config_string_list));
}

int config_string_list_contains_string(struct config_string_list *net_list,
				     char *netname)
{
  // if there is a list, we need to search for the net
  while (net_list != NULL) {
    if (strcmp(net_list->name, netname) == 0)
      return TRUE;
    net_list = net_list->next;
  }
  return FALSE;
}

void config_string_list_add_string(struct config_string_list **net_list,
				 char *netname)
{
  struct config_string_list **current = net_list;

  while (*current != NULL) {
    if (strcmp((*current)->name, netname) == 0)
      return;
    current = &(*current)->next;
  }
  initialize_config_string_list(current);
  (*current)->name = netname;
}

void dump_config_string_list(struct config_string_list *stringlist, 
			     char *title)
{
  if (!stringlist) {
    return;
  }

  printf("%s: \"%s\"\n", title,  stringlist->name);
  if (stringlist->next)
    dump_config_string_list(stringlist->next, title);

}

  /*******************/
 /* CONFIG_GLOBALS  */
/*******************/
void delete_config_globals(struct config_globals **tmp_globals)
{
  if (*tmp_globals == NULL)
    return;

  if ((*tmp_globals)->default_net)
    free((*tmp_globals)->default_net);

  if ((*tmp_globals)->allowed_nets)
    delete_config_string_list(&(*tmp_globals)->allowed_nets);

  FREE_STRING((*tmp_globals)->logfile);
  FREE_STRING((*tmp_globals)->ipc_group_name);
  FREE_STRING((*tmp_globals)->log_facility);
  FREE_STRING((*tmp_globals)->default_int);
  
  free (*tmp_globals);
  *tmp_globals = NULL;
}

void initialize_config_globals(struct config_globals **tmp_globals)
{
  if (*tmp_globals != NULL) {
    delete_config_globals(tmp_globals);
  }
  *tmp_globals = 
    (struct config_globals *)malloc(sizeof(struct config_globals));  
  if (*tmp_globals)
    memset(*tmp_globals, 0, sizeof(struct config_globals));

  // We want to set allmulti to have a default of 1.
  SET_FLAG((*tmp_globals)->flags, CONFIG_GLOBALS_ALLMULTI);
  SET_FLAG((*tmp_globals)->flags, CONFIG_GLOBALS_FIRMWARE_ROAM);

  // Auto associate.
  (*tmp_globals)->flags |= CONFIG_GLOBALS_ASSOC_AUTO;

  // Passive scanning.  XXX Removed for 1.2.4 release.
  // (*tmp_globals)->flags |= CONFIG_GLOBALS_PASSIVE_SCAN;

  (*tmp_globals)->stale_key_timeout = STALE_KEY_WARN_TIMEOUT;
  (*tmp_globals)->assoc_timeout = ASSOCIATION_TIMEOUT;
  (*tmp_globals)->passive_timeout = PASSIVE_TIMEOUT;
  (*tmp_globals)->active_timeout = RESCAN_TIMEOUT;
}

void dump_config_globals(struct config_globals *globals)
{
  if (!globals) {
    printf("No Globals\n");
    return;
  }
  printf("Default Net: \"%s\"\n", globals->default_net);
  if (globals->allowed_nets)
    dump_config_string_list(globals->allowed_nets, "Network Allowed");
  else 
    printf("Allowed Nets: ALL\n");

  if (globals->logfile)
    printf("Logfile: '%s'\n", globals->logfile);

  if (globals->log_facility)
    printf("Log Facility: '%s'\n", globals->log_facility);

  if (globals->ipc_group_name)
    printf("IPC Group Name : '%s' \n", globals->ipc_group_name);

  if (globals->default_int)
    printf("Default Interface : '%s' \n", globals->default_int);

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_AUTH_PER))
    printf("Auth Period: %d\n", globals->auth_period);
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_HELD_PER))
    printf("Held Period: %d\n", globals->held_period);
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_MAX_STARTS))
    printf("Max Starts: %d\n", globals->max_starts);

  if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
    {
      printf("Friendly Warnings : Yes\n");
    } else {
      printf("Friendly Warnings : No\n");
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ASSOC_AUTO))
    {
      printf("Auto Association : Yes\n");
    } else {
      printf("Auto Association : No\n");
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_FIRMWARE_ROAM))
    {
      printf("Roaming          : Firmware\n");
    } else {
      printf("Roaming          : Xsupplicant\n");
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_PASSIVE_SCAN))
    {
      printf("Passive Scanning : Yes\n");
    } else {
      printf("Passive Scanning : No\n");
    }

  printf("Passive Scan Timeout : %d\n", globals->passive_timeout);

  printf("Association Timeout : %d\n", globals->assoc_timeout);

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ALLMULTI))
    printf("ALLMULTI is enabled!\n");
  else
    printf("ALLMULTI is disabled!\n");

  printf("Destination: ");
  switch (globals->destination)
    {
    case DEST_AUTO:
      printf("Auto\n");
      break;

    case DEST_BSSID:
      printf("BSSID\n");
      break;

    case DEST_MULTICAST:
      printf("Multicast\n");
      break;

    case DEST_SOURCE:
      printf("Source\n");
      break;
    }
}

  /*******************/
 /* CONFIG_DATA     */
/*******************/
void delete_config_data(struct config_data **tmp_data)
{
  if (*tmp_data == NULL)
    return;

  if ((*tmp_data)->config_fname)
    free((*tmp_data)->config_fname);

  if ((*tmp_data)->globals)
    delete_config_globals(&(*tmp_data)->globals);

  if ((*tmp_data)->networks)
    delete_config_network(&(*tmp_data)->networks);
  
  free (*tmp_data);
  *tmp_data = NULL;
}

void initialize_config_data(struct config_data **tmp_data)
{
  if (*tmp_data != NULL) {
    delete_config_data(tmp_data);
  }

  *tmp_data = 
    (struct config_data *)malloc(sizeof(struct config_data));  
  if (*tmp_data)
    {
      memset(*tmp_data, 0, sizeof(struct config_data));
    } else {
      printf("Couldn't allocate memory for configuration data!\n");
    }
}

void dump_config_data(struct config_data *data)
{
  if (!data)
    return;
  printf("=-=-=-=-=-=-=-=-=-=-=-=-=\n");
  printf("Configuration File: %s\n", data->config_fname);
  dump_config_globals(data->globals);
  dump_config_network(data->networks);
  printf("=-=-=-=-=-=-=-=-=-=-=-=-=\n");
}
