/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <loggy.h>

#include "mempool.h"

#define SANITY_CHECK 1


static void _add_to_free_list(struct mem_pool* mp, struct mem_block* block);
static struct mem_block* _get_from_free_list(struct mem_pool* mp,
											 unsigned int size);

inline struct mem_block* HEAP_TO_BLOCK(void* heap)
{
	return ((struct mem_block*)heap) - 1;
}

int linear_mempool_malloc(struct mem_pool* mp,
						  unsigned int size, void** heap)
{
	int rc = 0;
	struct mem_block* block = NULL;

	size = (sizeof(struct mem_block) + size + 0x7f) & ~0x7f;
	//size = (sizeof(struct mem_block) + size + 0x7) & ~0x7;

	if (size > mp->remainingBytes) {
		block = _get_from_free_list(mp, size);
		if (block != NULL) {
			INIT_LIST_HEAD(&block->list);
			list_add(&block->list, &mp->allocated_blocks);

			*heap = (void*)(block + 1);

			LOGGY_DEBUG(mp->verbose,
				"mempool[alloc]: block(%u, 0x%08x/0x%08x, %u) recycled\n",
				block->id,
				(unsigned int)block, (unsigned int)block->heap,
				block->size);
		} else {
			*heap = NULL;
			rc = -1;
			LOGGY_WARNING(mp->verbose, "mempool[alloc]: Out of memory pool\n");
		}
	} else {
		block = (struct mem_block*)mp->priv.top;

		INIT_LIST_HEAD(&block->list);
		list_add(&block->list, &mp->allocated_blocks);

		block->id   = ((unsigned int)block - (unsigned int)mp->base) / 128;
		block->size = size;
		block->heap = (void*)(block + 1);
		*heap = block->heap;

		LOGGY_DEBUG(mp->verbose,
			"mempool[alloc]: block(%u, 0x%08x/0x%08x, %u) allocated\n",
			block->id,
			(unsigned int)block, (unsigned int)block->heap,
			block->size);

		mp->priv.top += size;
		mp->remainingBytes -= size;
	}

	return rc;
}

//#define offsetof(TYPE, MEMBER) ((unsigned int) &((TYPE *)0)->MEMBER)
//#define container_of(ptr, type, member) ((type *)(((unsigned int)ptr) - offsetof(type, member)))

int linear_mempool_free(struct mem_pool* mp, void* heap)
{
	struct list_head* pos;
	struct mem_block* entry;
//	struct mem_block* block = NULL;

#if SANITY_CHECK
	if (heap == NULL) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: Trying to free NULL pointer\n");
		while (1) { ; }
	}
#endif /* SANITY_CHECK */

	// Merge the free block back to the heap
	entry = HEAP_TO_BLOCK(heap);
#if SANITY_CHECK
	if ((unsigned int)entry < (unsigned int)mp->base) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: Invalidate heap @ 0x%08x\n", (unsigned int)heap);
		//while (1) { ; }
		return -1;
	}
	if ((unsigned int)entry >= (unsigned int)(mp->base + mp->maximumBytes)) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: Invalidate heap @ 0x%08x\n", (unsigned int)heap);
		//while (1) { ; }
		return -1;
	}
#endif /* SANITY_CHECK */

	LOGGY_DEBUG(mp->verbose,
				"mempool[free]: block(%u, 0x%08x/0x%08x/%u)\n",
				entry->id,
				(unsigned int)heap, (unsigned int)entry, entry->size);

	if (((unsigned int)entry + entry->size) == (unsigned int)mp->priv.top) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: heap merged(%u bytes@0x%08x/0x%08x)\n",
					  entry->size, entry, (unsigned int)mp->priv.top);
		list_del_init(&entry->list);
		//mp->priv.top -= block->size;
		mp->priv.top = (unsigned char*)entry;
		return 0;
	}

	//list_for_each_prev_safe(pos, tmp, &mp->allocated_blocks) {
//	list_for_each_prev(pos, &mp->allocated_blocks) {
	list_for_each(pos, &mp->allocated_blocks) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: pos/head(0x%08x/0x%08x)\n",
					  (unsigned int)pos, (unsigned int)&mp->allocated_blocks);
#if SANITY_CHECK
		if (pos == NULL) {
			LOGGY_ERROR(mp->verbose,
						"mempool[free]: Memory pool is screw-up.\n");
			while (1) { ; }
			//break; // EVIL HACK for BUS EARLY TERMINATION: REMOVE ME
		}
#endif /* SANITY_CHECK */

		entry = list_entry(pos, struct mem_block, list);
		if (entry->heap == heap) {
			LOGGY_WARNING(mp->verbose,
						  "mempool[free]: heap(0x%08x) found\n",
						  (unsigned int)heap);

			list_del_init(pos);
			_add_to_free_list(mp, HEAP_TO_BLOCK(heap));
			return 0;
			//break;
		}
	}

	LOGGY_WARNING(mp->verbose,
				  "mempool[free]: Unrecognized heap(0x%08x)\n",
				  heap);
	return -1;
/*
	if (pos == &mp->allocated_blocks) {
		LOGGY_WARNING(mp->verbose,
					  "mempool[free]: Unrecognized heap(0x%08x)\n",
					  heap);
		return -1;
	}

	return 0;
*/
}

static void _add_to_free_list(struct mem_pool* mp, struct mem_block* block)
{
	INIT_LIST_HEAD(&block->list);
	list_add(&block->list, &mp->free_blocks);
/*
	struct list_head* pos;
	struct mem_block* entry;

	list_for_each_prev(pos, &mp->free_blocks) {
		entry = list_entry(pos, struct mem_block, list);
		//LOGGY_DEBUG(mp->verbose,
		//	"mempool[free]: entry(%u), block(%u)\n",
		//	entry->id, block->id);
		if (entry->id < block->id) {
			list_add(&block->list, pos);
			LOGGY_DEBUG(mp->verbose,
				"mempool[free]: block(%u, 0x%08x) add to the free list(%u)\n",
				block->id, (unsigned int)block, entry->id);
			return;
		}
	}

	if (pos == &mp->free_blocks) {
		//list_add_tail(&block->list, &mp->free_blocks);
		list_add(&block->list, &mp->free_blocks);
		LOGGY_DEBUG(mp->verbose,
			"mempool[free]: block(%u, 0x%08x) add to the front of free list\n",
			block->id, (unsigned int)block);
	}
*/
}

static struct mem_block* _get_from_free_list(struct mem_pool* mp,
											 unsigned int size)
{
	struct list_head* pos;
	struct mem_block* entry;

	//list_for_each_prev(pos, &mp->free_blocks) {
	list_for_each(pos, &mp->free_blocks) {
		entry = list_entry(pos, struct mem_block, list);
		if (entry->size >= size) {
			list_del_init(pos);
			//list_add(&entry->list, &mp->allocated_blocks);

			LOGGY_DEBUG(mp->verbose,
				"mempool[get]: free block(%u, 0x%08x/0x%08x, %u) found\n",
				entry->id, (unsigned int)entry, (unsigned int)entry->heap,
				entry->size);

			return entry;
		}
	}

	return NULL;
}
