/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

//#include <stdlib.h>
//#include <string.h>
#include <common.h>

#include <loggy.h>

#include <enccommon.h>
#include <h264encapi_ext.h>
#include <source/h264/H264Instance.h>

#include "h264-coder.h"


static char* _streamType[2] = { "BYTE_STREAM", "NAL_UNITS" };


int h264_coder_verify_options(struct h264_coder_opts* opts)
{
	return -1;
}

/*
void h264_coder_initialize(struct h264_coder* coder,
						   struct h264_coder_opts* opts)
{
	coder->logger = log4c_category_get("com.acti.recoder.h264.coder");
	coder->options = opts;
}

struct h264_coder* h264_coders_create(int count,
									  struct h264_coder_opts* options)
{
	int i;
	struct h264_coder* coders;

	if (count == 0) return NULL;

	do {
		coders = calloc(count, sizeof(*coders));
		if (coders == NULL) break;

		for (i = 0; i < count; ++i) {
			coders[i].logger =
				log4c_category_get("com.acti.recoder.h264.coder");
			coders[i].options = &options[i];
		}
	} while (0);

	return coders;
}

void h264_coders_destroy(int count, struct h264_coder* coders)
{
	free(coders);
}
*/

static int _open_encoder(struct h264_coder* coder);
static int _alloc_resources(struct h264_coder* coder);
static int _free_resources(struct h264_coder* coder);
static int _close_encoder(struct h264_coder* coder);

int h264_coder_open(struct h264_coder* coder, struct h264_coder_opts* opts)
{
	int rc;

	coder->options = opts;

	rc = _open_encoder(coder);
	if (rc != 0) goto error;

	rc = _alloc_resources(coder);
	if (rc != 0) goto error;

	return rc;

error:
	h264_coder_close(coder);

	return rc;
}

static int _open_encoder(struct h264_coder* coder)
{
	H264EncRet ret;
	H264EncConfig           cfg;
	H264EncCodingCtrl       codingCfg;
	H264EncRateCtrl         rcCfg;
	H264EncPreProcessingCfg preProcCfg;

	memset(&cfg,        0, sizeof(cfg));
	memset(&codingCfg,  0, sizeof(codingCfg));
	memset(&rcCfg,      0, sizeof(rcCfg));
	memset(&preProcCfg, 0, sizeof(preProcCfg));

	if (coder->options->outputWidth == H264_CODER_DEFAULT ||
		coder->options->outputHeight == H264_CODER_DEFAULT)
	{
		LOGGY_DEBUG(coder->logger, "Output width or height is not specified.\n");
		return -1;
	}

	if (coder->options->rotation != 0) {
		cfg.width = coder->options->outputHeight;
		cfg.height = coder->options->outputWidth;
    } else {
		cfg.width = coder->options->outputWidth;
		cfg.height = coder->options->outputHeight;
	}

	cfg.frameRateDenom = coder->options->inputRateDenom;
	cfg.frameRateNum = coder->options->inputRateNumer;
	if (coder->options->byteStream != 0) {
		cfg.streamType = H264ENC_BYTE_STREAM;
	} else {
		cfg.streamType = H264ENC_NAL_UNIT_STREAM;
	}

	cfg.level = H264ENC_LEVEL_4_0;
	if (coder->options->level != H264_CODER_DEFAULT &&
		coder->options->level != 0)
	{
		cfg.level = (H264EncLevel)coder->options->level;
	}

	cfg.writeMvOutput = coder->options->mvOutput;

	LOGGY_TRACE(coder->logger,
				"\nInit config: size(%dx%d), %d/%d fps, %s, P&L %d, "
				"mv output %d\n",
				cfg.width, cfg.height,
				cfg.frameRateNum, cfg.frameRateDenom,
				_streamType[cfg.streamType], cfg.level,
				cfg.writeMvOutput);

	if ((ret = H264EncInit(&cfg, &coder->encoder)) != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncInit() failed. %s\n", h264_coder_err_to_str(ret));
		return -1;
	}

	if (coder->options->attachParameterSet != 0) {
		H264EncAttachParameterSetPerIntraFrame(coder->encoder,
										coder->options->attachParameterSet);
		LOGGY_TRACE(coder->logger, "Attach SPS and PPS to each Intra frame\n");
	}

	if (coder->options->intraPrevFavor != NULL) {
		ret = H264EncSetIntraPrevFavor(coder->encoder,
									   coder->options->intraPrevFavor);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetIntraPrevFavor() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	if (coder->options->intra16Favor != NULL) {
		ret = H264EncSetIntra16Favor(coder->encoder,
									 coder->options->intra16Favor);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetIntra16Favor() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	if (coder->options->interFavor != NULL) {
		ret = H264EncSetInterMbFavor(coder->encoder,
									 coder->options->interFavor);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetInterMbFavor() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	if (coder->options->skipSadPenalty != NULL) {
		ret = H264EncSetSkipSadPenalty(coder->encoder,
									   coder->options->skipSadPenalty);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetSkipSadPenalty() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	if (coder->options->diffMvPenalty != NULL) {
		ret = H264EncSetDiffMvPenalty(coder->encoder,
									  coder->options->diffMvPenalty);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetDiffMvPenalty() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	if (coder->options->diffMvPenalty4p != NULL) {
		ret = H264EncSetDiffMvPenalty4p(coder->encoder,
										coder->options->diffMvPenalty4p);
		if (ret != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"H264EncSetDiffMvPenalty4p() failed. %s\n",
						h264_coder_err_to_str(ret));
			return -1;
		}
	}

	/* Encoder setup: Rate control */
	if ((ret = H264EncGetRateCtrl(coder->encoder, &rcCfg)) != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncGetRateCtrl() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}

	LOGGY_TRACE(coder->logger,
				"Get rate control: qp %2d[%2d/%2d], %8d bps, "
				"picRc %d, mbRc %d, skipPic %d, hrd %d, cpbSize %d, "
				"gopLen %d, intraQpDelta %2d, "
				"fixedIntraQp %2d, mbQpAdjustment %d\n",
				rcCfg.qpHdr, rcCfg.qpMin, rcCfg.qpMax,
				rcCfg.bitPerSecond, rcCfg.pictureRc, rcCfg.mbRc,
				rcCfg.pictureSkip, rcCfg.hrd, rcCfg.hrdCpbSize,
				rcCfg.gopLen, rcCfg.intraQpDelta,
				rcCfg.fixedIntraQp, rcCfg.mbQpAdjustment);

	if (coder->options->qpHdr != H264_CODER_DEFAULT) {
		rcCfg.qpHdr = coder->options->qpHdr;
	}
	if (coder->options->qpMin != H264_CODER_DEFAULT) {
		rcCfg.qpMin = coder->options->qpMin;
	}
	if (coder->options->qpMax != H264_CODER_DEFAULT) {
		rcCfg.qpMax = coder->options->qpMax;
	}
	if (coder->options->picSkip != H264_CODER_DEFAULT) {
		rcCfg.pictureSkip = coder->options->picSkip;
	}
	if (coder->options->picRc != H264_CODER_DEFAULT) {
		rcCfg.pictureRc = coder->options->picRc;
	}
	if (coder->options->mbRc != H264_CODER_DEFAULT) {
		rcCfg.mbRc = (coder->options->mbRc != 0) ? 1 : 0;
	}
	if (coder->options->bitPerSecond != H264_CODER_DEFAULT) {
		rcCfg.bitPerSecond = coder->options->bitPerSecond;
	}
	if (coder->options->hrdConformance != H264_CODER_DEFAULT) {
		rcCfg.hrd = coder->options->hrdConformance;
	}
	if (coder->options->cpbSize != H264_CODER_DEFAULT) {
		rcCfg.hrdCpbSize = coder->options->cpbSize;
	}
	if (coder->options->intraPicRate != 0) {
		rcCfg.gopLen = MIN(coder->options->intraPicRate,
						   H264_CODER_MAX_GOP_LEN);
	}
	//if (coder->options->gopLength != H264_CODER_DEFAULT) {
	//	rcCfg.gopLen = coder->options->gopLength;
	//}
	if (coder->options->intraQpDelta != H264_CODER_DEFAULT) {
		rcCfg.intraQpDelta = coder->options->intraQpDelta;
	}
	rcCfg.fixedIntraQp = coder->options->fixedIntraQp;
	rcCfg.mbQpAdjustment = coder->options->mbQpAdjustment;

	LOGGY_TRACE(coder->logger,
				"Set rate control: qp %2d[%2d/%2d], %8d bps, "
				"picRc %d, mbRc %d, skipPic %d, hrd %d, cpbSize %d, "
				"gopLen %d, intraQpDelta %2d, "
				"fixedIntraQp %2d, mbQpAdjustment %d\n",
				rcCfg.qpHdr, rcCfg.qpMin, rcCfg.qpMax,
				rcCfg.bitPerSecond, rcCfg.pictureRc, rcCfg.mbRc,
				rcCfg.pictureSkip, rcCfg.hrd, rcCfg.hrdCpbSize,
				rcCfg.gopLen, rcCfg.intraQpDelta,
				rcCfg.fixedIntraQp, rcCfg.mbQpAdjustment);

	if ((ret = H264EncSetRateCtrl(coder->encoder, &rcCfg)) != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncSetRateCtrl() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}

    /* Encoder setup: coding control */
	ret = H264EncGetCodingCtrl(coder->encoder,&codingCfg);
	if (ret != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncGetCodingCtrl() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}

	if (coder->options->mbPerSlice != H264_CODER_DEFAULT) {
            codingCfg.sliceSize =
					coder->options->mbPerSlice / (cfg.width / 16);
	}
	if (coder->options->constIntraPred != H264_CODER_DEFAULT) {
		if (coder->options->constIntraPred != 0) {
			codingCfg.constrainedIntraPrediction = 1;
		} else {
			codingCfg.constrainedIntraPrediction = 0;
		}
	}
	if (coder->options->disableDeblocking != 0) {
		codingCfg.disableDeblockingFilter = 1;
	} else {
		codingCfg.disableDeblockingFilter = 0;
	}
//	if ((cml->disableDeblocking != 1) &&
//		((cml->filterOffsetA != 0) || (cml->filterOffsetB != 0))) {
//		codingCfg.disableDeblockingFilter = 1;
//	}
	if (coder->options->enableCabac != H264_CODER_DEFAULT) {
		codingCfg.enableCabac = coder->options->enableCabac;
		if (coder->options->cabacInitIdc != H264_CODER_DEFAULT) {
			codingCfg.cabacInitIdc = coder->options->cabacInitIdc;
		}
	}
//	if (coder->options->videoRange != DEFAULT) {
//		if (coder->options->videoRange != 0) {
//			codingCfg.videoFullRange = 1;
//		} else {
//			codingCfg.videoFullRange = 0;
//		}
//	}
	if (coder->options->sei) {
		codingCfg.seiMessages = 1;
	} else {
		codingCfg.seiMessages = 0;
	}

	codingCfg.transform8x8Mode = coder->options->trans8x8;
//	codingCfg.cirStart = coder->options->cirStart;
//	codingCfg.cirInterval = coder->options->cirInterval;
//	codingCfg.intraSliceMap1 = coder->options->intraSliceMap1;
//	codingCfg.intraSliceMap2 = coder->options->intraSliceMap2;
//	codingCfg.intraArea.enable = coder->options->intraAreaEnable;
//	codingCfg.intraArea.top = coder->options->intraAreaTop;
//	codingCfg.intraArea.left = coder->options->intraAreaLeft;
//	codingCfg.intraArea.bottom = coder->options->intraAreaBottom;
//	codingCfg.intraArea.right = coder->options->intraAreaRight;
//	codingCfg.roi1Area.enable = coder->options->roi1AreaEnable;
//	codingCfg.roi1Area.top = coder->options->roi1AreaTop;
//	codingCfg.roi1Area.left = coder->options->roi1AreaLeft;
//	codingCfg.roi1Area.bottom = coder->options->roi1AreaBottom;
//	codingCfg.roi1Area.right = coder->options->roi1AreaRight;
//	codingCfg.roi2Area.enable = coder->options->roi2AreaEnable;
//	codingCfg.roi2Area.top = coder->options->roi2AreaTop;
//	codingCfg.roi2Area.left = coder->options->roi2AreaLeft;
//	codingCfg.roi2Area.bottom = coder->options->roi2AreaBottom;
//	codingCfg.roi2Area.right = coder->options->roi2AreaRight;
//	codingCfg.roi1DeltaQp = coder->options->roi1DeltaQp;
//	codingCfg.roi2DeltaQp = coder->options->roi2DeltaQp;

	LOGGY_TRACE(coder->logger,
				"Set coding control: SEI %d, Slice %5d, deblocking %d, "
				"constrained intra %d, video range %d, "
				"cabac %d, cabac initial idc %d, Adaptive 8x8 transform %d\n",
				codingCfg.seiMessages, codingCfg.sliceSize,
				codingCfg.disableDeblockingFilter,
				codingCfg.constrainedIntraPrediction, codingCfg.videoFullRange,
				codingCfg.enableCabac, codingCfg.cabacInitIdc,
				codingCfg.transform8x8Mode);

//        if (codingCfg.cirStart)
//            printf("  CIR: %d %d\n",
//                codingCfg.cirStart, codingCfg.cirInterval);
//        if (codingCfg.intraSliceMap1 || codingCfg.intraSliceMap2)
//            printf("  IntraSliceMap: %u %u\n",
//                codingCfg.intraSliceMap1, codingCfg.intraSliceMap2);
//        if (codingCfg.intraArea.enable)
//            printf("  IntraArea: %dx%d-%dx%d\n",
//                codingCfg.intraArea.left, codingCfg.intraArea.top,
//                codingCfg.intraArea.right, codingCfg.intraArea.bottom);
//        if (codingCfg.roi1Area.enable)
//            printf("  ROI 1: %d  %dx%d-%dx%d\n", codingCfg.roi1DeltaQp,
//                codingCfg.roi1Area.left, codingCfg.roi1Area.top,
//                codingCfg.roi1Area.right, codingCfg.roi1Area.bottom);
//        if (codingCfg.roi2Area.enable)
//            printf("  ROI 2: %d  %dx%d-%dx%d\n", codingCfg.roi2DeltaQp,
//                codingCfg.roi2Area.left, codingCfg.roi2Area.top,
//                codingCfg.roi2Area.right, codingCfg.roi2Area.bottom);

	if ((ret = H264EncSetCodingCtrl(coder->encoder, &codingCfg)) != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncSetCodingCtrl() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}

	H264EncSetChromaQpIndexOffset(coder->encoder,
								  coder->options->chromaQpOffset);
	LOGGY_TRACE(coder->logger,
				"Set ChromaQpIndexOffset: %d\n",
				coder->options->chromaQpOffset);

	if (coder->options->quarterPixelMv != H264_CODER_DEFAULT) {
		H264EncSetQuarterPixelMv(coder->encoder,
								 coder->options->quarterPixelMv);
		LOGGY_TRACE(coder->logger,
					"Set Quarter Pixel MV: %d\n",
					coder->options->quarterPixelMv);
	}

/*
	if (cml->inputUserSkipSadPenalty != NULL) {
		FILE* file;

		file = fopen(cml->inputUserSkipSadPenalty, "rb");
		if (file != NULL) {
			int i;
			char* pch;
			char buffer[256];

			memset(buffer, 0, 256);
			fread(buffer, 1, 256, file);

			i = 0;
			pch = strtok(buffer, " ,\n");
			while (pch != NULL) {
				h264SkipSadPenaltyUser1[i] = atoi(pch);
				pch = strtok(NULL, " ,\n"); i++;
			}

			fclose(file);
		}
	}

	if (cml->enableUserSkipSadPenalty != 0) {
		printf("Switch to User Defined Skip Sad Penalty\n");
		H264EncSetSkipSadPenalty(encoder, h264SkipSadPenaltyUser1);
		printf("User Defined Skip Sad Penalty are:\n");
		printf("\t%u, %u, %u, %u, %u, %u, %u, %u, %u, %u,\n",
			   h264SkipSadPenaltyUser1[0], h264SkipSadPenaltyUser1[1],
			   h264SkipSadPenaltyUser1[2], h264SkipSadPenaltyUser1[3],
			   h264SkipSadPenaltyUser1[4], h264SkipSadPenaltyUser1[5],
			   h264SkipSadPenaltyUser1[6], h264SkipSadPenaltyUser1[7],
			   h264SkipSadPenaltyUser1[8], h264SkipSadPenaltyUser1[9]);
		printf("\t%u, %u, %u, %u, %u, %u, %u, %u, %u, %u,\n",
			   h264SkipSadPenaltyUser1[10], h264SkipSadPenaltyUser1[11],
			   h264SkipSadPenaltyUser1[12], h264SkipSadPenaltyUser1[13],
			   h264SkipSadPenaltyUser1[14], h264SkipSadPenaltyUser1[15],
			   h264SkipSadPenaltyUser1[16], h264SkipSadPenaltyUser1[17],
			   h264SkipSadPenaltyUser1[18], h264SkipSadPenaltyUser1[19]);
		printf("\t%u, %u, %u, %u, %u, %u, %u, %u, %u, %u,\n",
			   h264SkipSadPenaltyUser1[20], h264SkipSadPenaltyUser1[21],
			   h264SkipSadPenaltyUser1[22], h264SkipSadPenaltyUser1[23],
			   h264SkipSadPenaltyUser1[24], h264SkipSadPenaltyUser1[25],
			   h264SkipSadPenaltyUser1[26], h264SkipSadPenaltyUser1[27],
			   h264SkipSadPenaltyUser1[28], h264SkipSadPenaltyUser1[29]);
		printf("\t%u, %u, %u, %u, %u, %u, %u, %u, %u, %u,\n",
			   h264SkipSadPenaltyUser1[30], h264SkipSadPenaltyUser1[31],
			   h264SkipSadPenaltyUser1[32], h264SkipSadPenaltyUser1[33],
			   h264SkipSadPenaltyUser1[34], h264SkipSadPenaltyUser1[35],
			   h264SkipSadPenaltyUser1[36], h264SkipSadPenaltyUser1[37],
			   h264SkipSadPenaltyUser1[38], h264SkipSadPenaltyUser1[39]);
		printf("\t%u, %u, %u, %u, %u, %u, %u, %u, %u, %u,\n",
			   h264SkipSadPenaltyUser1[40], h264SkipSadPenaltyUser1[41],
			   h264SkipSadPenaltyUser1[42], h264SkipSadPenaltyUser1[43],
			   h264SkipSadPenaltyUser1[44], h264SkipSadPenaltyUser1[45],
			   h264SkipSadPenaltyUser1[46], h264SkipSadPenaltyUser1[47],
			   h264SkipSadPenaltyUser1[48], h264SkipSadPenaltyUser1[49]);
		printf("\t%u, %u\n",
			   h264SkipSadPenaltyUser1[50], h264SkipSadPenaltyUser1[51]);
	}
*/

	/* PreP setup */
	ret = H264EncGetPreProcessing(coder->encoder, &preProcCfg);
	if (ret != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncGetPreProcessing() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}
	LOGGY_TRACE(coder->logger,
				"Get PreP: input %4dx%d, offset %4dx%d, format %d, "
				"rotation %d, stab %d, cc %d\n",
				preProcCfg.origWidth, preProcCfg.origHeight,
				preProcCfg.xOffset, preProcCfg.yOffset,
				preProcCfg.inputType, preProcCfg.rotation,
				preProcCfg.videoStabilization,
				preProcCfg.colorConversion.type);

	preProcCfg.inputType = (H264EncPictureType)coder->options->inputFormat;
	preProcCfg.rotation = (H264EncPictureRotation)coder->options->rotation;

	preProcCfg.origWidth = coder->options->inputWidth;
	preProcCfg.origHeight = coder->options->inputHeight;

	if (coder->options->xOffset != H264_CODER_DEFAULT) {
		preProcCfg.xOffset = coder->options->xOffset;
	}
	if (coder->options->yOffset != H264_CODER_DEFAULT) {
		preProcCfg.yOffset = coder->options->yOffset;
	}
	if (coder->options->videoStab != H264_CODER_DEFAULT) {
		preProcCfg.videoStabilization = coder->options->videoStab;
	}
/*
	if(cml->colorConversion != DEFAULT)
		preProcCfg.colorConversion.type =
			(H264EncColorConversionType)cml->colorConversion;
	if(preProcCfg.colorConversion.type == H264ENC_RGBTOYUV_USER_DEFINED) {
		preProcCfg.colorConversion.coeffA = 20000;
		preProcCfg.colorConversion.coeffB = 44000;
		preProcCfg.colorConversion.coeffC = 5000;
		preProcCfg.colorConversion.coeffE = 35000;
		preProcCfg.colorConversion.coeffF = 38000;
	}
*/

//    printf
//        ("Set PreP: input %4dx%d, format %d, stab %d, stabDelta %4d\n",
//         preProcCfg.origWidth, preProcCfg.origHeight, preProcCfg.inputType,
//		 preProcCfg.videoStabilization, cml->videoStabDelta);
	LOGGY_TRACE(coder->logger,
				"Set PreP: input %4dx%d, offset %4dx%d, format %d, "
				"rotation %d, stab %d, cc %d\n",
				preProcCfg.origWidth, preProcCfg.origHeight,
				preProcCfg.xOffset, preProcCfg.yOffset,
				preProcCfg.inputType, preProcCfg.rotation,
				preProcCfg.videoStabilization,
				preProcCfg.colorConversion.type);

	ret = H264EncSetPreProcessing(coder->encoder, &preProcCfg);
	if (ret != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncSetPreProcessing() failed. %s\n",
					h264_coder_err_to_str(ret));
		return -1;
	}

	return 0;
}

static int _alloc_resources(struct h264_coder* coder)
{
	int ret;
	unsigned int pictureSize;
	unsigned int outbufSize;

	if (coder->options->inputFormat <= 1) {
		/* Input picture in planar YUV 4:2:0 format */
		pictureSize =
			((coder->options->inputWidth + 15) & (~15)) *
			coder->options->inputHeight * 3 / 2;
	} else if(coder->options->inputFormat <= 9) {
		/* Input picture in YUYV 4:2:2 or 16-bit RGB format */
		pictureSize =
			((coder->options->inputWidth + 15) & (~15)) *
			coder->options->inputHeight * 2;
    } else {
		/* Input picture in 32-bit RGB format */
		pictureSize =
			((coder->options->inputWidth + 15) & (~15)) *
			coder->options->inputHeight * 4;
	}

	LOGGY_TRACE(coder->logger,
				"Input %dx%d encoding at %dx%d\n",
				coder->options->inputWidth,
				coder->options->inputHeight,
				coder->options->outputWidth,
				coder->options->outputHeight);

	coder->frameSize = pictureSize;
	coder->outputMem.virtualAddress = NULL;
	coder->pictureMem.virtualAddress = NULL;
	coder->pictureStabMem.virtualAddress = NULL;

	/* Here we use the EWL instance directly from the encoder
	 * because it is the easiest way to allocate the linear memories */
	ret = EWLMallocLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
						  pictureSize, &coder->pictureMem);
	if (ret != EWL_OK) {
		LOGGY_DEBUG(coder->logger, "Failed to allocate input picture!\n");
		coder->pictureMem.virtualAddress = NULL;
		return -1;
	}

	if (coder->options->videoStab > 0) {
		ret = EWLMallocLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
							  pictureSize, &coder->pictureStabMem);
		if (ret != EWL_OK) {
			LOGGY_DEBUG(coder->logger, "Failed to allocate stab input picture!\n");
			coder->pictureStabMem.virtualAddress = NULL;
			return -1;
		}
	}

	outbufSize = 4 * coder->pictureMem.size < (1024 * 1024 * 8) ?
				 4 * coder->pictureMem.size : (1024 * 1024 * 8);

	ret = EWLMallocLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
						  outbufSize, &coder->outputMem);
	if (ret != EWL_OK) {
		LOGGY_DEBUG(coder->logger, "Failed to allocate output buffer!\n");
		coder->outputMem.virtualAddress = NULL;
		return -1;
	}

	LOGGY_TRACE(coder->logger,
				"\n"
				"Input buffer size:          %d bytes\n"
				"Input buffer bus address:   0x%08x\n"
				"Input buffer user address:  0x%08x\n"
				"Output buffer size:         %d bytes\n"
				"Output buffer bus address:  0x%08x\n"
				"Output buffer user address: 0x%08x\n",
				coder->pictureMem.size,
				coder->pictureMem.busAddress,
				(unsigned int)coder->pictureMem.virtualAddress,
				coder->outputMem.size,
				coder->outputMem.busAddress,
				(unsigned int)coder->outputMem.virtualAddress);

	return 0;
}

static int _free_resources(struct h264_coder* coder)
{
    if (coder->pictureMem.virtualAddress != NULL) {
        EWLFreeLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
					  &coder->pictureMem);
	}
    if (coder->pictureStabMem.virtualAddress != NULL) {
        EWLFreeLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
					  &coder->pictureStabMem);
	}
    if (coder->outputMem.virtualAddress != NULL) {
        EWLFreeLinear(((h264Instance_s*)coder->encoder)->asic.ewl,
					  &coder->outputMem);
	}

	return 0;
}

static int _close_encoder(struct h264_coder* coder)
{
	H264EncRet rc = H264ENC_OK;

	if (coder->encoder != NULL) {
		rc = H264EncRelease(coder->encoder);
		if (rc != H264ENC_OK) {
			LOGGY_DEBUG(coder->logger,
						"Failed to H264EncRelease(). %s\n",
						h264_coder_err_to_str(rc));
		}
	}

	return (rc == H264ENC_OK) ? 0 : -1;
}

int h264_coder_close(struct h264_coder* coder)
{
	_free_resources(coder);
	_close_encoder(coder);

	return 0;
}

const char* h264_coder_err_to_str(int val)
{
	const char* str;

	switch (val) {
	case H264ENC_ERROR: str = "H264ENC_ERROR"; break;
	case H264ENC_NULL_ARGUMENT: str = "H264ENC_NULL_ARGUMENT"; break;
	case H264ENC_INVALID_ARGUMENT: str = "H264ENC_INVALID_ARGUMENT"; break;
	case H264ENC_MEMORY_ERROR: str = "H264ENC_MEMORY_ERROR"; break;
	case H264ENC_EWL_ERROR: str = "H264ENC_EWL_ERROR"; break;
	case H264ENC_EWL_MEMORY_ERROR: str = "H264ENC_EWL_MEMORY_ERROR"; break;
	case H264ENC_INVALID_STATUS: str = "H264ENC_INVALID_STATUS"; break;
	case H264ENC_OUTPUT_BUFFER_OVERFLOW: str = "H264ENC_OUTPUT_BUFFER_OVERFLOW"; break;
	case H264ENC_HW_BUS_ERROR: str = "H264ENC_HW_BUS_ERROR"; break;
	case H264ENC_HW_DATA_ERROR: str = "H264ENC_HW_DATA_ERROR"; break;
	case H264ENC_HW_TIMEOUT: str = "H264ENC_HW_TIMEOUT"; break;
	case H264ENC_HW_RESERVED: str = "H264ENC_HW_RESERVED"; break;
	case H264ENC_SYSTEM_ERROR: str = "H264ENC_SYSTEM_ERROR"; break;
	case H264ENC_INSTANCE_ERROR: str = "H264ENC_INSTANCE_ERROR"; break;
	case H264ENC_HRD_ERROR: str = "H264ENC_HRD_ERROR"; break;
	case H264ENC_HW_RESET: str = "H264ENC_HW_RESET"; break;
	default: str = "UNDEFINED"; break;
	}

	return str;
}

void h264_ma_add_frame(struct h264_ma* ma, int frameSizeBits)
{
	ma->frame[ma->pos++] = frameSizeBits;

	if (ma->pos == ma->length) ma->pos = 0;

	if (ma->count < ma->length) ma->count++;
}

int  h264_ma_calculate(struct h264_ma* ma)
{
	int i;
	int sum = 0;

	for (i = 0; i < ma->count; i++) sum += ma->frame[i];

	if (!ma->frameRateDenom) return 0;

	return (sum / ma->length) * ma->frameRateNumer / ma->frameRateDenom;
}

int h264_coder_start(struct h264_coder* coder)
{
	H264EncRet rc;

	/* Set the window length for bitrate moving average calculation */
	coder->ma.frameRateNumer = coder->options->inputRateNumer;
	coder->ma.frameRateDenom = coder->options->inputRateDenom;
	coder->ma.length = H264_MOVING_AVG_FRAMES;

	coder->encInput.pNaluSizeBuf = NULL;
	coder->encInput.naluSizeBufSize = 0;

	coder->encInput.pOutBuf = coder->outputMem.virtualAddress;
	coder->encInput.busOutBuf = coder->outputMem.busAddress;
	coder->encInput.outBufSize = coder->outputMem.size;

	/* Allocate memory for NAL unit size buffer, optional */
//	coder->encInput.naluSizeBufSize = NALU_TABLE_SIZE * sizeof(u32);
//	coder->encInput.pNaluSizeBuf = (u32 *) malloc(encIn.naluSizeBufSize);
//	if(!encIn.pNaluSizeBuf) {
//		fprintf(H264ERR_OUTPUT,
//			"WARNING! Failed to allocate NAL unit size buffer.\n");
//	}

	/* Start stream */
	rc = H264EncStrmStart(coder->encoder, &coder->encInput, &coder->encOutput);
	if (rc != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncStrmStart() failed. %s\n",
					h264_coder_err_to_str(rc));
		return -1;
	}

	return 0;
}

int h264_coder_compress(struct h264_coder* coder, int target, int* actual)
{
	H264EncRet rc;

	coder->encInput.codingType = target;

	rc = H264EncStrmEncode(coder->encoder, &coder->encInput, &coder->encOutput);

	*actual = coder->encOutput.codingType;
	H264EncGetRateCtrl(coder->encoder, &coder->rateCtrl);
	h264_ma_add_frame(&coder->ma, coder->encOutput.streamSize * 8);

	coder->encInput.timeIncrement = coder->options->inputRateDenom;

	return rc;
}

int h264_coder_stop(struct h264_coder* coder)
{
	H264EncRet rc;

	rc = H264EncStrmEnd(coder->encoder, &coder->encInput, &coder->encOutput);
	if (rc != H264ENC_OK) {
		LOGGY_DEBUG(coder->logger,
					"H264EncStrmEnd() failed. %s\n",
					h264_coder_err_to_str(rc));
		return -1;
	}

	return 0;
}

void h264_coder_swap_linear_mem(EWLLinearMem_t* m1, EWLLinearMem_t* m2)
{
	EWLLinearMem_t swap;

	if (m1 == NULL || m2 == NULL) return;

	swap.virtualAddress = m1->virtualAddress;
	swap.busAddress = m1->busAddress;
	swap.size = m1->size;

	m1->virtualAddress = m2->virtualAddress;
	m1->busAddress = m2->busAddress;
	m1->size = m2->size;

	m2->virtualAddress = swap.virtualAddress;
	m2->busAddress = swap.busAddress;
	m2->size = swap.size;
}
