/*------------------------------------------------------------------------------
--                                                                            --
--       This software is confidential and proprietary and may be used        --
--        only as expressly authorized by a licensing agreement from          --
--                                                                            --
--                            Hantro Products Oy.                             --
--                                                                            --
--                   (C) COPYRIGHT 2006 HANTRO PRODUCTS OY                    --
--                            ALL RIGHTS RESERVED                             --
--                                                                            --
--                 The entire notice above must be reproduced                 --
--                  on all copies and should not be removed.                  --
--                                                                            --
--------------------------------------------------------------------------------
--
--  Abstract : Encoder Wrapper Layer for 6280/7280/8270/8290, common parts
--
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
    1. Include headers
------------------------------------------------------------------------------*/

#include "basetype.h"

#include "ewl.h"
#include "ewl_x280_common.h"

//#include <stdio.h>
//#include <stdlib.h>
//#include <string.h>

//#include <assert.h>
#define assert(...)  /* no assertion */

#include "../../access.h"
#include "../../register-map.h"
//#include "include/memory.h"


static const char *busTypeName[5] = { "UNKNOWN", "AHB", "OCP", "AXI", "PCI" };
static const char *synthLangName[3] = { "UNKNOWN", "VHDL", "VERILOG" };

volatile u32 asic_status = 0;


int MapAsicRegisters(hx280ewl_t * ewl)
{
	ewl->regSize = (96 * 4);
	ewl->regBase = HX8290_REG_BASE;
	ewl->pRegBase = (void*)HX8290_REG_BASE;

	return 0;
}

/*******************************************************************************
 Function name   : EWLReadAsicID
 Description     : Read ASIC ID register, static implementation
 Return type     : u32 ID
 Argument        : void
*******************************************************************************/
u32 EWLReadAsicID()
{
	u32 id = ~0;

	id = EWLReadReg(NULL, 0);

	PTRACE("EWLReadAsicID: 0x%08x at 0x%08x\r\n", id, HX8290_REG_BASE);

	return id;
}

/*******************************************************************************
 Function name   : EWLReadAsicConfig
 Description     : Reads ASIC capability register, static implementation
 Return type     : EWLHwConfig_t
 Argument        : void
*******************************************************************************/
EWLHwConfig_t EWLReadAsicConfig(void)
{
	u32  cfgval;
	u32* pRegs = (u32*)HX8290_REG_BASE;
	EWLHwConfig_t cfg_info;

	cfgval = pRegs[63];

    cfg_info.maxEncodedWidth = cfgval & ((1 << 11) - 1);
    cfg_info.h264Enabled = (cfgval >> 27) & 1;
    cfg_info.mpeg4Enabled = (cfgval >> 26) & 1;
    cfg_info.jpegEnabled = (cfgval >> 25) & 1;
    cfg_info.vsEnabled = (cfgval >> 24) & 1;
    cfg_info.rgbEnabled = (cfgval >> 28) & 1;
    cfg_info.searchAreaSmall = (cfgval >> 29) & 1;

    cfg_info.busType = (cfgval >> 20) & 15;
    cfg_info.synthesisLanguage = (cfgval >> 16) & 15;
    cfg_info.busWidth = (cfgval >> 12) & 15;

    PTRACE("EWLReadAsicConfig:\r\n"
           "    maxEncodedWidth   = %d\r\n"
           "    h264Enabled       = %s\r\n"
           "    jpegEnabled       = %s\r\n"
           "    mpeg4Enabled      = %s\r\n"
           "    vsEnabled         = %s\r\n"
           "    rgbEnabled        = %s\r\n"
           "    searchAreaSmall   = %s\r\n"
           "    busType           = %s\r\n"
           "    synthesisLanguage = %s\r\n"
           "    busWidth          = %d\r\n",
           cfg_info.maxEncodedWidth,
           cfg_info.h264Enabled == 1 ? "YES" : "NO",
           cfg_info.jpegEnabled == 1 ? "YES" : "NO",
           cfg_info.mpeg4Enabled == 1 ? "YES" : "NO",
           cfg_info.vsEnabled == 1 ? "YES" : "NO",
           cfg_info.rgbEnabled == 1 ? "YES" : "NO",
           cfg_info.searchAreaSmall == 1 ? "YES" : "NO",
           cfg_info.busType < 5 ? busTypeName[cfg_info.busType] : "ERROR",
           cfg_info.synthesisLanguage <
           3 ? synthLangName[cfg_info.synthesisLanguage] : "ERROR",
           cfg_info.busWidth * 32);

	return cfg_info;
}

/*******************************************************************************
 Function name   : EWLInit
 Description     : Allocate resources and setup the wrapper module
 Return type     : ewl_ret
 Argument        : void
*******************************************************************************/
const void *EWLInit(EWLInitParam_t * param)
{
	hx280ewl_t* enc = NULL;

    PTRACE("EWLInit: Start\r\n");

    /* Check for NULL pointer */
	if(param == NULL || param->clientType > 4) {
		PTRACE(("EWLInit: Bad calling parameters!\r\n"));
		return NULL;
	}

	/* Allocate instance */
	if ((enc = (hx280ewl_t *)EWLmalloc(sizeof(hx280ewl_t))) == NULL) {
		PTRACE("EWLInit: failed to alloc hx280ewl_t struct\r\n");
		return NULL;
	}

	/* New instance allocated */
	enc->clientType = param->clientType;
	enc->fd_mem = -1;
	enc->fd_enc = -1;
	enc->fd_memalloc = -1;

	/* map hw registers to user space */
	MapAsicRegisters(enc);

	PTRACE("EWLInit: mmap regs %d bytes --> %p\r\n", enc->regSize, enc->pRegBase);
	PTRACE("EWLInit: Return %0xd\r\n", (u32) enc);

	return enc;
}

/*******************************************************************************
 Function name   : EWLRelease
 Description     : Release the wrapper module by freeing all the resources
 Return type     : ewl_ret
 Argument        : void
*******************************************************************************/
i32 EWLRelease(const void *inst)
{
	hx280ewl_t *enc = (hx280ewl_t *) inst;

	assert(enc != NULL);
	if (enc == NULL) return EWL_OK;

	EWLfree(enc);

	PTRACE("EWLRelease: instance freed\r\n");

	return EWL_OK;
}

/*******************************************************************************
 Function name   : EWLWriteReg
 Description     : Set the content of a hadware register
 Return type     : void
 Argument        : u32 offset
 Argument        : u32 val
*******************************************************************************/
void EWLWriteReg(const void *inst, u32 offset, u32 val)
{
	hx280ewl_t *enc = (hx280ewl_t*)inst;

	assert(enc != NULL && offset < enc->regSize);

	if (offset == 0x04) {
		asic_status = val;
	}

	HAL_SETREG32((HX8290_REG_BASE + offset), val);

	PTRACE("EWLWriteReg 0x%02x with value %08x\r\n", offset, val);
}

/*------------------------------------------------------------------------------
    Function name   : EWLEnableHW
    Description     :
    Return type     : void
    Argument        : const void *inst
    Argument        : u32 offset
    Argument        : u32 val
------------------------------------------------------------------------------*/
void EWLEnableHW(const void *inst, u32 offset, u32 val)
{
    hx280ewl_t *enc = (hx280ewl_t *) inst;

    assert(enc != NULL && offset < enc->regSize);

	if (offset == 0x04) {
		asic_status = val;
	}

	HAL_SETREG32((HX8290_REG_BASE + offset), val);

	PTRACE("EWLEnableHW 0x%02x with value %08x\r\n", offset, val);
}

/*------------------------------------------------------------------------------
    Function name   : EWLDisableHW
    Description     :
    Return type     : void
    Argument        : const void *inst
    Argument        : u32 offset
    Argument        : u32 val
------------------------------------------------------------------------------*/
void EWLDisableHW(const void *inst, u32 offset, u32 val)
{
    hx280ewl_t *enc = (hx280ewl_t *) inst;

	assert(enc != NULL && offset < enc->regSize);

	HAL_SETREG32((HX8290_REG_BASE + offset), val);

	asic_status = val;

	PTRACE("EWLDisableHW 0x%02x with value %08x\r\n", offset, val);
}

/*******************************************************************************
 Function name   : EWLReadReg
 Description     : Retrive the content of a hadware register
                    Note: The status register will be read after every MB
                    so it may be needed to buffer it's content if reading
                    the HW register is slow.
 Return type     : u32
 Argument        : u32 offset
*******************************************************************************/
u32 EWLReadReg(const void *inst, u32 offset)
{
	return HAL_GETREG32((HX8290_REG_BASE + offset));
}

/*------------------------------------------------------------------------------
    Function name   : EWLMallocRefFrm
    Description     : Allocate a frame buffer (contiguous linear RAM memory)

    Return type     : i32 - 0 for success or a negative error code

    Argument        : const void * instance - EWL instance
    Argument        : u32 size - size in bytes of the requested memory
    Argument        : EWLLinearMem_t *info - place where the allocated memory
                        buffer parameters are returned
------------------------------------------------------------------------------*/
i32 EWLMallocRefFrm(const void *instance, u32 size, EWLLinearMem_t * info)
{
	hx280ewl_t* enc_ewl = (hx280ewl_t*)instance;
	EWLLinearMem_t* buf = (EWLLinearMem_t*)info;
	i32 ret;

	assert(enc_ewl != NULL);
	assert(buf != NULL);

	PTRACE("EWLMallocRefFrm\t%8d bytes\r\n", size);

	ret = EWLMallocLinear(enc_ewl, size, buf);

	PTRACE("EWLMallocRefFrm %08x -> %p\r\n", buf->busAddress, buf->virtualAddress);

	return ret;
}

/*------------------------------------------------------------------------------
    Function name   : EWLFreeRefFrm
    Description     : Release a frame buffer previously allocated with
                        EWLMallocRefFrm.

    Return type     : void

    Argument        : const void * instance - EWL instance
    Argument        : EWLLinearMem_t *info - frame buffer memory information
------------------------------------------------------------------------------*/
void EWLFreeRefFrm(const void *instance, EWLLinearMem_t * info)
{
	hx280ewl_t *enc_ewl = (hx280ewl_t *) instance;
	EWLLinearMem_t *buff = (EWLLinearMem_t *) info;

	assert(enc_ewl != NULL);
	assert(buff != NULL);

	EWLFreeLinear(enc_ewl, buff);

	PTRACE("EWLFreeRefFrm\t%p\r\n", buff->virtualAddress);
}

/*------------------------------------------------------------------------------
    Function name   : EWLMallocLinear
    Description     : Allocate a contiguous, linear RAM  memory buffer

    Return type     : i32 - 0 for success or a negative error code

    Argument        : const void * instance - EWL instance
    Argument        : u32 size - size in bytes of the requested memory
    Argument        : EWLLinearMem_t *info - place where the allocated memory
                        buffer parameters are returned
------------------------------------------------------------------------------*/
i32 EWLMallocLinear(const void *instance, u32 size, EWLLinearMem_t * info)
{
    hx280ewl_t* enc_ewl = (hx280ewl_t *) instance;
    EWLLinearMem_t* buff = (EWLLinearMem_t *) info;

	assert(enc_ewl != NULL);
	assert(buff != NULL);

	PTRACE("EWLMallocLinear\t%8d bytes\r\n", size);

	buff->size = size;
	buff->virtualAddress = (void*)dma_memory_alloc(size);
	buff->busAddress = (unsigned int)buff->virtualAddress;

	PTRACE("EWLMallocLinear: 0x%08x(BUS) <--> %p(CPU)\r\n",
		   buff->busAddress, buff->virtualAddress);

	return EWL_OK;
}

/*------------------------------------------------------------------------------
    Function name   : EWLFreeLinear
    Description     : Release a linera memory buffer, previously allocated with
                        EWLMallocLinear.

    Return type     : void

    Argument        : const void * instance - EWL instance
    Argument        : EWLLinearMem_t *info - linear buffer memory information
------------------------------------------------------------------------------*/
void EWLFreeLinear(const void *instance, EWLLinearMem_t * info)
{
	hx280ewl_t *enc_ewl = (hx280ewl_t *) instance;
	EWLLinearMem_t *buff = (EWLLinearMem_t *) info;

	assert(enc_ewl != NULL);
	assert(buff != NULL);

	if (buff->virtualAddress != NULL) {
		dma_memory_free(buff->virtualAddress);
	}

	PTRACE("EWLFreeLinear\t%p\r\n", buff->virtualAddress);
}

/*******************************************************************************
 Function name   : EWLReleaseHw
 Description     : Release HW resource when frame is ready
*******************************************************************************/
void EWLReleaseHw(const void *inst)
{
	u32 val;
	hx280ewl_t *enc = (hx280ewl_t *) inst;

	assert(enc != NULL);

	val = EWLReadReg(inst, 0x38);
	EWLWriteReg(inst, 0x38, val & (~0x01)); /* reset ASIC */

	PTRACE("EWLReleaseHw: HW released\r\n");
}

/* SW/SW shared memory */
/*------------------------------------------------------------------------------
    Function name   : EWLmalloc
    Description     : Allocate a memory block. Same functionality as
                      the ANSI C malloc()

    Return type     : void pointer to the allocated space, or NULL if there
                      is insufficient memory available

    Argument        : u32 n - Bytes to allocate
------------------------------------------------------------------------------*/
void *EWLmalloc(u32 n)
{
	void *p = (void*)dma_memory_alloc(n);

	PTRACE("EWLmalloc\t%8d bytes --> %p\r\n", n, p);

	return p;
}

/*------------------------------------------------------------------------------
    Function name   : EWLfree
    Description     : Deallocates or frees a memory block. Same functionality as
                      the ANSI C free()

    Return type     : void

    Argument        : void *p - Previously allocated memory block to be freed
------------------------------------------------------------------------------*/
void EWLfree(void *p)
{
	PTRACE("EWLfree\t%p\r\n", p);

	if (p != NULL) dma_memory_free(p);
}

/*------------------------------------------------------------------------------
    Function name   : EWLcalloc
    Description     : Allocates an array in memory with elements initialized
                      to 0. Same functionality as the ANSI C calloc()

    Return type     : void pointer to the allocated space, or NULL if there
                      is insufficient memory available

    Argument        : u32 n - Number of elements
    Argument        : u32 s - Length in bytes of each element.
------------------------------------------------------------------------------*/
void *EWLcalloc(u32 n, u32 s)
{
	void* p = (void*)dma_memory_alloc(n * s);
	memory_set(p, 0, n * s);

	PTRACE("EWLcalloc\t%8d bytes --> %p\r\n", n * s, p);

	return p;
}

/*------------------------------------------------------------------------------
    Function name   : EWLmemcpy
    Description     : Copies characters between buffers. Same functionality as
                      the ANSI C memcpy()

    Return type     : The value of destination d

    Argument        : void *d - Destination buffer
    Argument        : const void *s - Buffer to copy from
    Argument        : u32 n - Number of bytes to copy
------------------------------------------------------------------------------*/
void *EWLmemcpy(void *d, const void *s, u32 n)
{
	return memory_copy(d, s, (size_t) n);
}

/*------------------------------------------------------------------------------
    Function name   : EWLmemset
    Description     : Sets buffers to a specified character. Same functionality
                      as the ANSI C memset()

    Return type     : The value of destination d

    Argument        : void *d - Pointer to destination
    Argument        : i32 c - Character to set
    Argument        : u32 n - Number of characters
------------------------------------------------------------------------------*/
void* EWLmemset(void *d, i32 c, u32 n)
{
	return memory_set(d, (int) c, (size_t) n);
}
