/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/mp4decapi.h>


extern void reset_memorypool(void);
extern void* memory_alloc(unsigned int size);
extern unsigned char colorbar_divx[];
extern unsigned int colorbar_divx_size;


enum _properties {
	FRAME_WIDTH  = 144,
	FRAME_HEIGHT = 96,

	LUMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT,
	CHROMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT / 4,
	PIC_SIZE = FRAME_WIDTH * FRAME_HEIGHT * 3 / 2,
};


static void _print_decoder_info(MP4DecInfo* decInfo)
{
	printf("MPEG4 decoder information:\n");
	printf("    picWidth: %d\n", decInfo->frameWidth);
	printf("    picHeight: %d\n", decInfo->frameHeight);
	printf("    outputFormat: %d\n", decInfo->outputFormat);
}

int do_mpeg4dec_selftest(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int picDecodeNumber = 0; /* decoded picture ID */

	MP4DecRet ret;

	MP4DecInst    decoder;
	MP4DecInput   decIn;
	MP4DecInfo    decInfo;
	MP4DecOutput  decOut;
	MP4DecPicture decPic;

	unsigned char* input;
	unsigned char* output;

	memset(&decoder, 0, sizeof(decoder));
	memset(&decIn,   0, sizeof(decIn));
	memset(&decOut,  0, sizeof(decOut));
	memset(&decPic,  0, sizeof(decPic));
	memset(&decInfo, 0, sizeof(decInfo));

	reset_memorypool();

	input = memory_alloc(colorbar_divx_size);
	if (input == NULL) {
		ret = MP4DEC_PARAM_ERROR;
		printf("Failed to allocate memory block for MPEG4 frame\n");
		goto error;
	}
	memcpy(input, colorbar_divx, colorbar_divx_size);

	output = memory_alloc(PIC_SIZE);
	if (output == NULL) {
		ret = MP4DEC_PARAM_ERROR;
		printf("Failed to allocate memory block for output buffer\n");
		goto error;
	}

	/* Configure the decoder input structure for the first decoding unit */
	decIn.dataLen = colorbar_divx_size;
	decIn.pStream = (void*)input;
	decIn.streamBusAddress = (unsigned int)input;

	/* Decoder initialization, output reordering enabled */
	/* check for any initialization errors must be done */
	ret = MP4DecInit(&decoder, 0/* sorensonSpark */, 0/* useVideoFreezeConcealment */);
	if(ret != MP4DEC_OK) {
		printf("Failed to initialize HX9170 MPEG4 decoder\n");
		goto error;
	}

	do {
		/* Picture ID is the picture number in decoding order */
		decIn.picId = picDecodeNumber;

		/* decode the stream data */
		ret = MP4DecDecode(decoder, &decIn, &decOut);
		switch(ret) {
		case MP4DEC_HDRS_RDY:
			/* read stream info */
			ret = MP4DecGetInfo(decoder, &decInfo);
			if (ret == MP4DEC_OK) {
				_print_decoder_info(&decInfo);
			} else {
				printf("Failed to get decoder information\n");
				goto failed;
			}
			break;
		case MP4DEC_DP_HDRS_RDY:
			/* Data partitioning used in the stream.
			 * The decoder has to reallocate resources */
			printf("MP4DEC_DP_HDRS_RDY\n");
			break;
		case MP4DEC_PIC_DECODED:
			/* a picture was decoded */
			/* Increment decoding number after every decoded picture */
			MP4DecNextPicture(decoder, &decPic, 0);

			memcpy(output, decPic.pOutputPicture, PIC_SIZE);
			printf("Decoded frame is at: 0x%08x(%d)\n",
				   (unsigned int)output, PIC_SIZE);

			/* Increment decoding number after every decoded picture */
			picDecodeNumber++;

			ret = MP4DEC_OK;

			break;
		case MP4DEC_STRM_PROCESSED:
			/* Input stream was processed but no picture is ready */
			printf("MP4DEC_STRM_PROCESSED\n");
			break;
		case MP4DEC_STRM_ERROR:
			printf("MP4DEC_STRM_ERROR\n");
			break;
		default:
			/* all other cases are errors where decoding cannot continue */
			printf("Failed to decode video stream: %d\n", ret);
			goto failed;
		}

		if (decOut.dataLeft == 0) {
			/* new buffer needed. read more data !? */
			decIn.dataLen = decOut.dataLeft;
		} else {
			/* data left undecoded */
			decIn.dataLen = decOut.dataLeft;
			decIn.pStream = decOut.pStrmCurrPos;
			decIn.streamBusAddress = decOut.strmCurrBusAddress;
		}

		/* keep decoding until all data from input stream buffer consumed */
	} while (decIn.dataLen > 0);

failed:
	MP4DecRelease(decoder);

error:
	if (ret != MP4DEC_OK) {
		printf("Failed to decode a JPEG frame\n");
	}

	return 0;
}


U_BOOT_CMD(
	mpeg4dec_selftest,	CONFIG_SYS_MAXARGS,	1,	do_mpeg4dec_selftest,
	"- Perform selftest of MPEG4 decoder\n",
	"\n"
	"	- To decode a built-in color bar\n"
);
