/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx8290/include/jpegencapi.h>


enum _properties {
	FRAME_WIDTH  = 128,
	FRAME_HEIGHT = 96,
	OUTBUF_SIZE  = FRAME_WIDTH * FRAME_HEIGHT,

	LUMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT,
	CHROMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT / 4,
	PIC_SIZE = FRAME_WIDTH * FRAME_HEIGHT * 3 / 2,
};


//extern void reset_memorypool(void);
extern void* dma_memory_alloc(unsigned int size);
extern unsigned char colorbar_128x96_0[];


static void _init_encoder_config(JpegEncCfg* pCfg);
static int  _open_jpeg_encoder(JpegEncCfg* pCfg, JpegEncInst * pEnc);
static int  _allocate_resources(JpegEncInst* pEnc, JpegEncCfg* pCfg,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut);
static int  _free_resources(JpegEncIn* pEncIn);
static int  _start_jpeg_encode(JpegEncInst * pEnc,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut);


int do_jpegenc_selftest(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	int rc;
	JpegEncInst encoder;
	JpegEncCfg config;
	JpegEncIn encIn;
	JpegEncOut encOut;

	memset(&encoder, 0, sizeof(encoder));
	memset(&config,  0, sizeof(config));
	memset(&encIn,   0, sizeof(encIn));
	memset(&encOut,  0, sizeof(encOut));

	//reset_memorypool();

	//
	// Test case: JPEG encoding
	//

	// Step 1: Initial encoder
	_init_encoder_config(&config);
	rc = _open_jpeg_encoder(&config, &encoder);
	if (rc != 0) {
		printf("Failed to open JPEG encoder\n");
		goto error;
	}

	// Step 2: Allocate resource
	rc = _allocate_resources(&encoder, &config, &encIn, &encOut);
	if (rc != 0) {
		printf("Failed to open allocate resources for JPEG encoder\n");
		goto error;
	}

	// Step 3: Encode a frame
	rc = _start_jpeg_encode(&encoder, &encIn, &encOut);
	if (rc != 0) goto error;

	printf("JPEG encoded at 0x%08x(%u bytes)\n",
		   (unsigned int)encIn.pOutBuf, encOut.jfifSize);

	_free_resources(&encIn);
	JpegEncRelease(encoder);

	extern void dma_pool_info();
	dma_pool_info();

	return 0;

error:
	printf("Failed to encode a JPEG frame\n");

	return 0;
}


U_BOOT_CMD(
	jpegenc_selftest,	CONFIG_SYS_MAXARGS,	1,	do_jpegenc_selftest,
	"Perform selftest of JPEG encoder",
	"\n"
	"	- To Encode a built-in color bar"
);


static void _init_encoder_config(JpegEncCfg* pCfg)
{
	pCfg->qLevel = 5;
	pCfg->frameType = JPEGENC_YUV420_PLANAR;
	pCfg->markerType = JPEGENC_SINGLE_MARKER;
	pCfg->unitsType = JPEGENC_DOTS_PER_INCH;
	pCfg->xDensity = 72;
	pCfg->yDensity = 72;
	pCfg->inputWidth = FRAME_WIDTH;
	pCfg->inputHeight = FRAME_HEIGHT;
	pCfg->codingWidth = FRAME_WIDTH;
	pCfg->codingHeight = FRAME_HEIGHT;
	pCfg->xOffset = 0;
	pCfg->yOffset = 0;
	pCfg->rotation = JPEGENC_ROTATE_0;
	pCfg->codingType = JPEGENC_WHOLE_FRAME;
	pCfg->restartInterval = 0;
/*
	pCfg->rotation = 0;
	pCfg->inputWidth = FRAME_WIDTH;
	pCfg->inputHeight = FRAME_HEIGHT;

	pCfg->xOffset = 0;
	pCfg->yOffset = 0;

	pCfg->codingWidth = FRAME_WIDTH;
	pCfg->codingHeight = FRAME_HEIGHT;
	pCfg->xDensity = 1;
	pCfg->yDensity = 1;

	pCfg->qLevel = 5;

	pCfg->restartInterval = 0;
	pCfg->codingType = 0;
	pCfg->frameType = JPEGENC_YUV420_PLANAR;
	pCfg->unitsType = 0;
	pCfg->markerType = 0;
	pCfg->colorConversion.type = 0;
	pCfg->codingMode = 0;
*/
}

static int  _open_jpeg_encoder(JpegEncCfg* pCfg, JpegEncInst * pEnc)
{
	JpegEncRet ret;

	if ((ret = JpegEncInit(pCfg, pEnc)) != JPEGENC_OK) {
		return -1;
	}

	return 0;
}

static int  _allocate_resources(JpegEncInst* pEnc, JpegEncCfg* pCfg,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut)
{
    i32 ret;

	unsigned char* output;
	unsigned char* pattern;

	pattern = dma_memory_alloc(PIC_SIZE);
	if (pattern == NULL) return -1;
	memcpy(pattern, colorbar_128x96_0, PIC_SIZE);

	output = dma_memory_alloc(OUTBUF_SIZE);
	if (output == NULL) return -1;
	memset(output, 0, OUTBUF_SIZE);

	/* Setup encoder input */
	pEncIn->pOutBuf = (u8*)output;
	pEncIn->busOutBuf = (u32)output;
	pEncIn->outBufSize = OUTBUF_SIZE;
	pEncIn->frameHeader = 1;

	/* Set Full Resolution mode */
	ret = JpegEncSetPictureSize(*pEnc, pCfg);
	if (ret != JPEGENC_OK) {
		return 1;
	}

	/* Bus addresses of input picture, used by hardware encoder */
	pEncIn->busLum = (u32)pattern;
	pEncIn->busCb  = (u32)(pattern + LUMA_SIZE);
	pEncIn->busCr  = (u32)(pattern + LUMA_SIZE + CHROMA_SIZE);

	/* Virtual addresses of input picture, used by software encoder */
	pEncIn->pLum   = pattern;
	pEncIn->pCb    = pattern + LUMA_SIZE;
	pEncIn->pCr    = pattern + LUMA_SIZE + CHROMA_SIZE;

	return 0;
}

static int _free_resources(JpegEncIn* pEncIn)
{
	dma_memory_free(pEncIn->pOutBuf);
	dma_memory_free(pEncIn->pLum);
}

static int  _start_jpeg_encode(JpegEncInst * pEnc,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut)
{
	int ret;

	ret = JpegEncEncode(*pEnc, pEncIn, pEncOut);

	return (ret == JPEGENC_FRAME_READY) ? 0 : 1;
}
