/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/h264decapi.h>
//#include <codec/hx9170/source/h264high/h264decapi.h>


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);
extern unsigned char colorbar_h264[];
extern unsigned int colorbar_h264_size;


enum _properties {
	FRAME_WIDTH  = 144,
	FRAME_HEIGHT = 96,

	LUMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT,
	CHROMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT / 4,
	PIC_SIZE = FRAME_WIDTH * FRAME_HEIGHT * 3 / 2,
};


static void _print_decoder_info(H264DecInfo* decInfo)
{
	printf("H.264 decoder information:\n");
	printf("    picWidth: %d\n", decInfo->picWidth);
	printf("    picHeight: %d\n", decInfo->picHeight);
	printf("    outputFormat: 0x%08x\n", decInfo->outputFormat);
}

int do_h264dec_selftest(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int picDecodeNumber = 0; /* decoded picture ID */

	H264DecRet  ret;
	H264DecInst decoder;

	H264DecInput   decIn;
	H264DecOutput  decOut;
	H264DecPicture decPic;
	H264DecInfo    decInfo;

	unsigned char* input;
	unsigned char* output;

	memset(&decoder, 0, sizeof(decoder));
	memset(&decIn,   0, sizeof(decIn));
	memset(&decOut,  0, sizeof(decOut));
	memset(&decPic,  0, sizeof(decPic));
	memset(&decInfo, 0, sizeof(decInfo));

	dma_memorypool_reset();

	input = dma_memory_alloc(colorbar_h264_size);
	if (input == NULL) {
		ret = H264DEC_PARAM_ERROR;
		printf("Failed to allocate memory block for H.264 frame\n");
		goto error;
	}
	memcpy(input, colorbar_h264, colorbar_h264_size);

	output = dma_memory_alloc(PIC_SIZE);
	if (output == NULL) {
		ret = H264DEC_PARAM_ERROR;
		printf("Failed to allocate memory block for output buffer\n");
		goto error;
	}

	/* Configure the decoder input structure for the first decoding unit */
	decIn.dataLen = colorbar_h264_size;
	decIn.pStream = (void*)input;
	decIn.streamBusAddress = (unsigned int)input;

	/* Decoder initialization, output reordering enabled */
	/* check for any initialization errors must be done */
	ret = H264DecInit(&decoder, 1, 0);
	if (ret != H264DEC_OK) {
		printf("Failed to initialize HX9170 H.264 decoder\n");
		goto error;
	}

	do {
		/* Picture ID is the picture number in decoding order */
		decIn.picId = picDecodeNumber;

		/* decode the stream data */
		ret = H264DecDecode(decoder, &decIn, &decOut);
		switch (ret) {
		case H264DEC_HDRS_RDY:
			/* read stream info */
			ret = H264DecGetInfo(decoder, &decInfo);
			if (ret == H264DEC_OK) {
				_print_decoder_info(&decInfo);
			} else {
				printf("Failed to get decoder information\n");
				goto failed;
			}
			break;
		case H264DEC_ADVANCED_TOOLS:
			/* Arbitrary Slice Order/multiple slice groups noticed */
			/* in the stream. The decoder has to reallocate resources */
			printf("H264DEC_ADVANCED_TOOLS\n");
			break;
		case H264DEC_PIC_DECODED: /* a picture was decoded */
			/* The decoder output is ready, now obtain pictures in */
			/* display order, if any available */
			H264DecNextPicture(decoder, &decPic, 0);

			memcpy(output, decPic.pOutputPicture, PIC_SIZE);
			printf("Decoded frame is at: 0x%08x(%d)\n",
				   (unsigned int)output, PIC_SIZE);

			/* Increment decoding number after every decoded picture */
			picDecodeNumber++;

			ret = H264DEC_OK;

			break;
		case H264DEC_STRM_PROCESSED:
			/* Input stream was processed but no picture is ready */
			printf("H264DEC_STRM_PROCESSED\n");
			break;
		default:
			/* all other cases are errors where decoding cannot continue */
			printf("Failed to decode video stream: %d\n", ret);
			goto failed;
		}

		if (decOut.dataLeft == 0) {
			decIn.dataLen = decOut.dataLeft;
		} else {
			/* data left undecoded */
			decIn.dataLen = decOut.dataLeft;
			decIn.pStream = decOut.pStrmCurrPos;
			decIn.streamBusAddress = decOut.strmCurrBusAddress;
		}

		/* keep decoding until all data from input stream buffer consumed */
	} while (decIn.dataLen > 0);
	//} while (decIn.dataLen > 0 && picDecodeNumber < options->count);

failed:
	H264DecRelease(decoder);

error:
	if (ret != H264DEC_OK) {
		printf("Failed to decode a JPEG frame\n");
	}

	return 0;
}


U_BOOT_CMD(
	h264dec_selftest,	CONFIG_SYS_MAXARGS,	1,	do_h264dec_selftest,
	"Perform selftest of H.264 decoder\n",
	"\n"
	"	- To decode a built-in color bar"
);
