/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/h264decapi.h>


static char* _progress = "|/-\\";

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);


extern void dma_memorypool_reset(void);


static void _print_decoder_info(H264DecInfo* decInfo)
{
	printf("H.264 decoder information:\n");
	printf("    width: %d\n", decInfo->picWidth);
	printf("    height: %d\n", decInfo->picHeight);
	printf("    output format: 0x%08x\n", decInfo->outputFormat);
}

int do_h264dec_bm(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int start;
	unsigned int stop;

	unsigned int count;
	unsigned int picDecodeNumber = 0; /* decoded picture ID */

	H264DecRet  ret;
	H264DecInst decoder;

	H264DecInput   decIn;
	H264DecOutput  decOut;
	H264DecPicture decPic;
	H264DecInfo    decInfo;

	unsigned int   size;
	unsigned char* input;
	unsigned int   lastPic;

	unsigned int width = 0;
	unsigned int height = 0;

	//if (argc != 3) {
	if (argc < 3 || argc > 4) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	timer2_start(10);

	input = (unsigned char*)simple_strtoul(argv[1], NULL, 16);
	size  = simple_strtoul(argv[2], NULL, 10);
	printf("Decoding H.264 bitstream from: 0x%08x(%u bytes)\n",
		   (unsigned int)input, size);

	if (argc >= 4) {
		lastPic = simple_strtoul(argv[3], NULL, 10);
	} else {
		lastPic = 0;
	}

	memset(&decoder, 0, sizeof(decoder));
	memset(&decIn,   0, sizeof(decIn));
	memset(&decOut,  0, sizeof(decOut));
	memset(&decPic,  0, sizeof(decPic));
	memset(&decInfo, 0, sizeof(decInfo));

	dma_memorypool_reset();

	/* Configure the decoder input structure for the first decoding unit */
	decIn.dataLen = size;
	decIn.pStream = (void*)input;
	decIn.streamBusAddress = (unsigned int)input;

	/* Decoder initialization, output reordering disabled */
	ret = H264DecInit(&decoder, 1, 0);
	if (ret != H264DEC_OK) {
		printf("Failed to initialize HX9170 H.264 decoder\r\n");
		goto error;
	}

	count = 0;
	start = timer2_now();

	do {
		/* Picture ID is the picture number in decoding order */
		decIn.picId = picDecodeNumber;

		/* decode the stream data */
		ret = H264DecDecode(decoder, &decIn, &decOut);
		switch (ret) {
		case H264DEC_HDRS_RDY:
			/* read stream info */
			ret = H264DecGetInfo(decoder, &decInfo);
			if (ret == H264DEC_OK) {
				if (width == 0 && height == 0) {
					width  = decInfo.picWidth;
					height = decInfo.picHeight;

					_print_decoder_info(&decInfo);
				}
			} else {
				printf("\nFailed to get decoder information\n");
				goto failed;
			}
			break;
		case H264DEC_ADVANCED_TOOLS:
			/* Arbitrary Slice Order/multiple slice groups noticed */
			/* in the stream. The decoder has to reallocate resources */
			printf("\nH264DEC_ADVANCED_TOOLS\n");
			break;
		case H264DEC_PIC_DECODED: /* a picture was decoded */
			++count;

			/* The decoder output is ready, now obtain pictures in */
			/* display order, if any available */
			while (H264DecNextPicture(decoder, &decPic, 0) == H264DEC_PIC_RDY) {
				printf("\rframe#%u decoded %c",
					   picDecodeNumber, _progress[picDecodeNumber & 0x3]);
			}

			/* Increment decoding number after every decoded picture */
			picDecodeNumber++;
			ret = H264DEC_OK;

			break;
		case H264DEC_STRM_PROCESSED:
			/* Input stream was processed but no picture is ready */
			//printf("H264DEC_STRM_PROCESSED\n");
			break;
		default:
			/* all other cases are errors where decoding cannot continue */
			printf("\nFailed to decode video stream: %d\n", ret);
			goto failed;
		}

		if (lastPic != 0 && count >= lastPic) {
			decIn.dataLen = 0;
			continue;
		}

		if (decOut.dataLeft == 0) {
			if (lastPic != 0) {
				decIn.dataLen = size;
				decIn.pStream = (void*)input;
				decIn.streamBusAddress = (unsigned int)input;
			} else {
				decIn.dataLen = decOut.dataLeft;
			}
		} else {
			/* data left undecoded */
			decIn.dataLen = decOut.dataLeft;
			decIn.pStream = decOut.pStrmCurrPos;
			decIn.streamBusAddress = decOut.strmCurrBusAddress;
		}

		/* keep decoding until all data from input stream buffer consumed */
	} while (decIn.dataLen > 0);

	stop = timer2_now();

	printf("\nTotal %u frames decoded\n", count);
	printf("Start time: %u, End time: %u\n", start, stop);
	printf("Average frame rate: %u\n", count * 100 / (stop - start));
	//printf("Average frame rate: %u\n", picDecodeNumber * 100 / (stop - start));

failed:
	H264DecRelease(decoder);

error:
	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	h264dec_benchmark,	CONFIG_SYS_MAXARGS,	1,	do_h264dec_bm,
	"To benchmark H.264 decoder",
	"[address] [stream size] [last pic(optional)]"
);
