/*
    Copyright (C) 2011, W.L. Chuang <ponponli2000@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <common.h>

#include "lwip/opt.h"

#include "lwip/def.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include <lwip/stats.h>
#include <lwip/snmp.h>
#include "netif/etharp.h"
#include "netif/ppp_oe.h"

#include <common.h>
//DECLARE_GLOBAL_DATA_PTR;


typedef enum _IntrVector_E { int_eth110 = 3, } IntrVector_E;
extern void vic_disirq(IntrVector_E irq);
extern void vic_cleanirq(IntrVector_E irq);

extern int   printf(const char *fmt, ...);
extern void* dma_memory_alloc(unsigned int size);
extern void  dma_memory_free(void* ptr);
//extern void* memory_copy(void *dst, const void *src, unsigned int n);


/* Define those to better describe your network interface. */
#define IFNAME0 'e'
#define IFNAME1 'n'


struct _rx_desc
{
	unsigned int     rdes0;
	unsigned int     rdes1;
	unsigned char*   buffer;
	struct _rx_desc* next;
};

struct _tx_desc
{
	unsigned int     tdes0;
	unsigned int     tdes1;
	unsigned char*   buffer;
	struct _tx_desc* next;
};

enum {
	TX_DESC_COUNT   = 128,
	RX_DESC_COUNT   = 128,
	TX_BUFFER_SIZE  = 0x600 /* 1536 */,
	RX_BUFFER_SIZE  = 0x620 /* 1568 */,
	MAX_PACKET_SIZE = 1514,
};

enum {
	CSR00 = 0, CSR01 = 1, CSR02 = 2,
	CSR03 = 3, TX_DESC_BASE_REG = 3,
	CSR04 = 4, RX_DESC_BASE_REG = 4,
	CSR05 = 5, CSR06 = 6,  CSR07 = 7,
	CSR08 = 8, CSR09 = 9, CSR10 = 10, CSR11 = 11,
	CSR06_PBF      = 0x8,
	CSR06_PM       = 0x400,
	CSR06_SFE      = (0x1 << 17), /* Frame transmission will be started after full frame has been moved into FIFO */
	CSR06_OPMFD    = (0x1 << 4),  /* Full duplex mode */
	CSR06_TXCNTL   = (0x1 << 9),  /* Transmit Start/Stop command */
	CSR06_RXCNTL   = (0x1 << 8),  /* Receive Start/Stop command */
	INT_TRCMP      = (0x1 << 14), /* Transfer complete include TX RX*/
	INT_ABNRML     = (0x1 << 13), /* All abnormal happen */
	INT_TXCMP      = (0x1 << 12), /* Current frame transmit complete.*/
	INT_RXCMP      = (0x1 << 11), /* Current frame receive complete.*/
	INT_TXPROCSTP  = (0x1 << 10), /* TX process enter stop state. */
	INT_RXPROCSTP  = (0x1 << 9),  /* RX process enter stop state. */
	INT_TXUNDERF   = (0x1 << 5),  /* TX fifo have been underflow.*/
	INT_RXOVERF    = (0x1 << 4),  /* RX fifo have been overflow.*/
	INT_TXUNAVAIL  = (0x1 << 3),  /* The next Tx descriptor unavailable. */
	INT_RXUNAVAIL  = (0x1 << 2),  /* The next Rx descriptor unavailable. */
	INT_SBE        = (0x1 << 1),  /* System bus error. */
	INT_LSC        = (0x1 << 0),  /* Link status change. */
	SETUP_FRAME    = 0x04000000,
	TDES0_ERR_MASK = 0x4302,
};

/**
 * Helper struct to hold private data used to operate your ethernet interface.
 * Keeping the ethernet address of the MAC in this struct is not necessary
 * as it is already kept in the struct netif.
 * But this is only an example, anyway...
 */
struct umvp2500_if {
	//struct eth_addr  ethaddr;
	unsigned char    ethaddr[8];

	unsigned int*    registers;
	struct _rx_desc  rxDescs[RX_DESC_COUNT];
	struct _tx_desc  txDescs[TX_DESC_COUNT];

	struct _rx_desc* rxReadyDesc;

	struct _tx_desc* txReadyDesc;
	struct _tx_desc* txToSendDesc;
	unsigned int     freeTxDescs;
	unsigned int     sendingTxDescs;
};

/* Forward declarations. */
//static void umvp2500_if_input(struct netif *netif);

static void _initialize_netif(struct netif* netif)
{
	unsigned int i;
	char env_enetaddr[6];
	char *tmp, *end;
	//bd_t *bd = gd->bd;

	/* set MAC hardware address length */
	netif->hwaddr_len = ETHARP_HWADDR_LEN;

	tmp = getenv("ethaddr");
	if (tmp) {
		for (i = 0; i < 6; ++i) {
			env_enetaddr[i] = tmp ? simple_strtoul(tmp, &end, 16) : 0;
			if (tmp) tmp = (*end) ? end + 1 : end;
		}
	}

	/* set MAC hardware address */
	netif->hwaddr[0] = env_enetaddr[0];
	netif->hwaddr[1] = env_enetaddr[1];
	netif->hwaddr[2] = env_enetaddr[2];
	netif->hwaddr[3] = env_enetaddr[3];
	netif->hwaddr[4] = env_enetaddr[4];
	netif->hwaddr[5] = env_enetaddr[5];
/*
	netif->hwaddr[0] = bd->bi_enetaddr[0];
	netif->hwaddr[1] = bd->bi_enetaddr[1];
	netif->hwaddr[2] = bd->bi_enetaddr[2];
	netif->hwaddr[3] = bd->bi_enetaddr[3];
	netif->hwaddr[4] = bd->bi_enetaddr[4];
	netif->hwaddr[5] = bd->bi_enetaddr[5];
*/

//	printf("lwIP MAC address@0x%08x[%02x:%02x:%02x:%02x:%02x:%02x]\n",
//		   (unsigned int)netif->hwaddr,
//		   netif->hwaddr[0], netif->hwaddr[1], netif->hwaddr[2],
//		   netif->hwaddr[3], netif->hwaddr[4], netif->hwaddr[5]);

	/* maximum transfer unit */
	netif->mtu = 1500;

	/* device capabilities */
	/* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
	netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP;
}

static void _reset_dev(struct netif* netif)
{
	unsigned int i = 0;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	dev->registers[CSR00] = 0x1;
	for( ; i < 0x100000; ++i) { ; }
	dev->registers[CSR00] &= ~0x1;

	memcpy(dev->ethaddr, netif->hwaddr, 6);
//	dev->ethaddr.addr[0] = netif->hwaddr[0];
//	dev->ethaddr.addr[1] = netif->hwaddr[1];
//	dev->ethaddr.addr[2] = netif->hwaddr[2];
//	dev->ethaddr.addr[3] = netif->hwaddr[3];
//	dev->ethaddr.addr[4] = netif->hwaddr[4];
//	dev->ethaddr.addr[5] = netif->hwaddr[5];

	printf("lwIP MAC address@0x%08x = [%02x:%02x:%02x:%02x:%02x:%02x]\n",
		   (unsigned int)dev->ethaddr,
		   dev->ethaddr[0], dev->ethaddr[1], dev->ethaddr[2],
		   dev->ethaddr[3], dev->ethaddr[4], dev->ethaddr[5]);
}

#define IS_RX_DESC_OWN_BY_DEV(rdes0)  (((rdes0) & 0x80000000) != 0)
#define SET_RX_DESC_OWN_BY_DEV(rdes0)  ((rdes0) |= (1 << 31))
#define SET_RX_DESC_CHAINED(rdes1)     ((rdes1) |= (1 << 22))
#define SET_RX_DESC_BUFFER_SIZE(rdes1, size) ((rdes1) |= size)

#define SET_TX_DESC_INT_COMP(tdes1)    ((tdes1) |= (1 << 31))
#define SET_TX_DESC_CHAINED(tdes1)     ((tdes1) |= (1 << 23))

static void _initialize_dev(struct netif* netif)
{
	unsigned int i;
	unsigned char* buffer;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	dev->registers = (unsigned int*)0x90400000;

	buffer = (unsigned char*)dev + sizeof(*dev);

	// Chaining RX descriptors up
	for (i = 0; i < RX_DESC_COUNT; ++i, buffer += RX_BUFFER_SIZE) {
		memset(&dev->rxDescs[i], 0, sizeof(struct _rx_desc));
		SET_RX_DESC_OWN_BY_DEV(dev->rxDescs[i].rdes0);
		SET_RX_DESC_CHAINED(dev->rxDescs[i].rdes1);
		SET_RX_DESC_BUFFER_SIZE(dev->rxDescs[i].rdes1, RX_BUFFER_SIZE);
		dev->rxDescs[i].buffer = buffer;
		dev->rxDescs[i].next   = &dev->rxDescs[i + 1];
	}
	dev->rxDescs[RX_DESC_COUNT - 1].next = &dev->rxDescs[0];
	dev->registers[RX_DESC_BASE_REG] = (unsigned int)dev->rxDescs;
	dev->rxReadyDesc = &dev->rxDescs[0];

	// Chaining TX descriptors up
	buffer += RX_BUFFER_SIZE;
	for (i = 0; i < TX_DESC_COUNT; ++i, buffer += TX_BUFFER_SIZE) {
		memset(&dev->txDescs[i], 0, sizeof(struct _tx_desc));
		dev->txDescs[i].tdes0 = 0;
		SET_TX_DESC_INT_COMP(dev->txDescs[i].tdes1);
		SET_TX_DESC_CHAINED(dev->txDescs[i].tdes1);
		dev->txDescs[i].buffer = buffer;
		dev->txDescs[i].next   = &dev->txDescs[i + 1];
	}
	dev->txDescs[TX_DESC_COUNT - 1].next = &dev->txDescs[0];
	dev->registers[TX_DESC_BASE_REG] = (unsigned int)dev->txDescs;

	dev->txReadyDesc    = &dev->txDescs[0];
	dev->txToSendDesc   = NULL;
	dev->freeTxDescs    = TX_DESC_COUNT;
	dev->sendingTxDescs = 0;
}

static void _send_dev_setup_frame(struct netif* netif)
{
	unsigned int i;
	unsigned int* suptr;
	unsigned short* addrptr;
	struct _tx_desc* txdesc;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	txdesc = dev->txReadyDesc;
	suptr = (unsigned int*)txdesc->buffer;

	/* Node address */
	addrptr = (unsigned short*)dev->ethaddr;
	*suptr++ = addrptr[0];
	*suptr++ = addrptr[1];
	*suptr++ = addrptr[2];

	/* broadcast address */
	*suptr++ = 0xffff;
	*suptr++ = 0xffff;
	*suptr++ = 0xffff;

	/* fit the multicast address */
/*
	for (mcptr = dev->mc_list, i = 0; i < mc_cnt; i++, mcptr = mcptr->next) {
		addrptr = (u16 *) mcptr->dmi_addr;
		*suptr++ = addrptr[0];
		*suptr++ = addrptr[1];
		*suptr++ = addrptr[2];
	}
*/
	for (i = 0; i < 14; i++) {
		*suptr++ = 0x8001;
		*suptr++ = 0x00c2;
		*suptr++ = 0x0100;
	}

	/* prepare the setup frame */
	txdesc->tdes0 = 0x80000000;
	txdesc->tdes1 = 0x800000c0 | SETUP_FRAME;

	dev->freeTxDescs--;
	dev->sendingTxDescs++;
	if (dev->txToSendDesc == NULL) {
		dev->txToSendDesc = txdesc;
	}
	dev->txReadyDesc = txdesc->next;

	//update_cr6(db->cr6_data, dev->base_addr);
	dev->registers[CSR01] = 0x1; /* Issue Tx polling */
	//update_cr6(db->cr6_data, dev->base_addr);

	printf("Sending setup frame to GUCNET\n");
}

/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init(struct netif* netif)
{
	unsigned int status;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	vic_disirq(int_eth110);
	vic_cleanirq(int_eth110);

	memset(dev, 0, sizeof(*dev));

	_initialize_netif(netif);
	_reset_dev(netif);
	_initialize_dev(netif);
	_send_dev_setup_frame(netif);

	// Clear Mercury status
	status = dev->registers[CSR05];
	dev->registers[CSR05] = status;

	// Setup operation modes
	dev->registers[CSR06] = CSR06_OPMFD | CSR06_SFE;

	// Setup interrupt filters
	dev->registers[CSR07] = 0; // Uses polling mode for lwip
/*
	dev->registers[CSR07] = INT_TRCMP | INT_ABNRML | INT_TXCMP |
							INT_RXCMP | INT_TXPROCSTP | INT_RXPROCSTP |
							INT_TXUNDERF | INT_RXOVERF | INT_TXUNAVAIL |
							INT_RXUNAVAIL | INT_SBE | INT_LSC;
*/

	// Start to transfer and receive
	dev->registers[CSR06] |= (CSR06_TXCNTL | CSR06_RXCNTL);
	//dev->registers[CSR06] |= (CSR06_TXCNTL | CSR06_RXCNTL | CSR06_PBF | CSR06_PM);
}

static void umvp2500_dev_tx_complete(struct netif *netif)
{
	struct _tx_desc* txdesc;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	txdesc = dev->txToSendDesc;
	if (txdesc == NULL) {
		//printf("\numvp2500_if[tx_comp]: inconsistence of TX descripters(%u/%u).\n",
		//	   dev->freeTxDescs, dev->sendingTxDescs);
		return;
	}

	//printf("\numvp2500_dev_tx_complete[In]: %u/%u\n", dev->freeTxDescs, dev->sendingTxDescs);

	while (dev->sendingTxDescs > 0) {
		if (txdesc->tdes0 & 0x80000000) break;

		/* A packet sent completed */
		dev->freeTxDescs++;
		dev->sendingTxDescs--;

		if (txdesc->tdes0 != 0x7fffffff) {
			if (txdesc->tdes0 & TDES0_ERR_MASK) {
				if (txdesc->tdes0 & 0x0002) { // UnderRun
					if (!(dev->registers[CSR06] & CSR06_SFE)) {
						dev->registers[CSR06] |= CSR06_SFE;
					}
				}
			}
		}

		/* Transmit statistic counter */
/*
		if (txdesc->tdes0 != 0x7fffffff) {
			db->stats.collisions += (tdes0 >> 3) & 0xf;
			db->stats.tx_bytes += le32_to_cpu(txptr->tdes1) & 0x7ff;
			if (tdes0 & TDES0_ERR_MASK) {
				db->stats.tx_errors++;

				if (tdes0 & 0x0002) {	// UnderRun
					db->tx_fifo_underrun++;
					if (!(db->cr6_data & CR6_SFT)) {
						db->cr6_data =
						    db->cr6_data | CR6_SFT;
						update_cr6(db->cr6_data,
							   db->ioaddr);
					}
				}
				if (tdes0 & 0x0100)
					db->tx_excessive_collision++;
				if (tdes0 & 0x0200)
					db->tx_late_collision++;
				if (tdes0 & 0x0400)
					db->tx_no_carrier++;
				if (tdes0 & 0x0800)
					db->tx_loss_carrier++;
				if (tdes0 & 0x4000)
					db->tx_jabber_timeout++;
			}
		}
*/
		txdesc = txdesc->next;
	}

	/* Update TX remove pointer */
	if (dev->sendingTxDescs == 0) {
		dev->txToSendDesc = NULL;
	} else {
		dev->txToSendDesc = txdesc;
		dev->registers[CSR01] = 0x1; /* Issue Tx polling */
	}

	//printf("\numvp2500_dev_tx_complete[Out]: %u/%u\n", dev->freeTxDescs, dev->sendingTxDescs);
}

static void umvp2500_dev_input(struct netif *netif);

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */
static err_t low_level_output(struct netif *netif, struct pbuf *packet)
{
	struct pbuf* q;
	//unsigned int masks;
	unsigned char* ptr;
	struct _tx_desc* txdesc;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	if ((dev->freeTxDescs + dev->sendingTxDescs) != TX_DESC_COUNT) {
		printf("\numvp2500_if[tx_output]: inconsistence of TX descripters(%u/%u).\n",
			   dev->freeTxDescs, dev->sendingTxDescs);
//		printf("\numvp2500_if: inconsistence of TX descripters(%u/%u). %u @ %s\n",
//			   dev->freeTxDescs, dev->sendingTxDescs,
//			   __LINE__, __FILE__);
	}

	/* Too large packet check */
	if (packet->tot_len > MAX_PACKET_SIZE) return ERR_MEM;

	/* No Tx resource check, it never happen nromally */
	while (dev->freeTxDescs == 0) {
		/* Read Mercury status */
		unsigned int status = dev->registers[CSR05];

		printf("\numvp2500_if: dev->freeTxDescs == 0\n");

		//status = dev->registers[CSR05];
/*
		if (status & 0x800) {
			// Clear Mercury status
			dev->registers[CSR05] = 0x800;

			umvp2500_dev_input(netif);
		}
		if (status & 0x4) {
			// Clear Mercury status
			dev->registers[CSR05] = 0x4;

			// RX descriptor unavailable.
			// Kick rx engine off again
			dev->registers[CSR02] = 0x1;
		}
*/

		if (status & 0x1000) {
			// Clear Mercury status
			dev->registers[CSR05] = 0x1000;

			/* Free the transmitted descriptor */
			umvp2500_dev_tx_complete(netif);
		}
	}

	/* Disable all interrupt in CR7 to solve the interrupt edge problem */
	//masks = dev->registers[CSR07];
	//dev->registers[CSR07] = 0;

	/* Transmit this packet */
	txdesc = dev->txReadyDesc;
	for(ptr = txdesc->buffer, q = packet; q != NULL; ptr += q->len, q = q->next) {
		memory_copy(ptr, q->payload, q->len);
	}
	txdesc->tdes0 = 0x80000000;  /* Set owner bit */
	txdesc->tdes1 = 0xe1000000 | packet->tot_len;

	/* Point to next transmit free descriptor */
	dev->freeTxDescs--;
	dev->sendingTxDescs++;
	if (dev->txToSendDesc == NULL) {
		dev->txToSendDesc = txdesc;
	}
	dev->txReadyDesc = txdesc->next;

	dev->registers[CSR01] = 0x1; /* Issue Tx polling */

	/* Restore CR7 to enable interrupt mask */
	//dev->registers[CSR07] = masks;

	//printf("\numvp2500_if: transmitting a packet(%u)\n", packet->tot_len);

	return 0;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
static void umvp2500_if_input(struct netif *netif, struct pbuf* p)
{
	struct eth_hdr *ethhdr;

	/* points to packet payload, which starts with an Ethernet header */
	ethhdr = p->payload;
	//printf("umvp2500_if: packet type is %u\n", htons(ethhdr->type));

	switch (htons(ethhdr->type)) {
	/* IP or ARP packet? */
	case ETHTYPE_IP:
	case ETHTYPE_ARP:
		//printf("umvp2500_if: IP/ARP packet received\n");
		/* full packet send to tcpip_thread to process */
		if (netif->input(p, netif) != ERR_OK) {
			LWIP_DEBUGF(NETIF_DEBUG, "umvp2500_if: IP input error\n");
			pbuf_free(p);
			p = NULL;
		}
		break;
	default:
		//printf("umvp2500_if: Unsupported packet received\n");
		pbuf_free(p);
		p = NULL;
		break;
	}
}

/**
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return a pbuf filled with the received packet (including MAC header)
 *         NULL on memory error
 */
static void umvp2500_dev_input(struct netif *netif)
{
	unsigned int rxlen;
	struct _rx_desc* rxDesc;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	rxDesc = dev->rxReadyDesc;
	while (!IS_RX_DESC_OWN_BY_DEV(rxDesc->rdes0)/* packet owner check */) {
		/* A packet with First/Last flag */
		rxlen = ((rxDesc->rdes0 >> 16) & 0x3fff) - 4;

		/* error summary bit check */
		if (rxDesc->rdes0 & 0x8000) {
			/* This is a error packet */
		} else {
			struct pbuf* p;

			p = pbuf_alloc(PBUF_RAW, rxlen, PBUF_POOL);
			if (p != NULL) {
				memcpy(p->payload, rxDesc->buffer, rxlen);
				umvp2500_if_input(netif, p);
			} else {
				printf("\numvp2500_if: Failed to create packet buffer\n");
			}
		}

		SET_RX_DESC_OWN_BY_DEV(rxDesc->rdes0);
		rxDesc = rxDesc->next;
	}
	dev->rxReadyDesc = rxDesc;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
static inline void umvp2500_dev_poll(struct netif *netif)
{
	//unsigned int masks;
	unsigned int status;
	struct umvp2500_if* dev = (struct umvp2500_if*)netif->state;

	/* Read Mercury status */
	status = dev->registers[CSR05];
	// Clear Mercury status
	dev->registers[CSR05] = status;
	if (status == 0) return;

	/* Disable all interrupt in CR7 to solve the interrupt edge problem */
	//masks = dev->registers[CSR07];
	//dev->registers[CSR07] = 0;

	/* Check system status */
	if (status & 0x2) {
		/* system bus error happen */
		printf("umvp2500_if: System bus error happen\n");
		printf("umvp2500_if: Try to reset this device\n");
		low_level_init(netif);
		return;
	}

	/* Receives incoming packets */
	if (status & 0x800) {
		//printf("\numvp2500_if: rx complete\n");
		umvp2500_dev_input(netif);
	}
	if (status & 0x4) {
		// RX descriptor unavailable.
		// Kick rx engine off again
		dev->registers[CSR02] = 0x1;
	}

	/* TX completed */
	if (status & 0x1000) {
		//printf("\numvp2500_if: tx complete\n");
		/* Free the transmitted descriptor */
		umvp2500_dev_tx_complete(netif);
	}

	/* Restore CR7 to enable interrupt mask */
	//dev->registers[CSR07] = masks;
#if 0
	struct pbuf *p;
	struct eth_hdr *ethhdr;

	/* move received packet into a new pbuf */
	p = low_level_input(netif);
	/* no packet could be read, silently ignore this */
	if (p == NULL) {
		return;
	}

	/* points to packet payload, which starts with an Ethernet header */
	ethhdr = p->payload;
	//printf("umvp2500_if: packet type is %u\n", htons(ethhdr->type));

	switch (htons(ethhdr->type)) {
	/* IP or ARP packet? */
	case ETHTYPE_IP:
	case ETHTYPE_ARP:
		//printf("umvp2500_if: IP/ARP packet received\n");
		/* full packet send to tcpip_thread to process */
		if (netif->input(p, netif) != ERR_OK) {
			LWIP_DEBUGF(NETIF_DEBUG, "umvp2500_if: IP input error\n");
			pbuf_free(p);
			p = NULL;
		}
		break;
	default:
		printf("umvp2500_if: Unsupported packet received\n");
		pbuf_free(p);
		p = NULL;
		break;
	}
#endif /* 0 */
}

/**
 * This function should be called from main loop.
 * It polls the interface for received frames and passes
 * them on to higher layers of the stack.
 */
void umvp2500_if_poll(struct netif *netif)
{
	//printf("umvp2500_if_poll: packet received\n");
	umvp2500_dev_poll(netif);
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t umvp2500_if_init(struct netif *netif)
{
	struct umvp2500_if* priv;

	LWIP_ASSERT("netif != NULL", (netif != NULL));

	priv = dma_memory_alloc(sizeof(*priv) +
							TX_DESC_COUNT * TX_BUFFER_SIZE +
							RX_DESC_COUNT * RX_BUFFER_SIZE);
	if (priv == NULL) {
		LWIP_DEBUGF(NETIF_DEBUG, "umvp2500_if_init: out of memory\n");
		return ERR_MEM;
	}

#if LWIP_NETIF_HOSTNAME
	/* Initialize interface hostname */
	netif->hostname = "lwip";
#endif /* LWIP_NETIF_HOSTNAME */

	/*
	 * Initialize the snmp variables and counters inside the struct netif.
	 * The last argument should be replaced with your link speed, in units
	 * of bits per second.
	 */
	//NETIF_INIT_SNMP(netif, snmp_ifType_ethernet_csmacd, LINK_SPEED_OF_YOUR_NETIF_IN_BPS);

	netif->state = priv;
	netif->name[0] = IFNAME0;
	netif->name[1] = IFNAME1;

	/* We directly use etharp_output() here to save a function call.
	* You can instead declare your own function an call etharp_output()
	* from it if you have to do some checks before sending (e.g. if link
	* is available...) */
	netif->output = etharp_output;
	netif->linkoutput = low_level_output;

	//priv->ethaddr = (struct eth_addr *)&(netif->hwaddr[0]);

	/* initialize the hardware */
	low_level_init(netif);

	return ERR_OK;
}
