/****************************************************************************
 *
 *  Copyright (c) 2009 by Evatronix SA
 *
 *  Please review the terms of the license agreement before using this file.
 *  If you are not an authorized user, please destroy this source code file
 *  and notify Evatronix SA <ipcenter@evatronix.pl> immediately
 *  that you inadvertently received an unauthorized copy.
 *
 *****************************************************************************/
/**
 * \file
 * \author      Piotr Sroka \<piotr.sroka@evatronix.pl\>
 * \brief       NAND Flash controller types definitions
 * \version     $Revision: 1.5 $
 * \date        $Date: 2010-05-14 11:16:12 $
 *
 ******************************************************************************/

/*****************************************************************************/
#ifndef TYPES_H
#define TYPES_H
/*****************************************************************************/
#define LINUX_MODULE
/*****************************************************************************/
#ifdef LINUX_MODULE
#   include <linux/types.h>
#else
#   include <limits.h>
#endif


#include "nf_setup.h"

#ifndef NULL
#define NULL 		  0
#endif

#ifndef LINUX_MODULE
/*****************************************************************************/
/** Type which size is one byte */
typedef unsigned char uint8_t;
typedef signed char int8_t;
#   if UINT_MAX == 0xFFFF
/** Type which size is two bytes */
typedef unsigned int uint16_t;
typedef int int16_t;
#   elif UINT_MAX == 0xFFFFFFFF
/** Type which size is two bytes */
typedef unsigned short uint16_t;
typedef short int16_t;
#   endif
/** Type which size is four bytes */
typedef unsigned long uint32_t;
typedef long int32_t;
/*****************************************************************************/
#endif

#define dma_address_t uint32_t  

/* supported NAND information */
#define NAND_ID_ST_128W3A           0x73207320                                  
#define NAND_ID_ST_256W3A           0x75207520
#define NAND_ID_ST_256W4A           0x20005000
#define NAND_ID_SAMSUNG_K9LBG       0x72C5D7EC
#define NAND_ID_SAMSUNG_K9GAG       0x7284D5EC
#define NAND_ID_SAMSUNG_K9K8G       0x9551D3EC 
#define NAND_ID_SAMSUNG_K9F1G       0x9500F1EC 	/* A1 EVB NAND */


/*****************************************************************************/
/** 
 * @struct factory_bb_mark_possition
 * @brief Structure defines where manufacturer places mark of bad blocks
 *
 * @typedef factory_bb_mark_possition_t
 * @brief defines where manufacturer places mark of bad blocks
 */
typedef struct factory_bb_mark_possition factory_bb_mark_possition_t;
struct factory_bb_mark_possition {
    /// number of pages per block on which manufacturer 
    /// bad block mark can occur
    uint8_t page_addr_count;
    /// page address (inside block) on which manufacturer 
    /// bad block mark can occure
    uint32_t page_addresses[2];
    /// number of page offsets per page on which manufacturer 
    /// bad block mark can occure
    uint8_t page_offset_count;
    /// page offset on which manufacturer 
    /// bad block mark can occure
    uint16_t page_offsets_in_spare[3];
};
/*****************************************************************************/

#define BBT_RAM_STATUS_VALID    1
#define BBT_RAM_STATUS_INVALID  2

/*****************************************************************************/
/** @name Definitions of status code of nand flash bbt. */
/*****************************************************************************/
/** @{ */
/** unknown */
#define BBT_STATUS_PROBLEM  2
#define BBT_STATUS_OK       3
/** @} */
/*****************************************************************************/

/*****************************************************************************/
/** 
 * @struct bbt_ram
 * @brief Structure contains number of bad block, and info 
 * if this block is valid.
 *
 * @typedef bbt_ram_t
 * @brief contains number of bad block, and info if this block is valid.
 */
typedef struct bbt_ram bbt_ram_t;
struct bbt_ram {
    /** number of bad block */
    uint16_t block_address;
    /** status of record (it defines if it is valid or invalid) */
    uint8_t status;
};
/*****************************************************************************/


/*****************************************************************************/
/** 
 * @struct bbm_data
 * @brief Structure contains data using by bbm module. 
 *
 * @typedef bbm_data_t
 * @brief contains data using by bbm module
 */
typedef struct bbm_data bbm_data_t;
struct bbm_data {
    /* current status of bad block table */
    uint8_t bbt_status;
    /** table contains bad block numbers. It some kind of bad blocks table cache */
    bbt_ram_t bad_blocks[BBT_COUNT_BB_IN_RAM];
};
/*****************************************************************************/
/*****************************************************************************/
/** 
 * @struct nf_timings
 * @brief Structure contains information about timing settings.
 *
 * @typedef nf_timings_t
 * @brief contains information about timing setting.
 */
typedef struct nf_timings nf_timings_t;
/** struct defines controller timing parameters */
struct nf_timings
{
    /** Change column setup */
    uint8_t tccs_ns;
    /** ALE to data loading time for synchronous interface.
      ALE to data start time for asynchronous interface.*/
    uint8_t tadl_ns;
    /** Data output to command , address, or data input time 
     * for synchronous interface. 
     * RE# high to WE# low time for asynchronous interface */
    uint8_t trhw_ns;
    /** command cycle to data output time for synchronous interface 
     * WE# hight to RE# low time*/
    uint8_t twhr_ns;
    /** RE# or WE# high hold time. For asynchronous interface. */
    uint8_t trwh_ns;
    /** RE# or WE# pulse width. For asynchronous interface. */
    uint8_t trwp_ns;
    /** Command, Address, Data delay. For synchronous interface*/
    uint8_t tcad_ns;
    /** Busy time for interface change. 
     * It's the busy time when the interface changes from asynchronous 
     * to synchronous using the SET FEATURES (EFh) command 
     * or synchronous to asynchronous using the RESET (FFh) command.*/
    uint8_t twb_ns;
    /** Read high to Read low. TRR time period from rising edge on 
     * read/buys input line to the moment when the read enable signal can 
     * be asserted. */
    uint8_t trr_ns;
};
/*****************************************************************************/


/*****************************************************************************/
/** 
 * @struct nf_controller
 * @brief Structure contains settings of nand flash controller.
 *
 * @typedef nf_controller_t
 * @brief contains settings of nand flash controlle.
 */
struct nf_controller;
typedef struct nf_controller nf_controller_t;
/*****************************************************************************/

/*****************************************************************************/
/** 
 * @struct nf_memory
 * @brief Structure contains information about single nand flash memory.
 *
 * @typedef nf_memory_t
 * @brief contains information about single nand flash memory.
 */
typedef struct nf_memory nf_memory_t;
struct nf_memory
{
    /** Device ID */
    uint32_t nandid;
    /** Memory page size in bytes wihout of spare area */
    uint16_t page_size;
    /** Number of address bits in a page */
    uint8_t page_shift;
    /** Number of address bits in a block (without column address bits)*/
    uint8_t pages_per_block_shift;
    /** Spare area size in bytes */
    uint8_t spare_area_size;
    /** Memory manufacturer ID */
    uint8_t manufacturer_id;
	/** Address cycle count */
	uint8_t addr_cyc_cnt;
    /** Number of pages per block*/
    uint16_t pages_per_block;
    /** Memory size in giga bits*/
    uint8_t memory_size_gb;
    /** Number of all blocks in whole memory device*/
    uint16_t blocks_count;
    /** Memory supports cache programing if it is 1 */
    uint8_t cache_prog_supp;
    /** Memory supports two plane oprations if it is 1 */
    uint8_t two_plane_supp;   
    /** number planes per one LUN */
    uint8_t planes_per_lun;
    /** number of LUN per one chip */
    uint8_t number_of_lun;
    /** if it is 1 the copy back is supported by memory */
    uint8_t copy_back_support;
    /** chip enable signal number (0 - 7) */
    uint8_t ce;
    /** this field contains current memory work mode */
    uint8_t current_work_mode;
    /** pointer to nand flash controller object */
    nf_controller_t *pcontroller;
    /** it defines where manufacturer places mark of bad blocks */
    factory_bb_mark_possition_t *fbbm_possition;    
#if BBM_SUPPORT
    bbm_data_t bbm_data;
#endif
	struct nf_timings timings;
};
/*****************************************************************************/


/*****************************************************************************/
struct nf_controller
{
    /** start address of nand flash controller registers*/
    uint32_t reg_offset;
    /** dma burst type settings */
    uint16_t dma_burst_type;
    /** NAND flash IO width (8 or 16) */
    uint8_t organization;
    /** nand flash page size */
    uint16_t page_size;
    /** nand flash block size */
    uint16_t block_size;
    /** nand flash spare area size */
    uint8_t spare_area_size;
    /** controller data transfer mode. All of transfer modes are here @ref transfer_mode */
    uint8_t transfer_mode;
    /** ecc offset */
    uint16_t ecc_offset;
    /* if it is 0 then hardware ECC is disabled, 
     * if it is 1 hardware ECC is enabled**/
    uint8_t ecc_enabled;
    /** ECC code size per ecc unit in bytes */
    uint8_t ecc_size_per_unit;
    /* status of ecc controller setting */
    uint8_t ecc_status;
    uint8_t aux_buffer[NF_MAX_PAGE_SIZE + NF_MAX_SPARE_AREA_SIZE];
    
	uint8_t ce;

    /** Device ID */
    uint32_t nandid;

	/** Last Command **/
    uint32_t	 last_cmd;
	uint32_t	 column;
	uint32_t	 page_addr;

	uint32_t pio_bufptr;

	uint32_t debug;
#if USE_INDIRECT_BUFFER
    // physical address of buffer used by DMA
    dma_address_t dma_buff_address;
    // pointer to logical dma address 
    uint8_t *dma_buffer;
#endif
};
/*****************************************************************************/


#define WITH_ECC		1
#define WITHOUT_ECC		0

/*****************************************************************************/
/** 
 * @struct nf_ctrl_command
 * @brief Structure describes command to execute by NAND Flash controller.
 *
 * @typedef nf_ctrl_command_t
 * @brief describes command to execute by NAND Flash controller
 */
typedef struct nf_ctrl_command nf_ctrl_command_t;
struct nf_ctrl_command
{
    /** command code */
    uint32_t command_code;
    /** page address 0 */
    uint32_t page_address_0;
    /** page address 0 */
    uint32_t page_address_1;
    /** page offset 0*/
    uint16_t page_offset_0;
    /** page offset 1*/
    uint16_t page_offset_1;
    /** data size. If command doesn't need data it should be 0.*/
    uint16_t data_size;
    /** buffer with read/written data */
    uint8_t *buffer; 
    /** data transfer direction, they are defined here @ref directions  */
    uint8_t transfer_direction;
    /** ecc mode, (WITH_ECC or WITHOUT_ECC) */
    uint8_t ecc_mode;
};
/*****************************************************************************/

/******************************************************************************/
/** @name nand flash controller universal commands flags  */
/******************************************************************************/
/** @{ */
/** Enable Command 0 phase */
#define NF_CTRL_UNIV_CMD_CMD0_EN        (1 << 0)
/** Enable Command 1 phase */
#define NF_CTRL_UNIV_CMD_CMD1_EN        (1 << 1)
/** Enable Command 2 phase */
#define NF_CTRL_UNIV_CMD_CMD2_EN        (1 << 2)
/** Enable Command 3 phase */
#define NF_CTRL_UNIV_CMD_CMD3_EN        (1 << 3)
/** Enable address 0 phase */
#define NF_CTRL_UNIV_CMD_ADDR0_EN       (1 << 4)
/** Enable address 0 phase */
#define NF_CTRL_UNIV_CMD_ADDR1_EN       (1 << 5)
#define NF_CTRL_UNIV_CMD_COL_ADDR_EN    (1 << 6)
/** @} */
/*****************************************************************************/

/******************************************************************************/
/** @name nand flash controller universal commands flags  */
/******************************************************************************/
/** @{ */
#define NF_CTRL_UNIV_CMD_DELLAY_BOTH_DISABLED   1
#define NF_CTRL_UNIV_CMD_DELLAY_0_ENABLED       2
#define NF_CTRL_UNIV_CMD_DELLAY_1_ENABLED       3
/** @} */
/*****************************************************************************/

/*****************************************************************************/
/** 
 * @struct nf_ctrl_universal_command
 * @brief Structure describes universal command to execute 
 * by NAND Flash controller.
 *
 * @typedef nf_ctrl_universal_command_t
 * @brief describes universal command to execute by NAND Flash controller
 */
typedef struct nf_ctrl_universal_command nf_ctrl_universal_command_t;
struct nf_ctrl_universal_command
{
    uint8_t cmd0;
    uint8_t cmd1;
    uint8_t cmd2;
    uint8_t cmd3;
    /** page address 0 */
    uint32_t page_address_0;
    /** page address 1 */
    uint32_t page_address_1;
    /** page offset 0*/
    uint16_t page_offset_0;
    /** page offset 1*/
    uint16_t page_offset_1;
    /** data size. If command doesn't need data it should be 0.*/
    uint16_t data_size;
    /** buffer with read/written data */
    uint8_t *buffer; 
    /** data transfer direction, they are defined here @ref directions  */
    uint8_t transfer_direction;
    /** ecc mode, (WITH_ECC or WITHOUT_ECC) */
    uint8_t ecc_mode;
    /** it defines sequence of command */
    uint8_t flags;
    /**  */
    uint8_t busy_delays;
};
/*****************************************************************************/





/*****************************************************************************/
#endif /* TYPES _H */
/*****************************************************************************/

