//  ***************************************************************************
// 
//   Copyright (c) 2009  Evatronix SA 
//
//  ***************************************************************************
//
//   Please review the terms of the license agreement before using     
//   this file. If you are not an authorized user, please destroy this 
//   source code file and notify Evatronix SA immediately that you     
//   inadvertently received an unauthorized copy.                      
//
//  ***************************************************************************
/// @file           nf_dma.c
/// @brief          NAND Flash controller DMA module driver.
/// @version        $Revision: 1.3 $
/// @author         Piotr Sroka
/// @date           $Date: 2010-04-27 09:39:11 $
//  ***************************************************************************
#include <linux/types.h>

#include <asm/arch/nand/nf_controller.h>
#include <asm/arch/nand/nf_error.h>
#include <asm/arch/nand/nf_setup.h>
#include <asm/arch/nand/nf_utils.h>
#include <asm/arch/nand/nf_dma.h>
#include <asm/arch/nand/nf_types.h>

/*****************************************************************************/
static void nf_dma_create_one_desc(uint8_t type, uint32_t block_address,
                                   uint16_t block_length, 
                                   uint32_t next_descriptor_address,
                                   uint32_t *descriptor)
{
    descriptor[0] = (uint32_t)block_length << 16 | (type & 3);
    descriptor[1] = block_address;

    if (type == NF_DMA_DESCRIPTOR_TYPE_LINK)
        descriptor[2] = next_descriptor_address;
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_dma_create_descriptors(uint32_t block_address[], 
                                  uint16_t block_length[],
                                  uint8_t descr_count, 
                                  uint32_t next_descriptor_address,
                                  uint32_t *descriptors)
{
    uint8_t i;

    if (!descr_count)
        return NF_ERR_INVALID_PARAMETER; 

    for (i = 0; i < descr_count - 1; i++){
        nf_dma_create_one_desc(NF_DMA_DESCRIPTOR_TYPE_SIMPLE,
                               block_address[i], block_length[i],
                               0, &descriptors[i * 2]);
    }

    if (next_descriptor_address){
        nf_dma_create_one_desc(NF_DMA_DESCRIPTOR_TYPE_LINK,
                               block_address[descr_count - 1], 
                               block_length[descr_count - 1],
                               next_descriptor_address, 
                               &descriptors[(descr_count - 1) * 2]);
    }
    else {
        nf_dma_create_one_desc(NF_DMA_DESCRIPTOR_TYPE_END,
                               block_address[descr_count - 1], 
                               block_length[descr_count - 1],
                               0, &descriptors[(descr_count - 1) * 2]);
    }

    return NF_ERR_NO_ERRORS;    
}
//EXPORT_SYMBOL(nf_dma_create_descriptors);
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_dma_prepare_transfer(nf_controller_t *pcontroller, 
                                uint32_t buffer_address, 
                                uint32_t size, uint8_t direction)
{
    uint32_t dma_ctrl = 0;

    if (buffer_address & 0x3){
        // error buffer is not aligned to 4
        return NF_ERR_DATA_NOT_ALIGNED_TO_4B;
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_DMA_ADDR, buffer_address);    

    if (pcontroller->transfer_mode == NF_TRANSFER_MODE_DMA_SFR){
        // set dma offset
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_DMA_ADDR_OFFSET, 0);
        // set data size 
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_DMA_CNT, size);
        // select DMA sfr mode
        dma_ctrl |= NF_SFR_DMA_CTRL_DMA_MODE_SFR;
    }
    else if (pcontroller->transfer_mode == NF_TRANSFER_MODE_DMA_SCATHER_GATHER){
        // select DMA scatter gather mode
        dma_ctrl |= NF_SFR_DMA_CTRL_DMA_MODE_SG;        
    }

    // set dma transfer direction
    if (direction == NF_TRANSFER_DIRECTION_READ){
        dma_ctrl |= NF_SFR_DMA_CTRL_DMA_DIR_READ;
    }
    else {
        dma_ctrl |= NF_SFR_DMA_CTRL_DMA_DIR_WRITE;    
    }

    // set burst type
    dma_ctrl |= pcontroller->dma_burst_type;

    // start dma transfer after command execute
    dma_ctrl |= NF_SFR_DMA_CTRL_DMA_START;

    // write settings do dma controll register 
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_DMA_CTRL, dma_ctrl);

    return NF_ERR_NO_ERRORS;               
}
//EXPORT_SYMBOL(nf_dma_prepare_transfer);
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_dma_finish_transfer(nf_controller_t *pcontroller)
{
    uint8_t status;
	uint8_t loop = 15;

    uint32_t dma_ctrl;

    status = wait_for_value(pcontroller->reg_offset + NF_SFR_DMA_CTRL, 
                            NF_SFR_DMA_READY_FLAG, 100, 1);

   while(status && (loop-- > 0)) {
            // one more chance 
            status = wait_for_value(pcontroller->reg_offset + NF_SFR_DMA_CTRL,
                            NF_SFR_DMA_READY_FLAG, 100, 1);
    }

    if (status) {
        return status;
    }


    dma_ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_DMA_CTRL);

    if (dma_ctrl & NF_SFR_DMA_ERR_FLAG)
        return NF_ERR_DMA_ERROR;

    return NF_ERR_NO_ERRORS;               
}
//EXPORT_SYMBOL(nf_dma_finish_transfer);
/*****************************************************************************/
