/*
 * Pool allocator ... wraps the pci_alloc_consistent page allocator, so
 * small blocks are easily used by drivers for bus mastering controllers.
 * This should probably be sharing the guts of the slab allocator.
 */
#define POOL_POISON_BYTE 0xa7

//#define printk(msg...)

struct ahb_pool * ahb_pool_create (const char *name, size_t size, size_t align, size_t allocation, int flags)
{
    struct ahb_pool        *retval;

    if (align == 0)
        align = 1;
    if (size == 0)
        return 0;
    else if (size < align)
        size = align;
    else if ((size % align) != 0)
    {
        size += align + 1;
        size &= ~(align - 1);
    }

    if (allocation == 0)
    {
        if (PAGE_SIZE < size)
            allocation = size;
        else
            allocation = PAGE_SIZE;
        // FIXME: round up for less fragmentation
    }
    else if (allocation < size)
        return 0;

    if (!(retval = kmalloc (sizeof *retval, flags)))
        return retval;

    strncpy (retval->name, name, sizeof retval->name);
    retval->name [sizeof retval->name - 1] = 0;

    INIT_LIST_HEAD (&retval->page_list);
    spin_lock_init (&retval->lock);
    retval->size = size;
    retval->flags = flags;
    retval->allocation = allocation;
    retval->blocks_per_page = allocation / size;
    init_waitqueue_head (&retval->waitq);
    return retval;
}

static inline int
is_ahb_page_busy (int blocks, unsigned long *bitmap)
{
    while (blocks > 0) {
        if (*bitmap++ != ~0UL)
            return 1;
        blocks -= BITS_PER_LONG;
    }
    return 0;
}

void ahb_free_consistent(size_t size,void *vaddr, dma_addr_t dma_handle)
{
    free_pages((unsigned long)vaddr, get_order(size));
}


static void
ahb_pool_free_page (struct ahb_pool *pool, struct ahb_page *page)
{
    dma_addr_t    dma = page->dma;

    if (pool->flags & SLAB_POISON)
        memset (page->vaddr, POOL_POISON_BYTE, pool->allocation);
    ahb_free_consistent (pool->allocation, page->vaddr, dma);
    list_del (&page->page_list);
    kfree (page);
}

void ahb_pool_destroy (struct ahb_pool *pool)
{
    unsigned long        flags;

    spin_lock_irqsave (&pool->lock, flags);
    while (!list_empty (&pool->page_list)) {
        struct ahb_page        *page;
        page = list_entry (pool->page_list.next,
                struct ahb_page, page_list);
        if (is_ahb_page_busy (pool->blocks_per_page, page->bitmap)) {
            /* leak the still-in-use consistent memory */
            list_del (&page->page_list);
            kfree (page);
        } else
            ahb_pool_free_page (pool, page);
    }
    spin_unlock_irqrestore (&pool->lock, flags);
    kfree (pool);
}

void *ahb_alloc_consistent(size_t size, dma_addr_t *dma_handle)
{
    void *ret;
    int gfp = GFP_ATOMIC;

        gfp |= GFP_DMA;
    ret = (void *)__get_free_pages(gfp, get_order(size));

    if (ret != NULL) {
//        printk("ahb_alloc_consistent:memory allocated\n");
        memset(ret, 0, size);
        
        *dma_handle = virt_to_bus(ret);
        
//        printk("ahb_alloc_consistent(), virt_to_bus(ret:%x)= *dma_handle:%x\n", ret, *dma_handle);
    }
    return ret;
}

static struct ahb_page *
ahb_pool_alloc_page (struct ahb_pool *pool, int mem_flags)
{
    struct ahb_page    *page;
    int        mapsize;

    mapsize = pool->blocks_per_page;
    mapsize = (mapsize + BITS_PER_LONG - 1) / BITS_PER_LONG;
    mapsize *= sizeof (long);

    page = (struct ahb_page *) kmalloc (mapsize + sizeof *page, mem_flags);
    if (!page)
        return 0;
    page->vaddr = ahb_alloc_consistent (pool->allocation,
                        &page->dma);
    if (page->vaddr) {
        memset (page->bitmap, 0xff, mapsize);    // bit set == free
        if (pool->flags & SLAB_POISON)
            memset (page->vaddr, AHB_POOL_POISON_BYTE, pool->allocation);
        list_add (&page->page_list, &pool->page_list);
    } else {
        kfree (page);
        page = 0;
    }
    return page;
}

void * ahb_pool_alloc (struct ahb_pool *pool, int mem_flags, dma_addr_t *handle)
{
    unsigned long        flags;
    struct list_head    *entry;
    struct ahb_page        *page;
    int            map, block;
    size_t            offset;
    void            *retval;

restart:
    spin_lock_irqsave (&pool->lock, flags);
    list_for_each (entry, &pool->page_list) {
        int        i;
        page = list_entry (entry, struct ahb_page, page_list);
        /* only cachable accesses here ... */
        for (map = 0, i = 0;
                i < pool->blocks_per_page;
                i += BITS_PER_LONG, map++) {
            if (page->bitmap [map] == 0)
                continue;
            block = ffz (~ page->bitmap [map]);
            if ((i + block) < pool->blocks_per_page) {
                clear_bit (block, &page->bitmap [map]);
                offset = (BITS_PER_LONG * map) + block;
                offset *= pool->size;
                goto ready;
            }
        }
    }
    if (!(page = ahb_pool_alloc_page (pool, mem_flags))) {
        if (mem_flags == SLAB_KERNEL) {
            DECLARE_WAITQUEUE (wait, current);

            current->state = TASK_INTERRUPTIBLE;
            add_wait_queue (&pool->waitq, &wait);
            spin_unlock_irqrestore (&pool->lock, flags);

            schedule_timeout (AHB_POOL_TIMEOUT_JIFFIES);

            current->state = TASK_RUNNING;
            remove_wait_queue (&pool->waitq, &wait);
            goto restart;
        }
        retval = 0;
        goto done;
    }

    clear_bit (0, &page->bitmap [0]);
    offset = 0;
ready:
    retval = offset + page->vaddr;
    *handle = offset + page->dma;
done:
    spin_unlock_irqrestore (&pool->lock, flags);
    return retval;
}


static struct ahb_page *
ahb_pool_find_page (struct ahb_pool *pool, dma_addr_t dma)
{
    unsigned long        flags;
    struct list_head    *entry;
    struct ahb_page        *page;

    spin_lock_irqsave (&pool->lock, flags);
    list_for_each (entry, &pool->page_list) {
        page = list_entry (entry, struct ahb_page, page_list);
        if (dma < page->dma)
            continue;
        if (dma < (page->dma + pool->allocation))
            goto done;
    }
    page = 0;
done:
    spin_unlock_irqrestore (&pool->lock, flags);
    return page;
}


void ahb_pool_free (struct ahb_pool *pool, void *vaddr, dma_addr_t dma)
{
    struct ahb_page        *page;
    unsigned long        flags;
    int            map, block;

    if ((page = ahb_pool_find_page (pool, dma)) == 0) {
//        printk (KERN_ERR "pci_pool_free %s/%s, %p/%x (bad dma)\n",
//            pool->dev ? pool->dev->slot_name : NULL,
//            pool->name, vaddr, (int) (dma & 0xffffffff));
        return;
    }


    block = dma - page->dma;
    block /= pool->size;
    map = block / BITS_PER_LONG;
    block %= BITS_PER_LONG;
    

    if (pool->flags & SLAB_POISON)
        memset (vaddr, AHB_POOL_POISON_BYTE, pool->size);

    spin_lock_irqsave (&pool->lock, flags);
    set_bit (block, &page->bitmap [map]);
    if (waitqueue_active (&pool->waitq))
        wake_up (&pool->waitq);
    /*
     * Resist a temptation to do
     *    if (!is_ahb_page_busy(bpp, page->bitmap)) pool_free_page(pool, page);
     * it is not interrupt safe. Better have empty pages hang around.
     */
    spin_unlock_irqrestore (&pool->lock, flags);
}


