//#include <linux/config.h>
//#include <asm/arch/platform.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <asm/arch-umvp/dma.h>
#include <asm/hardware/vic.h>
#include <mach/hardware.h>
#include <mach/platform2500.h>

#define MAX_CHANNEL_NR	2


#define DMAP_BASE           IO_ADDRESS(UMVP_DMAP_BASE)

#define DMAP_IntStat        (DMAP_BASE+0x00)
#define DMAP_IntTCStat      (DMAP_BASE+0x04)
#define DMAP_IntTCClr       (DMAP_BASE+0x08)
#define DMAP_IntErrStat     (DMAP_BASE+0x0C)
#define DMAP_IntErrClr      (DMAP_BASE+0x10)
#define DMAP_RawIntTCStat   (DMAP_BASE+0x14)
#define DMAP_RawIntErrStat  (DMAP_BASE+0x18)
#define DMAP_EnbldChns      (DMAP_BASE+0x1C)
#define DMAP_Sync           (DMAP_BASE+0x24)

/* Each channel's base address */
static unsigned int dmap_channel_bases[MAX_CHANNEL_NR] =
	{
        DMAP_BASE + 0x100,
        DMAP_BASE + 0x110,
	};

/* offsets of each channel's control registers */
#define DMAP_Cn_SrcAddr_Offset  0x0
#define DMAP_Cn_DestAddr_Offset 0x4
#define DMAP_Cn_Ctrl_Offset     0x8
#define DMAP_Cn_Config_Offset   0xC

#define UMVP_READ_REG(r)      (*((volatile unsigned int *) (r)))
#define UMVP_WRITE_REG(r,v)   (*((volatile unsigned int *) (r)) = ((unsigned int)   (v)))


struct umvp_dmap_sts
{
	int isFree;
	umvp_dmap_intr_cb cb;
	unsigned long param;
};

struct umvp_dmap_sts dmap_sts[MAX_CHANNEL_NR];
static spinlock_t dmap_lock;

//irqreturn_t dmap_isr(int irq, void *dev_id, struct pt_regs *regs)
irqreturn_t dmap_isr(int irq, void *dev_id)
{
	unsigned int nIntStatVal, i;
	register unsigned int nIntTermVal;
	register unsigned int nIntErrVal;


	/* query channel information */
    nIntStatVal = UMVP_READ_REG(DMAP_IntStat);
                                                                                                
                                                                                                
    /* Clear interrupt status on the IP site.
        Interrupt on the VIC site would be cleared automatically
        while this function is called */
                                                                                                
    /* clear term. interrupt */
    nIntTermVal = UMVP_READ_REG(DMAP_IntTCStat);
    if (nIntTermVal)
        UMVP_WRITE_REG(DMAP_IntTCClr, nIntTermVal);
                                                                                                
    /* clear error interrupt if there is any */
    nIntErrVal = UMVP_READ_REG(DMAP_IntErrStat);
    if (nIntErrVal)
        UMVP_WRITE_REG(DMAP_IntErrClr, nIntErrVal);

	for (i = 0; i < MAX_CHANNEL_NR; ++i)
	{
		if ((nIntStatVal & (1 << i)) && (dmap_sts[i].cb))
		{
			(dmap_sts[i].cb)(dmap_sts[i].param);
		}
	}

	return (IRQ_HANDLED);
}

int umvp_dmap_request_channel(umvp_dmap_intr_cb cbf, unsigned long data)
{
	int nr;
	register unsigned int nVal;

	spin_lock(&dmap_lock);
	nVal = UMVP_READ_REG(DMAP_EnbldChns);
	for (nr = 0; nr < MAX_CHANNEL_NR; ++nr)
	{
		if (dmap_sts[nr].isFree && !(nVal & (1 << nr)))
		{
			dmap_sts[nr].isFree = 0;
			dmap_sts[nr].cb = cbf;
			dmap_sts[nr].param = data;
			break;
		}
	}
	spin_unlock(&dmap_lock);

	if (nr >= MAX_CHANNEL_NR)
		return (-1); /* fail */

	return (nr); /* success */
}
EXPORT_SYMBOL_GPL(umvp_dmap_request_channel);

int umvp_dmap_request_some_channel(int chn, umvp_dmap_intr_cb cbf, unsigned long data)
{

	dmap_sts[chn].isFree = 0;
	dmap_sts[chn].cb = cbf;
	dmap_sts[chn].param = data;


	return chn;
}
EXPORT_SYMBOL_GPL(umvp_dmap_request_some_channel);

static void umvp_dmap_disable_channel(int nChannel)
{
	register unsigned int nRegAddr;

	nRegAddr = dmap_channel_bases[nChannel];
	nRegAddr += DMAP_Cn_Config_Offset;

	UMVP_WRITE_REG(nRegAddr, 0);
}

void umvp_dmap_release_channel(int nChannel)
{
	spin_lock(&dmap_lock);

	umvp_dmap_disable_channel(nChannel);
	dmap_sts[nChannel].isFree = 1;
	dmap_sts[nChannel].cb = NULL;
	dmap_sts[nChannel].param=0;
	UMVP_WRITE_REG(DMAP_IntTCClr, (1 << nChannel));
	UMVP_WRITE_REG(DMAP_IntErrClr, (1 << nChannel));

	spin_unlock(&dmap_lock);
}
EXPORT_SYMBOL_GPL(umvp_dmap_release_channel);

int umvp_dmap_start(int nChannel, struct umvp_dmap_params *params)
{
	register unsigned int base;
	register unsigned int value;
	unsigned int divider;

	if (nChannel < 0 || nChannel >= MAX_CHANNEL_NR ||
		dmap_sts[nChannel].isFree)
		return (-1);

	base = dmap_channel_bases[nChannel];

	/* source address */
	UMVP_WRITE_REG((DMAP_Cn_SrcAddr_Offset+base), params->src_addr_phy);

	/* destination address */
	UMVP_WRITE_REG((DMAP_Cn_DestAddr_Offset+base), params->dst_addr_phy);

	/* control */
	value = 0;
	if (params->dst_addr_inc)
		value |= (1 << 31);
	if (params->src_addr_inc)
		value |= (1 << 30);
	divider = 1;

	/* burst type */ /* Ted added */
	value |= (1 << 18);	// source burst type
	value |= (1 << 21);	// dest burst type

	switch (params->transf_width)
	{
	case transf_width_byte:
		divider = 1;
		break;

	case transf_width_halfword:
		divider = 2;
		value |= (1 << 27);
		value |= (1 << 24);
		break;

	case transf_width_word:
		divider = 4;
		value |= (2 << 27);
		value |= (2 << 24);
		break;
	};
	value |= (params->transf_bytes/divider);
	UMVP_WRITE_REG((DMAP_Cn_Ctrl_Offset+base), value);

	/* config */
	value = 0;
	value |= ((1 << 15) + (1 << 14));
	switch (params->transf_type)
	{
	case memory_to_memory:
		break;

	case memory_to_peripheral:
		value |= (1 << 11);
		switch (params->peripheral_type) /* dest. peripheral */
		{

		case umvp_dai_tx:
			value |= (1 << 6);
			break;

		case umvp_dai_rx:
			break;

		case memory:
		default:
			printk("DMAP: illegal input parameters");
			return (-1);
		};
		break;

	case peripheral_to_memory:
		value |= (2 << 11);
		switch (params->peripheral_type) /* source peripheral */
		{

		case umvp_dai_tx:
			value |= (1 << 1);
			break;

		case umvp_dai_rx:
			break;

		case memory:
		default:
			printk("DMAP: illegal input parameters");
			return (-1);
		};
		break;
	};

    switch (params->transf_mode)
	{
		case transf_mode_src_burst:
			 value |= (1 << 18);
			 break;
	    case transf_mode_dst_burst:
			 value |= (1 << 19);
			 break;
		case transf_mode_both_burst:
			 value |= (3 << 18);
			 break;
	    case transf_mode_sigle:
		default:	
		     break;	 
	};


	value |= 1;
	UMVP_WRITE_REG((DMAP_Cn_Config_Offset+base), value);

	return (0);
}
EXPORT_SYMBOL_GPL(umvp_dmap_start);

void umvp_dmap_stop(int nChannel)
{
	umvp_dmap_disable_channel(nChannel);
}
EXPORT_SYMBOL_GPL(umvp_dmap_stop);

int umvp_dmap_is_busy(int n)
{
	register unsigned int r;

	r = UMVP_READ_REG(DMAP_EnbldChns);

	return (r & (1 << n));
}
EXPORT_SYMBOL_GPL(umvp_dmap_is_busy);

int __init umvp_dmap_init(void)
{
	int i;

	/*
	 * initialize the data structures
	 */
	spin_lock_init(&dmap_lock);
	
	for (i = 0; i < MAX_CHANNEL_NR; ++i)
	{
		dmap_sts[i].isFree = 1;
		dmap_sts[i].cb = NULL;
		dmap_sts[i].param = 0;
	}

	/*
	 * clear pending interrupt if there is any
	 */
	UMVP_WRITE_REG(DMAP_IntTCClr,  0xFF);
	UMVP_WRITE_REG(DMAP_IntErrClr, 0xFF);

	/*
	 * register interrupt for DMAP
	 */
	// SA_INTERRUPT removed after 2.6.19
	//request_irq(INT_DMAP, dmap_isr, SA_INTERRUPT, "dmap", NULL);
	//request_irq(INT_DMAP, dmap_isr, IRQF_DISABLED, "dmap", NULL);
	request_irq(INT_DMAP, dmap_isr, 0, "dmap", NULL); 
	//request_irq(INT_DMAP, dmap_isr, IRQF_SHARED, "dmap", NULL);
	//enable_irq(INT_DMAP);

	return (0);
}

void __exit umvp_dmap_exit(void)
{
}

module_init(umvp_dmap_init);
module_exit(umvp_dmap_exit);

