/*
 * TW2865 ALSA SoC (ASoC) codec driver
 *
 * Author: Betta Lin <timur@freescale.com>
 *
 * Copyright 2010 ACTI.  This file is licensed under
 * the terms of the GNU General Public License version 2.  This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 * This is an ASoC device driver for TW2865.
 *
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <sound/core.h>
#include <sound/soc.h>
#include <sound/initval.h>
#include <linux/i2c.h>
#include <linux/delay.h>

#include "tw2865.h"
#define ACKI

/* If I2C is defined, then we support software mode.  However, if we're
   not compiled as module but I2C is, then we can't use I2C calls. */
#if defined(CONFIG_I2C) || (defined(CONFIG_I2C_MODULE) && defined(MODULE))
#define USE_I2C
#endif

/* Private data for the TW2865 */
struct tw2865_private {
	unsigned int mclk; /* Input frequency of the MCLK pin */
	unsigned int mode; /* The mode (I2S or left-justified) */
};

/*
 * The codec isn't really big-endian or little-endian, since the I2S
 * interface requires data to be sent serially with the MSbit first.
 * However, to support BE and LE I2S devices, we specify both here.  That
 * way, ALSA will always match the bit patterns.
 */
#define TW2865_FORMATS (SNDRV_PCM_FMTBIT_S8      | \
			SNDRV_PCM_FMTBIT_S16_LE  | SNDRV_PCM_FMTBIT_S16_BE  | \
			SNDRV_PCM_FMTBIT_U8)
			
#define TW2865_RATES (SNDRV_PCM_RATE_8000 | \
		SNDRV_PCM_RATE_16000 | \
		SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100 | \
		SNDRV_PCM_RATE_48000) 
		

#ifdef USE_I2C
/* TW2865 registers addresses */
#define SRESET      0x80 /* resets the device to its default state
                          * but all register content remain unchanged.
                          * This bit is self-resetting.
                          */

#define CLK_PLL_CTL		0x60 
#define AU_CLK_CTL		0x70	/* Audio Clock Control , ref : p64 */ // for sample rate 
#define AUVD_DET_STS		0x74	//0x01 /* Status of Video and Audio detection */ 
#define MIX_MUTE_A5		0x7e 
#define MIXRATIO5		0x7f	//0x80 /* Audio input AIN5 ratio value for audio mixing */

#define LUMADY_HFILTER_CTL		0x93	/* Luma Delay and H filter Control */
#define SER_MO_CTL		0xcf	/* Serial mode control */
#define NUM_AU_REC		0xd2  //0x80/0x81 /* Number of Audio to be Recorded */
#define SEQ_AU_REC		0xd7	/* Number of Audio to be Recorded */
#define MSR_CTL		0xdb	/* Master Control, DAC/ADC on/off */
#define MIX_OUY_SEL		0xe0	/* Mix Output Selection */
#define AU_DET_PT		0xe1	/* Audio Detection Period and Audio Detection Threshold */
#define AU_DET_T1		0xe2	/* Audio Detection Threshold */
#define AU_DET_T2		0xe3	/* Audio Detection Threshold */

#define ACKN_L		0xf3	/* Audio Clock Number , ref : p53 , sample rate related*/
#define ACKN_M		0xf4 	/* Audio Clock Number , ref : p53 */
#define ACKN_H		0xf5 	/* Audio Clock Number , ref : p53 */
#define ACKI_L		0xf0
#define ACKI_M		0xf1
#define ACKI_H		0xf2

#define SCDIV		0xf6 /* Serial Clock divider */
#define ACLKCTL		0xf8  /* Audio Clock Control */
#define OEN_CLKO_CTL		0xfa

//#define AINGAIN0		0xD0
//#define AINGAIN1		0xD1
//#define AINGAIN2		0x7F

/****/
/* TW2865 registers addresses */


/*
 * Clock Ratio Selection for Master Mode with I2C enabled
 *
 * The data for this chart is taken from Table 5 of the TW2865 reference
 * manual.
 *
 * This table is used to determine how to program the Mode Control register.
 * It is also used by tw2865_set_dai_sysclk() to tell ALSA which sampling
 * rates the TW2865 currently supports.
 *
 * Each element in this array corresponds to the ratios in mclk_ratios[].
 * These two arrays need to be in sync.
 *
 * 'speed_mode' is the corresponding bit pattern to be written to the
 * MODE bits of the Mode Control Register
 *
 * 'mclk' is the corresponding bit pattern to be wirten to the MCLK bits of
 * the Mode Control Register.
 *
 * In situations where a single ratio is represented by multiple speed
 * modes, we favor the slowest speed.  E.g, for a ratio of 128, we pick
 * double-speed instead of quad-speed.  However, the TW2865 errata states
 * that Divide-By-1.5 can cause failures, so we avoid that mode where
 * possible.
 *
 * ERRATA: There is an errata for the TW2865 where divide-by-1.5 does not
 * work if VD = 3.3V.  If this effects you, select the
 * CONFIG_SND_SOC_TW2865_VD33_ERRATA Kconfig option, and the driver will
 * never select any sample rates that require divide-by-1.5.
 */

#if 0 
static struct {
	unsigned int ratio;
	u8 speed_mode;
	u8 mclk;
} tw2865_mode_ratios[] = {

	{512, 0, 0},
};
#endif

/* The number of MCLK/LRCK ratios supported by the TW2865 */
#define NUM_MCLK_RATIOS		ARRAY_SIZE(tw2865_mode_ratios)

static int tw2865_write_reg(struct snd_soc_codec *codec, unsigned int reg,
	unsigned int value)
{
	u8 data[2];

	/* data is
	 *   D15..D9 WM8731 register offset
	 *   D8...D0 register data
	 */
	data[0] = reg;
	data[1] = value;

	if (codec->hw_write(codec->control_data, data, 2) == 2)
	{
		printk("reg = 0x%02X, value = 0x%02X\n",reg, value);
		return 0;
	}	
	else
		return -EIO;       
        
}


static unsigned int tw2865_read_reg(struct snd_soc_codec *codec, unsigned int reg)
{
        

        struct i2c_client *client = codec->control_data;
        struct i2c_msg msgs[2];
        unsigned char pbuf0[1];
        unsigned char pbuf[1];
				int errorCode;

        pbuf0[0] = reg & 0xFF;
        msgs[0].addr = client->addr;
        msgs[0].flags = client->flags;
        msgs[0].len = 1;
        msgs[0].buf = pbuf0;

        msgs[1].addr = client->addr;
        msgs[1].flags = client->flags | I2C_M_RD;
        msgs[1].len = 1;
        msgs[1].buf = pbuf;

        errorCode = i2c_transfer(client->adapter, msgs, 2);
        if ((errorCode != 1) && (errorCode != 2)) {
                printk("tw2865_read_reg(error) %d [0x%x]\n", errorCode, reg);
                return -EIO;
        } else {
                return pbuf[0];
                
        }
}

static unsigned char tw2865_audio_default_regs[]=
{
	OEN_CLKO_CTL, 0x42,
	CLK_PLL_CTL , 0x15, /* Clock PLL Control */ 

//	MIX_MUTE_A5	, 0xa3, /* bit5 For AIN5 mute function 0:normal, 1:mute */
//	MIXRATIO5	 , 0x80 /* Audio input AIN5 ratio value for audio mixing */
	LUMADY_HFILTER_CTL	 , 0x30, /* Luma Delay and H filter Control */
	SER_MO_CTL, 0x80, /* Serial mode control */
	NUM_AU_REC, 0x81, /* Number of Audio to be Recorded */
	SEQ_AU_REC, 0x32, /* Sequence of Audio to be Recorded */
	MSR_CTL, 0xc5,	/* Master Control, DAC(bit7)/ADC(bit6) on/off */
	MIX_OUY_SEL, 0x10, /* Mix Output Selection */
	AU_DET_PT, 0xc0, /* Audio Detection Period and Audio Detection Threshold */
	AU_DET_T1, 0xaa, /* Audio Detection Threshold */
	AU_DET_T2, 0xaa, /* Audio Detection Threshold */

	SCDIV, 0x00, /* Serial Clock divider */
	ACLKCTL, 0xc4 /* Audio Clock Control */


};
static int tw2865_write_reg_array(struct snd_soc_codec *codec)
{
        int array_size, i, ret = 0;
        
        unsigned char subaddr, data;
        
				array_size = sizeof(tw2865_audio_default_regs);
        for (i=0; i<array_size; i+=2)
        {
                subaddr =  tw2865_audio_default_regs[i]; data = tw2865_audio_default_regs[i+1];
                ret = tw2865_write_reg(codec, subaddr, data);
        }
        return ret;


}

static void tw2865_reset(struct snd_soc_codec *codec)
{
        unsigned int data = 0;

   
        data = tw2865_read_reg(codec, SRESET);

        tw2865_write_reg(codec, SRESET, 0x01 | data);
        msleep(1);
}



/*
 * Determine the TW2865 samples rates.
 *
 * 'freq' is the input frequency to MCLK.  The other parameters are ignored.
 *
 * The value of MCLK is used to determine which sample rates are supported
 * by the TW2865.  The ratio of MCLK / Fs must be equal to one of nine
 * support values: 64, 96, 128, 192, 256, 384, 512, 768, and 1024.
 *
 * This function calculates the nine ratios and determines which ones match
 * a standard sample rate.  If there's a match, then it is added to the list
 * of support sample rates.
 *
 * This function must be called by the machine driver's 'startup' function,
 * otherwise the list of supported sample rates will not be available in
 * time for ALSA.
 *
 * Note that in stand-alone mode, the sample rate is determined by input
 * pins M0, M1, MDIV1, and MDIV2.  Also in stand-alone mode, divide-by-3
 * is not a programmable option.  However, divide-by-3 is not an available
 * option in stand-alone mode.  This cases two problems: a ratio of 768 is
 * not available (it requires divide-by-3) and B) ratios 192 and 384 can
 * only be selected with divide-by-1.5, but there is an errate that make
 * this selection difficult.
 *
 * In addition, there is no mechanism for communicating with the machine
 * driver what the input settings can be.  This would need to be implemented
 * for stand-alone mode to work.
 */
static int tw2865_set_dai_sysclk(struct snd_soc_dai *codec_dai,
				 int clk_id, unsigned int freq, int dir)
{
	//struct snd_soc_codec *codec = codec_dai->codec;
#if 0 //betta	
	struct tw2865_private *tw2865 = codec->private_data;
	unsigned int rates = 0;
	unsigned int rate_min = -1;
	unsigned int rate_max = 0;
	


	tw2865->mclk = freq;

	for (i = 0; i < NUM_MCLK_RATIOS; i++) {
		unsigned int rate = freq / tw2865_mode_ratios[i].ratio;
		rates |= snd_pcm_rate_to_rate_bit(rate);
		if (rate < rate_min)
			rate_min = rate;
		if (rate > rate_max)
			rate_max = rate;
	}
	/* FIXME: soc should support a rate list */
	rates &= ~SNDRV_PCM_RATE_KNOT;

	if (!rates) {
		printk(KERN_ERR "tw2865: could not find a valid sample rate\n");
		return -EINVAL;
	}

	codec_dai->playback.rates = rates;
	codec_dai->playback.rate_min = rate_min;
	codec_dai->playback.rate_max = rate_max;

	codec_dai->capture.rates = rates;
	codec_dai->capture.rate_min = rate_min;
	codec_dai->capture.rate_max = rate_max;
#endif

	return 0;
}

/*
 * Configure the codec for the selected audio format
 *
 * This function takes a bitmask of SND_SOC_DAIFMT_x bits and programs the
 * codec accordingly.
 *
 * Currently, this function only supports SND_SOC_DAIFMT_I2S and
 * SND_SOC_DAIFMT_LEFT_J.  The TW2865 codec also supports right-justified
 * data for playback only, but ASoC currently does not support different
 * formats for playback vs. record.
 */
static int tw2865_set_dai_fmt(struct snd_soc_dai *codec_dai,
			      unsigned int format)
{
	//struct snd_soc_codec *codec = codec_dai->codec;
	//struct tw2865_private *tw2865 = codec->private_data;
	int ret = 0;

#if 0 //betta
	switch (format & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
	case SND_SOC_DAIFMT_LEFT_J:
		//tw2865->mode = format & SND_SOC_DAIFMT_FORMAT_MASK;
		break;
	default:
		printk(KERN_ERR "tw2865: invalid DAI format\n");
		ret = -EINVAL;
	}
#endif

	return ret;
}

/*
 * A list of addresses on which this TW2865 could use.  I2C addresses are
 * 7 bits.  For the TW2865, the upper four bits are always 1001, and the
 * lower three bits are determined via the AD2, AD1, and AD0 pins
 * (respectively).
 */
static const unsigned short normal_i2c[] = {
	0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, I2C_CLIENT_END
};
I2C_CLIENT_INSMOD;

/*
 * Pre-fill the TW2865 register cache.
 *
 * We use the auto-increment feature of the TW2865 to read all registers in
 * one shot.
 */
static int tw2865_fill_cache(struct snd_soc_codec *codec)
{
	return 0;
}

/*
 * Program the TW2865 with the given hardware parameters.
 *
 * The .ops functions are used to provide board-specific data, like
 * input frequencies, to this driver.  This function takes that information,
 * combines it with the hardware parameters provided, and programs the
 * hardware accordingly.
 */
static int tw2865_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_device *socdev = rtd->socdev;
	struct snd_soc_codec *codec = socdev->codec;
	struct tw2865_private *tw2865 = codec->private_data;
	
	int ret = 0;	
	
	
	
	tw2865_write_reg_array(codec); //betta m
/* The followings are sampling rate supported by WM8731 */
	switch (params_rate(params)) {
	case 8000:
#ifdef 	ACKI	
		tw2865_write_reg(codec, ACKI_L, 0x83);
		tw2865_write_reg(codec, ACKI_M, 0xb5);
		tw2865_write_reg(codec, ACKI_H, 0x09);
		tw2865_write_reg(codec, ACKN_L, 0x78);
		tw2865_write_reg(codec, ACKN_M, 0x85);
		tw2865_write_reg(codec, ACKN_H, 0x00);
#endif		
		tw2865_write_reg(codec, AU_CLK_CTL, 0x08);
		break;
	case 16000:
#ifdef 	ACKI	
		tw2865_write_reg(codec, ACKI_L, 0x07);
		tw2865_write_reg(codec, ACKI_M, 0x6b);
		tw2865_write_reg(codec, ACKI_H, 0x13);
		tw2865_write_reg(codec, ACKN_L, 0xef);
		tw2865_write_reg(codec, ACKN_M, 0x0a);
		tw2865_write_reg(codec, ACKN_H, 0x01);
#endif			
		tw2865_write_reg(codec, AU_CLK_CTL, 0x09);
		break;
	case 32000:
#ifdef 	ACKI	
		tw2865_write_reg(codec, ACKI_L, 0x0e);
		tw2865_write_reg(codec, ACKI_M, 0xd6);
		tw2865_write_reg(codec, ACKI_H, 0x26);
		tw2865_write_reg(codec, ACKN_L, 0xde);
		tw2865_write_reg(codec, ACKN_M, 0x15);
		tw2865_write_reg(codec, ACKN_H, 0x02);
#endif		
		tw2865_write_reg(codec, AU_CLK_CTL, 0x0a);
		break;

	case 44100:
#ifdef 	ACKI	
		tw2865_write_reg(codec, ACKI_L, 0x65);
		tw2865_write_reg(codec, ACKI_M, 0x85);
		tw2865_write_reg(codec, ACKI_H, 0x35);
		tw2865_write_reg(codec, ACKN_L, 0xbc);
		tw2865_write_reg(codec, ACKN_M, 0xdf);
		tw2865_write_reg(codec, ACKN_H, 0x02);
#endif	
		tw2865_write_reg(codec, AU_CLK_CTL, 0x0b);
		break;
				
	case 48000:
#ifdef 	ACKI
		tw2865_write_reg(codec, ACKI_L, 0x15);
		tw2865_write_reg(codec, ACKI_M, 0x41);
		tw2865_write_reg(codec, ACKI_H, 0x3a);
		tw2865_write_reg(codec, ACKN_L, 0xcd);
		tw2865_write_reg(codec, ACKN_M, 0x20);
		tw2865_write_reg(codec, ACKN_H, 0x03);
#endif		
		tw2865_write_reg(codec, AU_CLK_CTL, 0x0c);
		break;

	default:
		printk(KERN_ERR "tw2865: unknown sampling rate\n");
		return -EINVAL;	
	}


	
	/* Figure out which MCLK/LRCK ratio to use */
#if 0

	rate = params_rate(params);	/* Sampling rate, in Hz */
	ratio = tw2865->mclk / rate;	/* MCLK/LRCK ratio */
	

	for (i = 0; i < NUM_MCLK_RATIOS; i++) {
		if (tw2865_mode_ratios[i].ratio == ratio)
			break;
	}


	if (i == NUM_MCLK_RATIOS) {
		/* We did not find a matching ratio */
		printk(KERN_ERR "tw2865: could not find matching ratio\n");
		return -EINVAL;
	}
#endif

	/* Freeze and power-down the codec */


	/* Program the mode control register */


	switch (tw2865->mode) {
	case SND_SOC_DAIFMT_I2S:
		
		break;
	default:
		printk(KERN_ERR "tw2865: unknown format\n");
		return -EINVAL;
	}


	/* Disable auto-mute.  This feature appears to be buggy, because in
	   some situations, auto-mute will not deactivate when it should. */


	/* Disable automatic volume control.  It's enabled by default, and
	 * it causes volume change commands to be delayed, sometimes until
	 * after playback has started.
	 */


	/* Thaw and power-up the codec */




	return ret;
}



/*
 * Set the TW2865 mute
 *
 */
static int tw2865_mute(struct snd_soc_dai *dai, int mute)
{
	//struct snd_soc_codec *codec = dai->codec;
	int ret = 0;
	//unsigned int data = snd_soc_read(codec, MSR_CTL);

	if (mute)
	{
		;
	}	
	else
		;

	return ret;
}

#endif

static int tw2865_i2c_probe(struct i2c_client *, const struct i2c_device_id *);

#if 0
/* A list of non-DAPM controls that the TW2865 supports */
static const struct snd_kcontrol_new tw2865_snd_controls[] = {
	SOC_DOUBLE_R("Master Playback Volume",
		TW2865_VOLA, TW2865_VOLB, 0, 0xFF, 1)
};
#endif

static const struct i2c_device_id tw2865_id[] = {
	{"tw2865", 0},
	{}
};
MODULE_DEVICE_TABLE(i2c, tw2865_id);

static struct i2c_driver tw2865_i2c_driver = {
	.driver = {
		.name = "TW2865 I2C",
		.owner = THIS_MODULE,
	},
	.id_table = tw2865_id,
	.probe = tw2865_i2c_probe,
};

/*
 * Global variable to store socdev for i2c probe function.
 *
 * If struct i2c_driver had a private_data field, we wouldn't need to use
 * tw2865_socdec.  This is the only way to pass the socdev structure to
 * tw2865_i2c_probe().
 *
 * The real solution to tw2865_socdev is to create a mechanism
 * that maps I2C addresses to snd_soc_device structures.  Perhaps the
 * creation of the snd_soc_device object should be moved out of
 * tw2865_probe() and into tw2865_i2c_probe(), but that would make this
 * driver dependent on I2C.  The TW2865 supports "stand-alone" mode, whereby
 * the chip is *not* connected to the I2C bus, but is instead configured via
 * input pins.
 */
static struct snd_soc_device *tw2865_socdev;

/*
 * Initialize the I2C interface of the TW2865
 *
 * This function is called for whenever the I2C subsystem finds a device
 * at a particular address.
 *
 * Note: snd_soc_new_pcms() must be called before this function can be called,
 * because of snd_ctl_add().
 */
static int tw2865_i2c_probe(struct i2c_client *i2c_client,
	const struct i2c_device_id *id)
{
	struct snd_soc_device *socdev = tw2865_socdev;
	struct snd_soc_codec *codec = socdev->codec;

	int ret = 0;

	/* Probing all possible addresses has one drawback: if there are
	   multiple TW2865s on the bus, then you cannot specify which
	   socdev is matched with which TW2865.  For now, we just reject
	   this I2C device if the socdev already has one attached. */
	if (codec->control_data)
		return -ENODEV;

	/* Note: codec_dai->codec is NULL here */

#if 0
	codec->reg_cache = kzalloc(TW2865_NUMREGS, GFP_KERNEL);
	if (!codec->reg_cache) {
		printk(KERN_ERR "tw2865: could not allocate register cache\n");
		ret = -ENOMEM;
		goto error;
	}
#endif

	/* Verify that we have a TW2865 */

	/* *************************** */
	codec->control_data = i2c_client;
	codec->read = tw2865_read_reg;
	codec->write = tw2865_write_reg;
	//codec->reg_cache_size = TW2865_NUMREGS;

	/* The I2C interface is set up, so pre-fill our register cache */

	ret = tw2865_fill_cache(codec);
	if (ret < 0) {
		printk(KERN_ERR "tw2865: failed to fill register cache\n");
		goto error;
	}

	/* Add the non-DAPM controls */
#if 0
	for (i = 0; i < ARRAY_SIZE(tw2865_snd_controls); i++) {
		struct snd_kcontrol *kctrl =
		snd_soc_cnew(&tw2865_snd_controls[i], codec, NULL);

		ret = snd_ctl_add(codec->card, kctrl);
		if (ret < 0)
			goto error;
	}
#endif

	i2c_set_clientdata(i2c_client, codec);

printk("Exit tw2865_i2c_probe\n");
	return 0;

error:
	codec->control_data = NULL;

	kfree(codec->reg_cache);
	codec->reg_cache = NULL;
	codec->reg_cache_size = 0;

	return ret;
}



struct snd_soc_dai tw2865_dai = {
	.name = "TW2865",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = TW2865_RATES,
		.formats = TW2865_FORMATS,
	},
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = TW2865_RATES,
		.formats = TW2865_FORMATS,
	},
#if 0	
	.ops = {
		//.prepare = wm8731_pcm_prepare,
		.hw_params = wm8731_hw_params,
		//.shutdown = wm8731_shutdown,
		.digital_mute = wm8731_mute,
		.set_sysclk = wm8731_set_dai_sysclk,
		.set_fmt = wm8731_set_dai_fmt,
#endif		
};
EXPORT_SYMBOL_GPL(tw2865_dai);

static int tw2865_add_i2c_device(struct platform_device *pdev,
				 const struct tw2865_setup_data *setup)
{
	struct i2c_board_info info;
	struct i2c_adapter *adapter;
	struct i2c_client *client;
	int ret;

	ret = i2c_add_driver(&tw2865_i2c_driver);
	if (ret != 0) {
		dev_err(&pdev->dev, "can't add i2c driver\n");
		return ret;
	}

	memset(&info, 0, sizeof(struct i2c_board_info));
	info.addr = setup->i2c_address;
	strlcpy(info.type, "tw2865", I2C_NAME_SIZE);

printk("i2c_bus = %d, i2c_address =0x%x\n", setup->i2c_bus, setup->i2c_address);
	adapter = i2c_get_adapter(setup->i2c_bus);
	if (!adapter) {
		dev_err(&pdev->dev, "can't get i2c adapter %d\n",
			setup->i2c_bus);
		printk("can't get i2c adapter %d\n", setup->i2c_bus);
		goto err_driver;
	}

	client = i2c_new_device(adapter, &info);
	i2c_put_adapter(adapter);
	if (!client) {
		dev_err(&pdev->dev, "can't add i2c device at 0x%x\n",
			(unsigned int)info.addr);
		goto err_driver;
	}

	return 0;

err_driver:
	i2c_del_driver(&tw2865_i2c_driver);
	return -ENODEV;
}

/*
 * ASoC probe function
 *
 * This function is called when the machine driver calls
 * platform_device_add().
 */
static int tw2865_probe(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec;
	struct tw2865_setup_data *setup;
	int ret = 0;

	printk(KERN_INFO "TW2865 ALSA SoC Codec\n");

	setup = socdev->codec_data;
	/* Allocate enough space for the snd_soc_codec structure
	   and our private data together. */
	codec = kzalloc(ALIGN(sizeof(struct snd_soc_codec), 4) +
			sizeof(struct tw2865_private), GFP_KERNEL);
	
	
	
	if (codec == NULL)
		return -ENOMEM;
					
	if (!codec) {
		printk(KERN_ERR "tw2865: Could not allocate codec structure\n");
		return -ENOMEM;
	}

	mutex_init(&codec->mutex);
	INIT_LIST_HEAD(&codec->dapm_widgets);
	INIT_LIST_HEAD(&codec->dapm_paths);

	codec->name = "TW2865";
	codec->owner = THIS_MODULE;
	codec->dai = &tw2865_dai;
	codec->num_dai = 1;
	codec->private_data = (void *) codec +
		ALIGN(sizeof(struct snd_soc_codec), 4);

	socdev->codec = codec;

	/* Register PCMs */

	ret = snd_soc_new_pcms(socdev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1);
	if (ret < 0) {
		printk(KERN_ERR "tw2865: failed to create PCMs\n");
		goto error_free_codec;
	}

#ifdef USE_I2C
	tw2865_socdev = socdev;

	//ret = i2c_add_driver(&tw2865_i2c_driver);

	if (setup->i2c_address) {
printk("codec->hw_write = (hw_write_t)i2c_master_send;\n");
		codec->hw_write = (hw_write_t)i2c_master_send;
		ret = tw2865_add_i2c_device(pdev, setup);
	}

	if (ret) {
		printk("tw2865: failed to attach driver");
		goto error_free_pcms;
	}

	/* Did we find a TW2865 on the I2C bus? */
	if (codec->control_data) {
		/* Initialize codec ops */
		tw2865_dai.ops.hw_params = tw2865_hw_params;
		tw2865_dai.ops.set_sysclk = tw2865_set_dai_sysclk;
		tw2865_dai.ops.set_fmt = tw2865_set_dai_fmt;

		tw2865_dai.ops.digital_mute = tw2865_mute;

	} else
		printk(KERN_INFO "tw2865: no I2C device found, "
			"using stand-alone mode\n");
#else
	printk(KERN_INFO "tw2865: I2C disabled, using stand-alone mode\n");
#endif

	ret = snd_soc_init_card(socdev);
	if (ret < 0) {
		printk(KERN_ERR "tw2865: failed to register card\n");
		goto error_del_driver;
	}

	//tw2865_reset(codec);  //betta 
	
	
	printk("Exit tw2865_probe\n");
	return 0;

error_del_driver:
#ifdef USE_I2C
	i2c_del_driver(&tw2865_i2c_driver);

error_free_pcms:
#endif
	snd_soc_free_pcms(socdev);

error_free_codec:
	kfree(socdev->codec);
	socdev->codec = NULL;

	return ret;
}

static int tw2865_remove(struct platform_device *pdev)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec = socdev->codec;

	snd_soc_free_pcms(socdev);

#ifdef USE_I2C
	i2c_unregister_device(codec->control_data);
	i2c_del_driver(&tw2865_i2c_driver);
#endif

	kfree(socdev->codec);
	socdev->codec = NULL;

	return 0;
}

/*
 * ASoC codec device structure
 *
 * Assign this variable to the codec_dev field of the machine driver's
 * snd_soc_device structure.
 */
struct snd_soc_codec_device soc_codec_device_tw2865 = {
	.probe = 	tw2865_probe,
	.remove = 	tw2865_remove
};
EXPORT_SYMBOL_GPL(soc_codec_device_tw2865);

static int __init tw2865_init(void)
{
	return snd_soc_register_dai(&tw2865_dai);
}
module_init(tw2865_init);

static void __exit tw2865_exit(void)
{
	snd_soc_unregister_dai(&tw2865_dai);
}
module_exit(tw2865_exit);

MODULE_AUTHOR("Betta Lin <Betta.Lin@acti.com>");
MODULE_DESCRIPTION("TW2865 ALSA SoC Codec Driver");
MODULE_LICENSE("GPL");
