
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/ptrace.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/version.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/skbuff.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/mii.h>

#include <asm/irq.h>
#include <asm/processor.h>
#include <linux/crc32.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/uaccess.h>
#include <mach/hardware.h>

#define GUC_NAME	"umvp_eth"
#define MY_VERSION	"2.00.01"
#define MY_RELDATE	"2012-11-28"


/* Board/System/Debug information/definition ---------------- */

#define SETUP_FRAME      	0x04000000
#define TX_MAX_SEND_CNT  	0x1			/* Maximum tx packet per time */
#define TX_DESC_CNT      	0x10		/* Allocated Tx descriptors */
#define RX_DESC_CNT      	0x1a		/* Allocated Rx descriptors */
#define TX_FREE_DESC_CNT 	(TX_DESC_CNT - 2)	/* Max TX packet count */
#define TX_WAKE_DESC_CNT 	(TX_DESC_CNT - 3)	/* TX wakeup count */
#define DESC_ALL_CNT     	(TX_DESC_CNT + RX_DESC_CNT)
#define TX_BUF_ALLOC     	0x600
#define RX_ALLOC_SIZE    	0x620

#define GUCNET_RESET     	1
#define CR0_DEFAULT      	0x0		   /* set 1 for MAC SW Reset @R:20121004*/

#define CR6_DEFAULT     	(0x10000|0x10)	/* [16].FCE=1 (Flow Control), [4].OM=1 (Full Duplex) */
#define CR7_DEFAULT      	0x1ff7
#define CR15_DEFAULT     	0x06		/* TxJabber RxWatchdog */

#define TDES0_ERR_MASK   	0x4302		/* TXJT, LC, EC, FUE */
#define MAX_PACKET_SIZE  	1514
#define GUCNET_MAX_MULTICAST 14
#define RX_COPY_SIZE	 	0
#define MAX_CHECK_PACKET 	0x8000

#define GUCNET_10MHF     	0
#define GUCNET_100MHF    	1
#define GUCNET_10MFD     	4
#define GUCNET_100MFD    	5
#define GUCNET_AUTO      	8
#define GUCNET_1M_HPNA   	0x10

#if 0  //no used
	#define GUCNET_TXTH_72	 	0x400000	/* TX TH 72 byte */
	#define GUCNET_TXTH_96	 	0x404000	/* TX TH 96 byte */
	#define GUCNET_TXTH_128	 	0x0000		/* TX TH 128 byte */
	#define GUCNET_TXTH_256	 	0x4000		/* TX TH 256 byte */
	#define GUCNET_TXTH_512	 	0x8000		/* TX TH 512 byte */
	#define GUCNET_TXTH_1K		0xC000		/* TX TH 1K  byte */
#endif

#define GUCNET_TIMER_WUT  	(jiffies + HZ * 1)	/* timer wakeup time : 1 second */
#define GUCNET_TX_TIMEOUT 	((16*HZ)/2)	/* tx packet time-out time 8 s" */
#define GUCNET_TX_KICK 		(4*HZ/2)	/* tx packet Kick-out time 2 s" */



#define GUC_DBUG(dbug_now, msg, value) if (gucnet_debug || (dbug_now)) printk(KERN_ERR GUC_NAME ": %s 0x%lx\n", (msg), (long) (value))

/* CR9 definition: SROM/MII */
#define CR9_SROM_READ   	0x4800
#define CR9_SRCS        	0x1
#define CR9_SRCLK       	0x2
#define CR9_CRDOUT      	0x8
#define SROM_DATA_0     	0x0
#define SROM_DATA_1     	0x4
#define PHY_DATA_1      	0x20
#define PHY_DATA_0      	0x00000
#define MDCLKH          	0x10
#define MII_READ        	0x40
#define MDIO_DELAY_TICK 	1
#define PHY_POWER_DOWN		0x800

#define SROM_V41_CODE   	0x14

#define SROM_CLK_WRITE(data, ioaddr) 						\
		outl(data|CR9_SROM_READ|CR9_SRCS,ioaddr);			\
		udelay(5);											\
		outl(data|CR9_SROM_READ|CR9_SRCS|CR9_SRCLK,ioaddr);	\
		udelay(5);											\
		outl(data|CR9_SROM_READ|CR9_SRCS,ioaddr);			\
		udelay(5);											\

/* Sten Check */
#define DEVICE net_device

#define reg_inl(a)			(*(volatile unsigned int  *)(a))
#define reg_oul(v,a)		(*(volatile unsigned int  *)(a) = (v))

extern unsigned char A1_macaddr[6];

/* Structure/enum declaration ------------------------------- */
struct tx_desc {
	u32 tdes0, tdes1, tdes2, tdes3;	/* Data for the card */
	char *tx_buf_ptr;	/* Data for us */
	struct tx_desc *next_tx_desc;
} __attribute__ ((aligned(32)));

struct rx_desc {
	u32 rdes0, rdes1, rdes2, rdes3;	/* Data for the card */
	struct sk_buff *rx_skb_ptr;		/* Data for us */
	struct rx_desc *next_rx_desc;
} __attribute__ ((aligned(32)));

struct gucnet_board_info {
	u32 chip_id;				/* Chip vendor/Device ID */
	u32 chip_revision;			/* Chip revision */
	struct DEVICE *next_dev;	/* next device */
	//struct pci_dev *pdev;     /* PCI device */
	spinlock_t lock;

	long ioaddr;				/* I/O base address */
	u32 cr0_data;
	u32 cr5_data;
	u32 cr6_data;
	u32 cr7_data;
	u32 cr15_data;

	/* pointer for memory physical address */
	dma_addr_t buf_pool_dma_ptr;	/* Tx buffer pool memory */
	dma_addr_t buf_pool_dma_start;	/* Tx buffer pool align dword */
	dma_addr_t desc_pool_dma_ptr;	/* descriptor pool memory */
	dma_addr_t first_tx_desc_dma;
	dma_addr_t first_rx_desc_dma;

	/* descriptor pointer */
	unsigned char *buf_pool_ptr;	/* Tx buffer pool memory */
	unsigned char *buf_pool_start;	/* Tx buffer pool align dword */
	unsigned char *desc_pool_ptr;	/* descriptor pool memory */
	struct tx_desc *first_tx_desc;
	struct tx_desc *tx_insert_ptr;
	struct tx_desc *tx_remove_ptr;
	struct rx_desc *first_rx_desc;
	struct rx_desc *rx_insert_ptr;
	struct rx_desc *rx_ready_ptr;	/* packet come pointer */
	unsigned long tx_packet_cnt;	/* transmitted packet count */
	unsigned long tx_queue_cnt;	    /* wait to send packet count */
	unsigned long rx_avail_cnt;		/* available rx descriptor count */
	unsigned long interval_rx_cnt;	/* rx packet count a callback time */

	u16 PHY_reg4;			/* Saved Phyxcer register 4 value */
	u8 media_mode;			/* user specify media mode */
	u8 op_mode;				/* real work media mode */
	u8 phy_addr;
	u8 link_failed;			/* Ever link failed */
	u8 wait_reset;			/* Hardware failed, need to reset */
	u8 gucnet_chk_mode;		/* Operating mode check */
	struct timer_list timer;

	/* System defined statistic counter */
	struct net_device_stats stats;

	/* Driver defined statistic counter */
	unsigned long tx_fifo_underrun;
	unsigned long tx_loss_carrier;
	unsigned long tx_no_carrier;
	unsigned long tx_late_collision;
	unsigned long tx_excessive_collision;
	unsigned long tx_jabber_timeout;
	unsigned long reset_count;
	unsigned long reset_cr8;
	unsigned long reset_fatal;
	unsigned long reset_TXtimeout;

	/* NIC SROM data */
	unsigned char srom[128];

};

enum gucnet_offsets {
	DCR0  = 0x00, DCR1  = 0x04, DCR2  = 0x08, DCR3  = 0x0c, DCR4  = 0x10,
	DCR5  = 0x14, DCR6  = 0x18, DCR7  = 0x1c, DCR8  = 0x20, DCR9  = 0x24,
	DCR10 = 0x28, DCR11 = 0x2c, DCR12 = 0x30, DCR13 = 0x34, DCR14 = 0x38,
	DCR15 = 0x3c
};

// IGDNET001A
// 1. CR6_PAM = 0x80 In IGDNET001A is set Force Collision Mode
// 2. CR6_PM, CR6_PAM map to Receive All
// 3. CR6_PBF, map to reserve
#if 0
enum gucnet_CR6_bits {
	CR6_RXSC = 0x100, CR6_PBF   = 0x8,   CR6_PM    = 0x400,    CR6_PAM = 0x80,
	CR6_FDM  = 0x10,  CR6_TXSC  = 0x200, CR6_BUST_LEN = 0x100000, CR6_FWC = 0x10000,
	CR6_SFT  = 0x20000, CR6_RXA = 0x400, CR6_NO_PURGE = 0x20000000
};
#else
enum gucnet_CR6_bits {
	CR6_RXSC = 0x100, CR6_PBF   = 0x8,   CR6_PM = 0x400, CR6_PAM = 0x400,
	CR6_FDM  = 0x10,  CR6_TXSC  = 0x200, CR6_BUST_LEN = 0x100000, CR6_FWC = 0x10000,
	CR6_SFT  = 0x20000
};
#endif
/* Global variable declaration ----------------------------- */
static int __devinitdata printed_version = 0;
static char version[] __devinitdata =
						KERN_INFO GUC_NAME ": A1 net driver, version "
						MY_VERSION " (" MY_RELDATE ")\n";
/* Debug Level --------------------------------------------- */
#define ETH_MSG_NONE		0
#define ETH_MSG_ERR			1
#define ETH_MSG_INFO		2
#define ETH_MSG_LEVEL		ETH_MSG_INFO

#define ETH_PHY_ADDR		0x01
/* Extension MII Registers in IP101GA ---------------------- */
#define MII_PHY_CTRL		16 /* UTP PHY Specific Control Register, reg 16 */
static int gucnet_open_cnt = 0;

static int gucnet_debug = 0;

static struct net_device *gucnet_dev = NULL;

unsigned long CrcTable[256] = {
	0x00000000L, 0x77073096L, 0xEE0E612CL, 0x990951BAL,
	0x076DC419L, 0x706AF48FL, 0xE963A535L, 0x9E6495A3L,
	0x0EDB8832L, 0x79DCB8A4L, 0xE0D5E91EL, 0x97D2D988L,
	0x09B64C2BL, 0x7EB17CBDL, 0xE7B82D07L, 0x90BF1D91L,
	0x1DB71064L, 0x6AB020F2L, 0xF3B97148L, 0x84BE41DEL,
	0x1ADAD47DL, 0x6DDDE4EBL, 0xF4D4B551L, 0x83D385C7L,
	0x136C9856L, 0x646BA8C0L, 0xFD62F97AL, 0x8A65C9ECL,
	0x14015C4FL, 0x63066CD9L, 0xFA0F3D63L, 0x8D080DF5L,
	0x3B6E20C8L, 0x4C69105EL, 0xD56041E4L, 0xA2677172L,
	0x3C03E4D1L, 0x4B04D447L, 0xD20D85FDL, 0xA50AB56BL,
	0x35B5A8FAL, 0x42B2986CL, 0xDBBBC9D6L, 0xACBCF940L,
	0x32D86CE3L, 0x45DF5C75L, 0xDCD60DCFL, 0xABD13D59L,
	0x26D930ACL, 0x51DE003AL, 0xC8D75180L, 0xBFD06116L,
	0x21B4F4B5L, 0x56B3C423L, 0xCFBA9599L, 0xB8BDA50FL,
	0x2802B89EL, 0x5F058808L, 0xC60CD9B2L, 0xB10BE924L,
	0x2F6F7C87L, 0x58684C11L, 0xC1611DABL, 0xB6662D3DL,
	0x76DC4190L, 0x01DB7106L, 0x98D220BCL, 0xEFD5102AL,
	0x71B18589L, 0x06B6B51FL, 0x9FBFE4A5L, 0xE8B8D433L,
	0x7807C9A2L, 0x0F00F934L, 0x9609A88EL, 0xE10E9818L,
	0x7F6A0DBBL, 0x086D3D2DL, 0x91646C97L, 0xE6635C01L,
	0x6B6B51F4L, 0x1C6C6162L, 0x856530D8L, 0xF262004EL,
	0x6C0695EDL, 0x1B01A57BL, 0x8208F4C1L, 0xF50FC457L,
	0x65B0D9C6L, 0x12B7E950L, 0x8BBEB8EAL, 0xFCB9887CL,
	0x62DD1DDFL, 0x15DA2D49L, 0x8CD37CF3L, 0xFBD44C65L,
	0x4DB26158L, 0x3AB551CEL, 0xA3BC0074L, 0xD4BB30E2L,
	0x4ADFA541L, 0x3DD895D7L, 0xA4D1C46DL, 0xD3D6F4FBL,
	0x4369E96AL, 0x346ED9FCL, 0xAD678846L, 0xDA60B8D0L,
	0x44042D73L, 0x33031DE5L, 0xAA0A4C5FL, 0xDD0D7CC9L,
	0x5005713CL, 0x270241AAL, 0xBE0B1010L, 0xC90C2086L,
	0x5768B525L, 0x206F85B3L, 0xB966D409L, 0xCE61E49FL,
	0x5EDEF90EL, 0x29D9C998L, 0xB0D09822L, 0xC7D7A8B4L,
	0x59B33D17L, 0x2EB40D81L, 0xB7BD5C3BL, 0xC0BA6CADL,
	0xEDB88320L, 0x9ABFB3B6L, 0x03B6E20CL, 0x74B1D29AL,
	0xEAD54739L, 0x9DD277AFL, 0x04DB2615L, 0x73DC1683L,
	0xE3630B12L, 0x94643B84L, 0x0D6D6A3EL, 0x7A6A5AA8L,
	0xE40ECF0BL, 0x9309FF9DL, 0x0A00AE27L, 0x7D079EB1L,
	0xF00F9344L, 0x8708A3D2L, 0x1E01F268L, 0x6906C2FEL,
	0xF762575DL, 0x806567CBL, 0x196C3671L, 0x6E6B06E7L,
	0xFED41B76L, 0x89D32BE0L, 0x10DA7A5AL, 0x67DD4ACCL,
	0xF9B9DF6FL, 0x8EBEEFF9L, 0x17B7BE43L, 0x60B08ED5L,
	0xD6D6A3E8L, 0xA1D1937EL, 0x38D8C2C4L, 0x4FDFF252L,
	0xD1BB67F1L, 0xA6BC5767L, 0x3FB506DDL, 0x48B2364BL,
	0xD80D2BDAL, 0xAF0A1B4CL, 0x36034AF6L, 0x41047A60L,
	0xDF60EFC3L, 0xA867DF55L, 0x316E8EEFL, 0x4669BE79L,
	0xCB61B38CL, 0xBC66831AL, 0x256FD2A0L, 0x5268E236L,
	0xCC0C7795L, 0xBB0B4703L, 0x220216B9L, 0x5505262FL,
	0xC5BA3BBEL, 0xB2BD0B28L, 0x2BB45A92L, 0x5CB36A04L,
	0xC2D7FFA7L, 0xB5D0CF31L, 0x2CD99E8BL, 0x5BDEAE1DL,
	0x9B64C2B0L, 0xEC63F226L, 0x756AA39CL, 0x026D930AL,
	0x9C0906A9L, 0xEB0E363FL, 0x72076785L, 0x05005713L,
	0x95BF4A82L, 0xE2B87A14L, 0x7BB12BAEL, 0x0CB61B38L,
	0x92D28E9BL, 0xE5D5BE0DL, 0x7CDCEFB7L, 0x0BDBDF21L,
	0x86D3D2D4L, 0xF1D4E242L, 0x68DDB3F8L, 0x1FDA836EL,
	0x81BE16CDL, 0xF6B9265BL, 0x6FB077E1L, 0x18B74777L,
	0x88085AE6L, 0xFF0F6A70L, 0x66063BCAL, 0x11010B5CL,
	0x8F659EFFL, 0xF862AE69L, 0x616BFFD3L, 0x166CCF45L,
	0xA00AE278L, 0xD70DD2EEL, 0x4E048354L, 0x3903B3C2L,
	0xA7672661L, 0xD06016F7L, 0x4969474DL, 0x3E6E77DBL,
	0xAED16A4AL, 0xD9D65ADCL, 0x40DF0B66L, 0x37D83BF0L,
	0xA9BCAE53L, 0xDEBB9EC5L, 0x47B2CF7FL, 0x30B5FFE9L,
	0xBDBDF21CL, 0xCABAC28AL, 0x53B39330L, 0x24B4A3A6L,
	0xBAD03605L, 0xCDD70693L, 0x54DE5729L, 0x23D967BFL,
	0xB3667A2EL, 0xC4614AB8L, 0x5D681B02L, 0x2A6F2B94L,
	0xB40BBE37L, 0xC30C8EA1L, 0x5A05DF1BL, 0x2D02EF8DL
};

/* function declaration ------------------------------------- */
static int gucnet_open(struct DEVICE *);
static int gucnet_start_xmit(struct sk_buff *, struct DEVICE *);
static int gucnet_stop(struct DEVICE *);
static struct net_device_stats *gucnet_get_stats(struct DEVICE *);
static void gucnet_set_filter_mode(struct DEVICE *);
static int gucnet_do_ioctl(struct DEVICE *, struct ifreq *, int);
static irqreturn_t gucnet_interrupt(int, void *);
#ifdef CONFIG_NET_POLL_CONTROLLER
static void poll_umvp (struct net_device *dev);
#endif		
static void gucnet_descriptor_init(struct gucnet_board_info *, unsigned long);
static void allocate_rx_buffer(struct gucnet_board_info *);
static void update_cr6(u32, unsigned long);
static void send_filter_frame(struct DEVICE *, int);
static u16 phy_read(unsigned long, u8, u8);
static void phy_write(unsigned long, u8, u8, u16);
static void phy_write_1bit(unsigned long, u32);
static u16 phy_read_1bit(unsigned long);
static void gucnet_timer(unsigned long);
static void gucnet_rx_packet(struct DEVICE *, struct gucnet_board_info *);
static void gucnet_free_tx_pkt(struct DEVICE *, struct gucnet_board_info *);
static void gucnet_reuse_skb(struct gucnet_board_info *, struct sk_buff *);
static void gucnet_dynamic_reset(struct DEVICE *);
static void gucnet_free_rxbuffer(struct gucnet_board_info *);
static void gucnet_init_board(struct DEVICE *);
//static inline u32 cal_CRC(unsigned char *, unsigned int, u8);
static unsigned long cal_CRC(unsigned char *, unsigned int, u8);
//static void gucnet_set_phyxcer(struct gucnet_board_info *);     //R:20121127
int a1_mac_addr(struct net_device *dev, void *p);

static const struct net_device_ops umvp_netdev_ops = {
	.ndo_open               = gucnet_open,
	.ndo_stop               = gucnet_stop,
	.ndo_start_xmit         = gucnet_start_xmit,
//  .ndo_tx_timeout         = pcnet32_tx_timeout,
	.ndo_get_stats          = gucnet_get_stats,
	.ndo_set_multicast_list = gucnet_set_filter_mode,
	.ndo_do_ioctl           = gucnet_do_ioctl,
    .ndo_set_mac_address    = eth_mac_addr,
    .ndo_change_mtu         = eth_change_mtu,
    .ndo_validate_addr      = eth_validate_addr,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller	= poll_umvp,
#endif
};

static int gucnet_set_ioaddr(struct gucnet_board_info *db)
{

#ifndef IO_ADDRESS
	db->ioaddr = (long) ioremap(UMVP_ETHERNET_BASE, 256);
	if (!db->ioaddr) return -ENOMEM;
#else
	db->ioaddr = IO_ADDRESS(UMVP_ETHERNET_BASE);
#endif
	return 0;
}

static void gucnet_phy_reset(struct gucnet_board_info *db)
{
	u16 bmcr = 0; /* mii register 0 */
	u16 advertise = (ADVERTISE_PAUSE_CAP|ADVERTISE_PAUSE_ASYM); /* mii register 4 */

	/* before reset the phy, set some configuration registers to fit my applications */
	/* set Auto-Negotiation Advertisement Register (MII 4) */
	switch(db->media_mode) {
		case GUCNET_10MHF: /* force to 10mbps, half duplex */
			bmcr = 0;
			advertise |= ADVERTISE_10HALF;
			break;
		case GUCNET_100MHF: /* force to 100mbps, half duplex */
			bmcr = BMCR_SPEED100;
			advertise |= ADVERTISE_100HALF;
			break;
		case GUCNET_10MFD:  /* force to 10mbps, full duplex */
			bmcr = BMCR_FULLDPLX;
			advertise |= ADVERTISE_10FULL;
			break;
		case GUCNET_100MFD: /* force to 100mbps, full duplex */
			bmcr = (BMCR_SPEED100|BMCR_FULLDPLX);
			advertise |= ADVERTISE_100FULL;
			break;
		case GUCNET_AUTO: /* auto mode with full capability */
			bmcr = BMCR_RESET;
			advertise |= ADVERTISE_ALL;
			break;
		default: /* not supported media mode, use auto mode */
			printk(	KERN_ERR "[%s.%s]: unsupport media mode %d\n",
					GUC_NAME, __func__, db->media_mode);
			bmcr = BMCR_RESET;
			advertise |= ADVERTISE_ALL;
	}
	phy_write( db->ioaddr, db->phy_addr, MII_ADVERTISE, advertise);
	/* disable power saving function (UTP PHY Specific Control Register, MII 16) */
	phy_write(db->ioaddr, db->phy_addr, MII_PHY_CTRL, 0x0080);
	/* set mode control register */
	phy_write(db->ioaddr, db->phy_addr, MII_BMCR, bmcr);
}

static void gucnet_phylink_update(struct gucnet_board_info *db)
{
	//u16 bmcr = 0;
	  u16 bmsr = 0;
	//u8  cr6_data = 0;

	/* read mii reg 1 */
	bmsr = phy_read(db->ioaddr, db->phy_addr, MII_BMSR);
	
	if(bmsr & BMSR_LSTATUS) { /* link up */
		#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
		if(db->link_failed) {
			printk( KERN_INFO "[%s.%s]: link up\n"
				 	"mii reg0=%04x, reg1=%04x, reg4=0x%04x, reg16=%04x\n",
					GUC_NAME, __func__,
					phy_read(db->ioaddr, db->phy_addr, MII_BMCR),
					bmsr,
					phy_read(db->ioaddr, db->phy_addr, MII_ADVERTISE),
					phy_read(db->ioaddr, db->phy_addr, MII_PHY_CTRL));
		}
		#endif
		/* read the link state to match the mac deplex mode and flow control */
		db->link_failed = 0;
		
		/*  R:20121128 :
			When MAC operating mode is set to half-duplex , network was not work.
			So it's always set to full-duplex
		*/
		#if 0
		/* read current mac operatiing mode */
		db->cr6_data = reg_inl(db->ioaddr + DCR6);
		/* read mii0 phy control register to see if the phy is auto mode */
		bmcr = phy_read(db->ioaddr, db->phy_addr, MII_BMCR);
		if(bmcr & BMCR_RESET) {
			printk( KERN_WARNING "[%s.%s]: phy is in resetting. reg0=%04x\n",
					GUC_NAME, __func__, bmcr);
					
		} else if(bmcr & BMCR_ANENABLE) { /* phy is auto mode */
			if((bmsr & BMSR_100FULL) || (bmsr & BMSR_10FULL)) { /* full duplex */
				cr6_data = CR6_FDM;
			} else { /* half duplex */
				cr6_data = 0;
			}
		} else { /* phy is forced mode */
			if(bmcr & BMCR_FULLDPLX) { /* phy is force to full duplex */
				cr6_data = CR6_FDM;
			} else {
				cr6_data = 0;
			}
		}


		if((db->cr6_data & CR6_FDM) != cr6_data) { /* need to update mac duplex mode */
		   int i;
			db->cr6_data ^= CR6_FDM;
			reg_oul(db->cr6_data, db->ioaddr + DCR6);

			#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
			printk( KERN_INFO "-->[%s.%s]: update mac cr6 to %x, mii reg0=%04x, reg1=%04x\n",
					GUC_NAME, __func__, db->cr6_data, bmcr, bmsr);
			#endif
		}
		#endif
		
	} else { /* link down */
		#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
		if(db->link_failed == 0) {
			printk( KERN_INFO "[%s.%s]: link down. reg0=%04x, reg1=%04x\n",
					GUC_NAME, __func__,
					phy_read(db->ioaddr, db->phy_addr, MII_BMCR),
					phy_read(db->ioaddr, db->phy_addr, MII_BMSR));
		}
		#endif
		db->link_failed = 1;
	}
}

static int
gucnet_probe(void)
{
	struct gucnet_board_info *db;	/* board information structure */
	struct net_device *dev;
	int err;

	#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
	printk(KERN_INFO "[%s.%s]: entered\n", GUC_NAME, __func__);
	#endif
	/* Init network device */
	dev = alloc_etherdev(sizeof (struct gucnet_board_info));
	if (dev == NULL) return -ENOMEM;

	/* Init system & device */
	db = netdev_priv(dev);
	
	if (!printed_version++)	printk(version);

	/* Allocate Tx/Rx descriptor memory */
	db->desc_pool_ptr = dma_alloc_coherent(NULL, sizeof(struct tx_desc) * DESC_ALL_CNT, &db->desc_pool_dma_ptr, GFP_KERNEL);
	if (db->desc_pool_ptr == NULL) {
		err = -ENOMEM;
		goto err_out_res;
	}
		
	db->buf_pool_ptr  = dma_alloc_coherent(NULL, TX_BUF_ALLOC * TX_DESC_CNT, &db->buf_pool_dma_ptr, GFP_KERNEL);
	if (db->buf_pool_ptr == NULL) {
		err = -ENOMEM;
		goto err_out_res;
	}

	db->first_tx_desc      = (struct tx_desc *) db->desc_pool_ptr;
	db->first_tx_desc_dma  = db->desc_pool_dma_ptr;
	db->buf_pool_start     = db->buf_pool_ptr;
	db->buf_pool_dma_start = db->buf_pool_dma_ptr;

	if(gucnet_set_ioaddr(db)) return -ENOMEM;

	/*db->pdev = pdev; */
	dev->base_addr = db->ioaddr;
	dev->irq = INT_ETHERNET;	// pdev->irq;
	spin_lock_init(&db->lock);
	
	/* Get A1_MAC_ADDR from U-BOOT via ATAG list */
	memcpy(dev->dev_addr, A1_macaddr, 6);
	dev->netdev_ops = &umvp_netdev_ops;
	
	err = register_netdev(dev);
	if (err)
		goto err_out_res;
	
	gucnet_dev = dev;
	return 0;

err_out_res:
	//kfree(dev);
	if (db->desc_pool_ptr)
		dma_free_coherent(NULL, sizeof (struct tx_desc) * DESC_ALL_CNT, db->desc_pool_ptr, db->desc_pool_dma_ptr);
	if (db->buf_pool_ptr)
		dma_free_coherent(NULL, TX_BUF_ALLOC * TX_DESC_CNT, db->buf_pool_ptr, db->buf_pool_dma_ptr);
	if (dev)
		free_netdev(dev);
	return err;
}

static int
gucnet_remove(void)
{
	struct gucnet_board_info *db;
	struct net_device *dev = gucnet_dev;

	if (dev) {
		db = netdev_priv(dev);
		unregister_netdev(dev);
#ifndef IO_ADDRESS
		iounmap((int *) db->ioaddr);
#endif
		dma_free_coherent(NULL, sizeof (struct tx_desc) * DESC_ALL_CNT,
				 		  db->desc_pool_ptr, db->desc_pool_dma_ptr);
		dma_free_coherent(NULL, TX_BUF_ALLOC * TX_DESC_CNT,
				 		  db->buf_pool_ptr, db->buf_pool_dma_ptr);

		free_netdev(dev);
		gucnet_dev = NULL;
		#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
		printk(KERN_INFO "[%s.%s]: leave\n", GUC_NAME, __func__);
		#endif
	}

	return 0;
}

/*
 *	Open the interface.
 *	The interface is opened whenever "ifconfig" actives it.
 */

static int
gucnet_open(struct DEVICE *dev)
{

	int ret;
	struct gucnet_board_info *db = netdev_priv(dev);

	#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
	gucnet_open_cnt ++;
	printk(KERN_INFO "[%s.%s]: entered, count=%d\n", GUC_NAME, __func__, gucnet_open_cnt);
	#endif

	ret = request_irq(dev->irq, &gucnet_interrupt, IRQF_SHARED, dev->name, dev);
	if (ret) {
		GUC_DBUG(1, "gucnet_open: request_irq fail", 0);
		return ret;
	}

	/* system variable init */
	db->cr6_data      = CR6_DEFAULT;
	db->tx_packet_cnt = 0;
	db->tx_queue_cnt  = 0;
	db->rx_avail_cnt  = 0;
	db->link_failed   = 1;
	db->wait_reset    = 0;
	db->PHY_reg4      = (ADVERTISE_ALL|ADVERTISE_PAUSE_CAP|ADVERTISE_PAUSE_ASYM);

	/* CR6 operation mode decision */
	db->gucnet_chk_mode = 4;	/* Enter the normal mode */

	/* Initilize GUCNET board */
	db->phy_addr   = ETH_PHY_ADDR;  /* hardcoded and decided by hardware desing in PHY chip */
	db->media_mode = GUCNET_AUTO;   /* default link mode is auto negotiation mode */
	db->op_mode    = GUCNET_AUTO;
	gucnet_init_board(dev);
	
	/* Active System Interface */
	netif_wake_queue(dev);

	/* set and active a timer process */
	init_timer(&db->timer);
	db->timer.expires = GUCNET_TIMER_WUT + HZ * 4;
	
	db->timer.data = (unsigned long) dev;
	db->timer.function = &gucnet_timer;
	add_timer(&db->timer);
	return 0;
}

 /*     Initilize TX/Rx descriptor chain structure
  *     Send the set-up frame
  *     Enable Tx/Rx machine
  */

static void
gucnet_init_board(struct DEVICE *dev)
{
	struct gucnet_board_info *db = netdev_priv(dev);
	unsigned long ioaddr = db->ioaddr;
	
    /* Reset A1 MAC controller */	
	reg_oul(GUCNET_RESET, ioaddr + DCR0);	/* RESET MAC */
	udelay(100);
	reg_oul(CR0_DEFAULT,  ioaddr + DCR0);
	udelay(5);
	
	/* reset phy */
	#if (ETH_MSG_LEVEL >= ETH_MSG_INFO)
	printk(KERN_INFO "[%s.%s]: reset phy\n", GUC_NAME, __func__);
	#endif
	gucnet_phy_reset(db);

	/* update MAC CR6 full duplex mode and flow control enabled */
	db->cr6_data = reg_inl(ioaddr + DCR6);
	
	#if 0
		if((db->media_mode == GUCNET_10MHF) || (db->media_mode == GUCNET_100MHF)) {
			/* half duplex with flow control */
			db->cr6_data |= CR6_FWC;
		} else { /* full duplex with flow control */
			db->cr6_data |= (CR6_FDM | CR6_FWC);
		}
	#else
		db->cr6_data |= (CR6_FDM | CR6_FWC);
	#endif	

	reg_oul(db->cr6_data, ioaddr + DCR6);

	
	/* Initiliaze Transmit/Receive decriptor and CR3/4 */
	gucnet_descriptor_init(db, ioaddr);

	/* R:20120911: Send setup frame */
	send_filter_frame(dev, dev->mc_count);
	
	/* Init CR7, interrupt active bit */
	db->cr7_data = CR7_DEFAULT;
	reg_oul(db->cr7_data, ioaddr + DCR7);
	
	/* Enable GUCNET Tx/Rx function */
	db->cr6_data  = reg_inl(ioaddr + DCR6);
	db->cr6_data |= CR6_RXSC | CR6_TXSC | CR6_SFT;
	reg_oul(db->cr6_data, ioaddr + DCR6);
	
	/* time to check the phy link status to sync the mac */
	gucnet_phylink_update(db);
}

/*
 *	Hardware start transmission.
 *	Send a packet to media from the upper layer.
 */

static int
gucnet_start_xmit(struct sk_buff *skb, struct DEVICE *dev)
{

	struct gucnet_board_info *db = netdev_priv(dev);
	struct tx_desc *txptr;
	unsigned long flags;

	netif_stop_queue(dev);

	/* Too large packet check */
	if (skb->len > MAX_PACKET_SIZE) {
		GUC_DBUG(1, "big packet len ", skb->len);
		dev_kfree_skb(skb);
		return NETDEV_TX_OK;
	}
	
	spin_lock_irqsave(&db->lock, flags);

	/* No Tx resource check, it never happen nromally */
	if (db->tx_queue_cnt >= TX_FREE_DESC_CNT) {
		spin_unlock_irqrestore(&db->lock, flags);
		GUC_DBUG(0, "No Tx resource ", db->tx_queue_cnt);
		return NETDEV_TX_BUSY;
	}

	/* Disable NIC interrupt */
	reg_oul(0, dev->base_addr + DCR7);
	
	/* transmit this packet */
	txptr = db->tx_insert_ptr;
	memcpy(txptr->tx_buf_ptr, skb->data, skb->len);
	txptr->tdes1 = cpu_to_le32(0xe1000000 | skb->len);
	
	/* Point to next transmit free descriptor */
	db->tx_insert_ptr = txptr->next_tx_desc;
	
	/* Transmit Packet Process */
	if ((!db->tx_queue_cnt) && (db->tx_packet_cnt < TX_MAX_SEND_CNT)) {
		txptr->tdes0 = cpu_to_le32(0x80000000);	/* Set owner bit */
		db->tx_packet_cnt++;					/* Ready to send */
		reg_oul(0x1, dev->base_addr + DCR1);	/* Issue Tx polling */
		dev->trans_start = jiffies;				/* saved time stamp */
	} else {
		db->tx_queue_cnt++;						/* queue TX packet */
		reg_oul(0x1, dev->base_addr + DCR1);	/* Issue Tx polling */
	}
	
	/* Tx resource check */
	if (db->tx_queue_cnt < TX_FREE_DESC_CNT) {
		netif_wake_queue(dev);
	}
	
	/* Restore CR7 to enable interrupt */
	spin_unlock_irqrestore(&db->lock, flags);
	reg_oul(db->cr7_data, dev->base_addr + DCR7);
	
	/* free this SKB */
	dev_kfree_skb(skb);
	
	return NETDEV_TX_OK;
}

/*
 *	Stop the interface.
 *	The interface is stopped when it is brought.
 */

static int
gucnet_stop(struct DEVICE *dev)
{
	struct gucnet_board_info *db = netdev_priv(dev);

	/* disable system */
	netif_stop_queue(dev);

	/* deleted timer */
	del_timer_sync(&db->timer);

	/* Reset & stop GUCNET board */
	#if 0
	/* MY: TODO! no need to reset the phy when close the nic */
	reg_oul(GUCNET_RESET, dev->base_addr + DCR0);
	udelay(5);
	phy_write(db->ioaddr, db->phy_addr, 0, BMCR_RESET);
	#endif

	/* free interrupt */
	free_irq(dev->irq, dev);

	/* free allocated rx buffer */
	gucnet_free_rxbuffer(db);
	return 0;
}

/*
 *	Insterrupt handler
 *	receive the packet to upper layer, free the transmitted packet
 */
static irqreturn_t
gucnet_interrupt(int irq, void *dev_id)
{
	struct DEVICE *dev = dev_id;
	struct gucnet_board_info *db = (struct gucnet_board_info *) netdev_priv(dev);
	unsigned long ioaddr = dev->base_addr;
	unsigned long flags;

	if (!dev) return IRQ_NONE;

	spin_lock_irqsave(&db->lock, flags);
	
	/* Got Mercury status */
	db->cr5_data = reg_inl(ioaddr + DCR5);
	reg_oul(db->cr5_data, ioaddr + DCR5);
	if (!db->cr5_data) {
		spin_unlock_irqrestore(&db->lock, flags);
		return IRQ_NONE;
	}

	/* Disable all interrupt in CR7 to solve the interrupt edge problem */
	reg_oul(0, ioaddr + DCR7);

	/* Check system status */
	if (db->cr5_data & 0x2) {
		/* system bus error happen */
		GUC_DBUG(1, "System bus error happen. CR5=", db->cr5_data);
		db->reset_fatal++;
		db->wait_reset = 1;	/* Need to RESET */
		/* R:20120913: Restore CR7 to enable interrupt mask */
		reg_oul(db->cr7_data, ioaddr + DCR7);
		spin_unlock_irqrestore(&db->lock, flags);
		return IRQ_HANDLED;
	}

	/* Received the coming packet */
	if ((db->cr5_data & 0x800) && db->rx_avail_cnt)
		gucnet_rx_packet(dev, db);
		
	if (db->cr5_data & 0x4) {
		reg_oul(0x1, ioaddr + DCR2);	// rx kick off again;
	}
	
	/* reallocate rx descriptor buffer */
	if (db->rx_avail_cnt < RX_DESC_CNT)
		allocate_rx_buffer(db);

	/* Free the transmitted descriptor */
	if (db->cr5_data & 0x1000)
		gucnet_free_tx_pkt(dev, db);

	/* Restore CR7 to enable interrupt mask */
	reg_oul(db->cr7_data, ioaddr + DCR7);
	spin_unlock_irqrestore(&db->lock, flags);

	return IRQ_HANDLED;
}

#ifdef CONFIG_NET_POLL_CONTROLLER
/*
 * Polling 'interrupt' - used by things like netconsole to send skbs
 * without having to re-enable interrupts. It's not called while
 * the interrupt routine is executing.
 */

static void poll_umvp (struct net_device *dev)
{
	/* disable_irq here is not very nice, but with the lockless
	   interrupt handler we have no other choice. */
	disable_irq(dev->irq);
	gucnet_interrupt (dev->irq, dev);
	enable_irq(dev->irq);
}
#endif

/*
 *	Free TX resource after TX complete
 */

static void
gucnet_free_tx_pkt(struct DEVICE *dev, struct gucnet_board_info *db)
{
	struct tx_desc *txptr;
	unsigned long ioaddr = dev->base_addr;
	u32 tdes0;

	txptr = db->tx_remove_ptr;
	while (db->tx_packet_cnt) {
		tdes0 = le32_to_cpu(txptr->tdes0);
		if (tdes0 & 0x80000000)
			break;

		/* A packet sent completed */
		db->tx_packet_cnt--;
		db->stats.tx_packets++;

		/* Transmit statistic counter */
		if (tdes0 != 0x7fffffff) {
			db->stats.collisions += (tdes0 >> 3) & 0xf;
			db->stats.tx_bytes += le32_to_cpu(txptr->tdes1) & 0x7ff;
			if (tdes0 & TDES0_ERR_MASK) {
				db->stats.tx_errors++;
				if (tdes0 & 0x0002) {	/* UnderRun */
					db->tx_fifo_underrun++;
					if (!(db->cr6_data & CR6_SFT)) {
						db->cr6_data = db->cr6_data | CR6_SFT;
						update_cr6(db->cr6_data, db->ioaddr);
					}
				}
				if (tdes0 & 0x0100)
					db->tx_excessive_collision++;
				if (tdes0 & 0x0200)
					db->tx_late_collision++;
				if (tdes0 & 0x0400)
					db->tx_no_carrier++;
				if (tdes0 & 0x0800)
					db->tx_loss_carrier++;
				if (tdes0 & 0x4000)
					db->tx_jabber_timeout++;
			}
		}

		txptr = txptr->next_tx_desc;
	}/* End of while */

	/* Update TX remove pointer to next */
	db->tx_remove_ptr = txptr;

	/* Send the Tx packet in queue */
	if ((db->tx_packet_cnt < TX_MAX_SEND_CNT) && db->tx_queue_cnt) {
		txptr->tdes0 = cpu_to_le32(0x80000000);	/* Set owner bit */
		db->tx_packet_cnt++;			/* Ready to send */
		db->tx_queue_cnt--;
		reg_oul(0x1, ioaddr + DCR1);	/* Issue Tx polling */
		dev->trans_start = jiffies;		/* saved time stamp */
	}

	/* Resource available check */
	if (db->tx_queue_cnt < TX_WAKE_DESC_CNT)
		netif_wake_queue(dev);	/* Active upper layer, send again */
}

/*
 *	Receive the come packet and pass to upper layer
 */
#define   	PHY_ERR		0x0008
static void
gucnet_rx_packet(struct DEVICE *dev, struct gucnet_board_info *db)
{

	struct rx_desc *rxptr;
	struct sk_buff *skb;
	int rxlen;
	u32 rdes0;

	rxptr = db->rx_ready_ptr;

	while (db->rx_avail_cnt) {
		rdes0 = le32_to_cpu(rxptr->rdes0);
		if (rdes0 & 0x80000000)	/* packet owner check */
			break;

		db->rx_avail_cnt--;
		db->interval_rx_cnt++;

		pci_unmap_single(NULL, le32_to_cpu(rxptr->rdes2), RX_ALLOC_SIZE, PCI_DMA_FROMDEVICE);
		if (0) {
			/* A packet without First/Last flag */
			/* reuse this SKB */
			gucnet_reuse_skb(db, rxptr->rx_skb_ptr);
		} else {
			/* A packet with First/Last flag */
			rxlen = ((rdes0 >> 16) & 0x3fff) - 4;

			/* error summary bit check */
			if (rdes0 & 0x8000) {
				/* This is a error packet */
				GUC_DBUG(0, "error packet rdes0 = ", rdes0);
				db->stats.rx_errors++;
				if (rdes0 & 1)
					db->stats.rx_fifo_errors++;
				if (rdes0 & 2)
					db->stats.rx_crc_errors++;
				if (rdes0 & 0x80)
					db->stats.rx_length_errors++;
			}

			if (!(rdes0 & 0x8000) ||
			    ((db->cr6_data & CR6_PM) && (rxlen > 6))) {
				skb = rxptr->rx_skb_ptr;

				/* Received Packet CRC check need or not */
				if ((db->gucnet_chk_mode & 1) &&
					(cal_CRC(skb->tail, rxlen, 1) != (*(u32 *) (skb->tail + rxlen)))) {	/* FIXME (?) */
						/* Found a error received packet */
						gucnet_reuse_skb(db, rxptr->rx_skb_ptr);
						db->gucnet_chk_mode = 3;
				} else {
						/* Good packet, send to upper layer */
						/* Shorst packet used new SKB */
						if (rxlen < RX_COPY_SIZE) {
							skb = dev_alloc_skb(rxlen + 2);
							if (skb != NULL) {
								/* size less than COPY_SIZE, allocate a rxlen SKB */
								skb->dev = dev;
								skb_reserve(skb, 2);	/* 16byte align */
								memcpy(skb_put(skb, rxlen),
						       		   skb_tail_pointer(rxptr->rx_skb_ptr),
									   rxlen);
								gucnet_reuse_skb(db, rxptr->rx_skb_ptr);
							}
						} else {
							if (skb->tail != skb->data) {
								skb->tail = skb->head + 0x10;
								skb->data = skb->tail;
								skb->len  = 0;
							}
							skb->dev = dev;
							skb_put(skb, rxlen);
						}
	
						skb->protocol =  eth_type_trans(skb, dev);
						netif_rx(skb);
						dev->last_rx = jiffies;
						db->stats.rx_packets++;
						db->stats.rx_bytes += rxlen;
				}
			} else {
			    /* rdes0 bit15-> ES:Error Summary */
			    if ((rdes0 & 0x8000)&&(rdes0 & PHY_ERR)) {
			    	 db->wait_reset = 1;
			    	 GUC_DBUG(0, "physical Error, rdes0", rdes0);
				}	
				/* Reuse SKB buffer when the packet is error */
				 gucnet_reuse_skb(db, rxptr->rx_skb_ptr);
			}
		}

		rxptr = rxptr->next_rx_desc;
	} /* end of while */

	db->rx_ready_ptr = rxptr;
}

/*
 *	Get statistics from driver.
 */

static struct net_device_stats *
gucnet_get_stats(struct DEVICE *dev)
{
	struct gucnet_board_info *db =
	    (struct gucnet_board_info *) netdev_priv(dev);
	return &db->stats;
}

/*
 * Set GUCNET multicast address
 */

static void
gucnet_set_filter_mode(struct DEVICE *dev)
{
	struct gucnet_board_info *db = netdev_priv(dev);
	unsigned long flags;

	spin_lock_irqsave(&db->lock, flags);

	if (dev->flags & IFF_PROMISC) {
		db->cr6_data |= CR6_PM | CR6_PBF;
		update_cr6(db->cr6_data, db->ioaddr);
		spin_unlock_irqrestore(&db->lock, flags);
		return;
	}

	/* I use one record for EAPOL multicast MAC address. therefore, the max. size of
	 * multicast address filter table is GUCNET_MAX_MULTICAST-1 */
	if (dev->flags & IFF_ALLMULTI || dev->mc_count > (GUCNET_MAX_MULTICAST-1)) {
		#if 0
		db->cr6_data &= ~(CR6_PM | CR6_PBF);
		db->cr6_data |= CR6_PAM;
		#endif		
		spin_unlock_irqrestore(&db->lock, flags);
		return;
	}
	
	send_filter_frame(dev, dev->mc_count);
	spin_unlock_irqrestore(&db->lock, flags);
}

/*
 *	Process the ethtool ioctl command
 */

static int
gucnet_ethtool_ioctl(struct net_device *dev, void *useraddr)
{
	struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO };
	u32 ethcmd;

	if (copy_from_user(&ethcmd, useraddr, sizeof (ethcmd)))
		return -EFAULT;

	switch (ethcmd) {
	case ETHTOOL_GDRVINFO:
		strcpy(info.driver,  GUC_NAME);
		strcpy(info.version, MY_VERSION);
		sprintf(info.bus_info, "AHB 0x%lx %d", dev->base_addr, dev->irq);		
		if (copy_to_user(useraddr, &info, sizeof (info)))
			return -EFAULT;
		return 0;
	}

	return -EOPNOTSUPP;
}

/*
 *	Process the upper socket ioctl command
 */

static int
gucnet_do_ioctl(struct DEVICE *dev, struct ifreq *ifr, int cmd)
{
	struct mii_ioctl_data  *data = if_mii(ifr);
	struct gucnet_board_info *db = netdev_priv(dev);
	int err = 0;
	u16 mii_regval;

	if (!netif_running(dev))
		return -ENODEV;

	switch (cmd) {
	case SIOCGMIIPHY:
		data->phy_id = db->phy_addr;

		/* fallthru */
	case SIOCGMIIREG:
		spin_lock_irq(&db->lock);
		mii_regval =  phy_read(db->ioaddr, db->phy_addr, data->reg_num & 0x1f);
		spin_unlock_irq(&db->lock);
		data->val_out = mii_regval;
		return err;

	case SIOCSMIIREG:
		spin_lock_irq(&db->lock);
		phy_write(db->ioaddr, db->phy_addr, data->reg_num & 0x1f, data->val_in);
		
		/* monitor the MII register set command. If it writes mii register 0 */
		if((data->reg_num & 0x1f) == 0)
		{
			if(data->val_in & (BMCR_RESET|BMCR_ANENABLE)) { /* set the PHY to AUTO mode */
				db->media_mode = GUCNET_AUTO;
			} else {
				if(data->val_in & BMCR_SPEED100) { /* force mode at speed 100Mbps */
					if(data->val_in & BMCR_FULLDPLX) { /* force mode at full duplex mode */
						db->media_mode = GUCNET_100MFD;
					} else { /* force mode at half duplex mode */
						db->media_mode = GUCNET_100MHF;
					}
				} else { /* force mode at speed 10Mbps */
					if(data->val_in & BMCR_FULLDPLX) { /* force mode at full duplex mode */
						db->media_mode = GUCNET_10MFD;
					} else { /* force mode at half duplex mode */
						db->media_mode = GUCNET_10MHF;
					}
				}
			}
		}
		//printk(GUC_NAME "[%s]: media_mode = %d\n", __func__, db->media_mode);
		
		spin_unlock_irq(&db->lock);
		return err;

	case SIOCETHTOOL:
		return gucnet_ethtool_ioctl(dev, (void *) ifr->ifr_data);
	default:
		/* do nothing */
		break;
	}
	return -EOPNOTSUPP;
}


/*
 *	A periodic timer routine
 *	Dynamic media sense, allocate Rx buffer...
 */

static void
gucnet_timer(unsigned long data)
{
	struct DEVICE *dev = (struct DEVICE *) data;
	struct gucnet_board_info *db =  (struct gucnet_board_info *) netdev_priv(dev);
	unsigned long flags;
	u8 link_failed = 1;

	spin_lock_irqsave(&db->lock, flags);

	/* Operating Mode Check */
	if ((db->gucnet_chk_mode & 0x1) && (db->stats.rx_packets > MAX_CHECK_PACKET)){
		db->gucnet_chk_mode = 0x4;
	}	
	db->interval_rx_cnt = 0;
	
	#if 0
	/* R:20121002: TX polling kick monitor */
	if ( db->tx_packet_cnt && time_after(jiffies, dev->trans_start + GUCNET_TX_KICK) )
	{
		reg_oul(0x1, dev->base_addr + DCR1);	/* Issue Tx polling */

		/* TX Timeout */
		if ( time_after(jiffies, dev->trans_start + GUCNET_TX_TIMEOUT) ) {
			db->reset_TXtimeout++;
			db->wait_reset = 1;
			GUC_DBUG(0, "timeout - resetting", db->reset_TXtimeout);
		}
	}
	#endif
	//
	
	if (db->wait_reset) {
		gucnet_dynamic_reset(dev);
		db->timer.expires = GUCNET_TIMER_WUT;
		add_timer(&db->timer);
		spin_unlock_irqrestore(&db->lock, flags);
		return;
	}

	/* backup the current link state */
	link_failed = db->link_failed;
	gucnet_phylink_update(db);

	if(link_failed !=  db->link_failed) { /* link status was changed. */
		 if(db->link_failed) { /* phy link was from link up to link down */
		 	netif_stop_queue(dev);
		 } else { /* phy link was from link down to link up */
		 	netif_wake_queue(dev);
		 }
	}
	
	#if 0
	unsigned char tmp_phy1;
	tmp_phy1 = phy_read(db->ioaddr, db->phy_addr, 1);   //Status Register of PHY

	if (!(tmp_phy1 & 0x4) && !db->link_failed) {
		db->link_failed = 1;
		netif_stop_queue(dev);
	} else if ((tmp_phy1 & 0x4) && db->link_failed) {
		db->link_failed = 0;
		gucnet_set_phyxcer(db);
		netif_wake_queue(dev);
	}
	#endif
	/* Timer active again */
	db->timer.expires = GUCNET_TIMER_WUT;
	add_timer(&db->timer);
	spin_unlock_irqrestore(&db->lock, flags);
}


//R:20121002-01
static void gucnet_reset_prepare(struct net_device *dev)
{
	struct gucnet_board_info *db = netdev_priv(dev);

	
	/* Sopt MAC controller */
	db->cr6_data &= ~(CR6_RXSC | CR6_TXSC);	/* Disable Tx/Rx */
	update_cr6(db->cr6_data, dev->base_addr);
	
	reg_oul(0, dev->base_addr + DCR7);		/* Disable Interrupt */
	reg_oul(reg_inl(dev->base_addr + DCR5), dev->base_addr + DCR5);
	
	/* Disable upper layer interface */
	netif_stop_queue(dev);
	
	/* Free Rx Allocate buffer */
	gucnet_free_rxbuffer(db);
	
	/* system variable init */
	db->tx_packet_cnt     = 0;
	db->tx_queue_cnt      = 0;
	db->rx_avail_cnt      = 0;
	db->link_failed       = 1;
	db->wait_reset        = 0;
	db->reset_count++;

}

/*
 *	Dynamic reset
 *	Free Tx/Rx allocated memory
 */
static void
gucnet_dynamic_reset(struct DEVICE *dev)
{
	struct gucnet_board_info *db = netdev_priv(dev);
	
	GUC_DBUG(0, __func__ , db->reset_count);
	
	gucnet_reset_prepare(dev);
	
	/* Re-initilize GUCNET board */
	gucnet_init_board(dev);
	
	/* Restart upper layer interface */
	netif_wake_queue(dev);
}

/*
 *	free all allocated rx buffer
 */

static void
gucnet_free_rxbuffer(struct gucnet_board_info *db)
{
	/* free allocated rx buffer */
	while (db->rx_avail_cnt) {
		dev_kfree_skb(db->rx_ready_ptr->rx_skb_ptr);
		db->rx_ready_ptr = db->rx_ready_ptr->next_rx_desc;
		db->rx_avail_cnt--;
	}
}

/*
 *	Reuse the SK buffer
 */

static void
gucnet_reuse_skb(struct gucnet_board_info *db, struct sk_buff *skb)
{
	struct rx_desc *rxptr = db->rx_insert_ptr;

	if (!(rxptr->rdes0 & cpu_to_le32(0x80000000))) {
		rxptr->rx_skb_ptr = skb;
		rxptr->rdes2 = cpu_to_le32(pci_map_single(NULL, skb->tail, RX_ALLOC_SIZE, PCI_DMA_FROMDEVICE));
		wmb();
		rxptr->rdes0 = cpu_to_le32(0x80000000);
		db->rx_avail_cnt++;
		db->rx_insert_ptr = rxptr->next_rx_desc;
	} else
		printk(KERN_ERR GUC_NAME ":SK Buffer reuse method error, %lx\n", (long)(db->rx_avail_cnt));
}

/*
 *	Initialize transmit/Receive descriptor
 *	Using Chain structure, and allocate Tx/Rx buffer
 */

static void
gucnet_descriptor_init(struct gucnet_board_info *db, unsigned long ioaddr)
{
	struct tx_desc *tmp_tx;
	struct rx_desc *tmp_rx;
	unsigned char  *tmp_buf;
	dma_addr_t tmp_tx_dma, tmp_rx_dma;
	dma_addr_t tmp_buf_dma;
	int i;

	/* tx descriptor start pointer */
	db->tx_insert_ptr = db->first_tx_desc;
	db->tx_remove_ptr = db->first_tx_desc;
	reg_oul(db->first_tx_desc_dma, ioaddr + DCR3);	/* TX DESC address */

	/* rx descriptor start pointer */
	db->first_rx_desc     = (void *) db->first_tx_desc + sizeof (struct tx_desc) * TX_DESC_CNT;
	db->first_rx_desc_dma =  db->first_tx_desc_dma + sizeof (struct tx_desc) * TX_DESC_CNT;
	db->rx_insert_ptr     = db->first_rx_desc;
	db->rx_ready_ptr      = db->first_rx_desc;
	reg_oul(db->first_rx_desc_dma, ioaddr + DCR4);	/* RX DESC address */

	/* Init Transmit chain */
	tmp_buf     = db->buf_pool_start;
	tmp_buf_dma = db->buf_pool_dma_start;
	tmp_tx_dma  = db->first_tx_desc_dma;
	
	for (tmp_tx = db->first_tx_desc, i = 0; i < TX_DESC_CNT; i++, tmp_tx++) {
		tmp_tx->tx_buf_ptr = tmp_buf;
		tmp_tx->tdes0 = cpu_to_le32(0);
		tmp_tx->tdes1 = cpu_to_le32(0x81000000);	/* IC, chain */
		tmp_tx->tdes2 = cpu_to_le32(tmp_buf_dma);
		tmp_tx_dma += sizeof (struct tx_desc);
		tmp_tx->tdes3 = cpu_to_le32(tmp_tx_dma);
		tmp_tx->next_tx_desc = tmp_tx + 1;
		tmp_buf = tmp_buf + TX_BUF_ALLOC;
		tmp_buf_dma = tmp_buf_dma + TX_BUF_ALLOC;
	}
	(--tmp_tx)->tdes3 = cpu_to_le32(db->first_tx_desc_dma);
	tmp_tx->next_tx_desc = db->first_tx_desc;

	/* Init Receive descriptor chain */
	tmp_rx_dma = db->first_rx_desc_dma;
	for (tmp_rx = db->first_rx_desc, i = 0; i < RX_DESC_CNT; i++, tmp_rx++) {
		tmp_rx->rdes0 = cpu_to_le32(0);
		tmp_rx->rdes1 = cpu_to_le32(0x01000600);
		tmp_rx_dma += sizeof (struct rx_desc);
		tmp_rx->rdes3 = cpu_to_le32(tmp_rx_dma);
		tmp_rx->next_rx_desc = tmp_rx + 1;
	}
	(--tmp_rx)->rdes3 = cpu_to_le32(db->first_rx_desc_dma);
	tmp_rx->next_rx_desc = db->first_rx_desc;

	/* pre-allocate Rx buffer */
	allocate_rx_buffer(db);
}

/*
 *	Update CR6 value
 *	Firstly stop GUCNET , then written value and start
 */

static void
update_cr6(u32 cr6_data, unsigned long ioaddr)
{

#if 0
	u32 cr6_tmp;
	
	cr6_tmp = cr6_data & ~0x2002;	/* stop Tx/Rx */
	reg_oul(cr6_tmp, ioaddr + DCR6);
	udelay(5);
	reg_oul(cr6_data, ioaddr + DCR6);
	udelay(5);
#endif

	reg_oul(cr6_data, ioaddr + DCR6);
	udelay(5);
	
}

/*
 *	Send a setup frame
 *	This setup frame initilize addres filter mode
 */

static void
send_filter_frame(struct DEVICE *dev, int mc_cnt)
{
	struct gucnet_board_info *db = netdev_priv(dev);
	struct dev_mc_list *mcptr;
	struct tx_desc *txptr;
	u16 *addrptr;
	u32 *suptr;
	int i;

	txptr = db->tx_insert_ptr;
	suptr = (u32 *) txptr->tx_buf_ptr;

	/* Node address */
	addrptr = (u16 *) dev->dev_addr;
	*suptr++ = addrptr[0];
	*suptr++ = addrptr[1];
	*suptr++ = addrptr[2];

	/* broadcast address */
	*suptr++ = 0xffff;
	*suptr++ = 0xffff;
	*suptr++ = 0xffff;

	/* add EAPOL (802.1x multicast address 01:80:c2:00:00:03 */
	*suptr++ = 0x8001;
	*suptr++ = 0x00c2;
	*suptr++ = 0x0300;

	/* fit the multicast address */
	for (mcptr = dev->mc_list, i = 0; i < mc_cnt; i++, mcptr = mcptr->next) {
		addrptr = (u16 *) mcptr->dmi_addr;
		*suptr++ = addrptr[0];
		*suptr++ = addrptr[1];
		*suptr++ = addrptr[2];
	}

//	for (; i < 14; i++) {
	/* I use one record for EAPOL multicast MAC address */
	for (; i < (GUCNET_MAX_MULTICAST-1); i++) {
		*suptr++ = 0xffff;
		*suptr++ = 0xffff;
		*suptr++ = 0xffff;
	}

	/* prepare the setup frame */
	db->tx_insert_ptr = txptr->next_tx_desc;
	txptr->tdes1 = cpu_to_le32(0x800000c0 | SETUP_FRAME);

	/* Resource Check and Send the setup packet */
	if (!db->tx_packet_cnt) {
		/* Resource Empty */
		db->tx_packet_cnt++;
		txptr->tdes0 = cpu_to_le32(0x80000000);
		update_cr6(db->cr6_data, dev->base_addr);
		reg_oul(0x1, dev->base_addr + DCR1);	/* Issue Tx polling */
		update_cr6(db->cr6_data, dev->base_addr);
		dev->trans_start = jiffies;
	} else
		db->tx_queue_cnt++;	/* Put in TX queue */
}

/*
 *	Allocate rx buffer,
 *	As possible as allocate maxiumn Rx buffer
 */

static void
allocate_rx_buffer(struct gucnet_board_info *db)
{
	struct rx_desc *rxptr;
	struct sk_buff *skb;

	rxptr = db->rx_insert_ptr;

	while (db->rx_avail_cnt < RX_DESC_CNT) {
		if ((skb = dev_alloc_skb(RX_ALLOC_SIZE)) == NULL)
			break;
		if (skb->len > 0) {
			dev_kfree_skb(skb);
			continue;
		}
		rxptr->rx_skb_ptr = skb;	/* FIXME (?) */
		rxptr->rdes2 = cpu_to_le32(pci_map_single(NULL, skb->tail, RX_ALLOC_SIZE, PCI_DMA_FROMDEVICE));
		wmb();
		rxptr->rdes0 = cpu_to_le32(0x80000000);
		rxptr = rxptr->next_rx_desc;
		db->rx_avail_cnt++;
	}

	db->rx_insert_ptr = rxptr;
}


#if 0   //R:20121127
/*
 *	Set 10/100 phyxcer capability
 *	AUTO mode : phyxcer register4 is NIC capability
 *	Force mode: phyxcer register4 is the force media
 */

#define PHY_ASY_FWC    	0x0800 /* PHY asymmetric flow control */
#define PHY_FWC       	0x0400 /* PHY Flow control */
#define LDPS_ENABLE    	0x0002 /* set hight to Enable Auto Power savind */
#define NWAY_PSAVE_DIS  0x0080 /* set hight to Disable power saving during auto-negotiation */
static void
gucnet_set_phyxcer(struct gucnet_board_info *db)
{
	u16 phy_reg , phy_reg16;
	u16 mii_regval = BMCR_ANENABLE | BMCR_ANRESTART;   //0x1200

	/* Phyxcer capability setting */
	phy_reg = phy_read(db->ioaddr, db->phy_addr, 4) & ~0x01e0;
	if (db->media_mode & GUCNET_AUTO) {
		/* AUTO Mode */
		phy_reg |= db->PHY_reg4;
	} else {
		/* Force Mode */
		switch (db->media_mode) {
		case GUCNET_10MHF:
			phy_reg |= 0x20;
			break;
		case GUCNET_10MFD:
			phy_reg |= 0x40;
			break;
		case GUCNET_100MHF:
			phy_reg |= 0x80;
			break;
		case GUCNET_100MFD:
			phy_reg |= 0x100;
			break;
		}
	}

	/* Write new capability to Phyxcer Reg4 */
	if (!(phy_reg & 0x01e0)) {
		phy_reg |= db->PHY_reg4;
		db->media_mode |= GUCNET_AUTO;
	}

	
 	/* Restart Auto-Negotiation */
	if (db->link_failed) {
		//R:20121005 enable phy flow control
		phy_reg |= (PHY_ASY_FWC | PHY_FWC);
		phy_write(db->ioaddr, db->phy_addr, 4, phy_reg);
		udelay(5);
	
		/* R:20121005: specific control register for auto power saving  */	
		phy_reg16 = phy_read(db->ioaddr, db->phy_addr, 16);
		phy_reg16 = ((phy_reg16 | NWAY_PSAVE_DIS) & ~LDPS_ENABLE);
		phy_write(db->ioaddr, db->phy_addr, 16, phy_reg16);
		udelay(5);
		//				
		phy_write(db->ioaddr, db->phy_addr, 0, mii_regval);
		udelay(50);
	}

	if (phy_reg & 0x100)
		db->cr6_data |= CR6_FDM;
	else
		db->cr6_data &= ~CR6_FDM;
}
#endif

/*
 *	Write a word to Phy register
 */

static void
phy_write(unsigned long iobase, u8 phy_addr, u8 offset, u16 phy_data)
{
	u16 i;
	unsigned long ioaddr;

	ioaddr = iobase + DCR9;

	/* Send 33 synchronization clock to Phy controller */
	for (i = 0; i < 32; i++)
		phy_write_1bit(ioaddr, PHY_DATA_1);

	/* Send start command(01) to Phy */
	phy_write_1bit(ioaddr, PHY_DATA_0);
	phy_write_1bit(ioaddr, PHY_DATA_1);

	/* Send write command(01) to Phy */
	phy_write_1bit(ioaddr, PHY_DATA_0);
	phy_write_1bit(ioaddr, PHY_DATA_1);

	/* Send Phy addres */
	for (i = 0x10; i > 0; i = i >> 1)
		phy_write_1bit(ioaddr, phy_addr & i ? PHY_DATA_1 : PHY_DATA_0);

	/* Send register addres */
	for (i = 0x10; i > 0; i = i >> 1)
		phy_write_1bit(ioaddr, offset & i ? PHY_DATA_1 : PHY_DATA_0);

	/* written trasnition */
	phy_write_1bit(ioaddr, PHY_DATA_1);
	phy_write_1bit(ioaddr, PHY_DATA_0);

	/* Write a word data to PHY controller */
	for (i = 0x8000; i > 0; i >>= 1)
		phy_write_1bit(ioaddr, phy_data & i ? PHY_DATA_1 : PHY_DATA_0);

}

/*
 *	Read a word data from phy register
 */

static u16
phy_read(unsigned long iobase, u8 phy_addr, u8 offset)
{
	int i;
	u16 phy_data;
	unsigned long ioaddr;

	ioaddr = iobase + DCR9;

	/* Send 33 synchronization clock to Phy controller */
	for (i = 0; i < 32; i++)
		phy_write_1bit(ioaddr, PHY_DATA_1);

	/* Send start command(01) to Phy */
	phy_write_1bit(ioaddr, PHY_DATA_0);
	phy_write_1bit(ioaddr, PHY_DATA_1);

	/* Send read command(10) to Phy */
	phy_write_1bit(ioaddr, PHY_DATA_1);
	phy_write_1bit(ioaddr, PHY_DATA_0);

	/* Send Phy addres */
	for (i = 0x10; i > 0; i = i >> 1)
		phy_write_1bit(ioaddr, phy_addr & i ? PHY_DATA_1 : PHY_DATA_0);

	/* Send register addres */
	for (i = 0x10; i > 0; i = i >> 1)
		phy_write_1bit(ioaddr, offset & i ? PHY_DATA_1 : PHY_DATA_0);

	/* Skip transition state */
	phy_read_1bit(ioaddr);

	/* read 16bit data */
	for (phy_data = 0, i = 0; i < 16; i++) {
		phy_data <<= 1;
		phy_data |= phy_read_1bit(ioaddr);
	}

	return phy_data;
}

/*
 *	Write one bit data to Phy Controller
 */

static void
phy_write_1bit(unsigned long ioaddr, u32 phy_data)
{
	reg_oul(phy_data, ioaddr);	/* MII Clock Low */
	udelay(MDIO_DELAY_TICK);
	reg_oul(phy_data | MDCLKH, ioaddr);	/* MII Clock High */
	udelay(MDIO_DELAY_TICK);
	reg_oul(phy_data, ioaddr);	/* MII Clock Low */
	udelay(MDIO_DELAY_TICK);
}

/*
 *	Read one bit phy data from PHY controller
 */

static u16
phy_read_1bit(unsigned long ioaddr)
{
	u16 phy_data;
	reg_oul(MII_READ | MDCLKH, ioaddr);
	udelay(MDIO_DELAY_TICK);
	phy_data = (reg_inl(ioaddr) >> 7) & 0x1;
	reg_oul(MII_READ, ioaddr);
	udelay(MDIO_DELAY_TICK);

	return phy_data;
}

/*
 *	Calculate the CRC valude of the Rx packet
 *	flag = 	1 : return the reverse CRC (for the received packet CRC)
 *		0 : return the normal CRC (for Hash Table index)
 */
#if 1
unsigned long
cal_CRC(unsigned char *Data, unsigned int Len, u8 flag)
{
	unsigned long Crc = 0xffffffff;

	while (Len--) {
		Crc = CrcTable[(Crc ^ *Data++) & 0xFF] ^ (Crc >> 8);
	}

	if (flag)
		return ~Crc;
	else
		return Crc;
}
#else
static inline u32 cal_CRC(unsigned char * Data, unsigned int Len, u8 flag)
{
	u32 crc = crc32(~0, Data, Len);
	if (flag) crc = ~crc;
	return crc;
}
#endif

MODULE_AUTHOR("simon");
MODULE_DESCRIPTION("Globalunichip mercury fast ethernet driver");
MODULE_LICENSE("GPL");

/*	Description:
 *	when user used insmod to add module, system invoked init_module()
 *	to initilize and register.
 */

static int __init
gucnet_init_module(void)
{
//  vic_set_intr_trigger(INT_ETHERNET, vicc_level_activeHigh);
	return gucnet_probe();
}

/*
 *	Description:
 *	when user used rmmod to delete module, system invoked clean_module()
 *	to un-register all registered services.
 */

static void __exit
gucnet_cleanup_module(void)
{
	gucnet_remove();
}

module_init(gucnet_init_module);

