//================================================================================
// FILE NAME:       xra1201.c
//--------------------------------------------------------------------------------
// Reason-ID        Reason
//--------------------------------------------------------------------------------
// R:20130724-00:   This driver is 16-bit GPIO expander with an I2c/SMBus interface
//================================================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/moduleparam.h>

#include <linux/delay.h>  /* mdeley() msleep() */
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <asm/uaccess.h>

#include <linux/slab.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <linux/cdev.h>
#include <linux/spinlock.h>

#include <linux/interrupt.h> /* tasklets */
#include <mach/hardware.h>
#include <mach/platform2500.h>
#include <asm/arch-umvp/a1_sys.h>
#include "umvp_gpio.h"
#include "umvp_xra1201.h"

/* -----------------------------------------------------------------------
	define debug macro
 ----------------------------------------------------------------------- */
static uint debug_ctrl = 1;  	
#define MSG_PFX  		"[XRA1201]"
#define dprintk(num, format, args...) \
		do { \
			if (debug_ctrl >= num) \
				printk(MSG_PFX format, ##args); \
		} while (0)

/* ----------------------------------------------------------------------- */
#define XRA1201_DEGUG		 1
#define _OK					 0
#define _ERR				-1

typedef struct {
	struct i2c_client 	*client;
	tXra1201Regs 		reg;
	spinlock_t   		lock;
	u_int 		 		irq;
	u_int 		 		DIs; 		/* default level for all DIs are 1s */
	u_int 		 		DOs; 		/* default level for all DOs are 0s */
	int	  		 		usPID;
	int	  		 		is_ready;
}xra1201_t;
static xra1201_t *XraDev;

enum gpio3_intr_cmd {
	GPIO3_INTR_LO_LEVEL = 0,	
	GPIO3_INTR_BOTH_EDGE,
	GPIO3_INTR_RISING_EDGE,
	GPIO3_INTR_ENABLE,
	GPIO3_INTR_DISABLE,
	GPIO3_INTR_CLEAR,	
	GPIO3_BIT29_INPUT,
	GPIO3_BIT29_RDDATA		
};

static int xra1201_reg_read(u_int start, u_int end, tXra1201Regs *reg);
static int xra1201_reg_write(u_int start, u_int end, tXra1201Regs *reg);

#if XRA1201_DEGUG
static void xra1201_reg_dump(void);
#endif

/* ----------------------------------------------------------------------- */
/* config gpio as interrupt source , gpio3 bit 29 */
static u_int gpio3_interrupt_set(int cmd_type)
{
	u_int gpio3_b29 = 0x00;
	
	switch(cmd_type)
	{
		case GPIO3_INTR_LO_LEVEL:		
			/* set  level trigger interrupt */			
			gpio3_b29 = GPIO3_REG(GPIOIS);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;
			GPIO3_REG(GPIOIS)  = gpio3_b29;
			/* set falling edge or low level trigger */			
			gpio3_b29 = GPIO3_REG(GPIOIEV);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIEV) = gpio3_b29;		
			break;
		case GPIO3_INTR_BOTH_EDGE:
			/* set edge trigger interrupt */			
			gpio3_b29 = GPIO3_REG(GPIOIS);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIS)  = gpio3_b29;
			/* set both edge trigger interrupt */			
			gpio3_b29 = GPIO3_REG(GPIOIBE);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;
			GPIO3_REG(GPIOIBE) = gpio3_b29;
			break;
		case GPIO3_INTR_RISING_EDGE:
			/* set edge trigger interrupt */			
			gpio3_b29 = GPIO3_REG(GPIOIS);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIS)  = gpio3_b29;
			/* clear both edge trigger interrupt */			
			gpio3_b29 = GPIO3_REG(GPIOIBE);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIBE) = gpio3_b29;
			/* set rising edge or high level trigger */			
			gpio3_b29 = GPIO3_REG(GPIOIEV);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIEV) = gpio3_b29;		
			break;
		case GPIO3_INTR_ENABLE:
			gpio3_b29 = GPIO3_REG(GPIOIE);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;
			GPIO3_REG(GPIOIE) = gpio3_b29;
			break;
		case GPIO3_INTR_CLEAR:
			gpio3_b29 = GPIO3_REG(GPIOIC);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;
			GPIO3_REG(GPIOIC) = gpio3_b29;
			/* do clear and disable */				
			//break;
		case GPIO3_INTR_DISABLE:
			gpio3_b29 = GPIO3_REG(GPIOIE);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIOIE) = gpio3_b29;
			break;
		case GPIO3_BIT29_INPUT:			
			gpio3_b29 = GPIO3_REG(GPIODIR);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;
			GPIO3_REG(GPIODIR) = gpio3_b29;
			/* set debounce */
			gpio3_b29 = GPIO3_REG(GPIODBEN);
			//gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;  //enable
			  gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x00000000;  //disable
			GPIO3_REG(GPIODBEN) =  gpio3_b29;
			break;
		case GPIO3_BIT29_RDDATA:
			/* set mask bit */
			gpio3_b29 = GPIO3_REG(GPIOMASK);
			gpio3_b29 = (gpio3_b29 & ~0x20000000) | 0x20000000;
			GPIO3_REG(GPIOMASK) = gpio3_b29;
			/* read data */
			gpio3_b29 = GPIO3_REG(GPIODATA);					
			break;

		default:
			break;
	}

	return gpio3_b29;
}

/* ----------------------------------------------------------------------- */
int is_xra1201_found(void)
{
	if (XraDev)
		return (XraDev->is_ready);
	else
		return 0;	
}

void xra1201_set_data_output(u_int DOs, u_int ctrl_pin)
{
    u_int data = 0;
    unsigned char tmp = 0;
    int i = 0;

	if (!XraDev) return;
	
	//spin_lock(&XraDev->lock);
	
    data  = XraDev->DOs & (~ctrl_pin);
    data |= DOs & ctrl_pin;
    data &= XRA1201_DOS_MASK;
    tmp = (unsigned char)data;
    XraDev->reg.OCR2 = 0;

    /* bit rotating to OCR2 */
    for(i = 0 ; i < 8  ; i ++) {
        if(tmp & (0x01<<i)) XraDev->reg.OCR2 |= 0x80>>i;
    }

    if(xra1201_reg_write(XRA1201_OCR2, XRA1201_OCR2, &XraDev->reg) == 0) {
		XraDev->DOs = data;
		dprintk(3,KERN_INFO "write %02xh/%02xh=> %02xh to %02xh (%02xh)\n",
				DOs, ctrl_pin, XraDev->reg.OCR2, XRA1201_OCR2, XraDev->DOs);
    }else{
    	dprintk(0,KERN_ERR "%s:Error. set DOs=%02X\n",__func__, XraDev->reg.OCR2);
    }

	//spin_unlock(&XraDev->lock);
}

u_int xra1201_read_data_input(void)
{
	u_int val;
	if (XraDev) {
		val = (XraDev->DOs << 8) & 0xff00;
		val = val | (XraDev->DIs & 0xff);
		return val;
	}

	return 0;	
}

void xra1201_input_data_update(void)
{
	if (!XraDev) return;

//	spin_lock(&XraDev->lock);

	#if XRA1201_DEGUG
		if(xra1201_reg_read(XRA1201_ISR1, XRA1201_ISR2, &XraDev->reg) == 0) {
			dprintk(3,KERN_INFO "ISR1(0x%02x) ISR2(0x%02x)\n", XraDev->reg.ISR1, XraDev->reg.ISR2);
		}
	#endif
	
    if(xra1201_reg_read(XRA1201_GSR1, XRA1201_GSR1, &XraDev->reg) == 0) {
		XraDev->DIs = (u_int)(XraDev->reg.GSR1 & XRA1201_DIS_MASK);
		dprintk(3,KERN_INFO "read DI=0x%02x/0x%02x\n", XraDev->DIs, XraDev->reg.GSR1);
    }

 //	spin_unlock(&XraDev->lock);
}

static int xra1201_init(void)
{
    XraDev->DIs &=   XRA1201_DIS_MASK;
    XraDev->DOs &= (~XRA1201_DOS_MASK);

    memset((char *)&XraDev->reg, 0, sizeof(tXra1201Regs));

    /* based on h/w design, the P0~P7 are DIs and P8~P15 are DOs */
    /* no inversion of parity for all GPIO pins */
    XraDev->reg.PIR1 = 0x00; /* PIR1: not invert the level in input pin */
    XraDev->reg.PIR2 = 0x00; /* PIR2: not invert the level in input pin */
    XraDev->reg.GCR1 = 0xff; /* GCR1: p0~p7 are input pins */
    XraDev->reg.GCR2 = 0x00; /* GCR2: p8~p15 are output pins */

 	XraDev->reg.PUR1 = 0xff; /* PUR1: pull-up p0~p7 */
    XraDev->reg.PUR2 = 0x00; /* PUR2: not pull-up p8~p15 */


    XraDev->reg.IER1 = 0x0f; /* IER1: enable interrupt form p0~p3 */
    XraDev->reg.IER2 = 0x00; /* IER2: no interrupt from p8~p15 */

    XraDev->reg.TSCR1 = 0x00; /* TSCR1: disable tri-state in p0~p7 */
    XraDev->reg.TSCR2 = 0x00; /* TSCR1: disable tri-state in p8~p15 */

    XraDev->reg.REIR1 = 0x0f; /* REIR1, issue an interrupt in raising edge: P0~P3 for DI1~DI3 */
    XraDev->reg.REIR2 = 0x00; /* REIR2, not issue any interrupt from p8~p15 */
    XraDev->reg.FEIR1 = 0x0f; /* FEIR1, issue an interrupt in falling edge: P0~P3 for DI1~DI3 */
    XraDev->reg.FEIR2 = 0x00; /* FEIR2, not issue any interrupt from p8~p15 */

    XraDev->reg.IFR1  = 0x0f; /* IFR1: enable time filter in p0~p3 */
    XraDev->reg.IFR2  = 0x00; /* IFR2: disable time filter in p8~p15 */

    XraDev->reg.OCR1  = 0xff;
    XraDev->reg.OCR2  = 0x00;

    /* setup input/output configuration first */
    if (xra1201_reg_write(XRA1201_GCR1, XRA1201_GCR2, &XraDev->reg))  return -EFAULT;
    if (xra1201_reg_write(XRA1201_PIR1, XRA1201_PIR2, &XraDev->reg))  return -EFAULT;
    if (xra1201_reg_write(XRA1201_PUR1, XRA1201_PUR2, &XraDev->reg))  return -EFAULT;
    /* setup the tri-state function */
    if (xra1201_reg_write(XRA1201_TSCR1, XRA1201_TSCR2,&XraDev->reg))return -EFAULT;
    /* setup interrupt configurations */
    if (xra1201_reg_write(XRA1201_REIR1, XRA1201_IFR2, &XraDev->reg)) return -EFAULT;
    /* enable interrupts */
    if (xra1201_reg_write(XRA1201_IER1, XRA1201_IER2,  &XraDev->reg))  return -EFAULT;

    if (xra1201_reg_write(XRA1201_OCR1, XRA1201_OCR2,  &XraDev->reg))  return -EFAULT;

    return _OK;
}


/* -------------------------------------------------------------------------- */
int xra1201_update_gpio_type(void)
{
	xra1201_init();
	
	#if XRA1201_DEGUG
		xra1201_reg_dump();
    #else
        dprintk(0,KERN_INFO "Don't dump xra1201.....\n");
	#endif
	
	/* clear bit29 of gpio3 interrup status */
	gpio3_interrupt_set(GPIO3_INTR_CLEAR);
	gpio3_interrupt_set(GPIO3_BIT29_INPUT);

#if 0
	dprintk(1,KERN_INFO "set the bit29 of gpio3 as in interrupt pin with ative low\n");
    /* set the bit29 of gpio3 as in interrupt pin with ative low */		
	gpio3_interrupt_set(GPIO3_INTR_LO_LEVEL);
    /* set the bit29 of gpio3 as in interrupt pin with both edge trigger */	
	gpio3_interrupt_set(GPIO3_INTR_BOTH_EDGE);
	/* set the bit29 of gpio3 as in interrupt pin with rising edge trigger */	
	gpio3_interrupt_set(GPIO3_INTR_RISING_EDGE);
#endif

    dprintk(1,KERN_INFO "set the bit29 of gpio3 as in interrupt pin with ative low\n");
    gpio3_interrupt_set(GPIO3_INTR_LO_LEVEL);
	
	/* First, read GSR1 to clear interrupt of XRA1201  */
	xra1201_input_data_update();
	gpio3_interrupt_set(GPIO3_INTR_ENABLE);

   // dprintk(1,KERN_INFO "set output gpio to 0xff(P15 ~ P8)\n");
   // xra1201_set_data_output(0,0x0f); /* turn off led & fog led */


    return _OK;
}

/* ---Read Command-------------------------------------------------------------------- */
int xra1201_i2c_read(unsigned char *buf, u_int address, u_int index, int num)
{
	struct i2c_client *client = XraDev->client;
    struct i2c_msg msgs[2];
    unsigned char  tmp[4];
    int            i = 0;

    msgs[0].addr  = address >> 1;
    msgs[1].addr  = address >> 1;
    msgs[0].flags = 0;
    msgs[0].len   = 1;
    msgs[0].buf   = tmp;
    msgs[1].flags = I2C_M_RD;
    msgs[1].len   = 1;
    msgs[1].buf   = &tmp[1];

    for(i = 0 ; i < num ; i ++) {
        tmp[0] = (unsigned char)((index+i)&0x0ff);
        if (i2c_transfer(client->adapter, &msgs[0], 1) != msgs[0].len) return -EFAULT;
		if (i2c_transfer(client->adapter, &msgs[1], 1) == msgs[1].len) buf[i] = tmp[1];
		else return -EFAULT;
		dprintk(3,KERN_INFO "num(%d)tmp0(0x%x)tmp1(0x%x)\n", num, tmp[0], tmp[1]);
    }
    return num;
}

static int xra1201_reg_read(u_int start, u_int end, tXra1201Regs *reg)
{
    int num = end - start + 1;
    unsigned char *p = (unsigned char *)reg;

    if (xra1201_i2c_read((p+start), XRA1201_I2C_ADDR, start, num) == num) return _OK;

    dprintk(0,KERN_ERR "%s:Error. read registers from 0x%02x (%d)\n", __func__, start, num);
    return -EFAULT;
}

/* ---Write Command-------------------------------------------------------------------- */
int xra1201_i2c_write(unsigned char *buf, u_int address, u_int index, int num)
{

	struct i2c_client *client = XraDev->client;
    struct i2c_msg msgs;
    unsigned char  tmp[4];
    int            i = 0;

    msgs.addr  = address >> 1;
    msgs.flags = 0;
    msgs.len   = 2;
    msgs.buf   = tmp;
    for (i = 0 ; i < num ; i ++) {
		tmp[0] = (unsigned char)((index+i)&0x0ff);
		tmp[1] = buf[i];
    	if (i2c_transfer(client->adapter, &msgs, 1) != msgs.len) return -EFAULT;
    }

    return num;
}

static int xra1201_reg_write(u_int start, u_int end, tXra1201Regs *reg)
{
	int num = end - start + 1;
	unsigned char *p = (unsigned char *)reg;
	
	if (xra1201_i2c_write((p+start), XRA1201_I2C_ADDR, start, num) == num) return _OK;
	
	dprintk(0,KERN_ERR "%s:Error. write %d registers from cmd 0x%02x (%d)\n", __func__, num, start, num);
	return -EFAULT;
}


#if XRA1201_DEGUG
static void xra1201_reg_dump(void)
{
    tXra1201Regs reg;

    memset((char *)&reg, 0, sizeof(reg));

    if(xra1201_reg_read(XRA1201_GSR1, XRA1201_IFR2, &reg) == 0) {
		printk(KERN_INFO "GSR1 =%02xh, GSR2 =%02xh\n", reg.GSR1,  reg.GSR2);
		printk(KERN_INFO "OCR1 =%02xh, OCR2 =%02xh\n", reg.OCR1,  reg.OCR2);
		printk(KERN_INFO "PIR1 =%02xh, PIR2 =%02xh\n", reg.PIR1,  reg.PIR2);
		printk(KERN_INFO "GCR1 =%02xh, GCR2 =%02xh\n", reg.GCR1,  reg.GCR2);
		printk(KERN_INFO "PUR1 =%02xh, PUR2 =%02xh\n", reg.PUR1,  reg.PUR2);
		printk(KERN_INFO "IER1 =%02xh, IER2 =%02xh\n", reg.IER1,  reg.IER2);
		printk(KERN_INFO "TSCR1=%02xh, TSCR2=%02xh\n", reg.TSCR1, reg.TSCR2);
		printk(KERN_INFO "ISR1 =%02xh, ISR2 =%02xh\n", reg.ISR1,  reg.ISR2);
		printk(KERN_INFO "REIR1=%02xh, REIR2=%02xh\n", reg.REIR1, reg.REIR2);
		printk(KERN_INFO "FEIR1=%02xh, FEIR2=%02xh\n", reg.FEIR1, reg.FEIR2);
		printk(KERN_INFO "IFR1 =%02xh, IFR2 =%02xh\n", reg.IFR1,  reg.IFR2);
        printk(KERN_INFO "OCR1 =%02xh, OCR2 =%02xh\n", reg.OCR1,  reg.OCR2);
    } else {
    	printk(KERN_ERR "Error. dump xra1201 registers\n");
    }
}
#endif

/* ----------------------------------------------------------------------- */
#if 0
/* Set up our tasklet */
void xra1201_do_tasklet(unsigned long);
DECLARE_TASKLET(xra1201_tasklet, xra1201_do_tasklet, 0);

void xra1201_do_tasklet (unsigned long unused)
{
	dprintk(5, KERN_INFO "(%s)\n", __func__);

	xra1201_input_data_update();
	gpio_send_signal2app(SIG_GPIO);	
	gpio3_interrupt_set(GPIO3_INTR_ENABLE);
}
#endif

static irqreturn_t xra1201_isr(int irq, void *dev_id)
{
	u_int gpio3_b29 = 0x00;
	u_int b29_mask  = 0x20000000;
	
	if (irq != INT_GPIO3) return IRQ_NONE;
	gpio3_b29 = GPIO3_REG(GPIOMIS);
	if (!(gpio3_b29 & b29_mask)) return IRQ_NONE;

	gpio3_interrupt_set(GPIO3_INTR_CLEAR);
//	tasklet_schedule(&xra1201_tasklet);
	xra1201_input_data_update();
	gpio_send_signal2app(SIG_GPIO);	
	gpio3_interrupt_set(GPIO3_INTR_ENABLE);

	return IRQ_HANDLED;
}

/* -----------------------------------------------------------------------------------
 * Generic i2c probe
 * concerning the addresses: i2c wants 7 bit (without the r/w bit), so '>>1'
 * -----------------------------------------------------------------------------------
 */
/* XRA12011 16-bit GPIO Exander id, reference include/linux/i2c-id.h */
#define I2C_DRIVERID_XRA1201	101	 	
#define I2C_NAME(s)  (s)->name

/* We have three I2C buses (adapters) whose id are 0 ~ 2 */
#define BUS_I2C0		0				

/* standard i2c insmod options */
static unsigned short normal_i2c[] = {
	XRA1201_I2C_ADDR >> 1,
	I2C_CLIENT_END
};
I2C_CLIENT_INSMOD;

static struct i2c_driver i2c_driver_xra1201;

static int xra1201_probe (struct i2c_adapter *adapter, int addr, int kind)
{
	struct i2c_client *client;	
	int ret = 0;

	if (adapter->nr != BUS_I2C0)       return 0;
	if (addr != (XRA1201_I2C_ADDR >>1))return 0;
	
	dprintk(1,KERN_INFO "probe client on address (0x%x) I2C_Bus(%d)\n", (addr << 1), adapter->nr);	

	client = kzalloc(sizeof (struct i2c_client), GFP_KERNEL);
	if (!client) {
		dprintk(0, KERN_ERR "client alloc fail\n");
		return -ENOMEM;
	}
	
	XraDev = kzalloc(sizeof(xra1201_t), GFP_KERNEL);
	if (!XraDev) {
		dprintk(0, KERN_ERR "XraDev alloc fail\n");
		kfree(client);
		return -ENOMEM;
	}

	memset(XraDev, 0, sizeof(xra1201_t));
	memset(client, 0, sizeof(struct i2c_client));

	client->addr    = addr;
	client->adapter = adapter;
	client->driver  = &i2c_driver_xra1201;
	client->flags   = 0;
	strlcpy(I2C_NAME(client), "xra1201", sizeof(I2C_NAME(client)));
	
	i2c_set_clientdata(client, XraDev);
	XraDev->client = client;

	ret = i2c_attach_client(client);
    if (ret) {
		dprintk(1, KERN_ERR "xra12011 attach fail\n");
		goto EXT;
	}

	/* register gpio isr */
   	ret = request_irq(INT_GPIO3, xra1201_isr, IRQF_SHARED, "XRA-1201",(void*)XraDev);
       if(ret < 0) {
        dprintk(1, KERN_ERR "register gpio3 irq fail\n");
		goto EXT;
    }

	/* initial data of XraDev  */
	XraDev->irq 	 = INT_GPIO3;
	XraDev->DIs 	 = 0xff; /* default level for all DIs are 1s */
	XraDev->DOs 	 = 0x00; /* default level for all DOs are 0s */
	XraDev->is_ready = 1;
	
	spin_lock_init(&XraDev->lock);
	xra1201_update_gpio_type();

	return _OK;

EXT:
	if (client) kfree(client);
	if (XraDev) kfree(XraDev);
	return ret;	
}

static int xra1201_attach_adapter (struct i2c_adapter *adapter)
{
	if (adapter->nr != BUS_I2C0) return 0;
	
	return i2c_probe(adapter, &addr_data, xra1201_probe);
}

static int xra1201_detach_client (struct i2c_client *client)
{
	xra1201_t *dev = i2c_get_clientdata(client);
	int err;

	if(client == NULL) return 0;
	if (!dev) return 0;
	
	dprintk(3, KERN_INFO "%s\n", __func__);

	err = i2c_detach_client(client);
	if (err) {
		dprintk(0, KERN_ERR "%s:detach fail\n", __func__);
		return err;
	}
	
	if (dev->irq == INT_GPIO3)
		free_irq(dev->irq, (void *)dev);

	i2c_set_clientdata(client, NULL);
	kfree(client);
	kfree(dev);		
	return _OK;
}

static struct i2c_driver i2c_driver_xra1201 = {
	.driver = {
		.name = "xra1201",
	},
	.id = I2C_DRIVERID_XRA1201,
	.attach_adapter = xra1201_attach_adapter,
	.detach_client  = xra1201_detach_client,
    .command = NULL
};

static int __init xra1201_mod_init (void)
{
	return i2c_add_driver(&i2c_driver_xra1201);
}

static void __exit xra1201_mod_exit (void)
{
	dprintk(0, KERN_INFO "%s\n", __func__);
	i2c_del_driver(&i2c_driver_xra1201);
}

MODULE_DESCRIPTION("GPIO EXPANDER");
MODULE_AUTHOR("ACTi Corporation");
MODULE_LICENSE("GPL");

module_init(xra1201_mod_init);
module_exit(xra1201_mod_exit);

