/*****************************************************************************************
 * umvp_gpio.c
 *    gpio driver for SoC A1 camera platform.
 *    revised by mingyoung.you, 2012/05/02
 *    in this driver, no interrupt mode in GPIOs. There is no blocking access in this
 * driver.
 * ########## pin definitions in the SoC A1 camera platform ##########
 * Function                  : I/O   , group and bitmap in GPIO Register
 * Reset Button              : Input , GPIO2_31
 * PT_STOP                   : Input , GPIO2_30
 * ISP Calibration Mode      : Input , GPIO3_26
 * PCBA Test Mode            : Input , GPIO3_27
 * Power LED Control         : Output, GPIO3_28
 * Day/Night Mode Indication : Input , GPIO3_30
 * DI1, DI2, DI3, DI4        : Input , GPIO0_00, GPIO0_01, GPIO0_02, GPIO0_03
 * DO1, DO2, DO3, DO4        : Outout, GPIO0_12, GPIO0_13, GPIO0_14, GPIO0_15
 ****************************************************************************************/
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/types.h>
#include <mach/hardware.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/poll.h> /* for POLLIN, etc. */
#include <linux/timer.h>
#include <asm/memory.h>
#include <asm/irq.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <asm/arch-umvp/a1_sys.h>
#include <mach/platform2500.h>
//#include <linux/spinlock.h>
#include "umvp_gpio.h"
#include "umvp_xra1201.h"

#define UMVP_GPIO_MAJOR			78

static atomic_t		gpio_drv_opened = ATOMIC_INIT(1);
static dev_t		umvp_gpio_dev;
static struct cdev	umvp_gpio_cdev;
static struct class *umvp_gpio_class = NULL;
static 	pid_t 		gusPID = 0;
static spinlock_t   umvp_gpio_lock = SPIN_LOCK_UNLOCKED;
/*****************************************************************************************
 * System Registers
 *****************************************************************************************/
#define MFS0_REG				(*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x0C))
#define MFS1_REG                (*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x10))
#define MFS2_REG                (*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x14))
#define MFS3_REG                (*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x18))
#define MFS4_REG                (*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x1C))
#define MFS5_REG                (*(volatile __u32 *)(IO_ADDRESS(UMVP_SYSTEM_BASE)+0x20))

/*****************************************************************************************
 * GPIO registers
 ****************************************************************************************/

#if 0 /*move here define to umvp_gpio.h because umvp_xra1201.c and ExISP.c will used */
	/* ##### GPIO group register base address */
	#define GPIO0_BASE_ADDR			IO_ADDRESS(UMVP_GPIO0_BASE)
	#define GPIO1_BASE_ADDR         IO_ADDRESS(UMVP_GPIO1_BASE)
	#define GPIO2_BASE_ADDR         IO_ADDRESS(UMVP_GPIO2_BASE)
	#define GPIO3_BASE_ADDR         IO_ADDRESS(UMVP_GPIO3_BASE)
	
	#define GPIO0_REG(offset)		(*(volatile __u32 *)(GPIO0_BASE_ADDR+(offset)))
	#define GPIO1_REG(offset)   	(*(volatile __u32 *)(GPIO1_BASE_ADDR+(offset)))
	#define GPIO2_REG(offset)   	(*(volatile __u32 *)(GPIO2_BASE_ADDR+(offset)))
	#define GPIO3_REG(offset)   	(*(volatile __u32 *)(GPIO3_BASE_ADDR+(offset)))
	
	/* ##### GPIO register offset ##### */
	#define GPIODATA				0x0000 /* The data register */
	#define GPIODIR					0x0004 /* Data direction control register */
	#define GPIOIS					0x0008 /* Interrupt sensitive control register */
	#define GPIOIBE					0x000C /* Both edge interrupt control register */
	#define GPIOIEV					0x0010 /* Interrupt event control register */
	#define GPIOIE					0x0014 /* Interrupt mask register */
	#define GPIOAFSEL				0x0018 /* Mode control register */
	#define GPIORIS					0x001C /* Raw interrupt status register */
	#define GPIOMIS					0x0020 /* Masked interrupt status register */
	#define GPIOIC					0x0024 /* Interrupt clear register */
	#define GPIOMASK				0x0028 /* Data read/write mask register */
	#define GPIODBEN				0x002C /* Debounce enable register */
#endif

static void gpio_read_data(struct gpio_data *gdata)
{
	unsigned int value;

	spin_lock(&umvp_gpio_lock);
	
	memset(gdata, 0, sizeof(struct gpio_data));
    /* probe if there is a xra1201 device */
	if (is_xra1201_found()) {
		value =	xra1201_read_data_input();
		gdata->DI = (unsigned char)(value&0x0f);
		gdata->DO = (unsigned char)((value>>8)&0x0f);
	}else{
		/* read DIs from GPIO 0: bit 3,2,1,0 (0000 000f) */
		GPIO0_REG(GPIOMASK)  = 0x0ff;
		value = GPIO0_REG(GPIODATA);
		gdata->DI = (unsigned char)(value&0x0f);
		gdata->DO = (unsigned char)((value>>4)&0x0f);
	}
	/* read reset button and PT_STOP from GPIO 2: bit 31,30 (c000 0000) */
	GPIO2_REG(GPIOMASK)  = 0xC0000000;
	value = GPIO2_REG(GPIODATA);
	if(value&0x40000000) gdata->pt_stop = 1;
	else                 gdata->pt_stop = 0;
	if(value&0x80000000) gdata->reset = 1;
	else                 gdata->reset = 0;
	/* read reset, isp_cal, pcba_test from GPIO 3: bit 30,27,26 (4c00 0000)*/
	GPIO3_REG(GPIOMASK)  = 0x4C000000;
	value = GPIO3_REG(GPIODATA);
	if(value&0x40000000) gdata->day_night = 1;
	else                 gdata->day_night = 0;
	if(value&0x08000000) gdata->pcba_test = 1;
	else                 gdata->pcba_test = 0;
	if(value&0x04000000) gdata->isp_cal = 1;
	else                 gdata->isp_cal = 0;
	
	spin_unlock(&umvp_gpio_lock);
}


static void gpio_setup(void)
{
	__u32 value;

	halSysSetMultiFunctionalPin(HAL_MFP_GPIO_AP);
	/* assume all gpio pins are configured by default to software control mode
	 * read in uboot (u-boot).
	 I don't set them to software control mode again here */

	/* setup the interested GPIO pins to default mode */
	/* ##### GPIO group 0: DO4,DO3,DO2,DO1,DI4,DI3,DI2,DI1 ##### */
	/* ##### the related bits are bit 7,6,5,4,3,2,1,0 (0000 00ff) ##### */
	
	GPIO0_REG(GPIOIE) = (GPIO0_REG(GPIOIE) & (~0x000000ff));    // R:201300408 edit for Di Do
	/* set direction. output(1): bit 7,6,5,4 input(0): bit 3,2,1,0 */
	value = GPIO0_REG(GPIODIR);
	GPIO0_REG(GPIODIR)  = ((value & 0xfffffff0) | 0x000000f0);  // R:201300408 edit for Di Do
	/* enable debounce for input pins, bit 3,2,1,0 */
	GPIO0_REG(GPIODBEN) = (GPIO0_REG(GPIODBEN) | 0x0f);         // R:201300408 edit for Di Do
	/* set low to output pins */
	GPIO0_REG(GPIOMASK)  = 0x0000000f0; /* bit 7,6,5,4 */
	GPIO0_REG(GPIODATA)  = 0x0;

	/* ##### GPIO group 2: Reset Button, PT_STOP ##### */
	/* ##### the related bits are bit 31,30 (c000 0000) ##### */
	GPIO2_REG(GPIOIE) = (GPIO2_REG(GPIOIE) & (~0xc0000000));
	/* set direction. input(0): bit 31,30 */
	GPIO2_REG(GPIODIR) = (GPIO2_REG(GPIODIR) & (~0xc0000000));
	/* enable debounce for input pins, bit 31,30 */
	GPIO2_REG(GPIODBEN) = (GPIO2_REG(GPIODBEN) | 0xc0000000);

	/* ##### GPIO group 3: ISP Calibration Mode, PCBA Test Mode, Power LED Control
	 * and Day/Night Mode Indication ##### */
	/* ##### the related bits are bit 30,28,27,26 (5c00 0000) ##### */
	/* disable interrupt */
	GPIO3_REG(GPIOIE) = (GPIO3_REG(GPIOIE) & (~0x5c000000));
	/* set direction. output(1): bit 28, input(0): bit 30,27.26 */
	value = GPIO3_REG(GPIODIR);
	GPIO3_REG(GPIODIR) = ((value & 0xB3ffffff) | 0x10000000);
	/* enable debounce for input pins, bit 30,27,26 */
	GPIO3_REG(GPIODBEN) = (GPIO3_REG(GPIODBEN) | 0x4c000000);
	/* not change the power LED level, leave it to applications */
}

static int gpio_ioctl(struct inode *inode, struct file *filp,
					  unsigned int cmd, unsigned long arg)
{
	struct gpio_data gdata;
	unsigned long    value;
	int              i;

	switch(cmd) {
		case GPIODATAGET: /* read all input GPIO levels */
			gpio_read_data(&gdata);
			if(copy_to_user((void __user *)arg, &gdata, sizeof(gdata)) == 0) return 0;
			printk(KERN_ERR "[GPIO.%s]:error. send data to application.\n", __func__);
			return -EFAULT;
			
		case GPIOSETDO: /* set the DO pins */
			if(copy_from_user(&gdata, (struct gpio_data *)arg, sizeof(gdata)) == 0)	{
			    /* probe if there is a xra1201 device */
				if (is_xra1201_found()) {
					xra1201_set_data_output(gdata.DO, gdata.DO_mask);
					return 0;
				}
				if(gdata.DO_mask) {
					/* DO starts from bit 12 in GPIO group 0 */
					value = 0;
					for(i = 0 ; i < GPIO_MAX_DOS ; i ++) {
						if(gdata.DO_mask & (1<<i)) value |= (1 << (4 + i));
					}
					GPIO0_REG(GPIOMASK)  = value;
					GPIO0_REG(GPIODATA)  = (((unsigned long)gdata.DO) << 4);
				}
				printk(KERN_INFO "[GPIO]:set DOs (%x,%x),(%x,%x)\n",
					   gdata.DO_mask, gdata.DO, (int)GPIO0_REG(GPIOMASK), (int)GPIO0_REG(GPIODATA));				
				return 0;
			}
			printk(KERN_ERR "[GPIO.%s]:error. get user data to set DOs\n", __func__);			
			return -EFAULT;
			
		case GPIOPOWERLED: /* turn on/off power LED */
			if(copy_from_user(&gdata, (struct gpio_data *)arg, sizeof(gdata)) == 0) {
				GPIO3_REG(GPIOMASK)  = 0x10000000; /* bit 28 */
				if(gdata.power_led) GPIO3_REG(GPIODATA) = 0x10000000;
				else                GPIO3_REG(GPIODATA) = 0x0;
				return 0;
			}
			printk(KERN_ERR "[GPIO]:error. get user data to set power LED\n");
			return -EFAULT;
			
		case GPIOSETPID:
			if(copy_from_user(&gusPID, (pid_t *)arg, sizeof(pid_t)) == 0) {
			    printk(KERN_INFO "[GPIO]: user space set PID (%d)\n", gusPID);
				return 0;			
			}
			gusPID = 0;
			printk(KERN_ERR "[GPIO]:error. get user space PID failed\n");
			return -EFAULT;	
			
		default:
			printk(KERN_ERR "[GPIO]:error. unknown command %x\n", cmd);
			return -EINVAL;
	}
}

static int gpio_open(struct inode *inode, struct file *filp)
{

	if(atomic_read(&gpio_drv_opened) > 1) {
		/* consifer applications might need to open /dev/gpio in different processes,
		 * I don't limit this driver just could be openned once. Show warning message */
	 	printk(KERN_INFO "[GPIO]:warning. gpio driver has been openned\n");
	}
	atomic_inc(&gpio_drv_opened);
	return 0;
}

static int gpio_release(struct inode *inode, struct file *filp)
{
	atomic_dec(&gpio_drv_opened);
	return 0;
}

static struct file_operations gpio_fops = {
	owner:THIS_MODULE,
	ioctl:gpio_ioctl,
	open:gpio_open,
	release:gpio_release,
};

static int gpio_init(void)
{
	int rc;

	umvp_gpio_dev = MKDEV(UMVP_GPIO_MAJOR, 0);
	rc = register_chrdev_region(umvp_gpio_dev, 1, "gpio");
	if(rc >= 0) {
		cdev_init(&umvp_gpio_cdev, &gpio_fops);
		umvp_gpio_cdev.owner = THIS_MODULE;
		umvp_gpio_cdev.ops = &gpio_fops;
		rc = cdev_add(&umvp_gpio_cdev, umvp_gpio_dev, 1);
		if(rc == 0) {
			umvp_gpio_class = class_create(THIS_MODULE, "gpio");
			if(IS_ERR(umvp_gpio_class) == 0) {
				device_create(umvp_gpio_class, NULL, umvp_gpio_dev, NULL, "gpio");
				/* configure the GPIOs for camera firmware */
				gpio_setup();
				printk(KERN_INFO "UMVP GPIO installed (major=%d)\n", UMVP_GPIO_MAJOR);
				return 0;
			}
			printk(KERN_ERR "[GPIO]:error. create gpio class\n");
			cdev_del(&umvp_gpio_cdev);
			umvp_gpio_class = NULL;
			rc = -EFAULT;
		} else {
			printk(KERN_ERR "[GPIO]:error. failed to add gpio device\n");
		}
		unregister_chrdev_region(umvp_gpio_dev, 1);
		return rc;
	}
	/* fail to register a character device for GPIO driver */
	printk(KERN_ERR "[GPIO]:error. register char device (major=%d)\n", UMVP_GPIO_MAJOR);
	return rc;
}

/* -----------------------------------------------------------------------
  gpio interrup isr process
-------------------------------------------------------------------------- */
#include <asm/siginfo.h>			//siginfo
#include <linux/rcupdate.h>			//rcu_read_lock
#define GDATA_DUMP		 0

enum gpio_intr_cmd {
	GPIO0_INTR_BOTH_EDGE = 0,
	GPIO0_INTR_ENABLE,
	GPIO0_INTR_DISABLE,
	GPIO0_INTR_CLEAR,
	GPIO2_INTR_BOTH_EDGE,		
	GPIO2_INTR_ENABLE,
	GPIO2_INTR_DISABLE,
	GPIO2_INTR_CLEAR,	
};

struct gpio_data last_gdata;	

void gpio_send_signal2app(int ev_type)
{
	struct siginfo     info;
	struct task_struct *t;
	union bits {
		struct gpio_data gdata;	
		unsigned int value;			
	}siData;
	
	if (gusPID <= 0)         return;
	if (ev_type != SIG_GPIO) return;
		
	/* find the task_struct associated with this pid */
	rcu_read_lock();
	t = find_task_by_vpid(gusPID);
	if(t == NULL){
		rcu_read_unlock();
		return;
	}
	rcu_read_unlock();	

	gpio_read_data(&siData.gdata);
	if (memcmp(&last_gdata ,&siData.gdata, sizeof(struct gpio_data)) == 0) {
		//printk("same last data \n");
		return;
	}
	memcpy(&last_gdata ,&siData.gdata, sizeof(struct gpio_data));	

	/* send signal to notify user space application */
	//spin_lock(&umvp_gpio_lock);	
	memset(&info, 0, sizeof(struct siginfo));	
	info.si_signo = ev_type;
	info.si_code  = SI_QUEUE;
	info.si_int   = siData.value;
	send_sig_info(ev_type, &info, t);    //send the signal
	//spin_unlock(&umvp_gpio_lock);
	
	#if GDATA_DUMP
	{
		struct gpio_data *p = &siData.gdata;
		printk(KERN_INFO "send signal(%d) notify user space application. value(0x%8x)\n",ev_type, siData.value);
		printk(KERN_INFO "Di(0x%x), Do(0x%x), DO_mask(0x%x)\n",p->DI, p->DO, p->DO_mask);
		printk(KERN_INFO "reset(%d), power_led(%d), pt_stop(%d), day_night(%d)\n",
			   p->reset, p->power_led, p->pt_stop, p->day_night);
	}
	#endif
}

/* config gpio as interrupt source , gpio0 bit0 ~ bit3 */
static u_int gpio0_b0b3_interrupt_set(int cmd_type)
{
	u_int gpio0_b0b3 = 0x00;	
	switch(cmd_type)
	{
		case GPIO0_INTR_BOTH_EDGE:
			/* set edge trigger interrupt */			
			gpio0_b0b3 = GPIO0_REG(GPIOIS);
			gpio0_b0b3 = (gpio0_b0b3 & ~0x0000000f) | 0x00000000;
			GPIO0_REG(GPIOIS) = gpio0_b0b3;
			/* set both edge trigger interrupt */			
			gpio0_b0b3 = GPIO0_REG(GPIOIBE);
			gpio0_b0b3 = (gpio0_b0b3 & ~0x0000000f) | 0x0000000f;
			GPIO0_REG(GPIOIBE) = gpio0_b0b3;
			break;
		case GPIO0_INTR_ENABLE:
			gpio0_b0b3 = GPIO0_REG(GPIOIE);
			gpio0_b0b3 = (gpio0_b0b3 & ~0x0000000f) | 0x0000000f;
			GPIO0_REG(GPIOIE) =  gpio0_b0b3;
			break;
		case GPIO0_INTR_CLEAR:
			gpio0_b0b3 = GPIO0_REG(GPIOIC);
			gpio0_b0b3 = (gpio0_b0b3 & ~0x0000000f) | 0x0000000f;
			GPIO0_REG(GPIOIC) = gpio0_b0b3;
			/* do clear and disable */	
			//break;
		case GPIO0_INTR_DISABLE:
			gpio0_b0b3 = GPIO0_REG(GPIOIE);
			gpio0_b0b3 = (gpio0_b0b3 & ~0x0000000f) | 0x00000000;
			GPIO0_REG(GPIOIE) = gpio0_b0b3;
			break;
		default:
			break;
	}
	return gpio0_b0b3;
}

/* config gpio as interrupt source , gpio3 bit 29 */
static u_int gpio2_b31_interrupt_set(int cmd_type)
{
	u_int gpio2_b31  = 0x00;
	switch(cmd_type)
	{
		case GPIO2_INTR_BOTH_EDGE:
			/* set edge trigger interrupt */			
			gpio2_b31 = GPIO2_REG(GPIOIS);
			gpio2_b31 = (gpio2_b31 & ~0x80000000) | 0x00000000;
			GPIO2_REG(GPIOIS) = gpio2_b31;
			/* set both edge trigger interrupt */			
			gpio2_b31 = GPIO2_REG(GPIOIBE);
			gpio2_b31 = (gpio2_b31 & ~0x80000000) | 0x80000000;
			GPIO2_REG(GPIOIBE) = gpio2_b31;
			break;
		case GPIO2_INTR_ENABLE:
			gpio2_b31 = GPIO2_REG(GPIOIE);
			gpio2_b31 = (gpio2_b31 & ~0x80000000) | 0x80000000;
			GPIO2_REG(GPIOIE) = gpio2_b31;
			break;
		case GPIO2_INTR_CLEAR:
			gpio2_b31 = GPIO2_REG(GPIOIC);
			gpio2_b31 = (gpio2_b31 & ~0x80000000) | 0x80000000;
			GPIO2_REG(GPIOIC) =  gpio2_b31;
			/* do clear and disable */	
			//break;
		case GPIO2_INTR_DISABLE:
			gpio2_b31 = GPIO2_REG(GPIOIE);
			gpio2_b31 = (gpio2_b31 & ~0x80000000) | 0x00000000;
			GPIO2_REG(GPIOIE) = gpio2_b31;
			break;
		default:
			break;
	}
	return gpio2_b31;
}

static irqreturn_t umvp_gpio0_isr(int irq, void *dev_id)
{
	u_int gpio0_b0b3 = 0x00;	
	u_int b0b3_mask  = 0x0000000f;

	if (irq != INT_GPIO0) return IRQ_NONE;	

	gpio0_b0b3 = GPIO0_REG(GPIOMIS);
	if (!(gpio0_b0b3 & b0b3_mask)) return IRQ_NONE;
	
	gpio0_b0b3_interrupt_set(GPIO0_INTR_CLEAR);
	gpio_send_signal2app(SIG_GPIO);	
	gpio0_b0b3_interrupt_set(GPIO0_INTR_ENABLE);
		
	return IRQ_HANDLED;
}

static irqreturn_t umvp_gpio2_isr(int irq, void *dev_id)
{
	u_int gpio2_b31 = 0x00;
	u_int b31_mask  = 0x80000000;
	
	if (irq != INT_GPIO2) return IRQ_NONE;	

	gpio2_b31 = GPIO2_REG(GPIOMIS);
	if (!(gpio2_b31 & b31_mask)) return IRQ_NONE;

	gpio2_b31_interrupt_set(GPIO2_INTR_CLEAR);
	gpio_send_signal2app(SIG_GPIO);
	xra1201_input_data_update(); //TestOnly	
	gpio2_b31_interrupt_set(GPIO2_INTR_ENABLE);
		
	return IRQ_HANDLED;
}

static int gpio_register_isr(void)
{
	int rc;
	/* register gpio isr */
	rc = request_irq(INT_GPIO0, umvp_gpio0_isr, IRQF_SHARED, "UMVP-GPIO0", (void*)umvp_gpio_dev);
	if(rc < 0) {
		printk(KERN_ERR "register gpio0 irq fail\n");
		return -EFAULT;
	}
	rc = request_irq(INT_GPIO2, umvp_gpio2_isr, IRQF_SHARED, "UMVP-GPIO2", (void*)umvp_gpio_dev);
	if(rc < 0) {
		printk(KERN_ERR "register gpio2 irq fail\n");
		free_irq(INT_GPIO0, (void *)umvp_gpio_dev);
		return -EFAULT;
	}

	/* set reset button interrupt on both egde trigger   */
	gpio2_b31_interrupt_set(GPIO2_INTR_BOTH_EDGE);
	/* First, clear masked interrupt status register */
	gpio2_b31_interrupt_set(GPIO2_INTR_CLEAR);
	/* enable reset button interrup */
	gpio2_b31_interrupt_set(GPIO2_INTR_ENABLE);
	
	/* set DIs(b0~b3) interrupt on both egde trigger   */
	gpio0_b0b3_interrupt_set(GPIO0_INTR_BOTH_EDGE);
	/* First, clear masked interrupt status register */	
	gpio0_b0b3_interrupt_set(GPIO0_INTR_CLEAR);
	/* enable DIs(b0~b3) interrupt */
	gpio0_b0b3_interrupt_set(GPIO0_INTR_ENABLE);
		
	return 0;
}

/* ----------------------------------------------------------------------- */
static int __init umvp_gpio_init_module(void)
{
	int err;
	
	err = gpio_init();
	if(err) return err;

	gpio_register_isr();
	
	return 0;
}

static void __exit umvp_gpio_cleanup_module(void)
{

	if(atomic_dec_and_test(&gpio_drv_opened)) {
		if(umvp_gpio_class) {
			unregister_chrdev_region(umvp_gpio_dev, 1);
			device_destroy(umvp_gpio_class, umvp_gpio_dev);
			class_destroy(umvp_gpio_class);
			//
			free_irq(INT_GPIO0, (void *)umvp_gpio_dev);
			free_irq(INT_GPIO2, (void *)umvp_gpio_dev);
			//
		}
	} else {
		printk(KERN_ERR "[GPIO]:error. this module was locked by applicateions\n");
	}
}

module_init(umvp_gpio_init_module);
module_exit(umvp_gpio_cleanup_module);

MODULE_LICENSE("GPL");


/* ----------------------------------------------------------------------- */


