/*******************************************************************
 *
 * File: xsup_ipc.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu, Terry.Simons@utah.edu
 *
 * $Id: xsup_ipc.c,v 1.52 2006/10/03 03:02:31 chessing Exp $
 * $Date: 2006/10/03 03:02:31 $
 * $Log: xsup_ipc.c,v $
 * Revision 1.52  2006/10/03 03:02:31  chessing
 * Fix a potential stack smash attack.
 *
 * Revision 1.51  2006/08/25 23:37:18  chessing
 * Numerous patches that have come in over the last month or two.
 *
 * Revision 1.50  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.49  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.48  2006/05/30 03:53:21  chessing
 * Some fixes to IPC support.
 *
 * Revision 1.47  2006/05/29 04:17:57  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.46  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.45  2006/05/22 04:02:09  chessing
 * Some small changes/fixes.
 *
 * Revision 1.44  2006/05/17 22:18:10  chessing
 * A couple of small changes to Xsupplicant, and some major changes to the GUI configuration/monitor tool.
 *
 * Revision 1.43  2006/05/17 02:40:06  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.42  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.41  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.40  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.39  2006/04/17 03:56:24  chessing
 * Added some support to enable/disable TNC support both via the configuration file, and via IPC.
 *
 * Revision 1.38  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.37  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.36  2005/09/25 00:39:48  shaftoe
 * - More work to pretty up syslog output.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.35  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.34  2005/08/20 19:26:42  chessing
 * Added option -s that will force removal of old unix domain control socket.  This option should only be used on startup, since it could cause some strange problems if it is used when an existing Xsupplicant instance is in operation.
 *
 * Revision 1.33  2005/08/18 19:12:48  chessing
 * Patch from Carsten Grohmann to fix xsup_ipc to use the correct socket name.
 *
 * Revision 1.32  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <net/if.h>
#include <grp.h>
#include <stdlib.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_ipc.h"
#include "xsup_common.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "ipc_callout.h"
#include "event_core.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

#define XSUP_SOCKET "/tmp/xsupplicant.sock."

// XXX Can get rid of this with a little work. ;)
static int ipc_sock = -1;

char socknamestr[256];

/**************************************************************************
 *
 * Determine the GID for a group name.  Used to determine what file 
 * permissions should be enabled on our IPC socket.
 *
 **************************************************************************/
int xsup_ipc_get_group_num(char *grp_name)
{
  struct group grp;
  struct group *grdata = &grp;
  struct group *tmpGrp;
  char *buffer = NULL;
  long buflen = 0;
  int retval;

  buflen = sysconf(_SC_GETGR_R_SIZE_MAX);
  if (buflen < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't determine the maximum buffer "
		   "size needed from getgrnam_r()\n");
      return -1;
    }

  buffer = (char *)malloc(buflen);
  if (buffer == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory needed to "
		   "obtain IPC group information!\n");
      return -1;
    }
  
  if (getgrnam_r(grp_name, grdata, buffer, buflen, &tmpGrp) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Unable to obtain the group index needed"
		   " to enable IPC functionality.  IPC will not be "
		   "available.\n");
      FREE(buffer);
      return -1;
    }
  
  retval = grp.gr_gid;

  if (buffer != NULL)
    {
      FREE(buffer);
    }

  return retval;
}

/******************************************************************
 *
 * Initalize the socket that we will use to communicate with a client/clients.
 * Also, set up any structures that may be needed.
 *
 ******************************************************************/
int xsup_ipc_init(struct interface_data *intdata)
{
  int sockErr = 0, ipc_gid = 0;
  char *error = NULL;
  struct sockaddr_un sa;
  struct config_globals *globals;

  xsup_assert((intdata != NULL), "intdata != NULL", TRUE);

  globals = config_get_globals();

  if (!xsup_assert((globals != NULL), "globals != NULL", FALSE))
    return XEGENERROR;

  sa.sun_family = AF_LOCAL;
  bzero(socknamestr, 256);
  strncpy(socknamestr, XSUP_SOCKET, 256-(strlen(intdata->intName)+1));
  strcat (socknamestr, intdata->intName);
  strncpy(sa.sun_path, socknamestr, sizeof(sa.sun_path));

  if (TEST_FLAG(intdata->flags, CLEAR_IPC))
    {
      // We need to clear the socket file if it exists.

      debug_printf(DEBUG_INT, "Clearing control socket %s.\n", socknamestr);
      remove(socknamestr);
    }

  // Socket we will be using to communicate.
  ipc_sock = socket(PF_UNIX, SOCK_STREAM, 0);

  if (ipc_sock == -1) {
    debug_printf(DEBUG_NORMAL, "Couldn't establish handler to daemon "
		    "socket!\n");
    return XENOSOCK;
  } 

  debug_printf(DEBUG_INT, "Opened socket descriptor #%d\n", ipc_sock);

  sockErr = bind(ipc_sock, (struct sockaddr *)&sa, sizeof(sa));
  if (sockErr == -1) 
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "An error occured binding to socket.  "
		      "(Error : %s)\n", error);
      close(ipc_sock);
      return XENOSOCK;
    }

  sockErr = listen(ipc_sock, 10);
  if (sockErr < 0)
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "An error occured listening on the socket! "
		   "(Error : %s)\n", error);
      close(ipc_sock);
      return XENOSOCK;
    }

  // Set the rights on the file.
  if (chmod(socknamestr, S_IREAD | S_IWRITE | S_IEXEC | S_IRGRP | S_IWGRP |
	     S_IXGRP | S_IROTH | S_IXOTH) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Can't set rights on socket file %s! (Error"
		   " : %s)\n", socknamestr, strerror(errno));
    }

  // Set the correct group ownership. 
  if (globals->ipc_group_name != NULL)
    {
      ipc_gid = xsup_ipc_get_group_num(globals->ipc_group_name);

      if (ipc_gid < 0)
	{
	  /* If we didn't get a valid response, then set the group to root. */
	  ipc_gid = 0;
	}
    } else {
      ipc_gid = 0;
    }

  if (ipc_gid >= 0)
    {
      if (chown(socknamestr, -1, ipc_gid) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Can't set group ownership on socket "
                       "file %s! (Error : %s)\n", socknamestr, strerror(errno));
	}
    }

  // And register our connection handler.
  event_core_register(ipc_sock, xsup_ipc_new_socket, LOW_PRIORITY,
		      "IPC master socket");

  return XENONE;
}

/**************************************************************
 *
 * Send a message to a client.
 *
 **************************************************************/
void xsup_ipc_send_message(int skfd, char *tosend, int tolen)
{
  int err;

  if ((!tosend) || (tolen <= 0))
    {
      debug_printf(DEBUG_NORMAL, "(IPC) Invalid data passed into "
		      "xsup_ipc_send_message()!\n");
      return;
    }

  debug_printf(DEBUG_INT, "(IPC) Sending (%d bytes) : \n", tolen);
  debug_hex_dump(DEBUG_INT, (uint8_t *) tosend, tolen);

  err = send(skfd, tosend, tolen, 0);
  if (err < 0)
    {
      debug_printf(DEBUG_NORMAL, "(IPC) Error writing to client! (Error :"
		      "%s)\n", strerror(errno));
      return;
    }

  if (err < tolen)
    {
      debug_printf(DEBUG_NORMAL, "(IPC) Runt message sent to client!\n");
    }
}

/**************************************************************
 *
 * Get a message from a client.  Validate it, and return the payload.
 * outsize should be passed in a valid that is the maximum size of
 * outbuf.  outsize will then be changed to the size of the result
 * buffer.
 *
 **************************************************************/
void xsup_ipc_get_message(int sockdesc, char *outbuf, int *outsize)
{
  int readStat = -1;

  if ((!outbuf) || (!outsize))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed into "
		      "xsup_ipc_get_message()!\n");
      return;
    }

  readStat = recv(sockdesc, outbuf, *outsize, 0);

  if (readStat <= 0)
    {
      if (readStat == 0)
	{
	  debug_printf(DEBUG_EVERYTHING, "(IPC) Disconnecting socket!\n");
	  //	  FD_CLR(sockdesc, &mset);
	  event_core_deregister(sockdesc);
	  close(sockdesc);
	} else {
	  debug_printf(DEBUG_NORMAL, "(IPC) Couldn't read message from client!"
		       " (Socket #%d) (Error %d -- %s)\n", sockdesc, errno, 
		       strerror(errno));
	}

      *outsize = 0;
    } else {
      debug_printf(DEBUG_INT, "(IPC) Got : \n");
      debug_hex_dump(DEBUG_INT, (uint8_t *) outbuf, readStat);
      
      *outsize = readStat;
    }
}

/**********************************************************     
 *
 * Send a message to all registered clients.
 *
 ***********************************************************/
int xsup_ipc_send_all(char *message, int msglen)
{
  int i = 0;

  if (!xsup_assert((msglen > 0), "msglen > 0", FALSE)) return -1;
  if (!xsup_assert((message != NULL), "message != NULL", FALSE)) return -1;

  event_core_reset_locate();

  while ((i = event_core_locate("client msg socket")) != -1)
    {
      xsup_ipc_send_message(i, message, msglen);
    }

  return XENONE;
}

/***********************************************************
 *
 * Push the current state to all clients.
 *
 ***********************************************************/
void xsup_ipc_send_auth_state(struct interface_data *workint)
{
  char buffer[50];
  int bufptr;
  struct ipc_cmd *cmd;

  bufptr = 0;
  cmd = (struct ipc_cmd *)buffer;
  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = AUTH_STATE;
  cmd->getset = IPC_SET;          // This is an unrequested push.
  cmd->len = 1;
  bufptr += sizeof(struct ipc_cmd);
  buffer[bufptr] = workint->statemachine->curState;
  bufptr += 1;

  xsup_ipc_send_all(buffer, bufptr);
  bufptr = 0;
}

/******************************************************************
 *
 * Send a normal log message out to all attached clients.
 *
 ******************************************************************/
void xsup_ipc_send_log(int level, char *msg)
{
  char buffer[1500];
  int bufptr;
  struct ipc_cmd *cmd;

  if (ipc_sock < 0) return;   // We can't do anything.
  if (msg == NULL) return;    // Don't send empty messages.

  bufptr = 0;
  cmd = (struct ipc_cmd *)buffer;
  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = LOG_MSG;
  cmd->getset = IPC_SET;       // An unrequested push.
  cmd->len = strlen(msg)+2;    // We want to send a NULL, and add one for
                               // the log level.
  bufptr+=sizeof(struct ipc_cmd);
  buffer[bufptr] = level;
  bufptr++;
  strncpy(&buffer[bufptr], msg, (1500 - (sizeof(struct ipc_cmd)+1)));
  bufptr += (strlen(msg)+1);

  xsup_ipc_send_all(buffer, bufptr);
}

/***********************************************************
 *
 * Push an EAP notification to any connected clients.
 *
 ***********************************************************/
void xsup_ipc_send_eap_notify(struct interface_data *workint, char *notify)
{
  char *buffer;
  int bufptr;
  struct ipc_cmd *cmd;

  buffer = malloc(strlen(notify)+1 + sizeof(struct ipc_cmd));
  if (buffer == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't send IPC EAP notify message!\n");
      return;
    }

  memset(buffer, 0x00, (strlen(notify)+1 + sizeof(struct ipc_cmd)));

  bufptr = 0;
  cmd = (struct ipc_cmd *)buffer;
  cmd->version = IPC_VERSION_NUM;
  cmd->attribute = NOTIFY;
  cmd->getset = IPC_SET;          // This is an unrequested push.
  cmd->len = strlen(notify);
  bufptr += sizeof(struct ipc_cmd);
  Strncpy((char *)&buffer[bufptr], notify, strlen(notify)+1);
  bufptr += strlen(notify);

  xsup_ipc_send_all(buffer, bufptr);
  bufptr =0;
  FREE(buffer);
}

/***********************************************************
 *
 * Handle the requests the client sent us.
 *
 ***********************************************************/
void xsup_ipc_process_message(struct interface_data *workint, int skfd,
			      char *buffer, int bufsize)
{
  char sendbuf[1520];
  int sendsize, bufptr;
  struct ipc_cmd *cmd;

  sendsize = 0;

  bufptr = 0;
  
  while (bufptr < bufsize)
    {
      cmd = (struct ipc_cmd *)&buffer[bufptr];

      if (workint != NULL)
	{
	  switch (cmd->attribute)
	    {
	    case AUTH_STATE:
	      debug_printf(DEBUG_EVERYTHING, "(IPC) Checking auth state!\n");
	      ipc_callout_auth_state(workint, &bufptr, buffer, 
				     bufsize, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case TEMPPASSWORD:
	      // Set a temporary password.
	      debug_printf(DEBUG_EVERYTHING, "(IPC) Setting temporary "
			      "password.\n");
	      ipc_callout_set_password(workint, &bufptr, buffer,
				       bufsize, (char *)&sendbuf, &sendsize);
	      break;

	    case TNCENABLE:
	      // Enable or disable TNC support.
	      debug_printf(DEBUG_EVERYTHING, "(IPC) Changing state of TNC.\n");
	      ipc_callout_set_tnc(workint, &bufptr, buffer, bufsize,
				  (char *)&sendbuf, &sendsize);
	      break;

	    case AUTH_CTRL:
	      // Change if we are attempting to authenticate or not.
	      debug_printf(DEBUG_EVERYTHING, "(IPC) Changeing authentication "
			   "control state.\n");
	      ipc_callout_set_auth_ctrl(workint, &bufptr, buffer, bufsize,
					(char *)&sendbuf, &sendsize);
	      break;

	    case PING:
	      debug_printf(DEBUG_EVERYTHING, "(IPC) Ping.\n");
	      ipc_callout_ping_resp(workint, (char *)&sendbuf, &sendsize);
	      break;
	      
	    case ERROR_MSG:
	      debug_printf(DEBUG_NORMAL, "(IPC) Got an error message from the"
			      " client. Your client is probably broken!\n");
	      break;
	      
	    default:
	      debug_printf(DEBUG_NORMAL, "(IPC) Unknown command %02X!\n",
			   buffer[bufptr]);
	      cmd = (struct ipc_cmd *)&buffer[bufptr];
	      
	      if ((bufptr+(cmd->len+sizeof(struct ipc_cmd))) > bufsize)
		{
		  debug_printf(DEBUG_NORMAL, "(IPC) Size included in command "
				  "is too high! Ignoring remaining part of "
				  "packet!\n");
		  bufptr = bufsize;
		} else {
		  bufptr+= (cmd->len+sizeof(struct ipc_cmd));
		}
	    }
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "(IPC) Error getting working interface "
		       "dumping current packet: \n");

	  debug_hex_dump(DEBUG_NORMAL, (uint8_t *) buffer, bufsize);

	  debug_printf(DEBUG_NORMAL, "(IPC) Ignoring remaining parts of "
			  "packet!\n");
	  bufptr = bufsize+1;
	}
    }
  
  // Make sure we actually have something to return.
  if (sendsize >= (sizeof(struct ipc_cmd)))
    {
      debug_printf(DEBUG_EVERYTHING, "(IPC) Sending IPC response!\n");
      xsup_ipc_send_message(skfd, sendbuf, sendsize);
    } else {
      debug_printf(DEBUG_EVERYTHING, "(IPC) sendsize = %d  -- Nothing to "
		      "send!\n", sendsize);
    }
}

/**************************************************************************
 *
 *  This callback is used when a GUI client asks us for some information.
 *  The functionality should be simple.  Gather the information, and answer
 *  as quickly as possible, so we can get on to other interesting things. ;)
 *
 **************************************************************************/
int xsup_ipc_process_socket(struct interface_data *ctx, int sock)
{
  char buffer[1520];
  int bufsize;

  bufsize = 1520;
  xsup_ipc_get_message(sock, (char *)&buffer, &bufsize);

  xsup_ipc_process_message(ctx, sock, (char *)&buffer, bufsize);

  return XENONE;
}

/**********************************************************************
 *
 *  This handler is called when the parent IPC socket gets a connection
 *  event.  All it should do is accept the connection, and register a new
 *  handler to handle communication with the client.
 *
 **********************************************************************/
int xsup_ipc_new_socket(struct interface_data *ctx, int sock)
{
  int newsock, len;
  struct sockaddr sa;

  // We got a request for a new IPC client connection.
  debug_printf(DEBUG_INT, "(IPC) Got a request to connect a new "
	       "client.\n");
  memset(&sa, 0x00, sizeof(sa));
  len = sizeof(sa);
  newsock = accept(ipc_sock, (struct sockaddr *)&sa, (u_int *)&len);
  if (newsock <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Got a request for a new IPC "
		   "client connection.  But, accept() returned"
		   " an error!\n");
      debug_printf(DEBUG_NORMAL, "Error was (%d) : %s\n", errno, 
		   strerror(errno));
    } else {
      debug_printf(DEBUG_INT, "Registering a new socket handler.\n");
      if (event_core_register(newsock, xsup_ipc_process_socket, LOW_PRIORITY,
			      "client msg socket") < 0)
	{
	  debug_printf(DEBUG_NORMAL, "No available socket handlers!\n");
	  close(newsock);
	}

      debug_printf(DEBUG_NORMAL, "Xsupplicant %s has connected a new client."
		   "\n", VERSION);
    }

  return XENONE;
}

/***********************************************************
 *
 * Clean up any structures used, and close out the communication socket.
 *
 ***********************************************************/
void xsup_ipc_cleanup()
{
  char *error;

  debug_printf(DEBUG_EVERYTHING, "Shutting down IPC socket!\n");
  debug_printf(DEBUG_INT, "Closing socket descriptor #%d\n", ipc_sock);
  if (close(ipc_sock) < 0)
    {
      error = strerror(errno);
      debug_printf(DEBUG_NORMAL, "Error closing socket!  (Error : %s)\n",
		   error);
    }

  unlink(socknamestr);
}

/*******************************************************************
 * 
 * Send a message to any GUI that is listening to let it know that we need
 * a password.
 *
 *******************************************************************/
int xsup_ipc_gui_prompt(char *interface, char *tempPwd, char *eapType, 
			char *challenge)
{
  char packet[512];
  int bufptr = 0, intidx;

  if (tempPwd == NULL)
    {
      // Ask the GUI to prompt for a password.
      debug_printf(DEBUG_AUTHTYPES, "Asking the GUI for a password.\n");
      debug_printf(DEBUG_AUTHTYPES, "EAP type is %s, challenge is %s\n",
		   eapType, challenge);

      bzero((char *)&packet[0], 512);

      if(interface)
	{
	  intidx = if_nametoindex(interface);
	  debug_printf(DEBUG_AUTHTYPES, "Interface : %s  Index : %d\n",
		       interface, intidx);
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "Invalid interface data in %s Line: %d\n",
		       __FUNCTION__, __LINE__);
	  return XEBADCONFIG;
	}

      ipc_callout_request_password(&bufptr, (char *)&packet[0], 512, 
				   eapType, challenge);
      debug_printf(DEBUG_AUTHTYPES, "Sending : \n");
      debug_hex_dump(DEBUG_AUTHTYPES, (uint8_t *) packet, bufptr);

      xsup_ipc_send_all((char *)&packet, bufptr);
      
      // Let the caller know we are asking for a password.
      return XPROMPT;
    }

  return XENONE;
}

