/*******************************************************************
 * Handle the EAPOL keying state machine.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: key_statemachine.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: key_statemachine.c,v 1.23 2006/06/01 22:49:49 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:49 $
 * $Log: key_statemachine.c,v $
 * Revision 1.23  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.22  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.21  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.20  2006/03/21 18:22:09  chessing
 * Fixes to EAP-AKA code.  Changed a few defaults to disable Xsupplicant based roaming, and passive scanning for this release.  (It would currently cause more problems than it would solve.
 *
 * Revision 1.19  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.18  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>

#include "key_statemachine.h"
#include "eapol_key_type1.h"
#include "eapol_key_type2.h"
#include "eapol_key_type254.h"
#include "xsup_debug.h"
#include "frame_structs.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

/*************************************************
 *
 * Call the processKey() function defined in the 802.1x standard.  Here, we
 * need to determine what type of key it is, and call the correct handler.
 *
 *************************************************/
void processKey(struct interface_data *thisint)
{
  struct key_packet *keydata;
  uint8_t *inframe;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  inframe = thisint->recvframe;

  keydata = (struct key_packet *)&inframe[OFFSET_TO_EAP];

  switch (keydata->key_descr)
    {
    case RC4_KEY_TYPE:
      eapol_key_type1_process(thisint);
      break;

    case WPA2_KEY_TYPE:                    // WPA2/802.11i keying
      eapol_key_type2_process(thisint);
      break;
      
    case WPA_KEY_TYPE:
      eapol_key_type254_process(thisint);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown EAPoL Key Descriptor (%d)!\n",
		   keydata->key_descr);
      break;
    }
}

/*************************************************
 *
 * Run the keying state machine that is defined in the 802.1x standard.  
 * Depending on the state, we may need to process a key.
 *
 *************************************************/
void key_statemachine_run(struct interface_data *thisint)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((thisint->statemachine != NULL), 
		   "thisint->statemachine != NULL", FALSE))
    return;

  if ((thisint->statemachine->initialize == TRUE) ||
      (thisint->statemachine->portEnabled == FALSE))
    {
      // Do the NO_KEY_RECIEVE part of the state machine.
    }

  if (thisint->statemachine->rxKey == TRUE)
    {
      processKey(thisint);
      thisint->statemachine->rxKey = FALSE;
    }
}
