/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsup_driver.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsup_driver.c,v 1.126 2006/10/05 22:33:28 chessing Exp $
 * $Date: 2006/10/05 22:33:28 $
 * $Log: xsup_driver.c,v $
 * Revision 1.126  2006/10/05 22:33:28  chessing
 * Proprietary Cisco LEAP functionality is now available to drivers that support it.
 *
 * Revision 1.125  2006/06/13 21:16:27  chessing
 * Updates to the new EAP-TNC code.
 *
 * Revision 1.124  2006/06/13 18:02:57  chessing
 * Fixed a problem that would result in a segfault if a HUP was received.  Created stub functions for implementation of EAP-TNC to support Trusted Network Connect.
 *
 * Revision 1.123  2006/06/05 01:40:41  chessing
 * Various small cleanups.
 *
 * Revision 1.122  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.121  2006/05/31 17:36:20  chessing
 * Some potential buffer overflow fixes, and memory leak patches from Carsten Grohmann.
 *
 * Revision 1.120  2006/05/29 20:36:11  chessing
 * Missed a few bits of the old madwifi code.  Bumpped the version number to 1.2.6.
 *
 * Revision 1.119  2006/05/29 20:31:00  chessing
 * Removed madwifi specific driver code.  (Wext works with the madwifi driver now.)
 *
 * Revision 1.118  2006/05/29 04:17:57  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.117  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.116  2006/05/17 02:40:06  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.115  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.114  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.112  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.111  2006/04/16 21:20:42  chessing
 * Added initial TNC support via patch from Mike McCauley.  Requires libtnc in order to work, which isn't yet available to the public.
 *
 * Revision 1.110  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.109  2006/01/19 22:23:36  chessing
 * Added the ability to associate to an AP without having wpa_group_cipher, or wpa_pairwise_cipher defined in the config file.  If they are defined, then the value in the config file will be used no matter what.  Also, changed a piece in xsup_driver.c so that we print the capabilities that the cards reports on startup.  (This should help debug some problems that are almost certain to pop up with this new code. ;)
 *
 * Revision 1.108  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.107  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.106  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.105  2005/12/02 03:15:21  chessing
 * A few small fixes to the madwifi driver code.
 *
 * Revision 1.104  2005/11/22 19:49:42  chessing
 * When the desired interface isn't available when Xsupplicant starts, Xsupplicant will wait for it to show up before starting to authenticate.  In this case, the interface isn't 'available' when an ifconfig doesn't show that the interface exists.  It has *NOTHING* to do with the up/down state of the interface.  This is mostly useful for situations where a cardbus wireless card is plugged in before linux is booted.
 *
 * Revision 1.103  2005/11/14 17:27:39  chessing
 * Removed the driver specific code for hostap, ipw, and ndiswrapper, since they all support WE18 now.
 *
 * Revision 1.102  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.101  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.100  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.99  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.98  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.97  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.96  2005/08/25 03:40:28  chessing
 * Small fix to resolve an issue with some new code that would have broken wired connections.
 *
 * Revision 1.95  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.94  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.93  2005/08/21 18:12:13  chessing
 * Added the ability to reload our configuration using a HUP.  Cleaned up a bunch of memory leaks in the config parse code.
 *
 * Revision 1.92  2005/08/20 19:26:42  chessing
 * Added option -s that will force removal of old unix domain control socket.  This option should only be used on startup, since it could cause some strange problems if it is used when an existing Xsupplicant instance is in operation.
 *
 * Revision 1.91  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/
/***
 *** This code implements 802.1X Authentication on a supplicant
 *** and supports multiple Authentication types.  
 ***/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <strings.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <errno.h>

#include "profile.h"
#include "xsupconfig.h"
#include "eap.h"
#include "config_ssid.h"
#include "statemachine.h"
#include "xsup_ipc.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "eapol.h"
#include "snmp.h"
#include "wireless_sm.h"
#include "cardif/cardif.h"
#include "timer.h"
#include "event_core.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

// This is needed to support the use of IW_ENCODE_TEMP keys for Cisco cards
// based on the airo driver.
#ifdef __LINUX__
#include "cardif/linux/cardif_linux_wext.h"
#endif
#ifdef  HAVE_TNC
#include <libtnctncc.h>
#endif
#define PIDBASE "/var/run/xsupplicant"

struct interface_data *intiface;
int dsd = 0;
char *config_path = NULL;
char *pid_filename = NULL;

/*********************************************
 *
 * Create a file and store our PID in it
 *
 *********************************************/
int create_pidfile()
{
  FILE *pidfile;
  
  if (pid_filename == NULL)
    {
      return FALSE;
    }

  pidfile = fopen(pid_filename, "w");
  if (pidfile)
    {
      fprintf(pidfile, "%d", getpid());
      fclose(pidfile);
      return TRUE;
    }

  return FALSE;
}

/*********************************************
 *
 * Delete the pid file
 *
 *********************************************/
int delete_pidfile()
{
  if (pid_filename == NULL)
    {
      return FALSE;
    }
    
  unlink(pid_filename);
  return TRUE;
}

/*********************************************
 *
 * Take a string for a driver name, and return a numeric value that indicates
 * which driver we should use.
 *
 *********************************************/
int xsup_driver_get_driver(char *drname)
{
  if (drname == NULL) return DRIVER_WEXT;

  lowercase(drname);

  if (strcmp(drname, "none") == 0) return DRIVER_NONE;
  if (strcmp(drname, "wext") == 0) return DRIVER_WEXT;

  if (strcmp(drname, "atmel") == 0) 
    {
      printf("Using atmel driver!\n");
      return DRIVER_ATMEL;
    }

  printf("Unknown driver '%s' requested.\n", drname);
  return DRIVER_WEXT;
}

/*********************************************
 *
 * Initialize the configuration data.
 *
 * NOTE : DO NOT debug_printf ANYTHING in this function, because the result
 * of debug_printf is undefined before a valid configuration is loaded.
 *
 *********************************************/
void xsup_driver_init_config(char *config)
{
  char *default_cfg = "/etc/xsupplicant.conf";

  if (config == NULL) 
    {
      config = default_cfg;
    }

  config_path = strdup(config);

  // Build up our config information.
  switch(config_setup(config))
    {
    case XECONFIGFILEFAIL:
      printf("\tPlease ensure that \"%s\" contains a valid"
	     " xsupplicant configuration.\n", config);
      exit(255);
      break;
    case XECONFIGPARSEFAIL:
      printf("There was a problem with the config file.  "
	     "We cannot continue.\n");
      exit(255);
    case XECONFIGALREADYLOADED:
      printf("config_setup() was called, but a "
	     "configuration is already loaded.\n");
      exit(255);
      break;
    }
}

/*********************************************
 *
 * Call what's needed to set up the interface to be used.
 *
 *********************************************/
int xsup_driver_init_interface(char *device, char driveridx, FDEPTH flags)
{
  xsup_assert((device != NULL), "device != NULL", TRUE);

  intiface = (struct interface_data *)malloc(sizeof(struct interface_data));
  if (intiface == NULL) return XEMALLOC;
  memset(intiface, 0, sizeof(struct interface_data));
  
  // Start by setting up the structure, and assigning the interface.
  if (init_interface_struct(intiface, device) != XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't init interface struct for device "
		   "%s!  Cannot continue!\n", device);
      exit(1);
    }

  // Set our flags.
  intiface->flags = flags;

  // Establish a handler to the interface. Only need to manually turn on WPA if
  // force is set, otherwise should only turn it on if AP indicates support.
  if (cardif_init(intiface, driveridx) < 0) 
    return XENOTINT;

  cardif_get_abilities(intiface);

  if (intiface->enc_capa == 0)
    {
      debug_printf(DEBUG_INT, "Interface has no encryption capabilities, or "
		   "unknown abilitites.\n");
    } else {
      debug_printf(DEBUG_NORMAL, "Card reported capabilitites :");
      if (intiface->enc_capa & DOES_WEP40) debug_printf_nl(DEBUG_NORMAL, 
							   " WEP40");
      if (intiface->enc_capa & DOES_WEP104) debug_printf_nl(DEBUG_NORMAL,
							    " WEP104");
      if (intiface->enc_capa & DOES_WPA) debug_printf_nl(DEBUG_NORMAL, " WPA");
      if (intiface->enc_capa & DOES_WPA2) debug_printf_nl(DEBUG_NORMAL, 
							  " WPA2");
      if (intiface->enc_capa & DOES_TKIP) debug_printf_nl(DEBUG_NORMAL, 
							  " TKIP");
      if (intiface->enc_capa & DOES_CCMP) debug_printf_nl(DEBUG_NORMAL, 
							  " CCMP");
      debug_printf_nl(DEBUG_NORMAL, "\n");
    }

  debug_printf(DEBUG_INT, "Interface initialized!\n");

  return XENONE;
}

/******************************************************************
 *
 * Initialize the log file that we will save information to.
 *
 ******************************************************************/
int xsup_driver_init_logfile(int xdaemon)
{
 struct config_globals *globals;

 debug_setdaemon(xdaemon);

 // This line *MUST* always come after the call to xsup_driver_init_config.
 // If config_get_globals() is called before it, then you will always get
 // a NULL value, and probably return an error.
 globals = config_get_globals();

 if (!globals)
   {
     // Do *NOT* debug_printf this function, or you will have problems!
     // debug_printf may not work until logfile_setup() is called!
     printf("No valid configuration globals in %s!\n", __FUNCTION__);
     return XEMALLOC;
   }
 
 return logfile_setup(globals->logfile);
}

/*********************************************
 *
 * Initialize all of the pieces that will be needed for our supplicant.
 * We need to initialize in the correct order, in order to make sure
 * that other pieces of the initialization happen with the correct 
 * information available.
 *
 * THIS FUNCTION SHOULD NEVER BE CALLED OUTSIDE OF THIS FILE!
 *
 *********************************************/
int global_init(char *device, char *drivernam, FDEPTH flags)
{
  struct interface_data *intcur;
  char driveridx = 0, retval = XENONE;
  struct config_globals *globals;

  xsup_assert((device != NULL), "device != NULL", TRUE);

  // Set our debug level.
  intiface = NULL;
  intcur = NULL;

  driveridx = xsup_driver_get_driver(drivernam);

  // This line *MUST* always come after the call to xsup_driver_init_config.
  // If config_get_globals() is called before it, then you will always get
  // a NULL value, and probably return an error.
  globals = config_get_globals();

  if (!globals)
    {
      // Do *NOT* debug_printf this function, or you will have problems!
      // debug_printf may not work until logfile_setup() is called!
      printf("No valid configuration globals in %s!\n", __FUNCTION__);
      return XEMALLOC;
    }

  if (device == NULL)
    {
      device = globals->default_int;
      if (device == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "No valid interface defined!  Cannot"
		       " continue!\n");
	  return XENOTINT;
	}

      debug_printf(DEBUG_NORMAL, "Interface not specified on the command line"
		   " we will use the default of '%s' instead!\n", device);
    }

  retval = xsup_driver_init_interface(device, driveridx, flags);
  if (retval != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't init interface!\n");
      return retval;
    }
  
  if (config_build(intiface, intiface->cur_essid) != TRUE)
    {
      // If we fail to build a config for this SSID, display a message so
      // that the user has a chance of knowing that there is a problem, but
      // continue to run so that we can authenticate if the user changes
      // SSIDs to something we know about.
      debug_printf(DEBUG_NORMAL, "Couldn't build config for network %s!\n",
		   intiface->cur_essid);
    }
  config_set_globals(intiface);

  // Initialize OpenSSL library
  SSL_library_init();
  SSL_load_error_strings();
  
  // Then, initialize EAPoL (which will initialize EAP methods).

  return XENONE;
}

/***************************************
 *
 * Trap a segfault, and exit cleanly.
 *
 ***************************************/
void global_sigseg()
{
    fprintf(stderr, "[FATAL] SIGSEGV  (Segmentation Fault)!!!\n");
    xsup_ipc_cleanup(intiface);
    fflush(stderr); fflush(stdout);
    delete_pidfile();
    exit(-1);
}

/****************************************
 *
 * Clean up any values that we have set up for use by the supplicant.  This
 * includes calling any clean up routines for other modules such as EAPoL
 * or EAP.
 *
 * THIS FUNCTION SHOULD NEVER BE CALLED OUTSIDE THIS FILE!
 *
 ****************************************/
void global_deinit()
{
  debug_printf(DEBUG_EVERYTHING, "Called %s()\n", __FUNCTION__);

  if (intiface)
    {
      xsup_ipc_cleanup(intiface);
      eapol_cleanup(intiface);

      debug_printf(DEBUG_INT, "Sending Logoff for int %s!\n",
		   intiface->intName);
      txLogoff(intiface);
      cardif_deinit(intiface);
      destroy_interface_struct(intiface);
    }

  if (intiface != NULL)
    {
      free(intiface);
      intiface = NULL;
    }

  debug_printf(DEBUG_CONFIG, "Cleaning up SSID structs.\n");
  config_ssid_clear();

  debug_printf(DEBUG_CONFIG, "Deinit wireless SM\n");
  wireless_sm_deinit();

  debug_printf(DEBUG_CONFIG, "Clean up timers\n");
  timer_cleanup();

  debug_printf(DEBUG_CONFIG, "Clean up event core\n");
  event_core_deinit();

  debug_printf(DEBUG_CONFIG, "Free up config\n");
  config_destroy();

  debug_printf(DEBUG_CONFIG, "Free memory used to parse config\n");
  config_terminate();

  debug_printf(DEBUG_CONFIG, "Clean up log file\n");
  logfile_cleanup();

  if (config_path)
    {
      debug_printf(DEBUG_CONFIG, "Clean up config path.\n");
      free(config_path);
      config_path = NULL;
    }

  debug_printf(DEBUG_CONFIG, "Clean up OpenSSL error strings\n");
  ERR_free_strings();

  debug_printf(DEBUG_CONFIG, "Clean up OpenSSL library data\n");
  EVP_cleanup();  // Clear memory allocated in SSL_library_init()

  debug_printf(DEBUG_CONFIG, "Clean up pid file\n");
  delete_pidfile();
  if (pid_filename != NULL)
    {
      free(pid_filename);
      pid_filename = NULL;
    }

  exit(0);
}

/****************************************
 *
 * Clear, and reload our config.
 *
 ****************************************/
void global_config_reload()
{
  config_destroy();
  config_setup(config_path);
  config_set_globals(intiface);
  signal(SIGHUP, global_config_reload);
}

/****************************************
 *
 * Wait for the interface we are looking for is up or inserted.
 *
 ****************************************/
int xsup_driver_int_wait()
{
  char newssid[33];

  // Check to see if this interface is wireless.
  if (cardif_int_is_wireless(intiface->intName) == TRUE)
    {
      SET_FLAG(intiface->flags, IS_WIRELESS);

      bzero(newssid, 33);
      if (cardif_GetSSID(intiface, (char *)&newssid) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't determine the current SSID!\n");
	} else {
	  debug_printf(DEBUG_NORMAL, "Your card is currently set for "
		       "wireless network \"%s\".\n", newssid);
	  debug_printf(DEBUG_NORMAL, "Looking for configuration information."
		       "\n");

	  intiface->cur_essid = strdup(newssid);

	  if (config_build(intiface, intiface->cur_essid) != TRUE)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't build a config for ESSID "
			   "%s!\n", intiface->cur_essid);
	      // XXX  Need to determine what to do when we fail.
	    }
	  config_set_globals(intiface);
	}
    } else {
      UNSET_FLAG(intiface->flags, IS_WIRELESS);
    }
  return XENONE;
}

/****************************************
 *
 * Display our usage information.
 *
 ****************************************/
void usage(char *prog)
{
  debug_printf(DEBUG_NORMAL, "\n\nXsupplicant %s\n", VERSION);
  debug_printf(DEBUG_NORMAL, "(c) Copyright 2002 - 2006 The Open1x Group\n");
  debug_printf(DEBUG_NORMAL, "Dual licensed under the GPL and BSD licenses."
	       "\n\n");
  debug_printf(DEBUG_NORMAL, "This product makes use of the OpenSSL libraries"
	       ". (http://www.openssl.org)\n\n");
  debug_printf(DEBUG_NORMAL, "Usage: %s "
	       "[-c config file] "
	       "[-i device] "
	       "[-d debug_level] "
	       "[-f] "
	       "[-D driver name] "
	       "[-z] "
	       "[-t] "
	       "[-q] "
	       "[-a] "
	       "[-s] "
	       "\n", prog);

  debug_printf(DEBUG_NORMAL, "\n\n");
  debug_printf(DEBUG_NORMAL, "-c <path> : Use the config file <path> instead "
	       "of the default.\n");
  debug_printf(DEBUG_NORMAL, "-i <interface> : Authenticate using <interface> "
	       "\n");
  debug_printf(DEBUG_NORMAL, "-d <debug_level/flags> : Set debug verbosity."
	       "\n");
  debug_printf(DEBUG_NORMAL, "-f : Run in forground mode.\n");
  debug_printf(DEBUG_NORMAL, "-D <driver name> : Use <driver name> support. "
	       "(Needed for WPA support.)\n");
  debug_printf(DEBUG_NORMAL, "-z : Reset WEP keys to 0s on roam. (Needed for "
	       "some drivers, such as Orinoco_cs.)\n");
  debug_printf(DEBUG_NORMAL, "-t : Don't use IW_ENCODE_TEMP in key setting "
	       "requests.  (Try this if you are getting errors setting keys.)\n");
  debug_printf(DEBUG_NORMAL, "-q : Terminate when defaulting to authenticated "
	       "state.\n");
  debug_printf(DEBUG_NORMAL, "-a : Watch alternate interface index for "
	       "wireless events.\n");
  debug_printf(DEBUG_NORMAL, "-s : Remove the existing control socket if found.  (Should only be used in system init scripts!)\n");
  debug_printf(DEBUG_NORMAL, "\n\n");

  debug_printf(DEBUG_NORMAL, " <driver name> can be any of : \n");
  debug_printf(DEBUG_NORMAL, "\twext : Interface uses Linux wireless "
	       "extensions. (Default)\n");
  debug_printf(DEBUG_NORMAL, "\tatmel : Interface uses atmel driver.\n");

  debug_printf(DEBUG_NORMAL, "\n");

  debug_printf(DEBUG_NORMAL, " <debug_level> can be any of : \n");
  debug_printf(DEBUG_NORMAL, "\t0..7 : Old style debug flags.\n");
  debug_printf(DEBUG_NORMAL, "\tA : Enable ALL debug flags.\n");
  debug_printf(DEBUG_NORMAL, "\tc : Enable CONFIG debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ts : Enable STATE debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ta : Enable AUTHTYPES debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\ti : Enable INT debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\tn : Enable SNMP debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\te : Enable EVERYTHING debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\tx : Enable EXCESSIVE debug flag.\n");
  debug_printf(DEBUG_NORMAL, "\nValues [0..7] cannot be combined with character attributes!\n");
}

/***************************************
 *
 * The main body of the program.  We should keep this simple!  Process any
 * command line options that were passed in, set any needed variables, and
 * enter the loop to handle sending an receiving frames.
 *
 ***************************************/
int main(int argc, char *argv[])
{
  int op, pid;
  char *theOpts = "c:i:d:D:Wwfztqas";
  char *config = NULL, *device = NULL;
  char *drivername = NULL;
  int xdaemon = 1, new_debug, zeros=0;
  FDEPTH flags = 0;

  new_debug = 0;
  config = NULL;

  // If we don't have at least a -i, then we can't start up!
  if (argc < 2)
    {
      usage(argv[0]);
      exit(0);
    }

  // Process any arguments we were passed in.
  while ((op = getopt(argc, argv, theOpts)) != EOF) 
    {
      switch (op)
	{
	case 'c':
	  // Path to config file.
	  config = optarg;
	  break;

	case 'i':
	  // Interface to use.
	  device = optarg;
	  break;

	case 'd':
	  // Set the debug level.
	  if ((atoi(optarg) == 0) && (optarg[0] != '0'))
	    {
	      debug_alpha_set_flags(optarg);
	    } else {
	      debug_set_flags(atoi(optarg));
	    }
	  break;

	case 'f':
	  // Force running in the foreground.
	  xdaemon = 2;
	  break;

	case 'D':
	  // The name of the wireless driver to use.
	  drivername = optarg;
	  break;

	case 'z':
	  // Reset the encryption key to zeros on roam.
	  zeros = 1;
	  break;

	case 't':
	  // Use IW_ENCODE_TEMP for setting keys.
	  SET_FLAG(flags, DONT_USE_TEMP);
	  break;

	case 'q':
	  // Terminate when we have exhausted the maximum number of starts we
	  // want to send.
	  SET_FLAG(flags, TERM_ON_FAIL);
	  break;

	case 'a':
	  // Enable "off by -1 mode" for wireless cards that provide the
	  // driver events on an interface that is (interface index)-1.
	  SET_FLAG(flags, ONEDOWN);
	  break;
	  
	case 's':
	  // Clear the IPC socket file, if it still exists.
	  SET_FLAG(flags, CLEAR_IPC);
	  break;

	  // added by npetroni, need to do something with bad options.
	  // for now, I say exit.
	default:
	  usage(argv[0]);
	  exit(0);
	  break;
	}
    }

  if (device == NULL)
    {
      // We don't have an interface listed to use!
      debug_printf(DEBUG_NORMAL, "You need to specify an interface to use!"
		   "\n\n");
      usage(argv[0]);
      exit(0);
    }

  if (xdaemon == 1)
    {
      printf("Starting XSupplicant v. %s\n",VERSION);
      // We should fork, and let the parent die.
      pid = fork();
      
      if (pid > 0) 
	{
	  // If we are the parent, die.
	  exit(0);
	}
      
      // Otherwise, keep going.
    }

  snmp_init();

  timer_init();

  xsup_driver_init_config(config);

  if (xsup_driver_init_logfile(xdaemon) != 0)
    {
      exit(255);
    }

#ifdef  HAVE_TNC
  // This specifies the location of a TNC configuration file
  // which contains the name and location of the loadable TNC IMC modules 
  // that are required to perform Integrity Measurement during 
  // EAP-TTLS authentications
 {
     char* tnc_config_file = "/etc/tnc_config";
     libtnc_tncc_Initialize(tnc_config_file);
 }
#endif

  // Wait for our interface to become available before we continue.
  cardif_wait_for_int(device);
  
  pid_filename = malloc(strlen(PIDBASE) + strlen(device) + 6);
  if (pid_filename == NULL)
    {
      /* Skip creating pid file in case of error */
      debug_printf(DEBUG_NORMAL, "Can't allocate memory for the pid filename!\n");
    } else {
      /* Create pid file ... */
      sprintf(pid_filename, "%s.%s.pid", PIDBASE, device);
      if(!(create_pidfile()))
        {
          debug_printf(DEBUG_NORMAL, "Failed to create the pid file!\n");
        }
    }

  event_core_init();

  // We have our options set, to initalize, then go in to our event loop.
  if (global_init(device, drivername, flags) != 0)
    {
      printf("Couldn't initalize!!!!\n");
      exit(255);
    }
           
  event_core_register(cardif_get_socket(intiface), eapol_withframe, 
		      LOW_PRIORITY, "frame handler");

  if (intiface == NULL)
    {
      printf("No valid interface found!\n");
      global_deinit();
      exit(255);
    }

  if (xsup_ipc_init(intiface) != 0)
    {
      printf("Couldn't initalize daemon socket!\n");
      global_deinit();
      exit(255);
    }
  
  // When we quit, cleanup.
  signal(SIGTERM, global_deinit);
  signal(SIGINT, global_deinit);
  signal(SIGQUIT, global_deinit);
  signal(SIGSEGV, global_sigseg);
  signal(SIGHUP, global_config_reload);

  // XXX Remove this in favor of using the wireless state machine.
  if (xsup_driver_int_wait() != XENONE)
    {
      printf("Error waiting for interface to become ready!\n");
      global_deinit();
      exit(255);
    }

  if (TEST_FLAG(intiface->flags, IS_WIRELESS))
  {
    wireless_sm_init(zeros, intiface);
  }

  while (1)
    {
      //      xsup_ipc_process(intiface);

      if (TEST_FLAG(intiface->flags, IS_WIRELESS))
	{
	  // If we are wireless, we have an association state machine that
	  // needs to be taking place.  It should handle calling the event
	  // core, once it is in the state needed to make that call.
	  wireless_sm_do_state(intiface);
	} else {
	  // If we are wired, we should just move right on to processing
	  // events.
	  event_core(intiface);
	}
    }

  return XENONE;
}
