/*******************************************************************
 * Handle keying for WPA2/802.11i keys.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eapol_key_type2.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapol_key_type2.c,v 1.34 2006/06/01 22:49:49 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:49 $
 * $Log: eapol_key_type2.c,v $
 * Revision 1.34  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.33  2006/05/29 04:17:57  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.32  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.31  2006/05/25 05:28:35  chessing
 * A couple of small fixes to the last patch.
 *
 * Revision 1.30  2006/05/25 05:20:44  chessing
 * Lots more WPA/WPA2 fixes.
 *
 * Revision 1.29  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.28  2006/05/17 03:56:09  chessing
 * Updates to WPA2.
 *
 * Revision 1.27  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.26  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.25  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.24  2006/04/10 04:21:18  chessing
 * Fixed problem where on a reassociation the backend state machine would throw us in authenticated state.
 *
 * Revision 1.23  2006/04/06 06:39:47  chessing
 * A few fixes for bugs found at Interop's Hot Stage.  We now work with madwifi-ng.
 *
 * Revision 1.22  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.21  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.20  2006/01/23 07:13:49  chessing
 * A couple of fixups to the WPA and WPA2 code.
 *
 * Revision 1.19  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.18  2006/01/19 05:39:52  chessing
 * One more small fix.
 *
 * Revision 1.17  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.16  2006/01/15 06:56:29  chessing
 * Some fixes to WPA/WPA2.  WPA is working correctly now.  WPA2 needs a little work to correctly identify key frames that are for the GTK.
 *
 * Revision 1.15  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.14  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.13  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.12  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 ******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "profile.h"
#include "xsupconfig.h"
#include "key_statemachine.h"
#include "eapol_key_type2.h"
#include "eapol.h"
#include "frame_structs.h"
#include "wpa_common.h"
#include "wpa2.h"
#include "wpa.h"
#include "psk.h"
#include "mic.h"
#include "snmp.h"
#include "config_ssid.h"
#include "cardif/cardif.h"
#include "eap_types/mschapv2/mschapv2.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

uint8_t wpa2_group_key_ver = 0, wpa2_pairwise_key_ver = 0;

/**************************************************************
 *
 * Given a frame, parse all of the data that is contained in it, and
 * provide a human readable output that is useful for debugging.
 *
 **************************************************************/
void eapol_key_type2_dump(struct interface_data *intdata, char *framedata)
{
  int value16=0;
  int need_comma = 0, encdata = 0;
  uint16_t keylen, version = 0;
  struct wpa2_key_packet *keydata;
  uint8_t *keypayload;
  uint8_t rc4_ek[32], key[1500];

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  if (!xsup_assert((framedata != NULL), "framedata != NULL", FALSE))
    return;

  keydata = (struct wpa2_key_packet *)&framedata[OFFSET_TO_EAPOL+4];

  debug_printf(DEBUG_INT, "Key Descriptor      = %d\n", keydata->key_descriptor);
  memcpy(&value16, keydata->key_information, 2);
  debug_printf(DEBUG_INT, "Key Information     = %04X  (Flags : ", ntohs(value16));
  if (ntohs(value16) & WPA2_PAIRWISE_KEY)
    {
      debug_printf_nl(DEBUG_INT, "Pairwise Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_INSTALL_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Install Key");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_KEY_ACK_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Key Ack");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_KEY_MIC_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "MIC");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_SECURE_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Secure");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_ERROR_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Error");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_REQUEST_FLAG)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Request");
      need_comma = 1;
    }
  if (ntohs(value16) & WPA2_ENCRYPTED_DATA)
    {
      if (need_comma) debug_printf_nl(DEBUG_INT, ", ");
      debug_printf_nl(DEBUG_INT, "Encrypted Data");
      need_comma = 1;
      encdata = 1;
    }

  debug_printf_nl(DEBUG_INT, ")\n");

  switch (ntohs(value16) & WPA2_KEYTYPE_MASK)
    {
    case 1:
      version = 1;
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-MD5 for MIC and RC4 for encryption.\n");
      break;

    case 2:
      version = 2;
      debug_printf(DEBUG_INT, "Key Descriptor Version : HMAC-SHA1-128 for MIC and AES for encryption.\n");
      break;
    }

  debug_printf(DEBUG_INT, "Key Length          = %d\n", 
	       ntohs(keydata->key_length));

  debug_printf(DEBUG_INT, "Key Replay Counter  = ");  
  debug_hex_printf(DEBUG_INT, keydata->key_replay_counter, 8);

  debug_printf(DEBUG_INT, "Key NONCE           = ");
  debug_hex_printf(DEBUG_INT, keydata->key_nonce, 32);

  debug_printf(DEBUG_INT, "Key IV              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_iv, 16);
  
  debug_printf(DEBUG_INT, "Key RSC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_rsc, 8);
  
  debug_printf(DEBUG_INT, "Key ID              = ");
  debug_hex_printf(DEBUG_INT, keydata->key_id, 8);

  debug_printf(DEBUG_INT, "Key MIC             = ");
  debug_hex_printf(DEBUG_INT, keydata->key_mic, 16);

  value16 = ntohs(keydata->key_material_len);
  debug_printf(DEBUG_INT, "Key Material Length = %d\n", value16);

  if (value16 > 0)
    {
      debug_printf(DEBUG_INT, "Key Data : (%d)\n", value16);
      debug_hex_dump(DEBUG_INT, keydata->keydata, value16);
    }

  if (encdata)
    {
      keylen = value16;
      memcpy(&value16, keydata->key_information, 2);
      
      keypayload = (uint8_t *)malloc(keylen);
      memcpy(keypayload, keydata->keydata, keylen);

      if (!intdata->statemachine->PTK)
	{
	  debug_printf(DEBUG_NORMAL, "PTK is NULL!  Can't decrypt data!\n");
	  return;
	}

      switch (version)
	{
	case 1:
	  bzero(rc4_ek, 32);
	  memcpy(rc4_ek, keydata->key_iv, 16);
	  memcpy(&rc4_ek[16], &intdata->statemachine->PTK[16], 16);
       	  rc4_skip(rc4_ek, 32, 256, keypayload, keylen);
	  break;
	  
	case 2:
	  bzero(key, 16);
	  debug_printf(DEBUG_INT, "PTK : ");
	  debug_hex_printf(DEBUG_INT, (uint8_t *)intdata->statemachine->PTK, 32);
	  debug_printf(DEBUG_INT, "\n");

	  if (aes_unwrap((uint8_t *) &intdata->statemachine->PTK[16], 
			 (keylen-8)/8, keypayload, key) != 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed AES unwrap. (Data will be"
			   " invalid!)\n");
	    }
	  keypayload = key;
	  break;
	  
	default:
	  debug_printf(DEBUG_NORMAL, "Unknown version %d!\n", version);
	  break;
	}

      debug_printf(DEBUG_INT, "Decypted data (%d) : \n", keylen);
      debug_hex_dump(DEBUG_INT, keypayload, keylen);
    }
}

/*******************************************************
 *
 * Generate the pre-Temporal key. (PTK) Using the authenticator, and 
 * supplicant nonces.  (Anonce, and Snonce.)  The PTK is used for keying
 * when we are ready.
 *
 *******************************************************/
char *eapol_key_type2_gen_ptk(struct interface_data *intdata, char *Anonce)
{
  char prfdata[76];  // 6*2 (MAC addrs) + 32*2 (nonces)
  char *retval;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return NULL;

  if (!xsup_assert((Anonce != NULL), "Anonce != NULL", FALSE))
    return NULL;

  bzero((char *)&prfdata, 76);

  if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac, 6) < 0)
    {
      memcpy((char *)&prfdata[0], (char *)&intdata->source_mac, 6);
      memcpy((char *)&prfdata[6], (char *)&intdata->dest_mac, 6);
    } 
  else if (memcmp((char *)&intdata->source_mac, (char *)&intdata->dest_mac,
		  6) > 0)
    {
      memcpy((char *)&prfdata[0], (char *)&intdata->dest_mac, 6);
      memcpy((char *)&prfdata[6], (char *)&intdata->source_mac, 6);
    } 
  else 
    {
      debug_printf(DEBUG_NORMAL, "Source and Destination MAC addresses "
		   "match!  The PTK won't be valid!\n");
      return NULL;
    }

  if (memcmp(intdata->statemachine->SNonce, Anonce, 32) < 0)
    {
      memcpy((char *)&prfdata[12], intdata->statemachine->SNonce, 32);
      memcpy((char *)&prfdata[44], Anonce, 32);
    } 
  else if (memcmp(intdata->statemachine->SNonce, Anonce, 32) > 0)
    {
      memcpy((char *)&prfdata[12], Anonce, 32);
      memcpy((char *)&prfdata[44], intdata->statemachine->SNonce, 32);
    } 
  else 
    {
      debug_printf(DEBUG_NORMAL, "ANonce and SNonce match!  The PTK won't"
		   " be valid!\n");
      return NULL;
    }
  
  retval = (char *)malloc(80);
  if (retval == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for retval in %s "
		   "at %d!\n", __FUNCTION__, __LINE__);
      return NULL;
    }

  debug_printf(DEBUG_INT, "PMK : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) intdata->statemachine->PMK, 32);
  wpa_PRF((uint8_t *) intdata->statemachine->PMK, 32, (uint8_t *) "Pairwise key expansion", 22,
	  (uint8_t *)&prfdata, 76, (uint8_t *) retval, 64);

  debug_printf(DEBUG_INT, "PTK : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) retval, 64);

  return retval;
}

/*****************************************************************
 *
 * When a MIC failure occures, we need to send the AP a request for
 * a new key. (Reference 802.11i-D3.0.pdf page 43, line 8)
 *
 *****************************************************************/
void eapol_key_type2_request_new_key(struct interface_data *intdata, 
				       char unicast)
{
  struct wpa2_key_packet *outkeydata;
  uint16_t value16, keyindex, len;
  int eapolver;
  char key[16];
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  outkeydata = (struct wpa2_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa2_key_packet));

  outkeydata->key_descriptor = WPA2_KEY_TYPE;

  value16 = (WPA2_REQUEST_FLAG | WPA2_ERROR_FLAG);

  if (unicast == 1)
    {
      // Set the key flags to indicate this is a pairwise key, with an
      // index of 0.
      keyindex = (WPA2_PAIRWISE_KEY | wpa2_pairwise_key_ver);
    } else {
      // Set the key flags to indicate this is a group key. We don't know the
      // index, so for now we will use 1.
      keyindex = ((1 << 4) | wpa2_group_key_ver);
    }

  value16 = (value16 | keyindex);

  memcpy(outkeydata->key_information, &value16, 2);

  // Build the response.
  len = sizeof(struct wpa2_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 
  
  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available!  We will not "
		   "be able to request a new key!\n");
      return;
    }

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
}


/*********************************************************************
 *
 * Process a GTK KDE, and set the keys as needed.
 *
 *********************************************************************/
char eapol_key_type2_do_gtk_kde(struct interface_data *intdata, uint8_t *key, 
				uint16_t kdelen, uint8_t keylen, 
				uint8_t *keyrsc, char version)
{
  char keyindex = 0, txkey = 0;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((key != NULL), "key != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((keyrsc != NULL), "keyrsc != NULL", FALSE))
    return XEMALLOC;

  debug_printf(DEBUG_INT, "KDE (%d) : \n", kdelen);
  debug_hex_dump(DEBUG_INT, key, kdelen);
  
  keyindex = WPA2_KEY_INDEX_MASK & key[0];
  
  if (WPA2_TX_KEY_MASK & key[0]) txkey = 1;
  
  kdelen -= 2;
  key += 2;

  // At this point, the only thing left should be the key.
  keylen = kdelen;

  debug_printf(DEBUG_INT, "Setting GTK of length %d with index of %d.\n", 
	       kdelen, keyindex);

  debug_printf(DEBUG_INT, "Key is : ");
  debug_hex_printf(DEBUG_INT, key, keylen);
  
  switch (version)
    {
    case 1:
      wpa_common_set_key(intdata, NULL, keyindex, txkey, (char *)key, keylen);
      break;

    case 2:
      wpa_common_set_key(intdata, NULL, keyindex, txkey, (char *)key, keylen);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown encryption/MAC type %d.\n", version);
      break;
    }

  debug_printf(DEBUG_NORMAL, "Set new group IEEE 802.11i/WPA2 key.\n");

  return XENONE;
}

/*********************************************************************
 *
 * Process an STAKey KDE and set keys as needed.
 *
 *********************************************************************/
char eapol_key_type2_do_stakey_kde(struct interface_data *intdata, 
				   uint8_t *key, uint16_t keylen, 
				   uint8_t *keyrsc, char version)
{
  debug_printf(DEBUG_NORMAL, "STAkeys are not supported at this time! "
	       "(Ignoring)\n");
  return XENONE;
}

/*********************************************************************
 *
 * Process a MAC address KDE.
 *
 *********************************************************************/
char eapol_key_type2_do_mac_kde(struct interface_data *intdata, uint8_t *key,
				uint16_t keylen, char version)
{
  debug_printf(DEBUG_NORMAL, "MAC KDEs are not supported at this time! "
	       "(Ignoring)\n");
  return XENONE;
}

/*********************************************************************
 *
 * Process a PMKID KDE.
 *
 *********************************************************************/
char eapol_key_type2_do_pmkid_kde(struct interface_data *intdata,
				  uint8_t *key, uint16_t keylen, char version)
{
  return XENONE;
}

/********************************************************
 *
 * Given the IE from the decrypted data, verify that it matches what
 * the AP originally told us.  If not, we should fail, and send a deauth.
 *
 ********************************************************/
int eapol_key_type2_cmp_ie(struct interface_data *intdata, uint8_t *iedata,
			   int len)
{
  uint8_t *apie, ielen;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  if (iedata[0] != WPA2_EID)
    {
      debug_printf(DEBUG_NORMAL, "Data presented doesn't appear to be a valid"
		   " WPA2/802.11i IE!\n");
      return -1;
    }

  debug_printf(DEBUG_INT, "Got an IE of        : \n");
  debug_hex_dump(DEBUG_INT, (uint8_t *) iedata, len);

  config_ssid_get_rsn_ie(&apie, &ielen);
  debug_printf(DEBUG_INT, "AP sent us an IE of : \n");
  debug_hex_dump(DEBUG_INT, (uint8_t *) apie, ielen);

  if (len != ielen)
    {
      debug_printf(DEBUG_NORMAL, "The length of the IE from the AP and IE in"
		   " key messages were different!  IEs are invalid!\n");
      free(apie);
      return -1;
    }

  if (memcmp(apie, iedata, len) != 0)
    {
      debug_printf(DEBUG_NORMAL, "The IE from the AP and the IE in the key "
		   "messages don't match!\n");
      free(apie);
      return -1;
    }

  return XENONE;
}

/*********************************************************************
 *
 * Given a string of bytes, go through it, and parse the information looking
 * for different Key Data Encapsulations (KDEs).  Depending on the KDE that
 * we find, we will pass the data on for further processing.
 *
 *********************************************************************/
char eapol_key_type2_process_keydata(struct interface_data *intdata, 
				     uint8_t *keydata, uint16_t len, 
				     uint8_t keylen, uint8_t *keyrsc, 
				     char version)
{
  uint8_t kdeval[3] = {0x00, 0x0f, 0xac};
  int i = 0, done = 0;
  uint8_t *p, kdelen;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((keydata != NULL), "keydata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((keyrsc != NULL), "keyrsc != NULL", FALSE))
    return XEMALLOC;

  while ((i < len) && (!done))
    {
      if (keydata[i] == 0x30)
	{
	  // We are testing the IE to see if it matches what the AP gave us.
	  if (eapol_key_type2_cmp_ie(intdata, &keydata[i], (keydata[i+1]+2))
	      != XENONE)
	    {
	      debug_printf(DEBUG_NORMAL, "IE presented by the AP doesn't "
			   "match the IE in the key message!  Assuming "
			   "an active attack is in progress!\n");
	      cardif_disassociate(intdata, DISASSOC_INVALID_IE);
	      return -1;
	    }
	}

      if (keydata[i] == WPA2_EXTENDED_KEY_DATA)
	{
	  kdelen = keydata[i+1];
	  if (kdelen == 0)
	    {
	      debug_printf(DEBUG_INT, "Remaining KDE data is padding.\n");
	      return XENONE;
	    }

	  if (memcmp(&keydata[i+2], kdeval, 3) == 0)
	    {
	      // This isn't a key type that we know how to deal with,
	      // ignore it.
	      p = (uint8_t *)&keydata[i+6];

	      switch (keydata[i+5])
		{
		case 0:
		  debug_printf(DEBUG_INT, "KDE : Reserved\n");
		  break;
		  
		case WPA2_EXTENDED_GTK_KEY:
		  debug_printf(DEBUG_INT, "KDE : GTK KDE\n");
		  if (eapol_key_type2_do_gtk_kde(intdata, p, kdelen-4, keylen, 
						 keyrsc, version) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't set the GTK! "
				   "Aborting the connection!\n");
		      return -1;
		    }
		  break;
		  
		case WPA2_EXTENDED_STA_KEY:
		  debug_printf(DEBUG_INT, "KDE : STAKey KDE\n");
		  if (eapol_key_type2_do_stakey_kde(intdata, p, kdelen-4, 
						    keyrsc, version) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't set STAKey!\n");
		      return -1;
		    }
		  break;
		  
		case WPA2_EXTENDED_MAC_ADDRESS:
		  debug_printf(DEBUG_INT, "KDE : MAC address KDE\n");
		  if (eapol_key_type2_do_mac_kde(intdata, p, kdelen-4, 
						 version) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't processing MAC "
				   "address KDE!\n");
		      return -1;
		    }
		  break;
		  
		case WPA2_EXTENDED_PMKID:
		  debug_printf(DEBUG_INT, "KDE : PMKID KDE\n");
		  if (eapol_key_type2_do_pmkid_kde(intdata, p, kdelen-4, 
						   version) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't process PMKID!\n");
		      return -1;
		    }
		  break;
		  
		default:
		  debug_printf(DEBUG_NORMAL, "Unknown KDE found!\n");
		  break;
		}
	    }
	  else 
	    {
	      debug_printf(DEBUG_NORMAL, "KDE contained an OUI that we don't"
			   " understand!\n");
	    }
	}
      i += (keydata[i+1] + 2);
    }
  i++;

  return XENONE;
}

/****************************************************************
 *
 * When we have completed the PTK piece, and the pairwise key has been
 * applied to the interface, we need to get the group key.  The authenticator
 * will send us a group key that is encrypted.  We should decrypt it, apply
 * it to our interface, and send the authenticator a message to let it know
 * that we have a group key.
 *
 ****************************************************************/
void eapol_key_type2_do_gtk(struct interface_data *intdata)
{
  struct wpa2_key_packet *inkeydata, *outkeydata;
  uint16_t value16, keyflags, version, len;
  int eapolver;
  unsigned char *keydata;
  char key[16], rc4_ek[32];
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inkeydata = (struct wpa2_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa2_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // First, make sure that the inkeydata replay counter is higher than
  // the last counter we saw.
  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) && 
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      intdata->recv_size = 0;
      return;
    }

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa2_key_packet));

  outkeydata->key_descriptor = WPA2_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA2_KEYTYPE_MASK);
  version = keyflags;
  wpa2_group_key_ver = version;

  // Verify that our flags are correct.  (We don't check for the install flag,
  // for now.  Since the WPA spec doesn't explicitly require that the install
  // flag be set.)
  if (!((value16 & WPA2_KEY_ACK_FLAG) &&
	(value16 & WPA2_KEY_MIC_FLAG) && (value16 & WPA2_SECURE_FLAG)))
    {
      debug_printf(DEBUG_NORMAL, "Invalid flags in GTK message 1!\n");
      return;
    }

  value16 = ntohs(inkeydata->key_material_len);
  
  keydata = (unsigned char *)malloc(value16+8);
  if (keydata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for key data!\n");
      return;
    }

  memcpy(keydata, inkeydata->keydata, value16);

  debug_printf(DEBUG_INT, "Setting GTK! (Version : %d  Length : %d)\n", 
	       version, value16);

  if (!intdata->statemachine->PTK)
    {
      debug_printf(DEBUG_NORMAL, "No valid PTK available!  We won't be able"
		   " to continue.\n");
      return;
    }

  switch (version)
    {
    case 1:
      // Decrypt the GTK.
      bzero(rc4_ek, 32);
      memcpy(rc4_ek, inkeydata->key_iv, 16);
      memcpy(&rc4_ek[16], &intdata->statemachine->PTK[16], 16);
      rc4_skip((uint8_t *) rc4_ek, 32, 256, keydata, value16);

      debug_printf(DEBUG_INT, "Decrypted data : (%d)\n", value16);
      debug_hex_dump(DEBUG_INT, keydata, value16);

      if (eapol_key_type2_process_keydata(intdata, keydata, value16, 
					  ntohs(inkeydata->key_length),
					  inkeydata->key_rsc, version) 
	  != XENONE)
	{
	  return;
	}
      break;

    case 2:
      // First, decrypt the GTK
      bzero(key, 16);
      if (aes_unwrap((uint8_t *) &intdata->statemachine->PTK[16], 
		     (value16-8)/8, keydata, (uint8_t *) key) != 0)
      {
	debug_printf(DEBUG_NORMAL, "Failed AES unwrap.\n");
      }

      debug_printf(DEBUG_INT, "Result : ");
      debug_hex_printf(DEBUG_INT, (uint8_t *)key, value16);

      if (eapol_key_type2_process_keydata(intdata, (uint8_t *)key, value16, 
					  ntohs(inkeydata->key_length),
					  inkeydata->key_rsc, version)
	  != XENONE)
	{
	  return;
	}
      break;
    }

  debug_printf(DEBUG_NORMAL, "Set new IEEE 802.11i/WPA2 group key.\n");

  // Build the response.
  len = sizeof(struct wpa2_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  value16 = (version | WPA2_KEY_MIC_FLAG | WPA2_SECURE_FLAG);
  value16 = htons(value16);
  memcpy(&outkeydata->key_information, &value16, 2);

  outkeydata->key_length = inkeydata->key_length;
  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 
  
  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type2_dump(intdata, (char *) intdata->sendframe);

  if (network_data->methods->method_num == WPA_PSK)
    {
      // If we are using PSK, and we made it here, then we are in 
      // AUTHENTICATED state.
      intdata->statemachine->eapSuccess = TRUE;
      cardif_drop_unencrypted(intdata, FALSE);
    } else {
      // Drop unencrypted frames.
      cardif_drop_unencrypted(intdata, TRUE);
    }
}

/***************************************************************
 *
 * Handle the first packet in the four-way handshake.
 *
 ***************************************************************/
void eapol_key_type2_do_type1(struct interface_data *intdata)
{
  struct wpa2_key_packet *inkeydata, *outkeydata;
  uint16_t keyflags, len, value16;
  int i, version, eapolver, ielen;
  char key[16];
  char wpa_ie[26];
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inkeydata = (struct wpa2_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa2_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) && 
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      intdata->recv_size = 0;
      return;
    }

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL+4], sizeof(struct wpa2_key_packet));

  // XXX Need to do this better.  Tie it in with Nonce code from SIM/AKA.
  for (i=0;i<32;i++)
    {
      outkeydata->key_nonce[i] = rand();
    }
  intdata->statemachine->SNonce = (char *)malloc(32);
  if (intdata->statemachine->SNonce == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for SNonce in "
		   "%s at %d!\n", __FUNCTION__, __LINE__);
      return;
    }
  memcpy(intdata->statemachine->SNonce, (char *)&outkeydata->key_nonce[0],
	 32);

  // Calculate the PTK.
  if (intdata->statemachine->PTK != NULL)
    {
      free(intdata->statemachine->PTK);
      intdata->statemachine->PTK = NULL;
    }
  intdata->statemachine->PTK = eapol_key_type2_gen_ptk(intdata,
						       (char *)&inkeydata->key_nonce);

  outkeydata->key_descriptor = WPA2_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);

  keyflags = 0;
  keyflags = (value16 & WPA2_KEYTYPE_MASK);
  version = keyflags;
  wpa2_pairwise_key_ver = version;
  keyflags |= (WPA2_PAIRWISE_KEY | WPA2_KEY_MIC_FLAG);
  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa2_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);

  cardif_get_wpa2_ie(intdata, wpa_ie, &ielen);

  memcpy(&intdata->sendframe[OFFSET_TO_EAPOL+4+sizeof(struct wpa2_key_packet)], 
	 &wpa_ie, ielen);
  value16 = ielen;
  value16 = htons(value16);
  intdata->send_size += ielen;

  outkeydata->key_material_len = value16;

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe);

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);

  // Dump what we built.
  eapol_key_type2_dump(intdata, (char *) intdata->sendframe);
}

/********************************************************
 *
 * Handle the third packet in the 4 way handshake.  We should be able to
 * generate the pairwise key at this point.
 *
 ********************************************************/
void eapol_key_type2_do_type3(struct interface_data *intdata)
{
  struct wpa2_key_packet *inkeydata, *outkeydata;
  uint8_t *keydata, *aesval;
  uint16_t keyflags, len, value16, keylen = 0;
  int eapolver, version = 0, encdata = 0;
  char key[32], rc4_ek[32];
  uint8_t framecpy[1530];
  int framesize;
  char zeros[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  bzero(key, 32);

  inkeydata = (struct wpa2_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];
  outkeydata = (struct wpa2_key_packet *)&intdata->sendframe[OFFSET_TO_EAPOL+4];

  // First, make sure that the inkeydata replay counter is higher than
  // the last counter we saw.
  if ((memcmp(inkeydata->key_replay_counter, 
	      intdata->statemachine->replay_counter, 8) <= 0) && 
      (memcmp(inkeydata->key_replay_counter, zeros, 8) != 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid replay counter!  Discarding!\n");
      intdata->recv_size = 0;
      return;
    }

  // Update our replay counter.
  memcpy(intdata->statemachine->replay_counter, inkeydata->key_replay_counter,
	 8);

  // Clear everything out.
  bzero(&intdata->sendframe[OFFSET_TO_EAPOL], 
	sizeof(struct wpa2_key_packet)+280);

  outkeydata->key_descriptor = WPA2_KEY_TYPE;

  memcpy(&value16, inkeydata->key_information, 2);
  value16 = ntohs(value16);
  
  keyflags = 0;
  keyflags = (value16 & WPA2_KEYTYPE_MASK);
  version = keyflags;  

  keyflags = 0;
  keyflags = (value16 & WPA2_KEYTYPE_MASK);
  keyflags |= (WPA2_PAIRWISE_KEY | WPA2_KEY_MIC_FLAG);

  // If the authenticator sets the secure flag, we need to do the same.
  if (value16 & WPA2_SECURE_FLAG) keyflags |= WPA2_SECURE_FLAG;
  if (value16 & WPA2_ENCRYPTED_DATA) encdata = 1;


  keyflags = htons(keyflags);

  memcpy(&outkeydata->key_information, &keyflags, 2);
  
  len = sizeof(struct wpa2_key_packet);
  intdata->send_size = len+OFFSET_TO_EAPOL+4;

  outkeydata->key_length = inkeydata->key_length;

  memcpy(&outkeydata->key_replay_counter, &inkeydata->key_replay_counter, 8);
  memcpy(&outkeydata->key_nonce, intdata->statemachine->SNonce, 32);

  memcpy(outkeydata->key_nonce, intdata->statemachine->SNonce,
	 32);

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_KEY, (intdata->send_size-OFFSET_TO_EAPOL-4), 
		     eapolver, (char *) intdata->sendframe); 

  memcpy(key, intdata->statemachine->PTK, 16);
  mic_wpa_populate((char *) intdata->sendframe, intdata->send_size+4, key, 16);  

  // Dump what we built.
  eapol_key_type2_dump(intdata, (char *) intdata->sendframe);

  // Get TK1
  value16 = ntohs(inkeydata->key_length);
  memcpy(key, (char *)&intdata->statemachine->PTK[32], value16);
  
  debug_printf(DEBUG_INT, "TK1 : ");
  debug_hex_printf(DEBUG_INT, (uint8_t *) key, value16);
  
  // We need to send the result frame now, but that will result in the
  // "sendframe" being zeroed out.  So, we need to make a copy.
  framesize = intdata->recv_size;
  memcpy(&framecpy, &intdata->recvframe, framesize);
  inkeydata = (struct wpa2_key_packet *)&framecpy[OFFSET_TO_EAPOL+4];

  cardif_sendframe(intdata);
  intdata->statemachine->eapolEap = FALSE;
  intdata->send_size = 0;

  keylen = ntohs(inkeydata->key_material_len);

  eapol_key_type2_dump(intdata, (char *) &framecpy);

  if ((encdata) && (keylen > 0))
    {
      keydata = (uint8_t *)malloc(keylen);

      memcpy(keydata, inkeydata->keydata, keylen);
      
      switch (version)
	{
	case 1:
	  bzero(rc4_ek, 32);
	  memcpy(rc4_ek, inkeydata->key_iv, 16);
	  memcpy(&rc4_ek[16], &intdata->statemachine->PTK[16], 16);
	  rc4_skip((uint8_t *) rc4_ek, 32, 256, keydata, keylen);
	  break;
	  
	case 2:
	  aesval = (uint8_t *)malloc(keylen);

	  bzero(aesval, keylen);

	  if (aes_unwrap((uint8_t *) &intdata->statemachine->PTK[16], 
			 ((keylen)/8)-1, keydata, aesval) != 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed AES unwrap!\n");
	      return;
	    } else {
	      free(keydata);
	      keydata = aesval;
	    }
	  break;
	  
	default:
	  debug_printf(DEBUG_NORMAL, "Unknown version ID! (Version = %d)\n",
		       version);
	  return;
	  break;
	}

      debug_printf(DEBUG_INT, "Keydata (%d) : \n", keylen);
      debug_hex_dump(DEBUG_INT, keydata, keylen);

      if (eapol_key_type2_process_keydata(intdata, keydata, keylen,
					  ntohs(inkeydata->key_length), 
					  inkeydata->key_rsc, version)
	  != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Error processing key data!\n");
	  return;
	}
    }

  if (network_data->methods->method_num == WPA_PSK)
    {
      // If we are using PSK, and we made it here, then we are in 
      // AUTHENTICATED state.
      intdata->statemachine->eapSuccess = TRUE;
      cardif_drop_unencrypted(intdata, FALSE);
    } else {
      // Drop unencrypted frames.
      cardif_drop_unencrypted(intdata, TRUE);
    }
  
  debug_printf(DEBUG_INT, "Setting PTK of length %d with index 0.\n",
	       value16);
  
  switch (version) 
    {
    case 1:
      wpa_common_set_key(intdata, intdata->dest_mac, 0, TRUE,
			 key, value16);
      break;

    case 2:
      wpa_common_set_key(intdata, intdata->dest_mac, 0, TRUE,
			 key, value16);
      break;
    }
  debug_printf(DEBUG_NORMAL, "Set new pairwise IEEE 802.11i/WPA2 key.\n");
}

void eapol_key_type2_determine_key(struct interface_data *intdata)
{
  struct wpa2_key_packet *keydata;
  int keyflags, eapolver;
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  keydata = (struct wpa2_key_packet *)&intdata->recvframe[OFFSET_TO_EAPOL+4];  
  memcpy(&keyflags, keydata->key_information, 2);

  keyflags = ntohs(keyflags);

  if (keyflags & WPA2_KEY_MIC_FLAG)
    {
      if (mic_wpa_validate((char *) intdata->recvframe, intdata->recv_size, 
			   intdata->statemachine->PTK, 16) == FALSE)
	{
	  intdata->statemachine->MICVerified = TRUE;
	  intdata->statemachine->IntegrityFailed = TRUE;
	  debug_printf(DEBUG_INT, "MIC failure!\n");
       	  return;   // Silently discard.
	} else {
	  intdata->statemachine->IntegrityFailed = FALSE;
	  intdata->statemachine->MICVerified = TRUE;
	}
    }
  if ((keyflags & WPA2_PAIRWISE_KEY) && (keyflags & WPA2_KEY_ACK_FLAG) &&
      (keyflags & WPA2_KEY_MIC_FLAG) && (keyflags & WPA2_INSTALL_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #3 (response) :\n");
      eapol_key_type2_do_type3(intdata);
    } 
  else if ((keyflags & WPA2_PAIRWISE_KEY) && (keyflags & WPA2_KEY_ACK_FLAG))
    {
      debug_printf(DEBUG_INT, "Key Packet #1 (response) :\n");
      eapol_key_type2_do_type1(intdata);
    } 
  else if ((keyflags & WPA2_KEY_MIC_FLAG) && (keyflags & WPA2_PAIRWISE_KEY))
    {
      debug_printf(DEBUG_NORMAL, "Got Key Packet #2!  (This shouldn't happen!"
		   ")\n");
      return;
    } 
  else if ((keyflags & WPA2_SECURE_FLAG) && 
	   (keyflags & WPA2_KEY_MIC_FLAG) && (keyflags & WPA2_PAIRWISE_KEY))
    {
      debug_printf(DEBUG_NORMAL, "Got Key Packet #4!  (This shouldn't happen"
		   "!)\n");
      return;
    } else if (!(keyflags & WPA2_PAIRWISE_KEY))
       {
	  // We have a group key packet.
	  eapol_key_type2_do_gtk(intdata);
       }

  if (intdata->recv_size > 0)
    {
      eapolver = eapol_get_eapol_ver();

      eapol_build_header(EAPOL_KEY, (intdata->recv_size-OFFSET_TO_EAPOL-4), 
			 eapolver, (char *) intdata->recvframe);
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }      
}

/*************************************************************
 *
 * Process a WPA frame that we get from the authenticator.
 *
 *************************************************************/
void eapol_key_type2_process(struct interface_data *intdata)
{
  struct config_wpa_psk *psk;
  uint8_t *inframe;
  int insize;
  char tpmk[256];
  struct config_network *network_data;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  inframe = intdata->recvframe;
  insize = intdata->recv_size;

  debug_printf(DEBUG_INT, "Processing WPA2 key message!\n");

  eapol_key_type2_dump(intdata, (char *) inframe);

  if (intdata->statemachine->PMK == NULL)
    {
      if (network_data->methods->method_num == WPA_PSK)
	{
	  psk = (struct config_wpa_psk *)network_data->methods->method_data;
	  if (psk->key)
	    {
	      if (intdata->cur_essid == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Unknown SSID, checking!\n");
		  intdata->cur_essid = (char *)malloc(99);
		  if (intdata->cur_essid == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! "
				   "(%s:%d)\n", __FUNCTION__, __LINE__);
		      return;
		    }

		  memset(intdata->cur_essid, 0x00, 99);

		  if (cardif_GetSSID(intdata, intdata->cur_essid) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't get ESSID!\n");
		      free(intdata->cur_essid);
		      return;
		    }
		}

	      // We have an ASCII password, so calculate it.
	      if (psk_wpa_pbkdf2(psk->key, (uint8_t *) intdata->cur_essid, 
				 strlen(intdata->cur_essid), (uint8_t *)&tpmk) 
		  == TRUE)
		{
		  if (intdata->statemachine->PMK)
		    {
		      free(intdata->statemachine->PMK);
		      intdata->statemachine->PMK = NULL;
		    }

		  intdata->statemachine->PMK = (char *)malloc(32);
		  if (intdata->statemachine->PMK == NULL)
		    {
		      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for"
				   " intdata->statemachine->PMK in %s:%d!\n",
				   __FUNCTION__, __LINE__);
		      return;
		    }
		  
		  memcpy(intdata->statemachine->PMK, (char *)&tpmk, 32);
		}
	    } else {
	      // We have a hex key, we need to convert it from ASCII to real
	      // hex.
	      if (strlen(psk->hex_key) != 64)
		{
		  debug_printf(DEBUG_NORMAL, "Invalid HEX key defined for "
			       "WPA2-PSK!\n");
		  return;
		}
	      process_hex(psk->hex_key, strlen(psk->hex_key), (char *)&tpmk);
	      intdata->statemachine->PMK = (char *)malloc(32);
	      if (intdata->statemachine->PMK == NULL)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for "
			       "intdata->statemachine->PMK in %s:%d!\n",
			       __FUNCTION__, __LINE__);
		  return;
		}
	    }
	} else {
	  debug_printf(DEBUG_NORMAL, "There is no PMK available!  WPA2 cannot"
		       " continue!\n");
	  return;
	}
    } 

  eapol_key_type2_determine_key(intdata);

  if (intdata->send_size > 0)
    {
      cardif_sendframe(intdata);
      intdata->statemachine->eapolEap = FALSE;
    }
}
