/*******************************************************************
 * Support for Trusted Network Connect EAP method
 * File: eaptnc.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eaptnc.c,v 1.2 2006/06/13 21:16:27 chessing Exp $
 * $Date: 2006/06/13 21:16:27 $
 * $Log: eaptnc.c,v $
 * Revision 1.2  2006/06/13 21:16:27  chessing
 * Updates to the new EAP-TNC code.
 *
 * Revision 1.1  2006/06/13 18:02:57  chessing
 * Fixed a problem that would result in a segfault if a HUP was received.  Created stub functions for implementation of EAP-TNC to support Trusted Network Connect.
 *
 *
 *******************************************************************/

#ifdef HAVE_TNC
#include <string.h>
#include <strings.h>
#include <stdlib.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "profile.h"
#include "xsupconfig.h"   
#include "eap.h"
#include "eaptnc.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

/*****************************************************
 *
 * Setup to handle TNC EAP requests
 *
 *****************************************************/
int eaptnc_setup(struct generic_eap_data *thisint)
{
  // Do anything special that might be needed for this EAP type to work.
  debug_printf(DEBUG_EVERYTHING, "Initalized EAP-TNC!\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  // Nothing to do here.

  return XENONE;
}

/*****************************************************
 *
 * Verify that we have a valid version, and flags.
 *
 *****************************************************/
int eaptnc_check_flags_ver(uint8_t flagsver)
{
  int retval = XENONE;

  // We don't want to terminate when we find an error.  Instead, we want
  // to kick back error message for each part of the check that fails.
  if ((flagsver & TNC_VERSION_MASK) > TNC_MAX_VERSION_SUPPORTED)
    {
      debug_printf(DEBUG_NORMAL, "Invalid version number %d!  We currently "
		   "only support up to version %d.\n", 
		   (flagsver & TNC_VERSION_MASK), TNC_MAX_VERSION_SUPPORTED);
      retval = XEINVALIDFLAGSVER;
    }

  if ((flagsver & TNC_RESERVED_FLAGS) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Invalid flags set for EAP-TNC!  Something "
		   "was set to 1 in bit positions 4 and/or 5.\n");
      retval = XEINVALIDFLAGSVER;
    }

  if ((flagsver & TNC_LENGTH_FLAG) != 0)
    {
      debug_printf(DEBUG_NORMAL, "The length flag was set coming from the "
		   "server.  But, there is no state in the standard that "
		   "allows for this!\n");
      retval = XEINVALIDFLAGSVER;
    }

  if ((flagsver & TNC_MORE_DATA) != 0)
    {
      debug_printf(DEBUG_NORMAL, "The 'more data' flag was set coming from "
		   "the server.  But, these is no state in the standard that "
		   "allows for this!\n");
      retval = XEINVALIDFLAGSVER;
    }

  return retval;
}

/*******************************************************
 *
 * Execute a TNC start.
 *
 *******************************************************/
int eaptnc_do_start()
{
  if (libtnc_tncc_BeginSession(0) != TNC_RESULT_SUCCESS)
    {
      debug_printf(DEBUG_NORMAL, "(EAP-TNC) Failed to start TNC session!\n");
      return XETNCLIBFAILURE;
    }

  debug_printf(DEBUG_NORMAL, "(EAP-TNC) Started IMC Handshake.\n");
  return XENONE;
}

/*******************************************************
 *
 * Process a bulk data TNC request.
 *
 *******************************************************/
int eaptnc_do_bulk_data(uint8_t *indata, int insize, uint8_t *outdata,
			int *outsize)
{
  int retval = XENONE;


  return retval;
}

/*******************************************************
 *
 * Process a version 1 TNC request.
 *
 *******************************************************/
int eaptnc_process_version_1(uint8_t *indata, int insize, uint8_t *outdata,
			     int *outsize)
{
  int retval = XENONE;
  uint8_t *indataptr = indata;

  if (indata[0] & TNC_START_FLAG)
    {
      // Process a start message.
      debug_printf(DEBUG_AUTHTYPES, "(EAP-TNC) Process Start\n");
      retval = eaptnc_do_start();
    }
  else
    {
      // Process a bulk data frame.
      debug_printf(DEBUG_AUTHTYPES, "(EAP-TNC) Process bulk data\n");
      retval = eaptnc_do_bulk_data(indata, insize, outdata, outsize);
    }

  return retval;
}

/*****************************************************
 *
 * Process TNC EAP Requests
 *
 ******************************************************/
int eaptnc_process(struct generic_eap_data *thisint, uint8_t *dataoffs, 
		   int insize, uint8_t *outframe, int *outsize)
{
  int retval = XENONE;
  //  struct config_eap_md5 *userdata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_data != NULL), "thisint->eap_data != NULL",
		   FALSE)) return XEMALLOC;

  if (!xsup_assert((dataoffs != NULL), "dataoffs != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outframe != NULL), "outframe != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outsize != NULL), "outsize != NULL", FALSE))
    return XEMALLOC;

  // The code, identifier, length, and type should be stripped at this point.
  // So start by validating the Flags/Ver byte, and process from there.
  retval = eaptnc_check_flags_ver(outframe[0]);
  if (retval != XENONE) return retval;

  // This is overkill for now, but since there is a version, we implement
  // this now, so we can expand it later.
  switch ((outframe[0] & TNC_VERSION_MASK))
    {
    case 1:
      // Process a version 1 request.
      retval = eaptnc_process_version_1(dataoffs, insize, outframe, outsize);
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown TNC version in request!\n");
      break;
    }

  return retval;
}

/*******************************************************
 *
 * Clean up after ourselves.  This will get called when we get a packet that
 * needs to be processed requests a different EAP type.  It will also be 
 * called on termination of the program.
 *
 *******************************************************/
int eaptnc_cleanup(struct generic_eap_data *thisint)
{
  // Clean up after ourselves.

  debug_printf(DEBUG_AUTHTYPES, "(EAP-TNC) Cleaning up.\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (thisint->eap_data != NULL)
    {
      free(thisint->eap_data);
      thisint->eap_data = NULL;
    }

  return XENONE;
}

/*******************************************************
 *
 * If we fail an authentication, we will call this routine.  It should clean
 * up anything that shouldn't live in to the next authentication attempt.
 *
 *******************************************************/
int eaptnc_failed(struct generic_eap_data *thisint)
{
  //  struct config_eap_md5 *userdata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_conf_data != NULL),
		   "thisint->eap_conf_data != NULL", FALSE))
    return XEMALLOC;

  return XENONE;
}


#endif /* HAVE_TNC */
