#include <common.h>
#include <linux/byteorder/swab.h>
#include <asm/arch/hardware.h>

flash_info_t flash_info[CONFIG_SYS_MAX_FLASH_BANKS];	/* info for FLASH chips    */
unsigned long sector_size;

#define FLASH_ID_NUYX		0x00890000
#define FLASH_MAN_NUYX		0x0089
#define FLASH_M29EW			0x227E
#define FLASH_M29EW_256Mb	0x2222	/* 32MB */
#define FLASH_M29EW_512Mb	0x2223	/* 64MB */
#define FLASH_M29EW_1Gb		0x2228	/* 128MB */
#define FLASH_M29EW_2Gb		0x2248	/* 256MB */

#define CONFIG_SYS_FLASH_ERASE_TOUT	250
#define CMD_AA					0xAA
#define CMD_90					0x90
#define CMD_55					0x55
#define CMD_80					0x80
#define CMD_10					0x10
#define CMD_30					0x30
#define CMD_A0					0xA0
#define CMD_PROG_RESET			0xf0
#define PHYS_FLASH_1	CONFIG_SYS_FLASH_BASE
#define ErasedWord			0xffffffff
/* Board support for 1 device */
#define FLASH_PORT_WIDTH8

#ifdef FLASH_PORT_WIDTH8
#define FLASH_PORT_WIDTH		uchar
#define FLASH_PORT_WIDTHV		vu_char
#endif

#define FPW	   FLASH_PORT_WIDTH
#define FPWV   FLASH_PORT_WIDTHV
#define mb() __asm__ __volatile__ ("" : : : "memory")


volatile FPW * nor = PHYS_FLASH_1;

static void flash_get_offsets (ulong base, flash_info_t * info);
void inline spin_wheel (void);
/*-----------------------------------------------------------------------
 * Functions
 */
unsigned long flash_init (void);
static ulong flash_get_size (FPW * addr, flash_info_t * info);
void flash_print_info (flash_info_t * info);
void flash_unprotect_sectors (FPWV * addr);
int flash_erase (flash_info_t * info, int s_first, int s_last);
int write_buff (flash_info_t * info, uchar * src, ulong addr, ulong cnt);
//static int write_data (flash_info_t * info, ulong dest, FPW data);


unsigned long flash_init (void)
{
	int i;
	u32 size = 0;

	// switch MFS to SMC
	A1_set_pin_mux (HAL_MFP_SMC);

	for (i = 0; i < CONFIG_SYS_MAX_FLASH_BANKS; i++) {
		switch (i) {
		case 0:
			flash_get_size ((FPW *) PHYS_FLASH_1, &flash_info[i]);
			flash_get_offsets (PHYS_FLASH_1, &flash_info[i]);
			break;
		default:
			panic ("configured too many flash banks!\n");
			break;
		}
		size += flash_info[i].size;
	}

	/* Protect monitor and environment sectors
	 */
//	flash_protect (FLAG_PROTECT_SET,
//			CONFIG_SYS_FLASH_BASE,
//			CONFIG_SYS_FLASH_BASE + monitor_flash_len - 1, &flash_info[0]);
//
//	flash_protect (FLAG_PROTECT_SET,
//			CONFIG_ENV_ADDR,
//			CONFIG_ENV_ADDR + CONFIG_ENV_SIZE - 1, &flash_info[0]);

	return size;
}

/*-----------------------------------------------------------------------
 */
static void flash_get_offsets (ulong base, flash_info_t * info)
{
	int i;

	if (info->flash_id == FLASH_UNKNOWN) {
		return;
	}

	if ((info->flash_id & FLASH_VENDMASK) == FLASH_ID_NUYX) {
		for (i = 0; i < info->sector_count; i++) {
				info->start[i] = base + (i * sector_size);
				info->protect[i] = 0;
		}
	}
}

/*-----------------------------------------------------------------------
 */
void flash_print_info (flash_info_t * info)
{
	int i;

	// switch MFS to SMC
	A1_set_pin_mux (HAL_MFP_SMC);

	if (info->flash_id == FLASH_UNKNOWN) {
		printf ("missing or unknown FLASH type\n");
		return;
	}

	switch (info->flash_id & FLASH_VENDMASK) {
	case FLASH_MAN_NUYX:
		printf ("Numonyx ");
		break;
	default:
		printf ("Unknown Vendor ");
		break;
	}

	switch (info->flash_id & FLASH_TYPEMASK) {
	case FLASH_M29EW_256Mb:
		printf ("FLASH M29EWHA\n");
		break;
	default:
		printf ("Unknown Chip Type\n");
		break;
	}

	printf ("  Size: %ld MB in %d Sectors\n",
			info->size >> 20, info->sector_count);

	printf ("  Sector Start Addresses:");
	for (i = 0; i < info->sector_count; ++i) {
		if ((i % 5) == 0)
			printf ("\n   ");
		printf (" %08lX%s",
			info->start[i], info->protect[i] ? " (RO)" : "     ");
	}
	printf ("\n");
	return;
}
/*
 * The following code cannot be run from FLASH!
 */
static ulong flash_get_size (FPW * addr, flash_info_t * info)
{
	volatile FPW value;

	/* Write auto select command: read Manufacturer ID */
	nor[0xAAA] = (FPW) 0xAA;
	nor[0x555] = (FPW) 0x55;
	nor[0xAAA] = (FPW) 0x90;

	mb ();
	value = addr[0];

	switch (value) {

	case (FPW) FLASH_MAN_NUYX:
		info->flash_id = FLASH_ID_NUYX;	//0x89
		break;

	default:
		info->flash_id = FLASH_UNKNOWN;
		info->sector_count = 0;
		info->size = 0;
		addr[0] = (FPW) 0xFF;	/* restore read mode */
		return (0);		/* no or unknown flash  */
	}

	mb ();
	value = addr[1<<1];	/* device ID        */
	switch (value) {

	case (FPW) FLASH_M29EW:
		info->flash_id += FLASH_M29EW;
		{
			value = addr[14<<1];

			switch(value) {
			case (FPW)	FLASH_M29EW_256Mb:	
				info->sector_count = 256;
				info->size = 0x02000000;
				break;
			case (FPW)	FLASH_M29EW_512Mb:
				info->sector_count = 512;
				info->size = 0x04000000;
				break;
			case (FPW)	FLASH_M29EW_1Gb:
				info->sector_count = 1024;
				info->size = 0x08000000;
				break;
			case (FPW)	FLASH_M29EW_2Gb:
				info->sector_count = 2048;
				info->size = 0x10000000;
				break;
			}
			sector_size = 0x20000;
		}
		break;			

	default:
		info->flash_id = FLASH_UNKNOWN;
		break;
	}

	if (info->sector_count > CONFIG_SYS_MAX_FLASH_SECT) {
		printf ("** ERROR: sector count %d > max (%d) **\n",
				info->sector_count, CONFIG_SYS_MAX_FLASH_SECT);
		info->sector_count = CONFIG_SYS_MAX_FLASH_SECT;
	}

	addr[0] = (FPW) 0x00F000F0;	/* restore read mode */

	return (info->size);
}

/* unprotects a sector for write and erase
 * on some intel parts, this unprotects the entire chip, but it
 * wont hurt to call this additional times per sector...
 */
//void flash_unprotect_sectors (FPWV * addr)
//{
//#define PD_FINTEL_WSMS_READY_MASK    0x0080
//
//	*addr = (FPW) 0x00500050;	/* clear status register */
//
//	/* this sends the clear lock bit command */
//	*addr = (FPW) 0x00600060;
//	*addr = (FPW) 0x00D000D0;
//}

/*-----------------------------------------------------------------------
 *  Description: Erase Flash by sector unit.
 *  flash_info_t: Flash info struct.
 *  s_first:      Start sector index. 
 *  s_last:       Last sector index.
 */
int flash_erase (flash_info_t * info, int s_first, int s_last)
{
	int flag, prot, sect;
	ulong type, start, last;
	int rcode = 0;

	if ((s_first < 0) || (s_first > s_last)) {
		if (info->flash_id == FLASH_UNKNOWN) {
			printf ("- missing\n");
		} else {
			printf ("- no sectors to erase\n");
		}
		return 1;
	}

	type = (info->flash_id & FLASH_VENDMASK);
	if ((type != FLASH_ID_NUYX)) {
		printf ("Can't erase unknown flash type %08lx - aborted\n",
				info->flash_id);
		return 1;
	}

	prot = 0;
	for (sect = s_first; sect <= s_last; ++sect) {
		if (info->protect[sect]) {
			prot++;
		}
	}

	if (prot) {
		printf ("- Warning: %d protected sectors will not be erased!\n",
				prot);
	} else {
		printf ("\n");
	}


	// switch MFS to SMC
	A1_set_pin_mux (HAL_MFP_SMC);

	nor[0xAAA] = (FPW) 0x00F000F0;/* resest to read mode */
	udelay(50);
	/* Disable interrupts which might cause a timeout here */
	flag = disable_interrupts ();

	/* Start erase on unprotected sectors */
	for (sect = s_first; sect <= s_last; sect++) {
		if (info->protect[sect] == 0) { /* not protected */
			FPWV *addr = (FPWV *) (info->start[sect]);
			FPW status;

			printf ("Erasing sector %2d ... ", sect);

			//flash_unprotect_sectors (addr);
			nor[0xAAA] = (FPW) 0xAA;
			nor[0x555] = (FPW) 0x55;
			nor[0xAAA] = (FPW) 0x80;
			nor[0xAAA] = (FPW) 0xAA;
			nor[0x555] = (FPW) 0x55;
			*addr = (FPW) 0x30;

			udelay(200);

			/* clear status register cmd.	*/
			nor[0xAAA] = (FPW) 0x00F000F0;/* resest to read mode */
			udelay(50);
			printf (" done\n");
		}
	}
	return rcode;
}


/*
 * Program Flash
 * 
 */
int flash_byte(ulong addr, u8 data)
{
	int i;

	// 4 Cycles
	nor[0xAAA] = (FPW) 0xAA;
	nor[0x555] = (FPW) 0x55;
	nor[0xAAA] = (FPW) 0xA0;
	*((volatile u8 *)(addr)) = data;

	// verify
	for (i = 0; i < 50; i++) {
		udelay(1);
		if (*((volatile u8 *)(addr)) == data)
			break;
	}

	if (i == 50) {
		printf("error at program byte at addr %08lx\n", addr);
		return ERR_PROG_ERROR;
	}
	return ERR_OK;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash, returns:
 * info: Flash info struct.
 * src : Source buffer.
 * bs: Base section num.
 * cnt:  Data length.
 * return: 0 - OK 1 - write timeout 2 - Flash not erased 4 - Flash not identified
 */
int write_buff (flash_info_t *info, uchar *src, ulong bs, ulong cnt)
{
	ulong i = 0, tmpaddr = 0 , tmpdata = 0;
  	u32 j;
	int rc = 0;

	// switch MFS to SMC
	A1_set_pin_mux (HAL_MFP_SMC);

	tmpaddr = info[0].start[bs];

	//Check every 4 bytes from aligned data address.
	for (i = 0; i < cnt; i += 4 ) {
		tmpdata = *((volatile u32 *)tmpaddr);
		if(tmpdata != ErasedWord) {
			printf ("Error at write_buff at address %lx, not erased !!\n", tmpaddr);
			return ERR_NOT_ERASED;
		}
		if (cnt - i > 4) {
			rc = flash_byte(tmpaddr++, *(src++));
			rc |= flash_byte(tmpaddr++, *(src++));
			rc |= flash_byte(tmpaddr++, *(src++));
			rc |= flash_byte(tmpaddr++, *(src++));
			spin_wheel();
		} 
		else {
			 for ( j = 0; j < cnt - i; j++ ) {
				rc = flash_byte(tmpaddr++, *(src++));
			 }
		}
		if (rc)
			return (rc);
	}

	return rc;
}

void inline spin_wheel (void)
{
	static int p = 0;
	static char w[] = "\\/-";

	printf ("\010%c", w[p]);
	if ( ++p == 3)
			p = 0;
}
