#include <common.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>

/* the number of clocks per CONFIG_SYS_HZ */
#define CONFIG_SYS_HZ_CLOCK	(CONFIG_SYS_CLK_FREQ / 2)
//#define TIMER_LOAD_VAL		CONFIG_SYS_HZ_CLOCK
#define TIMER_LOAD_VAL		0xFFFFFFFF
#define READ_TIMER (*(volatile ulong *)(A1_TIMER_BASE + 0x0))

static ulong timestamp = 0;
static ulong lastdec = 0;

int EnableTimer(int timer_id,
				unsigned int nMatch1,
				unsigned int nMatch2,
				unsigned int nNumPClks)
{
	unsigned int nRegValue;

	if (nMatch1 > nNumPClks || nMatch2 > nNumPClks)
		return (-1);

	switch (timer_id)
	{
		case 1:
			writel(nNumPClks, A1_TIMER_BASE + 0x0);
			writel(nNumPClks, A1_TIMER_BASE + 0x4);
			writel(nMatch1, A1_TIMER_BASE + 0x8);
			writel(nMatch2, A1_TIMER_BASE + 0xc);

			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFFFF0;
			nRegValue |= 1;
			writel(nRegValue, A1_TIMER_BASE + 0x30);
			break;

		case 2:
			writel(nNumPClks, A1_TIMER_BASE + 0x10);
			writel(nNumPClks, A1_TIMER_BASE + 0x14);
			writel(nMatch1, A1_TIMER_BASE + 0x18);
			writel(nMatch2, A1_TIMER_BASE + 0x1c);

			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFFF0F;
			nRegValue |= (1 << 4);
			writel(nRegValue, (A1_TIMER_BASE + 0x30));
			break;

		case 3:
			writel(nNumPClks, A1_TIMER_BASE + 0x20);
			writel(nNumPClks, A1_TIMER_BASE + 0x24);
			writel(nMatch1, A1_TIMER_BASE + 0x28);
			writel(nMatch2, A1_TIMER_BASE + 0x2c);

			/* fill 0101 to bit3~bit0 */
			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFF0FF;
			nRegValue |= (1 << 8);
			writel(nRegValue, A1_TIMER_BASE + 0x30);
			break;

	default:
		return (-1);
	}

	return (0);
}

void DisableTimer(int timer_id)
{
	unsigned int nRegValue;
	switch (timer_id)
	{
		case 1:
			writel(0, A1_TIMER_BASE + 0x0);
			writel(0, A1_TIMER_BASE + 0x4);
			writel(0, A1_TIMER_BASE + 0x8);
			writel(0, A1_TIMER_BASE + 0xc);

			/* fill 0101 to bit3~bit0 */
			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFFFF0;
			writel(nRegValue, A1_TIMER_BASE + 0x30);
			break;

		case 2:
			writel(0, A1_TIMER_BASE + 0x10);
			writel(0, A1_TIMER_BASE + 0x14);
			writel(0, A1_TIMER_BASE + 0x18);
			writel(0, A1_TIMER_BASE + 0x1c);

			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFFF0F;
			writel(nRegValue, A1_TIMER_BASE + 0x30);
			break;

		case 3:
			writel(0, A1_TIMER_BASE + 0x20);
			writel(0, A1_TIMER_BASE + 0x24);
			writel(0, A1_TIMER_BASE + 0x28);
			writel(0, A1_TIMER_BASE + 0x2c);

			nRegValue = readl(A1_TIMER_BASE + 0x30);
			nRegValue &= 0xFFFFF0FF;
			writel(nRegValue, A1_TIMER_BASE + 0x30);
			break;
	}
}

int timer_init (void)
{
	int32_t val;
	lastdec = TIMER_LOAD_VAL;

	/* Start the decrementer ticking down from 0xffffffff */
	writel(0x0, A1_TIMER_BASE + 0x0);
	writel(TIMER_LOAD_VAL, A1_TIMER_BASE + 0x4);
	writel(0x0, A1_TIMER_BASE + 0x8);
	writel(0x0, A1_TIMER_BASE + 0xc);

	val = readl(A1_TIMER_BASE + 0x30);
	val = (val & ~0xFL) | 0x1;

	writel(val, A1_TIMER_BASE + 0x30);

	/* init the timestamp and lastdec value */
//	reset_timer_masked();

	return 0;
}

/*
 * timer without interrupts
 */

void reset_timer (void)
{
	reset_timer_masked();	
}

ulong get_timer (ulong base)
{
	return get_timer_masked() - base;
}

void set_timer (ulong t)
{
	timestamp = t;	
}

void __udelay (unsigned long usec)
{
	udelay_masked(usec);
}

void reset_timer_masked (void)
{
	/* reset time */
	lastdec = READ_TIMER;
	timestamp = 0;
}

ulong get_timer_raw (void)
{
	ulong now = READ_TIMER;
	if (lastdec >= now) {
		/* normal mode */
		timestamp += lastdec - now;
	} else {
		/* we have an overflow ..... */
		timestamp += lastdec + TIMER_LOAD_VAL - now;
	}
	lastdec = now;

	return timestamp;
}

ulong get_timer_masked (void)
{
	return get_timer_raw()/ (CONFIG_SYS_HZ_CLOCK/CONFIG_SYS_HZ);
}

///* waits specified delay value and resets timestamp */
void udelay_masked (unsigned long usec)
{
	ulong tmo;
	ulong endtime;
	signed long diff;

	tmo = CONFIG_SYS_HZ_CLOCK / 1000;
	tmo *= usec;
	tmo /= 1000;

	endtime = get_timer_raw () + tmo;

	do {
		ulong now = get_timer_raw ();
		diff = endtime - now;
	} while (diff >= 0);
}




/*
 * This function is derived from PowerPC code (read timebase as long long).
 * On ARM it just returns the timer value.
 */
unsigned long long get_ticks(void)
{
	return get_timer(0);
}

/*
 * This function is derived from PowerPC code (timebase clock frequency).
 * On ARM it returns the number of timer ticks per second.
 */
ulong get_tbclk (void)
{
	return CONFIG_SYS_HZ;
}

