#include <asm/arch/nand/nf_controller.h>
#include <asm/arch/nand/nf_error.h>
#include <asm/arch/nand/nf_setup.h>
#include <asm/arch/nand/nf_utils.h>
#include <asm/arch/nand/nf_dma.h>
#include <asm/arch/nand/nf_types.h>
#include <asm/arch/nand/nf_memory.h>

// command using in all below functions
void dump_pcontroller(nf_controller_t *pcontroller);
void dump_nf_memory(nf_memory_t * nfmem);
uint8_t wait_until_target_busy(nf_memory_t *memory);

static nf_ctrl_command_t g_command;

/******************************************************************************/
static uint8_t nf_is_write_protected(nf_memory_t *memory)
{
    if (NF_IS_WP_ENABLED(memory->pcontroller)){
        return NF_ERR_MEMORY_WRITE_PROTECTED;
    }

    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/

/******************************************************************************/
static uint8_t nf_check_operation_status(nf_memory_t *memory, 
                                         uint8_t operation_type)
{
    uint8_t status, error_code, result = NF_ERR_NO_ERRORS;

    nf_controller_t *pcontroller = memory->pcontroller;

    // read controller
    uint32_t ecc_status;   

    switch(operation_type){
    case NF_OPERATION_TYPE_WRITE:
        // read status from device
        error_code = nf_mem_read_status(memory, &status);
        if (error_code) {
			printf( "[%s - NF_OPERATION_TYPE_WRITE] error_code = %d\n", __FUNCTION__, error_code);
            return error_code;
		}

        switch(status){
        case NF_MEMORY_STATUS_FAIL:
			printf( "[%s - NF_OPERATION_TYPE_WRITE]  NF_MEMORY_STATUS_FAIL\n", __FUNCTION__);
            result = NF_ERR_CURR_PAGE_OP_ERR;
            break;
        case NF_MEMORY_STATUS_FAILC:
			printf( "[%s - NF_OPERATION_TYPE_WRITE]  NF_MEMORY_STATUS_FAILC\n", __FUNCTION__);
            result = NF_ERR_PRIOR_PAGE_OP_FAIL;
            break;
        case NF_MEMORY_STATUS_FAIL | NF_MEMORY_STATUS_FAILC:       
			printf( "[%s - NF_OPERATION_TYPE_WRITE]  NF_MEMORY_STATUS_FAIL | NF_MEMORY_STATUS_FAILC\n", __FUNCTION__);
            result = NF_ERR_PRIOR_CURR_PAGE_OP_FAIL;
        }
        break;        

    case NF_OPERATION_TYPE_READ:
        if (NF_IS_ECC_ENABLED(pcontroller)){
			ecc_status =  IORD_32NF(pcontroller->reg_offset + NF_SFR_ECC_STAT);
            if (ecc_status & NF_SFR_ECC_CTR_ECC_ERR_UNCORRECT) {	// FIXME, try to revised define only !!, let default target is 0
				printf("=> %s:%d     NF_ERR_ECC_FATAL\n", __FILE__, __LINE__);
                result = NF_ERR_ECC_FATAL;
			}
            else if (ecc_status & NF_SFR_ECC_CTR_ECC_ERR_OVER) {	// FIXME, try to revised define only !!, let default target is 0
				printf("=> %s:%d     NF_SFR_ECC_CTR_ECC_ERR_OVER\n", __FILE__, __LINE__);
                result = NF_ERR_ECC_TRSH;
			}
        }
        break;

    case NF_OPERATION_TYPE_ERASE:
        // read status from device
        error_code = nf_mem_read_status(memory, &status);
        if (error_code) {
//			printf( "[%s - NF_OPERATION_TYPE_ERASE ] error_code = %d\n", __FUNCTION__, error_code);
            return error_code; 
		}

        // check if operation on current page succeed
        if (NF_MEMORY_STATUS_FAIL == status)   
            result = NF_ERR_CURR_PAGE_OP_ERR;
    }
    return result;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_select_target(nf_memory_t *memory)
{
    uint8_t status;
	uint8_t ecc_off;

    nf_controller_t *pcontroller =  memory->pcontroller;

    // set chip enable signal in controller
    nf_ctrl_select_target(pcontroller, memory->ce);

    // set page size in controller
    status = nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_PAGE_SIZE, 
                               memory->page_size);
    if (status)
        return status;

    // set block size in controller
    status = nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_BLOCK_SIZE, 
                               memory->pages_per_block);
    if (status)
        return status;

    // set spare area size in controller
    status = nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_SPARE_AREA_SIZE, 
                               memory->spare_area_size);
    if (status)
        return status;

    /* if memory mode is unknown state then first set controller 
     * to asynchronous mode 
     * and next execute reset command 
     * in other cases just set apriopriate controller work mode*/
    switch(memory->current_work_mode){
    case MEMORY_ASYNCHRONOUS_MODE:
        status = nf_ctrl_configure(memory->pcontroller, 
                                   NF_CONFIGURE_SET_ASYNCHRONOUS_MODE, 0);
        if (status)
            return status;
        break;
    default:
    case MEMORY_UNKNOWN_MODE:
        status = nf_ctrl_configure(memory->pcontroller, 
                                   NF_CONFIGURE_SET_ASYNCHRONOUS_MODE, 0);
        if (status)
            return status;
        status = nf_mem_reset(memory);
        if (status)
            return status;
        memory->current_work_mode = MEMORY_ASYNCHRONOUS_MODE;
        break;
    case MEMORY_SYNCHRONOUS_MODE:
        status = nf_ctrl_configure(memory->pcontroller, 
                                   NF_CONFIGURE_SET_SYNCHRONOUS_MODE, 0);
        if (status)
            return status;       
    }

// FIXME, temporay placement
	// set address cycle count
	nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ADDR0_CYCLE_COUNT, memory->addr_cyc_cnt);
	nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ADDR1_CYCLE_COUNT, memory->addr_cyc_cnt);

	// FIXME, ecc_off is hard-coded, sync with nand_ecclayout at acti_nand.c
	switch(memory->page_size){
		case 512:
			ecc_off = 0;
			break;
		case 2048:
			ecc_off = 32;
			break;
		case 4096:
			ecc_off = 160;
			break;
	}
	// confirgure possition of ecc code
	nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ECC_OFFSET, memory->page_size + ecc_off);

	// configure ECC ability and level
	if (memory->page_size == 512) {
		nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ECC_ABILITY, 2);
		nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ECC_ERR_LEVEL, 2);
	}
	else {
		nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ECC_ABILITY, 4);
		nf_ctrl_configure(pcontroller, NF_CONFIGURE_SET_ECC_ERR_LEVEL, 4);
	}
	
    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/

/******************************************************************************/
uint8_t wait_until_target_busy(nf_memory_t *memory)
{
    uint8_t status;
	uint8_t loop = 30;

    nf_controller_t *pcontroller =  memory->pcontroller;

    // wait for transfer sequence ended by NAND flash controller
   status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                            NF_SFR_STATUS_CTRL_STAT, 100, 0);

    while(status && (loop-- > 0)) {
			// one more chance 
   			status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                            NF_SFR_STATUS_CTRL_STAT, 100, 0);
	}

	if (status) {
        return status;
	}

    // wait until NAND flash memory device is busy
    status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                          NF_SFR_STATUS_MEMX_RDY(memory->ce), 500, 1);

	loop = 20;
    while(status && (loop-- > 0)) {
			// one more chance 
    		status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                          NF_SFR_STATUS_MEMX_RDY(memory->ce), 500, 1);
	}

    if(status) {
    	status = IORD_32NF(pcontroller->reg_offset + NF_SFR_STATUS);
		printf ( " [%s]:%d: status = %d\n", __FILE__, __LINE__, status);
	}
    return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_read_status(nf_memory_t *memory, uint8_t *memory_status)
{
    uint8_t status;
    nf_controller_t *pcontroller = memory->pcontroller;

    // prepare command
    NF_INIT_CMD_SIMPLE(g_command, NF_CMD_READ_STATUS);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);

    *memory_status = (uint8_t)IORD_32NF(pcontroller->reg_offset 
                                        + NF_SFR_READ_STATUS);    
    return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_reset(nf_memory_t *memory)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller = memory->pcontroller;

    // prepare command
    NF_INIT_CMD_SIMPLE(g_command, NF_CMD_RESET);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status)
        return status; 

    // wait for controller and device ready
    return wait_until_target_busy(memory);
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_read_id(nf_memory_t *memory, uint8_t id[8])
{
    uint8_t status = NF_ERR_NO_ERRORS; 
	uint8_t old_mode;

    nf_controller_t *pcontroller = memory->pcontroller;

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_READ_ID, 0, 0, id, 8, 
                                   NF_TRANSFER_DIRECTION_READ, WITHOUT_ECC);
    // execute command
	old_mode = pcontroller->transfer_mode;

	// use PIO mode to access device ID
    pcontroller->transfer_mode = NF_TRANSFER_MODE_MANUAL;
    status = nf_ctrl_execute_command(pcontroller, &g_command);

	// restore previous mode
    pcontroller->transfer_mode = old_mode;

	return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_read_unique_id(nf_memory_t *memory, uint8_t buffer[16]) 
{
    uint8_t i, j, wrong_id, status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller =  memory->pcontroller;
    uint8_t *buff =  memory->pcontroller->aux_buffer;

#if USE_INDIRECT_BUFFER
#if !COPY_DATA_TO_IB
    buffer =  memory->pcontroller->dma_buffer;
#endif
#endif

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_READ_UNIQUE_ID,
                                   0, 0, buff, 
                                   512, NF_TRANSFER_DIRECTION_READ, WITHOUT_ECC);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status)
        return status; 

    for (i = 0; i < 32; i += 1){
        wrong_id = 0;
        // check if ID data is correct
        for (j = 0; j < 16; j++){
            if ((buff[i + j] ^ buff[i + j + 16]) != 0xFF){
                wrong_id = 1;
                break;
            }
        }
        // if it is correct copy to buffer and return with no error
        if (!wrong_id){
            for (j = 0; j < 16; j++){
                buffer[j] = buff[i + j];
            }
            return NF_ERR_NO_ERRORS;
        }
    }    
    return NF_ERR_WRONG_UNIQUE_ID;
}
/******************************************************************************/

/******************************************************************************/
inline uint8_t nf_mem_page_empty(nf_memory_t *memory, uint32_t page_number)
{
    nf_controller_t *pcontroller = memory->pcontroller;
	uint32_t	tmp_buf = pcontroller->dma_buffer + pcontroller->page_size;
	
	nf_mem_page_read(memory, page_number, memory->page_size, tmp_buf, memory->spare_area_size);
	if ( (*(uint32_t *)(pcontroller->dma_buffer + pcontroller->ecc_offset)) == 0xffffffff)
		return 1;
	else
		return 0;
}

/******************************************************************************/
uint8_t nf_mem_page_read(nf_memory_t *memory, uint32_t page_number, 
                         uint16_t page_offset, uint8_t *buffer, 
                         uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller = memory->pcontroller;
	//int _verbose = 1;


    // wait for controller and device ready
    status = wait_until_target_busy(memory);
	
    if (status) {
		printf("%s:line %d\n", __FUNCTION__, __LINE__);		
        return status;  
	}
	

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_READ_PAGE, page_number,
                                   page_offset, buffer, buffer_size, 
                                   NF_TRANSFER_DIRECTION_READ, WITH_ECC);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);

    if (status) {
		printf( " [%s]: execute cmd status = 0x%x\n", __FUNCTION__, status);
        return status;
	}

	status = nf_check_operation_status(memory, NF_OPERATION_TYPE_READ);
	return status;

}

/******************************************************************************/
uint8_t nf_mem_page_read_oob(nf_memory_t *memory, uint32_t page_number, 
                         uint16_t page_offset, uint8_t *buffer, 
                         uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller = memory->pcontroller;
	unsigned int tmpbuf;

	pcontroller->ecc_enabled = 0;

    // wait for controller and device ready
    if (wait_until_target_busy(memory))
        return status;

	if (memory->nandid == NAND_ID_ST_128W3A) {
    	// prepare command
    	NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_READ_SPARE, page_number,
    	                               0, buffer, buffer_size,
    	                               NF_TRANSFER_DIRECTION_READ, WITHOUT_ECC);
    	// execute command
    	if (nf_ctrl_execute_command(pcontroller, &g_command))
    	    return status;

    	if (nf_check_operation_status(memory, NF_OPERATION_TYPE_READ))
			return status;

		// restore address to normal, issue a dummy read
		return nf_mem_page_read(memory, 0, 0, &tmpbuf, 0);
	}
	else
		return nf_mem_page_read (memory, page_number, page_offset, buffer, buffer_size);
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_page_program(nf_memory_t *memory, uint32_t page_number,
                            uint16_t page_offset, uint8_t *buffer, 
                            uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller = memory->pcontroller;

    status = nf_is_write_protected(memory);
    if (status) {
		printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
        return status; 
	}

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status) {
		printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
        return status;  
	}

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_PROGRAM_PAGE, page_number,
                                   page_offset, buffer, buffer_size, 
                                   NF_TRANSFER_DIRECTION_WRITE, WITH_ECC);    
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status) {
		printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
        return status;
	}

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status) {
		printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
        return status; 
	}

    return nf_check_operation_status(memory, NF_OPERATION_TYPE_WRITE);
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_page_program_oob(nf_memory_t *memory, uint32_t page_number, 
                         uint16_t page_offset, uint8_t *buffer, 
                         uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller = memory->pcontroller;
	unsigned int tmpbuf;

    status = nf_is_write_protected(memory);
    if (status) {
//		printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
        return status; 
	}
	pcontroller->ecc_enabled = 0;

    // wait for controller and device ready
    if (wait_until_target_busy(memory))
        return status;

	if (memory->nandid == NAND_ID_ST_128W3A) {
    	// prepare command
    	NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_READ_SPARE, page_number,
    	                               0, &tmpbuf, 0,
    	                               NF_TRANSFER_DIRECTION_READ, WITHOUT_ECC);
    	// execute command
    	if (nf_ctrl_execute_command(pcontroller, &g_command))
    	    return status;

		status = nf_mem_page_program(memory, page_number, 0, buffer, buffer_size);

		// restore address to normal, issue a dummy read
		nf_mem_page_read(memory, 0, 0, &tmpbuf, 0);
		return status;
	}
	else
		return nf_mem_page_program(memory, page_number, page_offset, buffer, buffer_size);
}

/******************************************************************************/
uint8_t nf_mem_block_erase(nf_memory_t *memory, uint32_t block_number)
{
    uint8_t status;
    nf_controller_t *pcontroller = memory->pcontroller;


    status = nf_is_write_protected(memory);
    if (status){
        return status; 
	}

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status) {
        return status; 
	}

    // prepare command
    if(memory->nandid == NAND_ID_ST_128W3A) {
   		NF_ERASE_CMD_WITH_ADDR(g_command, NF_CMD_ERASE_BLOCK_SEQ10, (block_number << memory->pages_per_block_shift), 0);
	}
	else {
	   	NF_INIT_CMD_WITH_ADDR(g_command, NF_CMD_ERASE_BLOCK, 
                          (block_number << memory->pages_per_block_shift), 0);
	}
    // execute command 
    // block address is converted to page address
    status = nf_ctrl_execute_command(pcontroller, &g_command);

    if (status) {
        return status; 
	}
	status = nf_check_operation_status(memory, NF_OPERATION_TYPE_ERASE);

	// make sure we have a clean FIFO, sort of work around for small page device 
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_FIFO_INIT, 1);

    return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_page_program_cache(nf_memory_t *memory, uint32_t page_number, 
                                  uint16_t page_offset, uint8_t *buffer, 
                                  uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller =  memory->pcontroller;

    status = nf_is_write_protected(memory);
    if (status)
        return status; 

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status;  

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_PROGRAM_PAGE_CACHE,
                                   page_number, page_offset, buffer, 
                                   buffer_size, NF_TRANSFER_DIRECTION_WRITE, 
                                   WITH_ECC);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status)
        return status;

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    status = nf_check_operation_status(memory, NF_OPERATION_TYPE_WRITE);

    return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_write_page(nf_memory_t *memory)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller =  memory->pcontroller;

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    // prepare command
    NF_INIT_CMD_SIMPLE(g_command, NF_CMD_WRITE_PAGE);    
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status)
        return status;

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    status = nf_check_operation_status(memory, NF_OPERATION_TYPE_WRITE);

    return status;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_page_random_read(nf_memory_t *memory, uint16_t page_offset, 
                                uint8_t *buffer, uint16_t buffer_size)
{
    uint8_t status = NF_ERR_NO_ERRORS;
    nf_controller_t *pcontroller =  memory->pcontroller;

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_CHANGE_READ_COLUMN, 0, 
                                   page_offset, buffer, buffer_size, 
                                   NF_TRANSFER_DIRECTION_READ, WITHOUT_ECC); 
    // execute command
    return nf_ctrl_execute_command(pcontroller, &g_command);
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_mem_internal_datamove(nf_memory_t *memory, uint32_t src_page_number, 
                                 uint32_t dest_page_number, 
                                 uint8_t *buffer, uint16_t buffer_size)
{
    uint8_t status;
    nf_controller_t *pcontroller =  memory->pcontroller;

    status = nf_is_write_protected(memory);
    if (status)
        return status; 

    // wait for controller and device ready
    status = wait_until_target_busy(memory);
    if (status)
        return status; 

    // prepare command
    NF_INIT_CMD_WITH_ADDR_AND_DATA(g_command, NF_CMD_COPYBACK_READ, 
                                   src_page_number, 0, buffer, buffer_size, 
                                   NF_TRANSFER_DIRECTION_READ, WITH_ECC);
    // execute command
    status = nf_ctrl_execute_command(pcontroller, &g_command);
    if (status)
        return status;

    // prepare command
    NF_INIT_CMD_WITH_ADDR(g_command, NF_CMD_COPYBACK_PROGRAM, dest_page_number, 
                          0);   
    // execute command
    return nf_ctrl_execute_command(pcontroller, &g_command);
}
/******************************************************************************/

