
//================================================================================
// FILE NAME:      umvp_audio_codec_drv.c
//--------------------------------------------------------------------------------
// Reason-ID       Reason
//--------------------------------------------------------------------------------
// R:2013114-00:   Create
//================================================================================

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>

//#include <linux/delay.h>  //mdeley() msleep()
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/string.h>
//#include <asm/uaccess.h>

#include <linux/slab.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#include "umvp_audio_codec.h"
#include "umvp_wm8940.h"

/* -----------------------------------------------------------------------
    define debug macro
 ----------------------------------------------------------------------- */
static u_int debug_ctrl = 1;
#define MSG_PFX         "[umvp_au]"
#define dprintk(num, format, args...) \
    do { \
        if (debug_ctrl >= num) \
            printk(MSG_PFX format, ##args); \
    } while (0)

/*-----------------------------------------------------------------------------
 * ssm2602 2 wire address is determined by GPIO5
 * state during powerup.
 *    low  = 0x1a
 *    high = 0x1b
 */
#define I2C_NAME(s)         (s)->name
#define I2C_BUS_ID           0        /* We have three I2C buses (adapters) whose id are 0 ~ 2 */
#define I2C_CMD_ADDR         0x1a     /* i2c address */
#define _OK                  0
#define _ERR                -1

au_codec_t au_codec;

/*-----------------------------------------------------------------------------*/
/*
 * ssm2602 register cache
 * We can't read the ssm2602 register space when we are
 * using 2 wire for device control, so we cache them instead.
 * There is no point in caching the reset register
 */
static const u16 ssm2602_reg[SSM2602_CACHEREGNUM] = {
    0x0017, 0x0017, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x000a,
    0x0000, 0x0000
};

static inline unsigned int au_ssm2602_read_reg_cache(u_int reg)
{
    u16 *cache = au_codec.reg_cache;

    if (reg == SSM2602_RESET)
        return 0;
    if (reg >= SSM2602_CACHEREGNUM)
        return -1;

    dprintk(4,KERN_INFO "ssm2602_read_cache: reg %d = 0x%04X\n", reg, cache[reg]);
    return cache[reg];

}

static inline void au_ssm2602_write_reg_cache(u_int reg, u_int value)
{
    u16 *cache = au_codec.reg_cache;

    if (reg >= SSM2602_CACHEREGNUM)
        return;
    cache[reg] = value;
}

static int au_ssm2602_write(u_int reg, u_int value)
{
    struct i2c_client *client = au_codec.client;
    u_char data[2];

    dprintk(4,KERN_INFO "ssm2602_write: reg %d = 0x%04X\n", reg, value);

    /* data is
     *   D15..D9 ssm2602 register offset
     *   D8...D0 register data
     */
    data[0] = (reg << 1) | ((value >> 8) & 0x0001);
    data[1] = value & 0x00ff;

    au_ssm2602_write_reg_cache(reg, value);
    if (i2c_master_send(client, data, 2) == 2)
        return 0;
    else
        return -EIO;
}

/*-----------------------------------------------------------------------------*/
struct _coeff_div {
    u32 mclk;
    u32 rate;
    u16 fs;
    u8 sr:4;
    u8 bosr:1;
    u8 usb:1;
};

/* codec mclk clock divider coefficients */
static const struct _coeff_div coeff_div[] = {
    /* 48k */
    {12288000, 48000, 256, 0x0, 0x0, 0x0},
    {18432000, 48000, 384, 0x0, 0x1, 0x0},
    {12000000, 48000, 250, 0x0, 0x0, 0x1},

    /* 32k */
    {12288000, 32000, 384, 0x6, 0x0, 0x0},
    {18432000, 32000, 576, 0x6, 0x1, 0x0},
    {12000000, 32000, 375, 0x6, 0x0, 0x1},

    /* 8k */
    {12288000, 8000, 1536, 0x3, 0x0, 0x0},
    {18432000, 8000, 2304, 0x3, 0x1, 0x0},
    {11289600, 8000, 1408, 0xb, 0x0, 0x0},
    {16934400, 8000, 2112, 0xb, 0x1, 0x0},
    {12000000, 8000, 1500, 0x3, 0x0, 0x1},

    /* 96k */
    {12288000, 96000, 128, 0x7, 0x0, 0x0},
    {18432000, 96000, 192, 0x7, 0x1, 0x0},
    {12000000, 96000, 125, 0x7, 0x0, 0x1},

    /* 44.1k */
    {11289600, 44100, 256, 0x8, 0x0, 0x0},
    {16934400, 44100, 384, 0x8, 0x1, 0x0},
    {12000000, 44100, 272, 0x8, 0x1, 0x1},

    /* 88.2k */
    {11289600, 88200, 128, 0xf, 0x0, 0x0},
    {16934400, 88200, 192, 0xf, 0x1, 0x0},
    {12000000, 88200, 136, 0xf, 0x1, 0x1},
};

static inline int get_coeff(int mclk, int rate)
{
    int i;

    for (i = 0; i < ARRAY_SIZE(coeff_div); i++) {
        if (coeff_div[i].rate == rate && coeff_div[i].mclk == mclk)
            return i;
    }
    return i;
}

int au_ssm2602_hw_params(hw_params_t *param)
{
    au_codec_t *codec = &au_codec;

    /* default setting wl=24 bits I2S mode */
    u16 iface = au_ssm2602_read_reg_cache(SSM2602_IFACE) & 0xfff3;
    int i     = get_coeff(codec->sysclk, param->rate);
    u16 srate;

    /*no match is found*/
    if (i == ARRAY_SIZE(coeff_div))
        return -EINVAL;

    srate = (coeff_div[i].sr << 2)   |
            (coeff_div[i].bosr << 1) | coeff_div[i].usb;

    au_ssm2602_write(SSM2602_ACTIVE, 0);
    au_ssm2602_write(SSM2602_SRATE, srate);

    /* set master/slave audio interface */
    switch (param->clk_fmt) {
    case A1_SOC_DAIFMT_CBM_CFM:
         /* cpu is slave; codec clk & frame master */
         iface |= 0x0040;
         break;
    case A1_SOC_DAIFMT_CBS_CFS:
         /* codec clk & FRM slave */
         break;
    default:
        return -EINVAL;
    }

    /* interface bit size format */
    switch (param->format) {
    case A1_SOC_FORMAT_S16_LE:
         break;
    case A1_SOC_FORMAT_S20_LE:
         iface |= 0x0004;
         break;
    case A1_SOC_FORMAT_S24_LE:
         iface |= 0x0008;
         break;
    case A1_SOC_FORMAT_S32_LE:
         iface |= 0x000c;
         break;
    }

    au_ssm2602_write(SSM2602_IFACE,  iface);
    dprintk(3,KERN_INFO "%s:iface(%x)\n", __func__, iface);

    au_ssm2602_write(SSM2602_ACTIVE, ACTIVE_ACTIVATE_CODEC);
    return 0;
}


int au_ssm2602_mic_sensitivity(int HiLo)
{
    u16 mute_reg = au_ssm2602_read_reg_cache(SSM2602_APANA) & ~APANA_ENABLE_MIC_BOOST;

    dprintk(1,KERN_INFO "%s: 0x%X\n", __func__, mute_reg );
    if (HiLo)
        au_ssm2602_write(SSM2602_APANA, mute_reg | APANA_ENABLE_MIC_BOOST);
    else
        au_ssm2602_write(SSM2602_APANA, mute_reg);

    return 0;
}

int au_ssm2602_mic_mute(int mute)
{
    u16 mute_reg = au_ssm2602_read_reg_cache(SSM2602_APANA) & ~APANA_ENABLE_MIC_MUTE;

    dprintk(1,KERN_INFO "%s: 0x%X\n", __func__, mute_reg );
    if (mute)
        au_ssm2602_write(SSM2602_APANA, mute_reg | APANA_ENABLE_MIC_MUTE);
    else
        au_ssm2602_write(SSM2602_APANA, mute_reg);

    return 0;
}

void au_ssm2602_stop(void)
{
    dprintk(1,KERN_INFO "%s\n", __func__);
    au_ssm2602_write(SSM2602_ACTIVE, 0);
}

void au_ssm2602_start(void)
{
    dprintk(1,KERN_INFO "%s\n", __func__);
    au_ssm2602_write(SSM2602_ACTIVE, ACTIVE_ACTIVATE_CODEC);
}

int au_ssm2602_set_sysclk(u_int freq)
{
    au_codec_t  *codec = &au_codec;

    dprintk(1,KERN_INFO "%s:freq(%u)\n", __func__, freq);
    switch (freq) {
        case 11289600:
        case 12000000:
        case 12288000:
        case 16934400:
        case 18432000:
             codec->sysclk = freq;
             return 0;
    }
    return -EINVAL;
}

int au_ssm2602_outvolume(int value)
{
    int reg;
    unsigned char val = value & 0x7f;
    /* -----------------------------
     * 000 0000 to 010 1111 = mute
     * 011 0000 = -73db
     * ....
     * 111 1001 = 0 db (default)
     * ...1 db steps up to
     * 111 1111 = +6db
     */

    reg = au_ssm2602_read_reg_cache(SSM2602_LOUT1V) & ~0x7f;
    au_ssm2602_write(SSM2602_LOUT1V, reg | val); /* set left ch DAC volum */
    dprintk(1,KERN_INFO "%s: L-volum(0x%X)\n", __func__, val);

    reg = au_ssm2602_read_reg_cache(SSM2602_ROUT1V) & ~0x7f;
    au_ssm2602_write(SSM2602_ROUT1V, reg | val); /* set right ch DAC volum */
    dprintk(1,KERN_INFO "%s: R-volum(0x%X)\n", __func__, val);

    return 0;
}

int au_ssm2602_dac_mute(int mute)
{
    u16 reg = au_ssm2602_read_reg_cache(SSM2602_APDIGI) & ~APDIGI_ENABLE_DAC_MUTE;

    dprintk(1,KERN_INFO "%s: 0x%X\n", __func__, reg);
    if (mute)
        au_ssm2602_write(SSM2602_APDIGI, reg | APDIGI_ENABLE_DAC_MUTE);
    else
        au_ssm2602_write(SSM2602_APDIGI, reg);

    return 0;
}

static int au_ssm2602_init(void)
{
    au_codec_t  *codec = &au_codec;
    hw_params_t *param = &codec->param;
    int reg;

    dprintk(0,KERN_INFO "%s\n", __func__);

    codec->reg_cache_size = sizeof(ssm2602_reg);
    codec->reg_cache      = kmemdup(ssm2602_reg, sizeof(ssm2602_reg), GFP_KERNEL);

    if (codec->reg_cache == NULL)
        return -ENOMEM;

    au_ssm2602_write(SSM2602_RESET, 0);

    /* mute first */
    reg = au_ssm2602_read_reg_cache(SSM2602_APDIGI);
    au_ssm2602_write(SSM2602_APDIGI, reg | APDIGI_ENABLE_DAC_MUTE);

    /*power on device*/
    au_ssm2602_write(SSM2602_ACTIVE, 0);
    /* set the update bits */
    reg = au_ssm2602_read_reg_cache(SSM2602_LINVOL);
    au_ssm2602_write(SSM2602_LINVOL, reg | LINVOL_LRIN_BOTH);

    reg = au_ssm2602_read_reg_cache(SSM2602_RINVOL);
    au_ssm2602_write(SSM2602_RINVOL, reg | RINVOL_RLIN_BOTH);

    reg = au_ssm2602_read_reg_cache(SSM2602_LOUT1V);
    au_ssm2602_write(SSM2602_LOUT1V, reg | LOUT1V_LRHP_BOTH | 0x79); /* set left ch DAC volum to 0db */
    reg = au_ssm2602_read_reg_cache(SSM2602_ROUT1V);
    au_ssm2602_write(SSM2602_ROUT1V, reg | ROUT1V_RLHP_BOTH | 0x79); /* set right ch DAC volum to 0db */
    /*select Micro in as default input*/
    au_ssm2602_write(SSM2602_APANA, APANA_SELECT_DAC | APANA_ENABLE_BYPASS | APANA_ADC_IN_SELECT | APANA_ENABLE_MIC_BOOST);

    au_ssm2602_write(0x12, 0xf9);
    au_ssm2602_write(SSM2602_PWR, 0);

#if 0
/* The followings are sampling rate supported by ssm2602 */
    switch (params_rate(params)) {
    case 8000:
    case 16000:
    case 48000:
    case 96000:
    /*  clk = 48000 x 256 fs */
        clk = 12288000;
        break;
    case 11025:
    case 22050:
    case 44100:
    /*  clk = 44100 x 256 fs */
        clk = 11289600;
        break;
    }
#endif

    /* set default sampling rate and bit size */
    au_ssm2602_set_sysclk(A1_SOC_SYSCLK_DEFAULT);
    param->rate     = A1_SOC_RATE_DEFAULT;
    param->format   = A1_SOC_FORMAT_S16_LE;
    param->clk_fmt  = A1_SOC_DAIFMT_CBM_CFM;

    au_ssm2602_hw_params(param);
    return 0;
}

/* -----------------------------------------------------------------------
 * Generic i2c probe concerning the addresses,
 * i2c wants 7 bit (without the r/w bit), so '>>1'
 * -----------------------------------------------------------------------
 */
static struct i2c_driver au_i2c_driver;

/* standard i2c insmod options */
#if 0
static unsigned short normal_i2c[] = {
    I2C_CMD_ADDR >> 1,
    I2C_CLIENT_END
};

#else
static unsigned short normal_i2c[] = {
    I2C_CMD_ADDR,
    I2C_CLIENT_END
};
#endif

/* create addr_date  */
I2C_CLIENT_INSMOD;

/*
enum DEVICE_TYPE {
     SSM2602 = 2602,
     WM8940  = 8940
};
*/
static int au_codec_get_chipId(struct i2c_client *client)
{
    struct i2c_msg xfer[2];

    /* WM8940 register 0 is device id that the default value is 0x8940 */
    u_char  reg  = 0;
    u_short data = 0;
    int ret;

    /* Write register */
    xfer[0].addr  = client->addr;
    xfer[0].flags = 0;
    xfer[0].len   = 1;
    xfer[0].buf   = &reg;

    ret = i2c_transfer(client->adapter, &xfer[0], 1);

    /* Read data */
    xfer[1].addr = client->addr;
    xfer[1].flags = I2C_M_RD;
    xfer[1].len   = 2;
    xfer[1].buf   = (u_char *)&data;

    ret = i2c_transfer(client->adapter, &xfer[1], 1);
    dprintk(0,KERN_INFO "ret(%d)..\n", ret);

    if (ret < 0) data = SSM2602;
    ret = (data >> 8) | ((data & 0xff) << 8);
    dprintk(0,KERN_INFO "ChipId(0x%04x - data(0x%04x)\n", ret, data);

    if (ret == 0x8940)
        return WM8940;
    else
        return SSM2602;
}

static int au_i2c_probe(struct i2c_adapter *adapter)
{
    struct i2c_client *client;
    int ret  = 0;
    int addr = I2C_CMD_ADDR;

    if (adapter->nr != I2C_BUS_ID) return _ERR;

    dprintk(1,KERN_INFO "robe audio i2c address(0x%x) adapter->nr(0x%x)\n", addr, adapter->nr);

    client = kzalloc(sizeof (struct i2c_client), GFP_KERNEL);
    if (!client) return -ENOMEM;

    memset(client, 0, sizeof(struct i2c_client));
    client->addr    = addr;
    client->adapter = adapter;
    client->driver  = &au_i2c_driver;
    client->flags   = 0;
    strlcpy(I2C_NAME(client), "au-ssm2602", sizeof(I2C_NAME(client)));

    i2c_set_clientdata(client, &au_codec);
    au_codec.client = client;

    ret = i2c_attach_client(client);
    if (ret) {
        dprintk(0,KERN_INFO "au_ssm2602 attach fail\n");
        goto EXT;
    }

    au_codec.chip_id = au_codec_get_chipId(client);
    if (au_codec.chip_id == WM8940) {
        ret = au_wm8940_init();
    }else {
        ret = au_ssm2602_init();
    }

    if (ret < 0) {
        dprintk(0,KERN_ERR "Codec(%d) initial fail\n", au_codec.chip_id);
        goto EXT;
    }

    /* initial umvp dai in umvp_dai.c */
    au_i2s_init();

    /* start codec */
    if (au_codec.chip_id == SSM2602)
        au_ssm2602_start();

    return _OK;
EXT:
    kfree(client);
    return ret;
}

static int au_i2c_remove(struct i2c_client *client)
{
    au_codec_t *codec = i2c_get_clientdata(client);

    dprintk(1,KERN_INFO "au_ssm2602_detach\n");

    /* release umvp dai in umvp_dai.c */
    au_i2s_release();

    if (codec){
        if (i2c_detach_client(codec->client) != 0)
            dprintk(0,KERN_INFO "au_ssm2602 detach fail\n");
        i2c_set_clientdata(client, NULL);
        kfree(codec->reg_cache);
        kfree(client);
    }
    return 0;
}


static struct i2c_driver au_i2c_driver = {
    .driver = {
        .name = "SSM2602 Codec",
        .owner = THIS_MODULE,
    },
    .id = 0,
    .attach_adapter = au_i2c_probe,
    .detach_client  = au_i2c_remove,
    .command = NULL
};

static int __init a1audio_init(void)
{
    int ret;

    ret = i2c_add_driver(&au_i2c_driver);
    if (ret)
        dprintk(0,KERN_ERR "Failed to register umvp_audio I2C driver: %d\n",ret);

    return ret;
}

static void __exit a1audio_exit(void)
{
    i2c_del_driver(&au_i2c_driver);
}

module_init(a1audio_init);
module_exit(a1audio_exit);

MODULE_DESCRIPTION("A1 Audio driver");
MODULE_AUTHOR("ACTi Corporation");
MODULE_LICENSE("GPL");

