/*
 * Port from IT8212 driver
 * linux/drivers/scsi/atlas.h
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#ifndef _ATLAS_H_
#define _ATLAS_H_

#include <linux/version.h>
#include <linux/types.h>
#include <linux/kdev_t.h>
//#include <asm/hardware.h>
//#include <asm/arch/platform.h>
#include <mach/hardware.h>

#define ATLAS_TESTMODE 0

/* Select system bus, PCI or AHB */
// #define ATLAS_FPGA	0
#define ATLAS_AHB	1
#define ATLAS_PCI	0

#define ATLAS_READ_AFTER_WRITE		0
#define ATLAS_READ_AFTER_WRITE_CHECK	0
#define ATLAS_BARRIER			0

#define FIND_DEVICE_WITH_NOP		0  // 0: Use NOP to find device, 1: use GetCBR to find device

//#if ATLAS_FPGA
//#define ATLAS_MAX_UDMA	4
#define ATLAS_MAX_UDMA		0
//#else // Test chip or release version
//#define ATLAS_MAX_UDMA	5
//#endif

#if (ATLAS_AHB && ATLAS_PCI) || (!defined(ATLAS_AHB) && !defined(ATLAS_PCI)) || (!ATLAS_AHB && !ATLAS_PCI)
#error Only one of ATLAS_AHB or ATLAS_PCI should be enabled!!
#endif

#ifdef ATLAS_AHB
#define ATLAS_AHB_RESOURCE_IRQ	INT_IDE
#define ATLAS_AHB_RESOURCE_ADDR	UMVP_IDE_BASE

#define ATLAS_AHB_RESOURCE_LEN	0x0000FFFF
#define ATLAS_AHB_RESOURCE_MASK	0xFFFFFFFF  // 0xFFFF0000
#endif
 
#define ATLAS_SIGNATURE	 0x0DD50509
#define ATLAS_SIG_OFFSET 0xFC
#define ATLAS_VENDOR_ID	 0x0dd5			/* Vendor ID (ATLAS)	*/
#define ATLAS_DEVICE_ID	 0x0509			/* Device IF (ATLAS)	*/
#define MAX_ADAPTERS	 1			/* 2 Max Board supported	*/
#define MAX_DEVICES	(MAX_ADAPTERS * 2)	/* 4 Max Dev supported	*/
#define LBA28_MAX_SECTORS		256
#define LBA48_MAX_SECTORS		65535

#define TRUE		1
#define FALSE 		0

#define FAULT_RETRY_EMULATION	0

/*
 * Undef macros which may conflict
 */
#undef	START_STOP

/************************************************************************
 * Debugging macro
 ************************************************************************/
#define message(msg...) do{ if(system_dbg) printk(msg); }while(0)
#if ATLAS_DEBUG
#define dprintk		message
#else
#define dprintk(msg...) do { } while(0)
#endif


/************************************************************************
 * Physical disk status definitions
 ************************************************************************/
#define DISK_KEY_OFF			0
#define DISK_OFF_LINE			1
#define DISK_ON_LINE			2
#define DISK_REBUILDING			3
#define DISK_PLUGGING			4
#define DISK_PLUGGING_OK		5

#define MaximumLBAOf28Bit		0x10000000

#define DisableChannel			1
#define EnableChannel			2

#define CABLE_40_PIN			1
#define CABLE_80_PIN			0

#define RaidActive                      0
#define RaidInactive			1

#define IDE_CLOCK_66			0
#define IDE_CLOCK_50			1

#define USE_ULTRA_DMA			0
#define USE_MULTIWORD_DMA		1

/************************************************************************
 *
 ************************************************************************/
typedef struct _PHYSICAL_DISK_STATUS
{
 u8		ModelNumber[40];		/* Byte 00-39		*/
 u32		UserAddressableSectors_LOW;	/* Byte 40-43		*/
 u32		UserAddressableSectors_HIGH;	/* Byte 44-47		*/
 u8		MultiWordDMASupport;		/* Byte 48		*/
 u8		MultiWordDMAActive;		/* Byte 49		*/
 u8		UltraDMASupport;		/* Byte 50		*/
 u8		UltraDMAActive;			/* Byte 51		*/
 u8		RaidType;			/* Byte 52		*/
 u8		RaidNumber;			/* Byte 53		*/
 u8		SerialNumber[20];		/* Byte 54-73		*/
 u8		DiskStatus;			/* Byte 74		*/
 u8		DiskOriginalLocation;		/* Byte 75		*/
 u8		Cable80Pin;			/* Byte 76		*/
 u8		BootableDisk;			/* Byte 77		*/
 u8		StorageSize[8];			/* Byte 78-85		*/
 u8		Reserved[35];			/* Byte 86-120		*/
 u8		UpdateYear;			/* Byte 121		*/
 u8		UpdateMonth;			/* Byte 122		*/
 u8		UpdateDay;			/* Byte 123		*/
 u8		FirmwareVer;			/* Byte 124		*/
 u8		RebuildStatus;			/* Byte 125		*/
 u8		StripeSize;			/* Byte 126		*/
 u8		AutoRebuildEnable;		/* Byte 127		*/
} PHYSICAL_DISK_STATUS, *PPHYSICAL_DISK_STATUS;

/************************************************************************
 *
 ************************************************************************/
typedef struct _Atlas_SET_CHIP_STATUS_INFO
{
 u16		RaidType;			/* Word 129		*/
 u16		ContainingDisks;		/* Word 130		*/
 u16		UltraDmaTiming01;		/* Word 131		*/
 u16		UltraDmaTiming23;		/* Word 132		*/
 u16		UltraDmaTiming45;		/* Word 133		*/
 u16		UltraDmaTiming6;		/* Word 134		*/
 u16		MultiWordDmaTiming01;		/* Word 135		*/
 u16		UltraDmaTiming2;		/* Word 136		*/
 u16		PioTiming4;			/* Word 137		*/
 u16		AutoRebuildEnable;		/* Word 138		*/
 u16		IdeClkUDma01;			/* Word 139		*/
 u16		IdeClkUDma23;			/* Word 140		*/
 u16		IdeClkUDma45;			/* Word 141		*/
 u16		IdeClkUDma6;			/* Word 142		*/
 u16		IdeClkMDma01;			/* Word 143		*/
 u16		IdeClkMDma2;			/* Word 144		*/
 u16		IdeClkPio4;			/* Word 145		*/
 u16		StripeSize;			/* Word 146		*/
 u16		BootableDisk;			/* Word 147		*/
 u16		CheckHotSwapInterval;		/* Word 148		*/
 u16		TargetSourceDisk;		/* Word 149		*/
 u16		RebuildBlockSize;		/* Word 150		*/
 u16		ResetInterval1;			/* Word 151		*/
 u16		ResetInterval2;			/* Word 152		*/
 u16		RebuildRetryTimes;		/* Word 153		*/
 u16		NewlyCreated;			/* Word 154		*/
} Atlas_SET_CHIP_STATUS_INFO, *PAtlas_SET_CHIP_STATUS_INFO;


/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

#define ATLAS_DRV_SIGNATURE		"ATLAS ATA/ATAPI CONTROLLER"


/************************************************************************
 * Extra IDE commands supported by Accusys
 ************************************************************************/
#define IDE_COMMAND_GET_CHIP_STATUS	0xFA
#define IDE_COMMAND_SET_CHIP_STATUS	0xFB
#define IDE_COMMAND_REBUILD		0xFC
#define IDE_COMMAND_REBUILD_STATUS	0xFD


/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

/************************************************************************
 * IDE registers offset
 ************************************************************************/
#define IDE_NR_PORTS			23

#define IDE_DATA_OFFSET			0
#define IDE_FEATURE_OFFSET		1
#define IDE_NSECTOR_OFFSET		2
#define IDE_LOCYL_OFFSET		3
#define IDE_MIDCYL_OFFSET		4
#define IDE_HCYL_OFFSET			5
#define IDE_SELECT_OFFSET		6
#define IDE_COMMAND_OFFSET		7
#define IDE_CONTROL_OFFSET		8
#define IDE_IRQ_OFFSET			9
#define	IDE_ERROR_OFFSET		10
#define IDE_STATUS_OFFSET		11
#define ATLAS_CONTROL			12
#define ATLAS_STATUS			13
#define ATLAS_PRD			14
#define ATLAS_TP1			15
#define ATLAS_TP2			16
#define ATLAS_MAC			17
#define ATLAS_XFER_SEC			18
#define ATLAS_TEST_CTRL			19
#define ATLAS_TEST_DATA			20
#define ATLAS_PKT_CTRL			21
#define ATLAS_SECTOR_SIZE		22

#define IDE_ALTERNATE_OFFSET		IDE_CONTROL_OFFSET			

/************************************************************************
 * ATAPI registers offset
 ************************************************************************/
#define ATAPI_DATA_OFFSET		0
#define ATAPI_ERROR_OFFSET		1
#define ATAPI_INTREASON_OFFSET		2
#define ATAPI_UNUSED1_OFFSET		3
#define ATAPI_LCYL_OFFSET		4
#define ATAPI_HCYL_OFFSET		5
#define ATAPI_SELECT_OFFSET		6
#define ATAPI_STATUS_OFFSET		7
#define ATAPI_CONTROL_OFFSET		8

#define ATAPI_COMMAND_OFFSET		ATAPI_STATUS_OFFSET
#define	ATAPI_FEATURE_OFFSET		ATAPI_ERROR_OFFSET


/* Atlas 32bit I/O wrapper function */
#define ATLAS_INOUT	1  /* Use ATLAS inport/outport wrapper function */
#define ATLAS_MEMIO	1  /* Use ATLAS memory i/o */
#define ATLAS_ALIGN	4

#define ATLAS_ALIGN_MASK (0xFFFFFFFC)
#define ATLAS_BYTE_MASK	 (0x000000FF)
#define ATLAS_WORD_MASK	 (0x0000FFFF)
#define ATLAS_DWORD_MASK (0xFFFFFFFF)


#if	ATLAS_MEMIO
#define	atlas_inx	readl
#define atlas_outx	writel
#define ATLAS_BAR(x)	(x->virtaddr)
#else
#define atlas_inx	inl
#define atlas_outx	outl
#define ATLAS_BAR(x)	(x->phyaddr)
#endif


#if ATLAS_READ_AFTER_WRITE_CHECK
static __inline__
void atlas_out_check(u32 p, u32 w, u32 r)
{
  u32 dmask = 0xffffffff;
  u32 pmask = 0x000000ff;
	
   // Change Data Mask
   if((p&pmask)==0x0c) // Atlas_Status[x], Atlas_Control[o]
    dmask = 0x00007fff;
   else if((p&pmask)==0x28) // TestModeCtrl will not be compared
    dmask = 0x0;
   else if((p&pmask)>0x2c) // Out of our register range
    dmask = 0x0;

   		
   if((w&dmask)!=(r&dmask))
   {
     printk("\n!!!!!!! Warning !!!!!!!\n");
     printk("Addr: 0x%08x, Write: 0x%08x, Read: 0x%08x\n", p, w, r);
   }
	
}
#endif


#if	ATLAS_INOUT_DBG
extern int inout_dbg;
extern int system_dbg;
#define ATLAS_INOUT_DBG_EN	(inout_dbg=1)
#define ATLAS_INOUT_DBG_DIS	(inout_dbg=0)

static __inline__
u32 atlas_inl(u32 port)
{
  u32 d1;	
  d1 = atlas_inx(port);
  if(inout_dbg)
   dprintk("%s: port 0x%08x, data 0x%08x\n", __FUNCTION__, port, d1);    
  return d1;
}
static __inline__
void atlas_outl(u32 d1, u32 port)
{
  atlas_outx(d1, port);
#if ATLAS_BARRIER  
  barrier();
#endif  
  if(inout_dbg)
   dprintk("%s: port 0x%08x, data 0x%08x\n", __FUNCTION__, port, d1);
#if ATLAS_READ_AFTER_WRITE
  {
    u32 d2;
    d2 = atlas_inl(port);	
#if ATLAS_READ_AFTER_WRITE_CHECK
    atlas_out_check(port, d1, d2);
#endif
  }
#endif       
}

#else // !ATLAS_INOUT_DBG

#define ATLAS_INOUT_DBG_EN	{}
#define ATLAS_INOUT_DBG_DIS	{}

static __inline__
u32 atlas_inl(u32 port)
{
  u32 d1;	
  d1 = atlas_inx(port);
  return d1;
}
static __inline__
void atlas_outl(u32 d1, u32 port)
{
  atlas_outx(d1, port);
#if ATLAS_BARRIER  
  barrier();
#endif  
#if ATLAS_READ_AFTER_WRITE
  {
    u32 d2;
    d2 = atlas_inl(port);	
#if ATLAS_READ_AFTER_WRITE_CHECK
    atlas_out_check(port, d1, d2);
#endif
  }
#endif   
}
#endif


#if	ATLAS_INOUT
static __inline__
u8 ATLAS_INB(u32 port)
{
  u32 d1;	
  d1 = atlas_inl(port&ATLAS_ALIGN_MASK);
  return (u8)(d1>>((port%ATLAS_ALIGN)*8));	
}

static __inline__
u16 ATLAS_INW(u32 port)
{
  u32 d1, d2;
  d1 = atlas_inl(port&ATLAS_ALIGN_MASK);  
  if((port%ATLAS_ALIGN)>=(ATLAS_ALIGN-1))
  {
   d2 = atlas_inl((port&ATLAS_ALIGN_MASK)+ATLAS_ALIGN);
   return (u16)((d1>>((port%ATLAS_ALIGN)*8))|(d2<<((1)*8)));
  }
  else	  
   return (u16)(d1>>((port%ATLAS_ALIGN)*8));	
}

static __inline__
u32 ATLAS_INL(u32 port)
{
  u32 d1, d2;
  d1 = atlas_inl(port&ATLAS_ALIGN_MASK);  
  if((port%ATLAS_ALIGN)>=(ATLAS_ALIGN-3))
  {
   d2 = atlas_inl((port&ATLAS_ALIGN_MASK)+ATLAS_ALIGN);
   return (u32)(((d1>>(port%ATLAS_ALIGN)*8))|(d2<<((ATLAS_ALIGN-(port%ATLAS_ALIGN))*8)));
  }
  else
   return (u32)(d1);
}

static __inline__
void ATLAS_OUTB(u8 d1p, u32 port)
{
   u32 d2;
   u32 d1 = (u32)d1p;
   d2 = atlas_inl(port&ATLAS_ALIGN_MASK);
   atlas_outl(((d2&~(((u32)ATLAS_BYTE_MASK)<<((port%ATLAS_ALIGN)*8)))|(((u32)d1)<<((port%ATLAS_ALIGN)*8))), (port&ATLAS_ALIGN_MASK));	
}
 

static __inline__
void ATLAS_OUTW(u16 d1, u32 port)
{
   u32 d2;
   	
   d2 = atlas_inl(port&ATLAS_ALIGN_MASK);	
	
   if((port%ATLAS_ALIGN)>=(ATLAS_ALIGN-1))
   {
      u8 v1 = (u8)d1, v2 = (u8)(d1>>8);
      atlas_outl((d2&~(((u32)ATLAS_WORD_MASK)<<((port%ATLAS_ALIGN)*8)))|(((u32)v1)<<((port%ATLAS_ALIGN)*8)), (port&ATLAS_ALIGN_MASK));
      d2 = atlas_inl((port&ATLAS_ALIGN_MASK)+ATLAS_ALIGN);
      atlas_outl((d2&~((u32)ATLAS_WORD_MASK>>(ATLAS_ALIGN-port%ATLAS_ALIGN)*8))|((u32)v2), (port&ATLAS_ALIGN_MASK)+ATLAS_ALIGN);            	
   }
   else
   {
      atlas_outl((d2&~(((u32)ATLAS_WORD_MASK)<<((port%ATLAS_ALIGN)*8)))|(((u32)d1)<<((port%ATLAS_ALIGN)*8)), (port&ATLAS_ALIGN_MASK));	   	
   }	
}

static __inline__
void ATLAS_OUTL(u32 d1, u32 port)
{
   u32 d2;
   	
   d2 = atlas_inl(port&ATLAS_ALIGN_MASK);	
	
   if((port%ATLAS_ALIGN)>=(ATLAS_ALIGN-3))
   {
      u32 v1 = (d1<<(port%ATLAS_ALIGN)*8);
      u32 v2 = (d1>>(ATLAS_ALIGN-(port%ATLAS_ALIGN))*8);
      atlas_outl((d2&~(((u32)ATLAS_DWORD_MASK)<<((port%ATLAS_ALIGN)*8)))|v1, port&ATLAS_ALIGN_MASK);
      d2 = atlas_inl((port/ATLAS_ALIGN)*ATLAS_ALIGN+ATLAS_ALIGN);
      atlas_outl((d2&~((u32)ATLAS_DWORD_MASK>>(ATLAS_ALIGN-port%ATLAS_ALIGN)*8))|v2, (port&ATLAS_ALIGN_MASK)+ATLAS_ALIGN);            	
   }
   else
   {
      atlas_outl(d1, port&ATLAS_ALIGN_MASK);	   	
   }      
}

static __inline__
void ATLAS_INSW(u32 port, void *to, s32 len)
{
	s32 i;

	for (i = 0; i < len; i++)
		((u16 *) to)[i] = ATLAS_INW(port);
}

static __inline__
void ATLAS_INSWB(u32 port, void *to, s32 len)
{
	ATLAS_INSW(port, to, len >> 2);
}

static __inline__
void ATLAS_OUTSW(u32 port, const void *from, s32 len)
{
	s32 i;

	for (i = 0; i < len; i++)
		ATLAS_OUTW(((unsigned short *) from)[i], port);		
}

static __inline__
void ATLAS_OUTSWB(u32 port, const void *from, s32 len)
{
	ATLAS_OUTSW(port, from, len >> 2);
}

#else  // ATLAS_INOUT
#define ATLAS_INB	inb
#define	ATLAS_INW	inw
#define ATLAS_INL	inl
#define ATLAS_OUTB	outb
#define ATLAS_OUTW	outw
#define ATLAS_OUTL	outl
#define ATLAS_INSW	insw
#define ATLAS_INSWB	inswb
#define ATLAS_OUTSW	outsw
#define ATLAS_OUTSWB	outswbs
#endif

#define ATLAS_ROWB(data, port)		ATLAS_OUTB(ATLAS_INB(port) | data, port)
#define ATLAS_ROWW(data, port)		ATLAS_OUTW(ATLAS_INW(port) | data, port)
#define ATLAS_ROWL(data, port)		ATLAS_OUTL(ATLAS_INL(port) | data, port)

#define ATLAS_RAWB(data, port)		ATLAS_OUTB(ATLAS_INB(port) & data, port)
#define ATLAS_RAWW(data, port)		ATLAS_OUTW(ATLAS_INW(port) & data, port)
#define ATLAS_RAWL(data, port)		ATLAS_OUTL(ATLAS_INL(port) & data, port)


/************************************************************************
 * ATLAS registers
 ************************************************************************/
//#define AHB_MHZ                 (UMVP2500_SYS_CLK_FREQ/UMVP2500_SYS_CLK_DIV)  /*  AHB bus speed */
#define AHB_MHZ                  	100 /*  IDE Host speed */
#define ATLAS_MAC_PCI			50       /* MHz */

#if ATLAS_PCI
#define ATLAS_MAC_CLOCK			ATLAS_MAC_PCI
#endif

#if ATLAS_AHB
#define ATLAS_MAC_CLOCK			AHB_MHZ
#endif

#define ATLAS_WAIT_TIME			400  /* ns */
#define ROUNDUP(x)			(((ATLAS_MAC_CLOCK * x + 999) /1000) - 1)
#define ATLAS_WAIT_PARM			ROUNDUP(ATLAS_WAIT_TIME)

/* Control Register */
#define ATLAS_CONTROL_START		(1<<0)
#define ATLAS_CONTROL_RW		(1<<3)  /* 1 as read */
#define ATLAS_CONTROL_XFER_NONE		(0<<8)
#define ATLAS_CONTROL_XFER_PIO		(1<<8)
#define ATLAS_CONTROL_XFER_MDMA		(2<<8)
#define ATLAS_CONTROL_XFER_UDMA		(3<<8)
#define ATLAS_CONTROL_48BIT		(1<<10)
#define ATLAS_CONTROL_HSRST		(1<<15)

/* Status Register */
#define ATLAS_STATUS_ACTIVE		(1<<0)
#define ATLAS_STATUS_CRF		(1<<1)
#define ATLAS_STATUS_INT		(1<<2)
#define ATLAS_STATUS_DESC_ERR		(1<<3)
#define ATLAS_STATUS_TX_ERR		(1<<4)
#define ATLAS_STATUS_RX_ERR		(1<<5)
#define ATLAS_STATUS_INIT_OK		(1<<6)	

/* Timing Parameter Register 1 */
#define ATLAS_TP1_DMA1(x)		((x&31)<<0)  /* 5 bits */
#define ATLAS_TP1_DMA2(x)		((x&31)<<8)
#define ATLAS_TP1_DMA3(x)		((x&31)<<16)
#define ATLAS_TP1_DMA4(x)		((x&31)<<24)

/* Timing Parameter Register 2 */
#define ATLAS_TP2_WAIT(x)		((x&127)<<0) /* 7 bits */
#define ATLAS_TP2_PIO(x)		((x&31)<<8)  /* 5 bits */

/* Device Control Register */
#define ATLAS_DC_ENABLE			(0<<0)
#define ATLAS_DC_NIEN			(1<<1)
#define ATLAS_DC_SRST			(1<<2)

/* MAC Register */
#define ATLAS_ATA_STATUS(x)		(x&255) /* 8 bits */
#define ATLAS_ATA_ERROR(x)		(x&255) /* 8 bits */
#define ATLAS_MAC_STATUS(x)		(x&1)  /* 1 bit */

#define ATLAS_TEST_FEATURE		(1<<0)
#define ATLAS_TEST_ERROR		(1<<0)
#define ATLAS_TEST_SECTOR		(1<<1)
#define ATLAS_TEST_LBAL			(1<<2)
#define ATLAS_TEST_LBAM			(1<<3)
#define ATLAS_TEST_LBAH			(1<<4)
#define ATLAS_TEST_DEVICE		(1<<5)
#define ATLAS_TEST_COMMAND		(1<<6)
#define ATLAS_TEST_STATUS		(1<<6)
#define ATLAS_TEST_PIODATA		(1<<7)
#define ATLAS_TEST_DEVCBR_SHIFT		8


#define ClearAtlasInt(pChan)	\
	ATLAS_ROWB(ATLAS_STATUS_INT, pChan->io_ports[ATLAS_STATUS]);
		
    
/************************************************************************
 * Atlas test mode structure
 ************************************************************************/
#if ATLAS_TESTMODE

#include "atlas_test.h"

#endif

/************************************************************************
 * Following structures are according to SPC-3 (by Chanel)
 ************************************************************************/
typedef struct _SCSI_MODE_SENSE6
{
 u8		OperationCode;
 u8		Reserved1 : 3;
 u8		Dbd : 1;
 u8		Reserved2 : 4;
 u8		PageCode : 6;
 u8		Pc : 2;
 u8		SubpageCode;
 u8		AllocationLength;
 u8		Control;
} SCSI_MODE_SENSE6, *PSCSI_MODE_SENSE6;

typedef struct _SCSI_MODE_SENSE10
{
 u8		OperationCode;
 u8		Reserved1 : 3;
 u8		Dbd : 1;
 u8		LLBAA : 1;
 u8		Reserved2 : 3;
 u8		PageCode : 6;
 u8		Pc : 2;
 u8		SubpageCode;
 u8		Reserved3[3];
 u8		AllocationLengthMsb;
 u8		AllocationLengthLsb;
 u8		Control;
} SCSI_MODE_SENSE10, *PSCSI_MODE_SENSE10;

typedef struct _SCSI_MODE_SELECT6
{
 u8		OperationCode;
 u8		SPBit : 1;
 u8		Reserved1 : 3;
 u8		PFBit : 1;
 u8		Reserved2 : 3;
 u8		Reserved3[2];
 u8		ParameterListLength;
 u8		Control;
} SCSI_MODE_SELECT6, *PSCSI_MODE_SELECT6;

typedef struct _SCSI_MODE_SELECT10
{
 u8		OperationCode;
 u8		SPBit : 1;
 u8		Reserved1 : 3;
 u8		PFBit : 1;
 u8		Reserved2 : 3;
 u8		Reserved3[5];
 u8		ParameterListLengthMsb;
 u8		ParameterListLengthLsb;
 u8		Control;
} SCSI_MODE_SELECT10, *PSCSI_MODE_SELECT10;

typedef struct _SCSI_MODE_PARAMETER_HEADER6
{
 u8		ModeDataLength;
 u8		MediumType;
 u8		DeviceSpecificParameter;
 u8		BlockDescriptorLength;
} SCSI_MODE_PARAMETER_HEADER6, *PSCSI_MODE_PARAMETER_HEADER6;

typedef struct _SCSI_MODE_PARAMETER_HEADER10
{
 u8		ModeDataLengthMsb;
 u8		ModeDataLengthLsb;
 u8		MediumType;
 u8		DeviceSpecificParameter;
 u8		Reserved[2];
 u8		BlockDescriptorLengthMsb;
 u8		BlockDescriptorLengthLsb;
} SCSI_MODE_PARAMETER_HEADER10, *PSCSI_MODE_PARAMETER_HEADER10;

typedef struct _SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER
{
 u8		DesityCode;
 u8		NumberOfBlocks2;
 u8		NumberOfBlocks1;
 u8		NumberOfBlocks0;
 u8		Reserved;
 u8		BlockLength2;
 u8		BlockLength1;
 u8		BlockLength0;
} SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER, *PSCSI_MODE_PARAMTER_BLOCK_DESCRIPTER;

/************************************************************************
 * IDE command definitions
 ************************************************************************/
#define IDE_COMMAND_NOP				0x00
#define IDE_COMMAND_ATAPI_RESET			0x08
#define IDE_COMMAND_RECALIBRATE			0x10
#define IDE_COMMAND_READ_SECTOR			0x20
#define IDE_COMMAND_READ_SECTOR_EXT		0x24
#define IDE_COMMAND_READ_DMA_EXT		0x25
#define IDE_COMMAND_READ_MULTIPLE_EXT		0x29
#define IDE_COMMAND_WRITE_SECTOR		0x30
#define IDE_COMMAND_WRITE_SECTOR_EXT		0x34
#define IDE_COMMAND_WRITE_DMA_EXT		0x35
#define IDE_COMMAND_WRITE_MULTIPLE_EXT		0x39
#define IDE_COMMAND_READ_VERIFY			0x40
#define IDE_COMMAND_READ_VERIFY_EXT		0x42
#define IDE_COMMAND_SEEK			0x70
#define IDE_COMMAND_SET_DRIVE_PARAMETERS	0x91
#define IDE_COMMAND_ATAPI_PACKET		0xA0
#define IDE_COMMAND_ATAPI_IDENTIFY		0xA1
#define IDE_COMMAND_READ_MULTIPLE		0xC4
#define IDE_COMMAND_WRITE_MULTIPLE		0xC5
#define IDE_COMMAND_SET_MULTIPLE		0xC6
#define IDE_COMMAND_READ_DMA			0xC8
#define IDE_COMMAND_WRITE_DMA			0xCA
#define IDE_COMMAND_GET_MEDIA_STATUS		0xDA
#define IDE_COMMAND_ENABLE_MEDIA_STATUS		0xEF
#define IDE_COMMAND_SET_FEATURE			0xEF
#define IDE_COMMAND_IDENTIFY			0xEC
#define IDE_COMMAND_MEDIA_EJECT			0xED

/************************************************************************
 * IDE status definitions
 ************************************************************************/
#define IDE_STATUS_ERROR			0x01
#define IDE_STATUS_INDEX			0x02
#define IDE_STATUS_CORRECTED_ERROR		0x04
#define IDE_STATUS_DRQ				0x08
#define IDE_STATUS_DSC				0x10
#define IDE_STATUS_DRDY				0x40
#define IDE_STATUS_IDLE				0x50
#define IDE_STATUS_BUSY				0x80

/************************************************************************
 * IDE drive control definitions
 ************************************************************************/
#define IDE_DC_DISABLE_INTERRUPTS		0x02
#define IDE_DC_RESET_CONTROLLER			0x04
#define IDE_DC_REENABLE_CONTROLLER		0x00

/************************************************************************
 * IDE error definitions
 ************************************************************************/
#define IDE_ERROR_BAD_BLOCK			0x80
#define IDE_ERROR_DATA_ERROR			0x40
#define IDE_ERROR_MEDIA_CHANGE			0x20
#define IDE_ERROR_ID_NOT_FOUND			0x10
#define IDE_ERROR_MEDIA_CHANGE_REQ		0x08
#define IDE_ERROR_COMMAND_ABORTED		0x04
#define IDE_ERROR_END_OF_MEDIA			0x02
#define IDE_ERROR_ILLEGAL_LENGTH		0x01

/************************************************************************
 * IDENTIFY data
 ************************************************************************/
typedef struct _IDENTIFY_DATA
{
 u16		GeneralConfiguration;		/* 00 00		*/
 u16		NumberOfCylinders;		/* 02  1		*/
 u16		Reserved1;			/* 04  2		*/
 u16		NumberOfHeads;			/* 06  3		*/
 u16		UnformattedBytesPerTrack;	/* 08  4		*/
 u16		UnformattedBytesPerSector;	/* 0A  5		*/
 u16		SectorsPerTrack;		/* 0C  6		*/
 u16		VendorUnique1[3];		/* 0E  7-9		*/
 u16		SerialNumber[10];		/* 14  10-19		*/
 u16		BufferType;			/* 28  20		*/
 u16		BufferSectorSize;		/* 2A  21		*/
 u16		NumberOfEccBytes;		/* 2C  22		*/
 u16		FirmwareRevision[4];		/* 2E  23-26		*/
 u16		ModelNumber[20];		/* 36  27-46		*/
 u8		MaximumBlockTransfer;		/* 5E  47		*/
 u8		VendorUnique2;			/* 5F			*/
 u16		DoubleWordIo;			/* 60  48		*/
 u16		Capabilities;			/* 62  49		*/
 u16		Reserved2;			/* 64  50		*/
 u8		VendorUnique3;			/* 66  51		*/
 u8		PioCycleTimingMode;		/* 67			*/
 u8		VendorUnique4;			/* 68  52		*/
 u8		DmaCycleTimingMode;		/* 69			*/
 u16		TranslationFieldsValid : 1;	/* 6A  53		*/
 u16		Reserved3 : 15;			/*			*/
 u16		NumberOfCurrentCylinders;	/* 6C  54		*/
 u16		NumberOfCurrentHeads;		/* 6E  55		*/
 u16		CurrentSectorsPerTrack;		/* 70  56		*/
 u32		CurrentSectorCapacity;		/* 72  57-58		*/
 u16		CurrentMultiSectorSetting;	/*     59		*/
 u32		UserAddressableSectors;		/*     60-61		*/
 u16		SingleWordDMASupport : 8;	/*     62		*/
 u16		SingleWordDMAActive : 8;	/*			*/
 u16		MultiWordDMASupport : 8;	/*     63		*/
 u16		MultiWordDMAActive : 8;		/*			*/
 u16		AdvancedPIOModes : 8;		/*     64		*/
 u16		Reserved4 : 8;			/*			*/
 u16		MinimumMWXferCycleTime;		/*     65		*/
 u16		RecommendedMWXferCycleTime;	/*     66		*/
 u16		MinimumPIOCycleTime;		/*     67		*/
 u16		MinimumPIOCycleTimeIORDY;	/*     68		*/
 u16		Reserved5[2];			/*     69-70		*/
 u16		ReleaseTimeOverlapped;		/*     71		*/
 u16		ReleaseTimeServiceCommand;	/*     72		*/
 u16		MajorRevision;			/*     73		*/
 u16		MinorRevision;			/*     74		*/
 u16		Reserved6[50];			/*     75-126		*/
 u16		SpecialFunctionsEnabled;	/*     127		*/
 u16		Reserved7[128];			/*     128-255		*/
} IDENTIFY_DATA, *PIDENTIFY_DATA;

/************************************************************************
 * Identify data without the Reserved4
 ************************************************************************/
typedef struct _IDENTIFY_DATA2
{
 u16		GeneralConfiguration;		/* 00			*/
 u16		NumberOfCylinders;		/* 01			*/
 u16		Reserved1;			/* 02			*/
 u16		NumberOfHeads;			/* 03			*/
 u16		Reserved2[2];			/* 04-05		*/
 u16		SectorsPerTrack;		/* 06			*/
 u16		Reserved3[3];			/* 07-09		*/
 u16		SerialNumber[10];		/* 10-19		*/
 u16		Reserved4[3];			/* 20-22		*/
 u16		FirmwareRevision[4];		/* 23-26		*/
 u16		ModelNumber[20];		/* 27-46		*/
 u16		MaximumBlockTransfer;		/* 47			*/
 u16		Reserved5;			/* 48			*/
 u16		Capabilities[2];		/* 49-50		*/
 u16		Reserved6[2];			/* 51-52		*/
 u16		ValidFieldIndicator;		/* 53			*/
 u16		NumberOfCurrentCylinders;	/* 54			*/
 u16		NumberOfCurrentHeads;		/* 55			*/
 u16		CurrentSectorsPerTrack;		/* 56			*/
 u16		CurrentSectorCapacityLow;	/* 57			*/
 u16		CurrentSectorCapacityHigh;	/* 58			*/
 u16		CurrentMultiSectorSetting;	/* 59			*/
 u32		UserAddressableSectors;		/* 60-61		*/
 u16		Reserved7;			/* 62			*/
 u8		MultiWordDMASupport;		/* 63			*/
 u8		MultiWordDMAActive;		/*			*/
 u16		AdvancedPIOModes;		/* 64			*/
 u16		MinimumMWXferCycleTime;		/* 65			*/
 u16		RecommendedMWXferCycleTime;	/* 66			*/
 u16		MinimumPIOCycleTime;		/* 67			*/
 u16		MinimumPIOCycleTimeIORDY;	/* 68			*/
 u16		Reserved8[6];			/* 69-74		*/
 u16		QueueDepth;			/* 75			*/
 u16		Reserved9[4];			/* 76-79		*/
 u16		MajorVersionNumber;		/* 80			*/
 u16		MinorVersionNumber;		/* 81			*/
 u32		CmdSetSupported;		/* 82-83		*/
 u16		CmdSetFeatureSupportedExt;	/* 84			*/
 u16		CmdSetFeatureEnabledLow;	/* 85			*/
 u16		CmdSetFeatureEnabledHigh;	/* 86			*/
 u16		CmdSetFeatureDefault;		/* 87			*/
 u8		UltraDMASupport;		/* 88			*/
 u8		UltraDMAActive;			/*			*/
 u16		SecurityEraseTime;		/* 89			*/
 u16		EnhancedSecurityEraseTime;	/* 90			*/
 u16		PowerManagementValue;		/* 91			*/
 u16		MasterPasswordRevision;		/* 92			*/
 u16		HwResetResult;			/* 93			*/
 u16		Reserved11[6];			/* 94-99		*/
 u32		Capacity_48bit_LOW;		/* 100-101		*/
 u32		Capacity_48bit_HIGH;		/* 102-103		*/
 u16		Reserved12[24];			/* 104-127		*/
 u16		SecurityStatus;			/* 128			*/
 u16		Reserved13[31];			/* 129-159 vendor spec  */
 u16		Reserved14[96];			/* 160-255		*/
} IDENTIFY_DATA2, *PIDENTIFY_DATA2;

#define IDENTIFY_DATA_SIZE sizeof(IDENTIFY_DATA)

/************************************************************************
 * IDENTIFY capability bit definitions.
 ************************************************************************/
#define IDENTIFY_CAPABILITIES_DMA_SUPPORTED	0x0100
#define IDENTIFY_CAPABILITIES_LBA_SUPPORTED	0x0200

/************************************************************************
 * IDENTIFY DMA timing cycle modes.
 ************************************************************************/
#define IDENTIFY_DMA_CYCLES_MODE_0		0x00
#define IDENTIFY_DMA_CYCLES_MODE_1		0x01
#define IDENTIFY_DMA_CYCLES_MODE_2		0x02

typedef struct _SENSE_DATA
{
 u8		ErrorCode:7;
 u8		Valid:1;
 u8		SegmentNumber;
 u8		SenseKey:4;
 u8		Reserved:1;
 u8		IncorrectLength:1;
 u8		EndOfMedia:1;
 u8		FileMark:1;
 u8		Information[4];
 u8		AdditionalSenseLength;
 u8		CommandSpecificInformation[4];
 u8		AdditionalSenseCode;
 u8		AdditionalSenseCodeQualifier;
 u8		FieldReplaceableUnitCode;
 u8		SenseKeySpecific[3];
} SENSE_DATA, *PSENSE_DATA;

/************************************************************************
 * Sense codes
 ************************************************************************/
#define SCSI_SENSE_NO_SENSE		0x00
#define SCSI_SENSE_RECOVERED_ERROR	0x01
#define SCSI_SENSE_NOT_READY		0x02
#define SCSI_SENSE_MEDIUM_ERROR		0x03
#define SCSI_SENSE_HARDWARE_ERROR	0x04
#define SCSI_SENSE_ILLEGAL_REQUEST	0x05
#define SCSI_SENSE_UNIT_ATTENTION	0x06
#define SCSI_SENSE_DATA_PROTECT		0x07
#define SCSI_SENSE_BLANK_CHECK		0x08
#define SCSI_SENSE_UNIQUE		0x09
#define SCSI_SENSE_COPY_ABORTED		0x0A
#define SCSI_SENSE_ABORTED_COMMAND	0x0B
#define SCSI_SENSE_EQUAL		0x0C
#define SCSI_SENSE_VOL_OVERFLOW		0x0D
#define SCSI_SENSE_MISCOMPARE		0x0E
#define SCSI_SENSE_RESERVED		0x0F

/************************************************************************
 * Additional Sense codes
 ************************************************************************/
#define SCSI_ADSENSE_NO_SENSE		0x00
#define SCSI_ADSENSE_MAN_INTERV		0x03
#define SCSI_ADSENSE_LUN_NOT_READY	0x04
#define SCSI_ADSENSE_ILLEGAL_COMMAND	0x20
#define SCSI_ADSENSE_ILLEGAL_BLOCK	0x21
#define SCSI_ADSENSE_INVALID_LUN	0x25
#define SCSI_ADSENSE_SELECT_TIMEOUT	0x45
#define SCSI_ADSENSE_MUSIC_AREA		0xA0
#define SCSI_ADSENSE_DATA_AREA		0xA1
#define SCSI_ADSENSE_VOLUME_OVERFLOW	0xA7

#define SCSI_ADSENSE_NO_MEDIA_IN_DEVICE	0x3A
#define SCSI_ADWRITE_PROTECT		0x27
#define SCSI_ADSENSE_MEDIUM_CHANGED	0x28
#define SCSI_ADSENSE_BUS_RESET		0x29
#define SCSI_ADSENSE_TRACK_ERROR	0x14
#define SCSI_ADSENSE_SEEK_ERROR		0x15
#define SCSI_ADSENSE_REC_DATA_NOECC	0x17
#define SCSI_ADSENSE_REC_DATA_ECC	0x18
#define SCSI_ADSENSE_ILLEGAL_MODE	0x64
#define SCSI_ADSENSE_BAD_CDB		0x24
#define SCSI_ADSENSE_BAD_PARM_LIST	0x26
#define SCSI_ADSENSE_CANNOT_READ_MEDIUM	0x30

#define SCSISTAT_CHECK_CONDITION	0x02

/************************************************************************
 * Inquiry buffer structure. This is the data returned from the target
 * after it receives an inquiry.
 *
 * This structure may be extended by the number of bytes specified
 * in the field AdditionalLength. The defined size constant only
 * includes fields through ProductRevisionLevel.
 *
 * The NT SCSI drivers are only interested in the first 36 bytes of data.
 ************************************************************************/

#define INQUIRYDATABUFFERSIZE 36

typedef struct _INQUIRYDATA
{
 u8		DeviceType : 5;
 u8		DeviceTypeQualifier : 3;
 u8		DeviceTypeModifier : 7;
 u8		RemovableMedia : 1;
 u8		Versions;
 u8		ResponseDataFormat;
 u8		AdditionalLength;
 u8		Reserved[2];
 u8		SoftReset : 1;
 u8		CommandQueue : 1;
 u8		Reserved2 : 1;
 u8		LinkedCommands : 1;
 u8		Synchronous : 1;
 u8		Wide16Bit : 1;
 u8		Wide32Bit : 1;
 u8		RelativeAddressing : 1;
 u8		VendorId[8];
 u8		ProductId[16];
 u8		ProductRevisionLevel[4];
 u8		VendorSpecific[20];
 u8		Reserved3[40];
} INQUIRYDATA, *PINQUIRYDATA;

#define DIRECT_ACCESS_DEVICE		0x00	/* Disks		*/

/************************************************************************
 * Read Capacity Data - returned in Big Endian format
 ************************************************************************/
typedef struct _READ_CAPACITY_DATA
{
 u32		LogicalBlockAddress;
 u32		BytesPerBlock;
} READ_CAPACITY_DATA, *PREAD_CAPACITY_DATA;

#define MAXIMUM_CDB_SIZE	12

/************************************************************************
 * CDB (Command Descriptor Block)
 ************************************************************************/
typedef union _CDB
{
 /*
  * Standard 6-byte CDB
  */
 struct _CDB6READWRITE
    {
     u8		OperationCode;		/* Opcode			*/
     u8		LogicalBlockMsb1 : 5;	/* Logical block MSB 5-bit	*/
     u8		LogicalUnitNumber : 3;	/* LUN				*/
     u8		LogicalBlockMsb0;	/* Logical block MSB 8-bit	*/
     u8		LogicalBlockLsb;	/* Logical block LSB 8-bit	*/
     u8		TransferBlocks;		/* Data length			*/
     u8		Control;		/* Control byte			*/
    } CDB6READWRITE, *PCDB6READWRITE;

 /*
  * Standard 10-byte CDB
  */
 struct _CDB10
    {
     u8		OperationCode;
     u8		Reserved1 : 5;
     u8		LogicalUnitNumber : 3;
     u8		LogicalBlockByte0;
     u8		LogicalBlockByte1;
     u8		LogicalBlockByte2;
     u8		LogicalBlockByte3;
     u8		Reserved2;
     u8		TransferBlocksMsb;
     u8		TransferBlocksLsb;
     u8		Control;
    } CDB10, *PCDB10;

 struct _START_STOP
    {
     u8		OperationCode;
     u8		Immediate: 1;
     u8		Reserved1 : 4;
     u8		LogicalUnitNumber : 3;
     u8		Reserved2[2];
     u8		Start : 1;
     u8		LoadEject : 1;
     u8		Reserved3 : 6;
     u8		Control;
    } START_STOP, *PSTART_STOP;

} CDB, *PCDB;

/************************************************************************
 * SCSI CDB operation codes
 ************************************************************************/
#define SCSIOP_TEST_UNIT_READY		0x00
#define SCSIOP_REZERO_UNIT		0x01
#define SCSIOP_REWIND			0x01
#define SCSIOP_REQUEST_BLOCK_ADDR	0x02
#define SCSIOP_REQUEST_SENSE		0x03
#define SCSIOP_FORMAT_UNIT		0x04
#define SCSIOP_READ_BLOCK_LIMITS	0x05
#define SCSIOP_REASSIGN_BLOCKS		0x07
#define SCSIOP_READ6			0x08
#define SCSIOP_RECEIVE			0x08
#define SCSIOP_WRITE6			0x0A
#define SCSIOP_PRINT			0x0A
#define SCSIOP_SEND			0x0A
#define SCSIOP_SEEK6			0x0B
#define SCSIOP_TRACK_SELECT		0x0B
#define SCSIOP_SLEW_PRINT		0x0B
#define SCSIOP_SEEK_BLOCK		0x0C
#define SCSIOP_PARTITION		0x0D
#define SCSIOP_READ_REVERSE		0x0F
#define SCSIOP_WRITE_FILEMARKS		0x10
#define SCSIOP_FLUSH_BUFFER		0x10
#define SCSIOP_SPACE			0x11
#define SCSIOP_INQUIRY			0x12
#define SCSIOP_VERIFY6			0x13
#define SCSIOP_RECOVER_BUF_DATA		0x14
#define SCSIOP_MODE_SELECT		0x15
#define SCSIOP_RESERVE_UNIT		0x16
#define SCSIOP_RELEASE_UNIT		0x17
#define SCSIOP_COPY			0x18
#define SCSIOP_ERASE			0x19
#define SCSIOP_MODE_SENSE		0x1A
#define SCSIOP_START_STOP_UNIT		0x1B
#define SCSIOP_STOP_PRINT		0x1B
#define SCSIOP_LOAD_UNLOAD		0x1B
#define SCSIOP_RECEIVE_DIAGNOSTIC	0x1C
#define SCSIOP_SEND_DIAGNOSTIC		0x1D
#define SCSIOP_MEDIUM_REMOVAL		0x1E
#define SCSIOP_READ_CAPACITY		0x25
#define SCSIOP_READ			0x28
#define SCSIOP_WRITE			0x2A
#define SCSIOP_SEEK			0x2B
#define SCSIOP_LOCATE			0x2B
#define SCSIOP_WRITE_VERIFY		0x2E
#define SCSIOP_VERIFY			0x2F
#define SCSIOP_SEARCH_DATA_HIGH		0x30
#define SCSIOP_SEARCH_DATA_EQUAL	0x31
#define SCSIOP_SEARCH_DATA_LOW		0x32
#define SCSIOP_SET_LIMITS		0x33
#define SCSIOP_READ_POSITION		0x34
#define SCSIOP_SYNCHRONIZE_CACHE	0x35
#define SCSIOP_COMPARE			0x39
#define SCSIOP_COPY_COMPARE		0x3A
#define SCSIOP_WRITE_DATA_BUFF		0x3B
#define SCSIOP_READ_DATA_BUFF		0x3C
#define SCSIOP_CHANGE_DEFINITION	0x40
#define SCSIOP_READ_SUB_CHANNEL		0x42
#define SCSIOP_READ_TOC			0x43
#define SCSIOP_READ_HEADER		0x44
#define SCSIOP_PLAY_AUDIO		0x45
#define SCSIOP_PLAY_AUDIO_MSF		0x47
#define SCSIOP_PLAY_TRACK_INDEX		0x48
#define SCSIOP_PLAY_TRACK_RELATIVE	0x49
#define SCSIOP_PAUSE_RESUME		0x4B
#define SCSIOP_LOG_SELECT		0x4C
#define SCSIOP_LOG_SENSE		0x4D
#define SCSIOP_MODE_SELECT10		0x55
#define SCSIOP_MODE_SENSE10		0x5A
#define SCSIOP_LOAD_UNLOAD_SLOT		0xA6
#define SCSIOP_MECHANISM_STATUS		0xBD
#define SCSIOP_READ_CD			0xBE

#define DRIVER_NAME		"Device Driver for ATLAS ATA/ATAPI Controller"
#define COMPANY_NAME		"Global Unichip, Inc."
#define CONTROLLER_NAME_Atlas	"ATLAS ATA/ATAPI Controller"
#define ATLAS_DEV_NAME		"atlas"
#define ATLAS_DEV_MINOR		0
#define ATLAS_MAX_CMDS		124

#define PCI_IOSEN	0x01	/* Enable IO space			*/
#define PCI_BMEN	0x04	/* Enable IDE bus master		*/

/************************************************************************
 * PRD (Physical Region Descriptor) = Scatter-gather table
 *
 * |  byte3   |	 byte2	 |  byte1   |   byte0   |
 * +--------------------------------------------+
 * | Memory Region Physical Base Address[31:1]  |
 * +----+----------------+----------------------+
 * |EOT |  reserved      |   Byte count[15:1]   |
 * +----+----------------+----------------------+
 ************************************************************************/
typedef struct _PRD_TABLE_ENTRY
{
 u32		PhysicalBaseAddress;		/* Byte0 - Byte3	*/
 u16		ByteCount;			/* Byte4 - Byte5	*/
 u16		EndOfTable;			/* Byte6 - Byte7	*/
} PRD_TABLE_ENTRY, *PPRD_TABLE_ENTRY;

#define SG_FLAG_EOT			0x8000	/* End of PRD		*/
#define MAX_SG_DESCRIPTORS		17	/* 17 -- maximum 64K	*/

#define NUM_OF_PRD_TABLE_ENTRY		0x10

/************************************************************************
 * Bus master register bits definition
 ************************************************************************/
#define BM_CMD_FLG_START		0x01
#define BM_CMD_FLG_WRTTOMEM		0x08
#define BM_CMD_FLG_WRTTODSK		0x00

#define BM_STAT_FLG_ACTIVE		0x01
#define BM_STAT_FLG_ERR			0x02
#define BM_STAT_FLG_INT			0x04
#define BM_DRV0_DMA_CAPABLE		0x20
#define BM_DRV1_DMA_CAPABLE		0x40

#define BM_PRD_FLG_EOT  		0x8000

/************************************************************************
 * SRB Functions
 ************************************************************************/
#define SRB_FUNCTION_EXECUTE_SCSI	0x00
#define SRB_FUNCTION_IO_CONTROL		0x02
#define SRB_FUNCTION_SHUTDOWN		0x07
#define SRB_FUNCTION_FLUSH		0x08

/************************************************************************
 * SRB Status
 ************************************************************************/
#define SRB_STATUS_PENDING		0x00
#define SRB_STATUS_SUCCESS		0x01
#define SRB_STATUS_ABORTED		0x02
#define SRB_STATUS_ABORT_FAILED		0x03
#define SRB_STATUS_ERROR		0x04
#define SRB_STATUS_BUSY			0x05
#define SRB_STATUS_INVALID_REQUEST	0x06
#define SRB_STATUS_INVALID_PATH_ID	0x07
#define SRB_STATUS_NO_DEVICE		0x08
#define SRB_STATUS_TIMEOUT		0x09
#define SRB_STATUS_SELECTION_TIMEOUT	0x0A
#define SRB_STATUS_COMMAND_TIMEOUT	0x0B
#define SRB_STATUS_MESSAGE_REJECTED	0x0D
#define SRB_STATUS_BUS_RESET		0x0E
#define SRB_STATUS_PARITY_ERROR		0x0F
#define SRB_STATUS_REQUEST_SENSE_FAILED	0x10
#define SRB_STATUS_NO_HBA		0x11
#define SRB_STATUS_DATA_OVERRUN		0x12
#define SRB_STATUS_UNEXPECTED_BUS_FREE	0x13
#define SRB_STATUS_BAD_SRB_BLOCK_LENGTH	0x15
#define SRB_STATUS_REQUEST_FLUSHED	0x16
#define SRB_STATUS_INVALID_LUN		0x20
#define SRB_STATUS_INVALID_TARGET_ID	0x21
#define SRB_STATUS_BAD_FUNCTION		0x22
#define SRB_STATUS_ERROR_RECOVERY	0x23
#define SRB_STATUS_NEED_REQUEUE		0x24

/************************************************************************
 * SRB Status Masks
 ************************************************************************/
#define SRB_STATUS_QUEUE_FROZEN		0x40
#define SRB_STATUS_AUTOSENSE_VALID	0x80

#define SRB_STATUS(Status)	\
	(Status & ~(SRB_STATUS_AUTOSENSE_VALID | SRB_STATUS_QUEUE_FROZEN))

/************************************************************************
 * SRB Flag Bits
 ************************************************************************/
#define SRB_FLAGS_DATA_IN		0x00000040
#define SRB_FLAGS_DATA_OUT		0x00000080

/************************************************************************
 * SRB Working flags define area
 ************************************************************************/
#define	SRB_WFLAGS_USE_INTERNAL_BUFFER	0x00000001
#define	SRB_WFLAGS_IGNORE_ARRAY		0x00000002
#define	SRB_WFLAGS_HAS_CALL_BACK	0x00000004
#define	SRB_WFLAGS_MUST_DONE		0x00000008
#define	SRB_WFLAGS_ON_MIRROR_DISK	0x00000010
#define	SRB_WFLAGS_ON_SOURCE_DISK	0x00000020
#define SRB_WFLAGS_ARRAY_IO_STARTED	0x10000000
#define SRB_WFLAGS_WATCHTIMER_CALLED	0x20000000
#define SRB_WFLAGS_USE_SG		0x40000000

/************************************************************************
 * SCSI I/O Request Block
 ************************************************************************/
typedef struct _SCSI_REQUEST_BLOCK
{
 u16			Length;
 u8			Function;
 u8			SrbStatus;
 u8			ScsiStatus;
 u8			TargetId;
 u8			Lun;
 u8			CdbLength;
 u8			SenseInfoBufferLength;
 u8			UseSg;
 u8			reseved[2];
 u32			WorkingFlags;
 u32			SrbFlags;
 u32			DataTransferLength;
 u32			TimeOutValue;
 void *			DataBuffer;
 void *			SenseInfoBuffer;
 u8			Cdb[16];
 Scsi_Cmnd *		pREQ;
 u32			emulate_fault;
} SCSI_REQUEST_BLOCK, *PSCSI_REQUEST_BLOCK;

#define SCSI_REQUEST_BLOCK_SIZE sizeof(SCSI_REQUEST_BLOCK)

/************************************************************************
 * Second device flags
 ***********************************************************************/
#define DFLAGS_REDUCE_MODE	        0x00010000
#define DFLAGS_DEVICE_DISABLED		0x00020000
#define DFLAGS_BOOTABLE_DEVICE		0x00080000
#define DFLAGS_BOOT_MARK		0x00100000
#define DFLAGS_NEW_ADDED		0x40000000
#define DFLAGS_REMAINED_MEMBER		0x80000000

/************************************************************************
 * Device Extension Device Flags
 ************************************************************************/
/*
 * Indicates that some device is present.
 */
#define DFLAGS_DEVICE_PRESENT		0x0001

/*
 * Indicates whether ATAPI commands can be used.
 */
#define DFLAGS_ATAPI_DEVICE		0x0002

/*
 * Indicates whether this is a tape device.
 */
#define DFLAGS_TAPE_DEVICE		0x0004

/*
 * Indicates whether device interrupts as DRQ is set after
 * receiving Atapi Packet Command.
 */
#define DFLAGS_INT_DRQ			0x0008

/*
 * Indicates that the drive has the 'removable' bit set in
 * identify data (offset 128)
 */
#define DFLAGS_REMOVABLE_DRIVE		0x0010

/*
 * Media status notification enabled.
 */
#define DFLAGS_MEDIA_STATUS_ENABLED	0x0020

/*
 * Indicates atapi 2.5 changer present.
 */
#define DFLAGS_ATAPI_CHANGER		0x0040

/*
 * Indicates multi-platter device, not conforming to the 2.5 spec.
 */
#define DFLAGS_SANYO_ATAPI_CHANGER	0x0080

/*
 * Indicates that the init path for changers has already been done.
 */
#define DFLAGS_CHANGER_INITED		0x0100
#define DFLAGS_CONFIG_CHANGED		0x0200

#define UDMA_MODE_5_6			0x80

/************************************************************************
 * Used to disable 'advanced' features.
 ************************************************************************/
#define MAX_ERRORS			4

/************************************************************************
 * ATAPI command definitions
 ************************************************************************/
#define ATAPI_MODE_SENSE		0x5A
#define ATAPI_MODE_SELECT		0x55
#define ATAPI_FORMAT_UNIT		0x24

/************************************************************************
 * User IOCTL structure
 * Notes:
 * (1) Data transfers are limited to PAGE_SIZE (4k on i386, 8k for alpha)
 ************************************************************************/
typedef struct	_uioctl_t
{
 u16		inlen;		/* Length of data written to device	*/
 u16		outlen;		/* Length of data read from device	*/
 void *		data;		/* Data read from devic starts here	*/
 u8		status;		/* Status return from driver		*/
 u8		reserved[3];	/* For 4-byte alignment			*/
}		uioctl_t;

/************************************************************************
 * IOCTL commands for RAID
 ************************************************************************/
#define ATLAS_IOCMAGIC			't'

#define ATLAS_IOC_GET_PHY_DISK_STATUS	_IO(ATLAS_IOCMAGIC, 1)
#define ATLAS_IOC_CREATE_DISK_ARRAY	_IO(ATLAS_IOCMAGIC, 2)
#define ATLAS_IOC_REBUILD_START		_IO(ATLAS_IOCMAGIC, 3)
#define ATLAS_IOC_GET_REBUILD_STATUS	_IO(ATLAS_IOCMAGIC, 4)
#define ATLAS_IOC_RESET_ADAPTER		_IO(ATLAS_IOCMAGIC, 5)
#define ATLAS_IOC_GET_DRIVER_VERSION	_IO(ATLAS_IOCMAGIC, 6)

/************************************************************************
 * _Channel
 ************************************************************************/
typedef struct _Channel
{
 u32			phyaddr;
 u32			virtaddr;
 u32			addrlen;	
 /*
  * IDE (ATAPI) io port address.
  */
//Jason
 unsigned long		io_ports[IDE_NR_PORTS];
 //unsigned long		io_ports[38];

 /*
  * DMA base address.
  */
 unsigned long		dma_base;

 /*
  * Flags word for each possible device.
  */
 u16			DeviceFlags[2];

 /*
  * Indicates number of platters on changer-ish devices.
  */
 u32			DiscsPresent[2];

 /*
  * Indicates expecting an interrupt.
  */
 u8			ExpectingInterrupt;

 /*
  * Indicate last tape command was DSC Restrictive.
  */
 u8			RDP;

 /*
  * Interrupt level.
  */
 u8			InterruptLevel;

 /*
  * Placeholder for status register after a GET_MEDIA_STATUS command.
  */
 u8			ReturningMediaStatus;

 /*
  * Remember the channel number (0, 1)
  */
 u8			channel;

 /*
  * Indicates cable status.
  */
 u8			Cable80[2];

 /*
  * Reserved for alignment.
  */
 u8			reserved1[0];

 /*
  * Data buffer pointer.
  */
 unsigned short *	DataBuffer;

 /*
  * Data words left.
  */
 u32			WordsLeft;

 /*
  * Retry count.
  */
 u32			RetryCount;

 /*
  * Keep Selected PIO/MDMA/MDMA type (PIO_FLOW or MULTIWORD_DMA or ULTRA_DMA) for each device.
  */
 u8			TraType[2];
 
 /*
  * Keep current PIO/MDMA/UDMA Mode
  */  
 u8			PioMode[2];
 u8			DmaMode[2];
 
 /*
  * Keep Best PIO/MDMA/UDMA mode
  */
 u8			BestPioMode[2];
 u8			BestUdmaMode[2];
 u8			BestMdmaMode[2];
  

 /*
  * Keep the active device for each channel.
  */
 u8			ActiveDevice;

 /*
  * Indicate whether we should perform DMA mode switch on this channel?
  */
 u8			DoSwitch;

 /*
  * ???
  */
 u8			ConvertCdb;

 /*
  * Use or do not use DMA.
  */
 u8			UseDma[2];

 /*
  * Reserved for alignment.
  */
 u8			reserved2[3];

 /*
  * Identify data for device.
  */
 IDENTIFY_DATA		FullIdentifyData;
 IDENTIFY_DATA2		IdentifyData[2];

 /*
  * DMA PRD table physical address.
  */
 dma_addr_t		dmatable_dma;

 /*
  * DMA PRD table virtual address.
  */
 unsigned long *	dmatable_cpu;

 /*
  * Point to SCATTER/GATHER data buffer.
  */
 struct scatterlist *	sg_table;

 /*
  * DMA read or write.
  */
 int			sg_dma_direction;
 
 /*
  * PIO data access.
  */
 int			pio_access;

 /*
  * Current request on controller.
  */
 PSCSI_REQUEST_BLOCK	CurrentSrb;

 /*
  * Original request on controller.
  */
 PSCSI_REQUEST_BLOCK	OriginalSrb;

 /*
  * Internal SRB.
  */
 SCSI_REQUEST_BLOCK	_Srb;

 /*
  * Remember the PCI device.
  */
 struct pci_dev *	pPciDev;

 /*
  * Placeholder for CDB.
  */
 u8			TempCdb[MAXIMUM_CDB_SIZE];
 u8 *			PIOVirt;
 u32			PIOPhy;

 /*
  * Read/Write retry counter.
  */
 u8			RWRetry;
} Channel, *PChannel;

/************************************************************************
 * _Adapter
 ************************************************************************/
typedef struct _Adapter
{
 char *			name;		/* Adapter's name		*/
 u8			num_channels;	/* How many channels support	*/
 u8			irq;		/* irq number			*/
 u8			irqOwned;	/* If any irq is use		*/
 u8			pci_bus;	/* PCI bus number		*/
 u8			devfn;		/* Device and function number	*/
 u8			offline;	/* On line or off line		*/
 u8			bypass_mode;	/* bypass or firware mode	*/
 u8			reserved2[1];	/* Reserved for alignment	*/
 Channel *		IDEChannel;	/* Atlas supports two channels	*/
 struct pci_dev *	pci_dev;	/* For PCI device		*/
}ATLAS_ADAPTER, *PATLAS_ADAPTER;




/************************************************************************
 * Beautification macros
 ************************************************************************/
#define ScheduleRetryProcess(pChan) do {		\
	pChan->retry_timer->expires = jiffies + 10;	\
	add_timer(pChan->retry_timer);			\
	} while (0)

#define CancelRetryProcess(pChan) del_timer(pChan->retry_timer)

#define GetStatus(pChan, Status)	\
    Status = ATLAS_INB(pChan->io_ports[IDE_STATUS_OFFSET]);

#define GetBaseStatus(pChan, Status)	\
    Status = ATLAS_INB(pChan->io_ports[IDE_STATUS_OFFSET]);

#define GetAtlasStatus(pChan, Status)	\
    Status = ATLAS_INB(pChan->io_ports[ATLAS_STATUS]);
    
#define GetError(pChan, Error)		\
    Error = ATLAS_INB(pChan->io_ports[IDE_ERROR_OFFSET]);

/*
#define ReadBuffer(pChan, Buffer, Count) \
    ATLAS_INSW(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);

#define WriteCommand(BaseIoAddress, Command) \
    ATLAS_OUTB(pChan->io_ports[IDE_COMMAND_OFFSET], Command);

#define WriteBuffer(pChan, Buffer, Count) \
    ATLAS_OUTSB(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);    
*/
#define ReadBuffer(pChan, Buffer, Count) \
	AtlasPIOData(pChan, Buffer, Count*2, ATLAS_CONTROL_RW);

#define WriteBuffer(pChan, Buffer, Count) \
	AtlasPIOData(pChan, Buffer, Count*2, 0);

#define WaitOnAtlasInitial(pChan, Status)	\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetAtlasStatus(pChan, Status);		\
     if (Status & ATLAS_STATUS_INIT_OK)		\
	{					\
	 break;					\
	} 					\
     else					\
	{					\
	 mdelay(25);				\
	 continue;				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitOnBusy(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	}					\
     else 					\
	{					\
	 break;					\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitOnDevSRST(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     Status = ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]);	\
     if (Status & ATLAS_DC_SRST)		\
	{					\
	 udelay(150);				\
	 continue;				\
	}					\
     else 					\
	{					\
	 break;					\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitOnBaseBusy(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	} 					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitForDrq(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 1000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitForBaseDrq(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define CheckBusyDrq(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if ((Status & IDE_STATUS_BUSY) ||		\
	!(Status & IDE_STATUS_DRQ))		\
	{					\
	 udelay(200);				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitShortForDrq(pChan, Status)		\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 2; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(100);				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitForDeviceReady(pChan, Status)	\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status == 0)				\
	{					\
	 break;					\
	}					\
     if ((Status & IDE_STATUS_BUSY) || (Status & IDE_STATUS_DRQ)) \
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define WaitForCommandComplete(pChan, Status)		\
{							\
 int		i;					\
 ATLAS_INOUT_DBG_DIS;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetStatus(pChan, Status);				\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE))			\
      || (Status == (IDE_STATUS_IDLE|IDE_STATUS_INDEX)))\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
 ATLAS_INOUT_DBG_EN;					\
}

#define WaitForBaseCommandComplete(pChan, Status)	\
{							\
 int		i;					\
 ATLAS_INOUT_DBG_DIS;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetBaseStatus(pChan, Status);			\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE)			\
      || (Status == (IDE_STATUS_IDLE|IDE_STATUS_INDEX)))\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
 ATLAS_INOUT_DBG_EN;					\
}


#define AtapiSoftReset(pChan, DevNum)		\
{						\
 unsigned char		statusByte;		\
 ATLAS_INOUT_DBG_DIS;				\
 ATLAS_OUTB((unsigned char)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 udelay(500);									  \
 ATLAS_OUTB(IDE_COMMAND_ATAPI_RESET, pChan->io_ports[IDE_COMMAND_OFFSET]);		  \
 mdelay(1000);									  \
 ATLAS_OUTB((unsigned char)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 WaitOnBusy(pChan, statusByte);			\
 udelay(500);					\
 ATLAS_INOUT_DBG_EN;				\
}

// Reset controller (HSRST)
#define ControllerReset(pChan)			\
{						\
   ATLAS_INOUT_DBG_DIS;				\
   atlas_controller_init(NULL, pChan);		\
   atlas_find_device(pChan, pChan->channel);	\
   AtlasSetBestTransferMode(NULL, pChan, pChan->channel);	\
   ATLAS_INOUT_DBG_EN;				\
}


// Reset device (SRST)
#define IdeHardReset(pChan, result)		\
{						\
  u8 devctl, status;				\
   printk("Going to reset device!!!\n");	\
   devctl = ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]); \
   ATLAS_ROWB(ATLAS_DC_SRST, pChan->io_ports[IDE_CONTROL_OFFSET]);	\
    mdelay(25);					\
   ATLAS_ROWB(ATLAS_STATUS_CRF, pChan->io_ports[ATLAS_STATUS]); \
    mdelay(25);					\
   ATLAS_ROWB(ATLAS_STATUS_CRF, pChan->io_ports[ATLAS_STATUS]);	\
   WaitOnDevSRST(pChan, status);		\
    if(status & ATLAS_DC_SRST)			\
     result = FALSE;				\
    else					\
    {						\
     ATLAS_OUTB(devctl, pChan->io_ports[ATLAS_STATUS]);	\
     result = TRUE; 				\
    }						\
}

#define AtlasSelectDevice(dev)			\
{						\
 int		_i;				\
 ATLAS_INOUT_DBG_DIS;				\
 ATLAS_RAWB(~ATLAS_CONTROL_RW, pChan->io_ports[ATLAS_CONTROL]);	\
 ATLAS_OUTB((u8)((dev << 4) | 0xA0), pChan->io_ports[ATLAS_TEST_DATA]);	\
 ATLAS_OUTW(ATLAS_TEST_DEVICE, pChan->io_ports[ATLAS_TEST_CTRL]); \
 for (_i = 0; _i < 50000; _i++)			\
    {						\
     if (ATLAS_INW(pChan->io_ports[ATLAS_TEST_CTRL]) == 0)	\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}

#define AtlasGetDeviceCBR(dev)			\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 ATLAS_ROWB(ATLAS_CONTROL_RW, pChan->io_ports[ATLAS_CONTROL]);	\
 ATLAS_OUTW((1<<(ATLAS_TEST_DEVCBR_SHIFT+dev)), pChan->io_ports[ATLAS_TEST_CTRL]);	\
 for (i = 0; i < 50000; i++)			\
    {						\
     if (ATLAS_INW(pChan->io_ports[ATLAS_TEST_CTRL]) == 0)	\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
    }						\
    if(i>=50000)		\
    {	\
     printk("AtlasGetDeviceCBR failed!!\n");			\
    }	\
 ATLAS_INOUT_DBG_EN;				\
}

#define AtlasWaitTestDone(pChan, status)	\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     status = ATLAS_INW(pChan->io_ports[ATLAS_TEST_CTRL]);	\
     if (status == 0)				\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}


#define AtlasTestWaitInterruptDone(pChan)	\
{						\
 int		i;				\
 ATLAS_INOUT_DBG_DIS;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     if (pChan->ExpectingInterrupt == 0)	\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
    }						\
 ATLAS_INOUT_DBG_EN;				\
}


static __inline__
u8 WaitForAtlasActiveDone(PChannel pChan)
{
  u8 status;
  int i;
 ATLAS_INOUT_DBG_DIS;
   for (i = 0; i < 50000; i++)
   {
     GetAtlasStatus(pChan, status);
     if(!(status & ATLAS_STATUS_ACTIVE))
     {
 ATLAS_INOUT_DBG_EN;    	
       return 1;
     }
     else  
       udelay(500);  
   }
 ATLAS_INOUT_DBG_EN;
 return 0;
}

static __inline__
u8 WaitForAtlasCommandComplete(PChannel pChan)
{
  u8 status;
  int i;
  
  WaitForAtlasActiveDone(pChan);
  
 ATLAS_INOUT_DBG_DIS;
 
  if(ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]) & IDE_DC_DISABLE_INTERRUPTS)
  {
    // Interrupt is disabled	
    ATLAS_INOUT_DBG_EN;
    return 1;	
  }
  else
  {
   for (i = 0; i < 50000; i++)
   {
     GetAtlasStatus(pChan, status);
     if(status & ATLAS_STATUS_INT)
     {
       ATLAS_INOUT_DBG_EN;    	
       ClearAtlasInt(pChan);
       return 1;	
     }
     else
       udelay(500);
       
   }
  }
 ATLAS_INOUT_DBG_EN;
 return 0;
}

static __inline__
u8 WaitForAtlasActiveDoneShort(PChannel pChan)
{
  u8 status;
  int i;
 ATLAS_INOUT_DBG_DIS;
   for (i = 0; i < 5000; i++)
   {
     GetAtlasStatus(pChan, status);
     if(!(status & ATLAS_STATUS_ACTIVE))
     {
 ATLAS_INOUT_DBG_EN;    	
       return 1;
     }
     else  
       udelay(100);
   }
 ATLAS_INOUT_DBG_EN;
 return 0;
}

static __inline__
u8 WaitForAtlasCommandCompleteShort(PChannel pChan)
{
  u8 status;
  int i;
  
  WaitForAtlasActiveDoneShort(pChan);
  
 ATLAS_INOUT_DBG_DIS;
 
  if(ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]) & IDE_DC_DISABLE_INTERRUPTS)
  {
    // Interrupt is disabled	
    ATLAS_INOUT_DBG_EN;
    return 1;	
  }
  else
  {
   for (i = 0; i < 5000; i++)
   {
     GetAtlasStatus(pChan, status);
     if(status & ATLAS_STATUS_INT)
     {
       ATLAS_INOUT_DBG_EN;    	
       ClearAtlasInt(pChan);
       return 1;	
     }
     else
       udelay(100);
       
   }
  }
 ATLAS_INOUT_DBG_EN;
 return 0;
}

#define WaitForAtlasCommandCompleteStatus(PChan, Status)	\
{								\
	WaitForAtlasCommandComplete(pChan);			\
	WaitForBaseCommandComplete(pChan, Status);		\
}



/************************************************************************
 * Function prototypes
 ************************************************************************/
int		atlas_detect (struct scsi_host_template *);
//  int		atlas_detect (Scsi_Host_Template *);
int		atlas_release (struct Scsi_Host *);
#if !ATLAS_TESTMODE
const char *	atlas_info (struct Scsi_Host *);
int		atlas_command (Scsi_Cmnd *);
int		atlas_queuecommand (Scsi_Cmnd *, void (*done)(Scsi_Cmnd *));
int		atlas_biosparam (struct scsi_device *, struct block_device *, sector_t , int *);
//  int		atlas_biosparam (Disk *, kdev_t, int *);
//int		atlas_proc_info (char *buffer, char **start, off_t offset,
//			int length, int hostno, int inout);
int		atlas_proc_info ( struct Scsi_Host *host, char *buffer, char **start, off_t offset, int length, int inout);
#endif			

#if !ATLAS_TESTMODE
static void	TaskStart(PChannel, Scsi_Cmnd *);
static void	TaskQueue(void);
static void	TaskDone(PChannel, PSCSI_REQUEST_BLOCK);
static u32	IdeSendCommand(PChannel, PSCSI_REQUEST_BLOCK);
static void	MapRequest(Scsi_Cmnd *, PSCSI_REQUEST_BLOCK);
static u8	AtlasResetAdapter(PATLAS_ADAPTER);
static void	AtapiStartIo(PChannel, PSCSI_REQUEST_BLOCK);
static u8	AtapiResetController(PATLAS_ADAPTER pAdap, PChannel pChan);
static irqreturn_t	AtapiInterrupt(PChannel);
#endif

//static void	IdeMediaStatus(u8, PChannel, u8);
#ifndef ATLAS_TEST_ONLY
static int	AtlasPIOData(PChannel pChan, void *buffer, u32 len, u16 direction);
static void	IdeSetupDma(PChannel, unsigned long, unsigned short);
static u8	IssueIdentify(PChannel, u8, u8);
static int	atlas_controller_init(PATLAS_ADAPTER pAdap, PChannel pChan);
static u8	atlas_find_device(PChannel pChan, u8 channel);
#endif
void		AtlasSetBestTransferMode(PATLAS_ADAPTER pAdap, PChannel pChan, u8 channel);
int		atlas_set_piodma_mode(PChannel pChan, int devid, int type, int mode);

#endif /* #ifndef _ATLAS_H_ */
