#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "certgen.h"
#include <openssl/pem.h>
#include <openssl/conf.h>
#include <openssl/x509v3.h>
#ifndef OPENSSL_NO_ENGINE
#include <openssl/engine.h>
#endif

#define READ_SIZE 1024

typedef struct stCertInfo {
	char Country [HTTPS_COUNTRY_BUFSIZE];	/* Max Len: 3 (include '\0') */
	char Province [HTTPS_PROVINCE_BUFSIZE];	/* Max Len: 41 (include '\0') */
	char Locality [HTTPS_LOCALITY_BUFSIZE];	/* Max Len: 41 (include '\0') */
	char Organization [HTTPS_ORGANIZATION_BUFSIZE];	/* Max Len: 41 (include '\0') */
	char Division [HTTPS_DIVISION_BUFSIZE];	/* Max Len: 41 (include '\0') */
	char CommonName [HTTPS_COMMONANME_BUFSIZE];	/* Max Len: 65 (include '\0') */
	int Validity;				/* 1~9999 */
	int KeyLen;				/* 512 / 1024 / 2048 */
	int Serial;
} tCertInfo;

int certgen(tCertInfo *sCertData);
int reqgen(tCertInfo *sCertReq);
int mkcert(X509 **x509p, EVP_PKEY **pkeyp, tCertInfo *sCertData);
int mkreq(X509_REQ **x509p, EVP_PKEY **pkeyp, tCertInfo *sCertReq);
int Decode_certificate_pemfile(char *filePath, char *Properties);
int Decode_certreq_pemfile(char *filePath, char *Properties);

int print_Properties(X509 *x, char *buf);
int print_csr_Properties(X509_REQ *x, char *buf);
int NAME_print(char *buf, X509_NAME *name, int obase);
int TIME_print(char *buf, ASN1_TIME *tm);
int GENERALIZEDTIME_print(char *buf, ASN1_GENERALIZEDTIME *tm);
int UTCTIME_print(char *buf, ASN1_UTCTIME *tm);
int RSA_print_n(char *buf, const RSA *x, int off);
int indent_n(char *buf, int indent, int max);
static int print_n(char *output, const char *number, BIGNUM *num, unsigned char *buf, int off);
int signature_print(char *buf, X509_ALGOR *sigalg, ASN1_STRING *sig);

int check_cert_value(int type, char *, tCertInfo *);          // if not, return 0
int is_Distinguish( char  );                    // if not, return 0
int is_Distinguish_Name( char * );              // if not, return 0
int is_Common( char  );                         // if not, return 0
int is_IPv4_Address( char * );                  // if not, return 0
int is_Host_Address( char * );                  // if not, return 0
int is_Common_Name( char * );                   // if not, return 0
int is_KeyLen( char * );                        /* if not, return 0;
                                                   if yes, return 512, 1024, or 2048 */
int is_Validity( char * );                      /* if not, return 0;
                                                   if yes, return the value */

#ifdef TEST
int main(int argc, char **argv) {
	char input[]= "TW,North,Taipei,ACTi,SSPO,www.Anne.com,1024,365";
	char csrinput[]= "AU,QLD,Portland,Mincom Pty. Ltd.,SSPO,www.Anne.com,1024";
	tHttpsApiCreate httpsCreate; 
	tHttpsApiProp 	httpsProp; 
	tHttpsApiName 	httpsName; 
#ifdef GEN_CERT

	/* Generate self-signed cert & Privkey*/
	strncpy (httpsCreate.inputArgs, input, strlen(input) + 1); // + 1 ('\0')
        httpsCreate.argsLen = strlen (httpsCreate.inputArgs);

	if (https_cert_create (&httpsCreate) == HTTPS_OK) {
		printf("<Cert pem>\n%s \n", httpsCreate.retMsg);
		printf("<Len> %d \n", httpsCreate.retLen);
	} else {
		printf("<msg> %s \n", httpsCreate.retMsg);
		printf("<Len> %d \n", httpsCreate.retLen);
		return HTTPS_ERROR;
	}

	if (https_get_cert_prop (&httpsProp) == HTTPS_OK) {
		printf("<Cert prop>\n%s \n", httpsProp.retMsg);
		printf("<Len> %d \n", httpsProp.retLen);
	} else {
		printf("<msg>%s \n", httpsProp.retMsg);
		printf("<Len> %d \n", httpsProp.retLen);
		return HTTPS_ERROR;
	}

	if (https_get_cert_common_name (&httpsName) == 0) {
		printf("<Common Name> %s \n", httpsName.retMsg);
		printf("<Len> %d \n", httpsName.retLen);
	} else {
		printf("<msg> %s \n", httpsName.retMsg);
		printf("<Len> %d \n", httpsName.retLen);
		return HTTPS_ERROR;
	}
#endif

#ifdef GEN_CERTREQ

	/* Generate CSR & Privkey*/
	strncpy (httpsCreate.inputArgs, csrinput, strlen(csrinput) + 1); // + 1 ('\0')
        httpsCreate.argsLen = strlen (httpsCreate.inputArgs);

	if (https_csr_create (&httpsCreate) == HTTPS_OK) {
		printf("<CSR pem>\n%s \n", httpsCreate.retMsg);
		printf("<Len> %d \n", httpsCreate.retLen);
	} else {
		printf("<msg>%s \n", httpsCreate.retMsg);
		printf("<Len> %d \n", httpsCreate.retLen);
		return HTTPS_ERROR;
	}

	if (https_get_csr_prop (&httpsProp) == HTTPS_OK) {
		printf("<CSR prop>\n%s \n", httpsProp.retMsg);
		printf("<Len> %d \n", httpsProp.retLen);
	} else {
		printf("<msg>%s \n", httpsProp.retMsg);
		printf("<Len> %d \n", httpsProp.retLen);
		return HTTPS_ERROR;
	}

	if (https_get_csr_common_name (&httpsName) == HTTPS_OK) {
		printf("<CSR Common Name> %s \n", httpsName.retMsg);
		printf("<Len> %d \n", httpsName.retLen);
	} else {
		printf("<msg>%s \n", httpsName.retMsg);
		printf("<Len> %d \n", httpsName.retLen);
		return HTTPS_ERROR;
	}
#endif
	return HTTPS_OK;
}
#endif

/*
 * Check to see if the file is existed or not
 * Input: filename with path
 * Output: NOT_EXIST
 *         EXIST
 */
enum {  
	NOT_EXIST = 0,
        EXIST
};

int isFileExists (const char *s){
	FILE *fd;

        if (s == NULL || strlen (s) == 0)
                return NOT_EXIST;

        fd = fopen (s, "r");
        if (fd) {
                fclose (fd);
                return EXIST;
        }
        return NOT_EXIST;
}

int get_pem_content(char *file_path, char *output) { 	// 0: Error, 1: Success 
	char line[1024];
	char *pLeft;
	char *pSymbol;
	FILE *fp;
	char tmpbuf [HTTPS_PROP_RETMSG_BUFSIZE]; 

	if((fp = fopen(file_path, "r")) != NULL) {
		pLeft = tmpbuf;
		while (fgets(line, sizeof(line), fp) != NULL) {
			strcpy(pLeft, line);
			pLeft = strrchr(tmpbuf, '\0');
		}

		if ((pSymbol = strstr (tmpbuf, HTTPS_CERT_BEGIN_SYMBOL)) == NULL) {
			if ((pSymbol = strstr (tmpbuf, HTTPS_CSR_BEGIN_SYMBOL)) == NULL) {
				fprintf(stderr, "<OpenSSL API> %s: Bad Certificate!! >\n", __func__); 
				return 0; 		
			} 
		}
		// If found BEGIN SYMBOL of CERT/CSR, move *pSymbol to the begin of encrypted data.  
		while ((pSymbol > tmpbuf) && *(--pSymbol) == '-');
		//*pSymbol++;	

		strcpy(output, pSymbol);
		fclose(fp);
		return 1;
	} else {
		return 0;
	}
}	

int https_cert_create (tHttpsApiCreate *api) { 
	tCertInfo sCertData;
	if (api->inputArgs == NULL || !strcmp("", api->inputArgs) || !api->argsLen || \
	    api->argsLen != strlen(api->inputArgs) || api->argsLen > HTTPS_INPUTARGS_BUFSIZE) {
		fprintf(stderr, "<OpenSSL API> %s: CERT: error inputArgs or argsLen.>\n", __func__);

		goto INTERNAL_ERR;
	}

	if(!check_cert_value(HTTPS_CERTTYPE_CERT, api->inputArgs, &sCertData))		 goto INVALID_ERR;

	sCertData.Serial = (int)time(NULL);	
	if(certgen(&sCertData)) {
		if(get_pem_content(DEFAULT_CERTFILE, api->retMsg)) {
			api->retLen = strlen (api->retMsg);
			return HTTPS_OK;
		}
	} 
INTERNAL_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: invalid parameters");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR;
}

int https_csr_create(tHttpsApiCreate *api) {
	tCertInfo sCertReq;

	if (api->inputArgs == NULL || !strcmp("", api->inputArgs) || !api->argsLen || \
	    api->argsLen != strlen(api->inputArgs) || api->argsLen > HTTPS_INPUTARGS_BUFSIZE) {
		fprintf(stderr, "<OpenSSL API> %s: CSR: error inputArgs or argsLen.>\n", __func__);

		goto INTERNAL_ERR;
	}

	if (!check_cert_value(HTTPS_CERTTYPE_CSR, api->inputArgs, &sCertReq)) 	goto INVALID_ERR;
	if (reqgen (&sCertReq)) {
		if (get_pem_content (DEFAULT_CERTREQFILE, api->retMsg)) { 	
			api->retLen = strlen (api->retMsg);	
			return HTTPS_OK;
		}
	} 
INTERNAL_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: invalid parameters");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR;
}

int https_get_cert_prop(tHttpsApiProp *api) {
	char *pLeft;
	if (Decode_certificate_pemfile (DEFAULT_CERTFILE, api->retMsg)) {
		pLeft = strchr(api->retMsg, '\0');
		if (get_pem_content (DEFAULT_CERTFILE, pLeft)) {
			api->retLen = strlen (api->retMsg);
			return HTTPS_OK;
		}
	}
INTERNAL_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: invalid parameters");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR;
}

int https_get_csr_prop(tHttpsApiProp *api) {
	char *pLeft;
	if(Decode_certreq_pemfile(DEFAULT_CERTREQFILE, api->retMsg)) {
		pLeft = strchr(api->retMsg, '\0');
		if(get_pem_content(DEFAULT_CERTREQFILE, pLeft)) {
			api->retLen = strlen(api->retMsg);
			return HTTPS_OK;
		}
	}
INTERNAL_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: invalid parameters");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR;
}

int https_get_cert_common_name(tHttpsApiName *api) {
	char *ptr = NULL; 
	X509 *x = NULL;
	char *cname;
	BIO *in = NULL;
	X509_NAME *subject = NULL;
//fprintf(stderr, "<OpenSSL API> %s: before overwrite... retMsg=%s<\n", __func__, api->retMsg);
	if ((in = BIO_new(BIO_s_file_internal())) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error new a BIO.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if (BIO_read_filename(in, DEFAULT_CERTFILE) <= 0) {
		fprintf(stderr, "<OpenSSL API> %s: Self-Signed Error reading BIO file.>\n", __func__);
		goto INTERNAL_ERR;
	}
	
	if ((x = PEM_read_bio_X509(in,NULL,0,NULL)) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error decoding pem file.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if ((subject = (X509_get_subject_name (x))) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error reading subject name.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if ((cname = X509_NAME_oneline (subject, NULL, 0)) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error reading Common Name.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if ((ptr = strstr (cname, HTTPS_CN_BEGIN_SYMBOL)) == NULL)
		goto INVALID_ERR;
	ptr += strlen (HTTPS_CN_BEGIN_SYMBOL); 	 	// move *ptr to value of common name
//fprintf(stderr, "<OpenSSL API> %s: ptr=%s, strlen=%d<\n", __func__, ptr, strlen(ptr));
	memset (api->retMsg, 0, HTTPS_NAME_RETMSG_BUFSIZE); 
	strncpy (api->retMsg, ptr, strlen(ptr));
	api->retMsg[strlen(ptr)] = '\0';
//fprintf(stderr, "<OpenSSL API> %s: retMsg=%s<\n", __func__, api->retMsg);
	api->retLen = strlen (api->retMsg); 
	return HTTPS_OK;	
INTERNAL_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: certificate format error.");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR_FILE; 
}

int https_get_csr_common_name (tHttpsApiName *api) {
	X509_REQ *x = NULL;
	char *ptr = NULL; 
	char *cname;
	BIO *in = NULL;
	X509_NAME *subject = NULL;
	
//fprintf(stderr, "<OpenSSL API> %s: before overwrite... retMsg=%s<\n", __func__, api->retMsg);
	if ((in = BIO_new(BIO_s_file_internal())) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error new a BIO.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if (BIO_read_filename(in, DEFAULT_CERTREQFILE) <= 0) {
		fprintf(stderr, "<OpenSSL API> %s: CSR Error reading BIO file.>\n", __func__);
		goto INTERNAL_ERR;
	}
	
	if ((x = PEM_read_bio_X509_REQ (in, NULL, 0, NULL)) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error decoding pem file.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if ((subject = (X509_REQ_get_subject_name (x))) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error reading subject name.>\n", __func__);
		goto INTERNAL_ERR;
	}

	if ((cname = X509_NAME_oneline (subject, NULL, 0)) == NULL) {
		fprintf(stderr, "<OpenSSL API> %s: Error reading Common Name.>\n", __func__);
		goto INTERNAL_ERR;
	}
	if ((ptr = strstr (cname, HTTPS_CN_BEGIN_SYMBOL)) == NULL)
		goto INVALID_ERR;
	ptr += strlen (HTTPS_CN_BEGIN_SYMBOL); 	 	// move *ptr to value of common name
//fprintf(stderr, "<OpenSSL API> %s: ptr=%s, strlen=%d<\n", __func__, ptr, strlen(ptr));
	memset (api->retMsg, 0, HTTPS_NAME_RETMSG_BUFSIZE); 
	strncpy (api->retMsg, ptr, strlen(ptr));
	api->retMsg[strlen(ptr)] = '\0';
//fprintf(stderr, "<OpenSSL API> %s: retMsg=%s<\n", __func__, api->retMsg);
	api->retLen = strlen (api->retMsg); 
	return HTTPS_OK;	
INTERNAL_ERR:
	snprintf (api->retMsg, sizeof (api->retMsg), "ERROR: internal error."); 
	api->retLen = strlen (api->retMsg);
	return HTTPS_ERROR; 	
INVALID_ERR:
        snprintf (api->retMsg, sizeof (api->retMsg), "ERROR: certificate format error.");
	api->retLen = strlen (api->retMsg);
        return HTTPS_ERROR_FILE;
}

/*
 * int https_cer (int op, tHttpsApiCerOp *api) 
 *
 * == DESCRIPTION ==
 * This function support three operations: remove, upload and query.
 *
 * op: HTTPS_CER_OP_QUERY | HTTPS_CER_OP_UPLOAD | HTTPS_CER_OP_REMOVE 
 * 	HTTPS_CER_OP_QUERY:  query certificate filename
 * 	HTTPS_CER_OP_UPLOAD: upload certificate to IP Camera
 * 	HTTPS_CER_OP_REMOVE: remove certificate and key files
 * *api: 
 * 	https_cer() must declares an pointer *api of struct tHttpsApiCerOp that defined in certgen.h .
 * 	api->inputArgs: char* | NULL
 * 			a string of file name for uploading
 * 			If operation mode is HTTPS_CER_OP_QUERY or HTTPS_CER_OP_REMOVE, api->inputArgs must be NULL or "".
 * 	api->argsLen: integer
 * 			The length of api->inputArgs.
 * 			If operation mode is HTTPS_CER_OP_QUERY or HTTPS_CER_OP_REMOVE, api->argsLen must be 0.
 * 	api->retMsg: char*
 * 			Return message 
 * 	api->retLen: integer
 * 			The Length of api->retMsg.	
 *
 * == RETURN VALUE ==
 * On success, HTTPS_OK(0) is returned; on error, https_cer() returns HTTPS_ERROR(-1).
 *
 * == EXAMPLE ==
 * If you don't know how to use these operations, there are some examples below.
 * 1. Querying filename operation
 * 	tHttpsApiCerOp httpsapi;
 *	//
 * 	// Do Something	
 *	//
 * 	if (https_cer (HTTPS_CER_OP_QUERY, &httpsapi) == HTTPS_OK) {
 * 		PrintFunction ("%s", httpsapi.retMsg);
 *		total_retLen += httpsapi.retLen; 
 *		//
 * 		// Do Something	
 *		//
 * 	} else{
 * 		// Handling Exception 
 * 	}
 * 2. Uploading operation:
 * 	tHttpsApiCerOp httpsapi;
 *	//
 * 	// Do Something
 *	//
 *	strcpy (httpsapi.inputArgs, fData->field[0].UploadFileName);
 *	httpsapi.argsLen = strlen (httpsapi.inputArgs);
 * 	if (https_cer (HTTPS_CER_OP_UPLOAD, &httpsapi) == HTTPS_OK) {
 * 		PrintFunction ("%s", httpsapi.retMsg);
 *		total_retLen += httpsapi.retLen; 
 *		//
 * 		// Do Something	
 *		//
 *	} else {
 * 		// Handling Exception 
 * 	}
 * 3. Removing operation:
 * 	tHttpsApiCerOp httpsapi;
 * 	int total_retLen = 0; 
 *	//
 * 	// Do Something	
 *	//
 * 	if (https_cer (HTTPS_CER_OP_REMOVE, &httpsapi) == HTTPS_OK) {
 * 		PrintFunction ("%s", httpsapi.retMsg);
 *		total_retLen += httpsapi.retLen; 
 *		//
 * 		// Do Something	
 *		//
 * 	} else {
 * 		// Handling Exception 
 * 	}
 */
int https_cer (int op, tHttpsApiCerOp *api) {
	char 	buf[READ_SIZE];
	FILE 	*fp_R = NULL, 
		*fp_W1 = NULL;
		////*fp_W1 = NULL,
		////*fp_W2 = NULL;
	int 	ret_read = 0, 
		ret_write1 = 0,
		////ret_write2 = 0,
		total_size = 0;
	if (!(op >= HTTPS_CER_OP_QUERY && op <= HTTPS_CER_OP_REMOVE)) {
		fprintf(stderr, "<OpenSSL API> %s: Error operation code.>\n", __func__);
		goto INTERNAL_ERR;
	}
 	/* Querying certificate name */
	if (op == HTTPS_CER_OP_QUERY && ((api->inputArgs == NULL) || !strcmp("", api->inputArgs))) {
		////if (!isFileExists(RUNTIME_CERTFILE)) 		snprintf(api->retMsg, (sizeof (api->retMsg) - 1), "none");
		if (!isFileExists(DEFAULT_CERTFILE)) 		snprintf(api->retMsg, (sizeof (api->retMsg) - 1), "none");
		else 						snprintf(api->retMsg, (sizeof (api->retMsg) - 1), "certSrv.pem");
	} else if (op == HTTPS_CER_OP_UPLOAD) { /* Uploading certificate */
		if (api->inputArgs != NULL && api->argsLen > 0 && \
		    api->argsLen == strlen(api->inputArgs) && api->argsLen <= HTTPS_INPUTARGS_BUFSIZE) {
			fp_R = fopen (api->inputArgs, "r");
			if (fp_R == NULL) {
                        	fprintf(stderr, "<OpenSSL API> %s: open file %s error!>\n", __func__, api->inputArgs);
				goto  FILE_ERR;
			}
			fp_W1 = fopen (DEFAULT_CERTFILE, "w");
			if (fp_W1 == NULL) {
                        	fprintf(stderr, "<OpenSSL API> %s: open file %s error!>\n", __func__, DEFAULT_CERTFILE);
				goto  FILE_ERR;
			}
			////fp_W2 = fopen (RUNTIME_CERTFILE, "w");
			////if (fp_W2 == NULL) {
                        ////	fprintf(stdout, "<OpenSSL API: open file %s error!>\n", RUNTIME_CERTFILE);
			////	goto  FILE_ERR;
			////}
			do {
				memset(buf, 0, READ_SIZE);
				ret_read = fread (buf, 1, READ_SIZE, fp_R);
				total_size += ret_read;
				if (total_size == 0) {
                        		fprintf(stderr, "<OpenSSL API> %s: read file %s error!>\n", __func__, api->inputArgs);
					goto  FILE_ERR;
				} else if (ret_read > 0) {
					if((ret_write1 = fwrite(buf, 1, ret_read, fp_W1)) <= 0) {
                        			fprintf(stderr, "<OpenSSL API> %s: write file %s error!>\n", 
							__func__, DEFAULT_CERTFILE);
						goto  FILE_ERR;
                			}
					////if((ret_write2 = fwrite(buf, 1, ret_read, fp_W2)) <= 0) {
                        		////	fprintf(stdout, "<OpenSSL API: write file %s error!>\n", RUNTIME_CERTFILE);
					////	goto  FILE_ERR;
                			////}
				} 
			} while (ret_read > 0);
			fclose (fp_R);
			fclose (fp_W1);
			////fclose (fp_W2);

			////if (!isFileExists(DEFAULT_CERTFILE) || !isFileExists(RUNTIME_CERTFILE)) { 
                        ////	fprintf(stdout, "<OpenSSL API: files not found!>\n");
			////	goto  FILE_ERR;
			////}
		} else {
                        fprintf(stderr, "<OpenSSL API> %s: Upload: error inputArgs or argsLen.>\n", __func__);
			goto  INTERNAL_ERR;
		}
		snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "OK"); 
	} else if (op == HTTPS_CER_OP_REMOVE && (api->inputArgs == NULL || !strcmp("", api->inputArgs))) { /* Removing certificate and key files */
    		unlink (DEFAULT_CERTFILE);
    		unlink (DEFAULT_CERTREQFILE);
    		unlink (DEFAULT_PRIVKEYFILE);
		snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "OK"); 
	} else {
		goto INVALID_ERR;
	}
	api->retLen = strlen(api->retMsg); 
	return HTTPS_OK; 	/* 0: Success */
INTERNAL_ERR: 	/* Wrong implementation that caused by developer. */
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: internal error.");
	api->retLen = strlen(api->retMsg); 
	return HTTPS_ERROR; 	
INVALID_ERR: 	/* Wrong parameters is given by user. */
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1), "ERROR: invalid parameters");
	api->retLen = strlen(api->retMsg); 
	return HTTPS_ERROR; 	
FILE_ERR:
	snprintf (api->retMsg, (sizeof (api->retMsg) - 1),  "ERROR: internal error.(Exception occurred during file upload.)");
	api->retLen = strlen(api->retMsg); 
	return HTTPS_ERROR; 	
}

/* 
 * int https_csr_remove (tHttpsApiCerOp *api) 
 * == DESCRIPTION ==
 * This function removes certificate and key files without checking file status.
 *
 * *api:
 * 	https_cer() must declares an pointer *api of struct tHttpsApiCerOp that defined in certgen.h .
 * 	api->retMsg: char*
 * 			Return message 
 * 	api->retLen: integer
 * 			The Length of api->retMsg.	
 *
 * == RETURN VALUE ==
 * The return code and return message are always 0(HTTPS_OK) and "OK".
 *
 * == EXAMPLE ==
 * If you don't know how to use these operations, there is an examples for you.
 * 	tHttpsApiCerOp httpsapi;
 * 	int total_retLen = 0; 
 *	//
 * 	// Do Something	
 *	//
 * 	if (https_csr_remove (&httpsapi) == HTTPS_OK) {
 * 		PrintFunction ("%s", httpsapi.retMsg);
 *		total_retLen += httpsapi.retLen; 
 *		//
 * 		// Do Something	
 *		//
 * 	} else {
 * 		// Handling Exception 
 * 	}
 */
int https_csr_remove (tHttpsApiCerOp *api) {
    	unlink (DEFAULT_CERTFILE);
    	unlink (DEFAULT_CERTREQFILE);
    	unlink (DEFAULT_PRIVKEYFILE);
	snprintf(api->retMsg, (strlen(api->retMsg) - 1), "OK");
	api->retLen = strlen(api->retMsg); 
	return HTTPS_OK; 	
}

static void callback(int p, int n, void *arg) 
{
	char c='B';

	if (p == 0) c='.';
	if (p == 1) c='+';
	if (p == 2) c='*';
	if (p == 3) c='\n';
	fputc(c,stderr);
	}


/*
*	Generate Cert API.
*	
*/
int certgen(tCertInfo *sCertData)
	{
	BIO *bio_err;
	X509 *x509=NULL;
	EVP_PKEY *pkey=NULL;
	FILE *pPrivkey, *pCert;
	////FILE *pCert_cp;

	CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);
	bio_err=BIO_new_fp(stderr, BIO_NOCLOSE);

	//make privkey & cert
	if(!mkcert(&x509,&pkey,sCertData))
	{
		printf("<<<OpenSSL API: Make Cert Err>>>\n");
		return(0);
	}

        //Write down Privkey into pem file
        pPrivkey = fopen(DEFAULT_PRIVKEYFILE, "w");
        if ((pPrivkey == NULL) || (!PEM_write_PrivateKey(pPrivkey,pkey,NULL,NULL,0,NULL, NULL)))
        {
                printf("<<<<<<<Err Privkey>>>>>>>>\n");
                return(0);
        }
	fclose(pPrivkey);

	//Write down Cert into pem file
	pCert = fopen(DEFAULT_CERTFILE, "w");	
	////pCert_cp = fopen(RUNTIME_CERTFILE, "w");	
	////if ((pCert == NULL) || (!PEM_write_X509(pCert,x509)) || (!PEM_write_X509(pCert_cp,x509)))
	if ((pCert == NULL) || (!PEM_write_X509(pCert,x509)))
	{
		printf("<<<<<<<Err Cert>>>>>>>>\n");
		return(0);
	}
	fclose(pCert);
	////fclose(pCert_cp);

	X509_free(x509);
	EVP_PKEY_free(pkey);

	CRYPTO_mem_leaks(bio_err);
	BIO_free(bio_err);

	unlink (DEFAULT_CERTREQFILE);
	return(1);
	}

int mkcert(X509 **x509p, EVP_PKEY **pkeyp, tCertInfo *sCertData)
	{
	X509 *x;
	EVP_PKEY *pk;
	RSA *rsa;
	X509_NAME *name=NULL;
	//X509_NAME_ENTRY *ne=NULL;
	//X509_EXTENSION *ex=NULL;

	
	if ((pkeyp == NULL) || (*pkeyp == NULL))
		{
		if ((pk=EVP_PKEY_new()) == NULL)
			{
			abort(); 
			return 0;
			}
		}
	else
		pk= *pkeyp;

	if ((x509p == NULL) || (*x509p == NULL))
		{
		if ((x=X509_new()) == NULL)
			goto err;
		}
	else
		x= *x509p;

	rsa=RSA_generate_key(sCertData->KeyLen,RSA_F4,callback,NULL);
	if (!EVP_PKEY_assign_RSA(pk,rsa))
		{
		abort();
		goto err;
		}
	rsa=NULL;

	X509_set_version(x,3);
	ASN1_INTEGER_set(X509_get_serialNumber(x),sCertData->Serial);
	X509_gmtime_adj(X509_get_notBefore(x),0);
	X509_gmtime_adj(X509_get_notAfter(x),(long)60*60*24*sCertData->Validity);
	X509_set_pubkey(x,pk);

	name=X509_get_subject_name(x);

	/* This function creates and adds the entry, working out the
	 * correct string type and performing checks on its length.
	 * Normally we'd check the return value for errors...
	 */
	X509_NAME_add_entry_by_txt(name,"C",
				MBSTRING_ASC, sCertData->Country, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"ST",
				MBSTRING_ASC, sCertData->Province, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"L",
				MBSTRING_ASC, sCertData->Locality, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"O",
				MBSTRING_ASC, sCertData->Organization, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"OU",
				MBSTRING_ASC, sCertData->Division, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"CN",
				MBSTRING_ASC, sCertData->CommonName, -1, -1, 0);

	X509_set_issuer_name(x,name);

#if 0
	/* Add extension using V3 code: we can set the config file as NULL
	 * because we wont reference any other sections. We can also set
         * the context to NULL because none of these extensions below will need
	 * to access it.
	 */
	ex = X509V3_EXT_conf_nid(NULL, NULL, NID_netscape_cert_type, "server");
	X509_add_ext(x,ex,-1);
	X509_EXTENSION_free(ex);

	ex = X509V3_EXT_conf_nid(NULL, NULL, NID_netscape_comment,
						"example comment extension");
	X509_add_ext(x,ex,-1);
	X509_EXTENSION_free(ex);

	ex = X509V3_EXT_conf_nid(NULL, NULL, NID_netscape_ssl_server_name,
							"www.openssl.org");

	X509_add_ext(x,ex,-1);
	X509_EXTENSION_free(ex);
#endif
	if (!X509_sign(x,pk,EVP_md5()))
		goto err;

	*x509p=x;
	*pkeyp=pk;
	return 1;
err:
	return 0;
	}



/*
*	Generate GSR API.
*	
*/
int reqgen(tCertInfo *sCertReq)
	{
	BIO *bio_err;
	X509_REQ *req=NULL;
	EVP_PKEY *pkey=NULL;
	//FILE *pStream;
	FILE *pPrivkey, *pCertReq;
	////FILE *pCertReq_cp;

	CRYPTO_mem_ctrl(CRYPTO_MEM_CHECK_ON);
	bio_err=BIO_new_fp(stderr, BIO_NOCLOSE);

	//make privkey & csr
	if(!mkreq(&req,&pkey,sCertReq))
	{
		printf("<<<<<<<Make CSR Err>>>>>>>>\n");
		return(0);
	}
#if 0
	//Get Properties
    	pStream = fmemopen((void *)sCertReq->Properties, sizeof(sCertReq->Properties), "w");
	if (pStream == NULL)
	{
		printf("<<<<<<<Err>>>>>>>>\n");
		return 0;
	}
	X509_REQ_print_fp(pStream, req);
	fclose(pStream);
#endif
	//Write down Privkey into pem file
	pPrivkey = fopen(DEFAULT_PRIVKEYFILE, "w");	
	if ((pPrivkey == NULL) || (!PEM_write_PrivateKey(pPrivkey,pkey,NULL,NULL,0,NULL, NULL)))
	{
		printf("<<<<<<<Err Privkey>>>>>>>>\n");
		return 0;
	}
	fclose(pPrivkey);

	//Write down CSR into pem file
	pCertReq = fopen(DEFAULT_CERTREQFILE, "w");	
	////pCertReq_cp = fopen(RUNTIME_CERTREQFILE, "w");	
	////if ((pCertReq == NULL) || (!PEM_write_X509_REQ(pCertReq, req)) || (!PEM_write_X509_REQ(pCertReq_cp, req)))
	if ((pCertReq == NULL) || (!PEM_write_X509_REQ(pCertReq, req)))
	{
		printf("<<<<<<<Err CertReq>>>>>>>>\n");
		return 0;
	}
	fclose(pCertReq);
	////fclose(pCertReq_cp);

	X509_REQ_free(req);
	EVP_PKEY_free(pkey);

#ifndef OPENSSL_NO_ENGINE
	ENGINE_cleanup();
#endif
	CRYPTO_cleanup_all_ex_data();

	CRYPTO_mem_leaks(bio_err);
	BIO_free(bio_err);
	unlink (DEFAULT_CERTFILE);
	return 1;
	}

int mkreq(X509_REQ **req, EVP_PKEY **pkeyp, tCertInfo *sCertReq)
	{
	X509_REQ *x;
	EVP_PKEY *pk;
	RSA *rsa;
	X509_NAME *name=NULL;
	STACK_OF(X509_EXTENSION) *exts = NULL;
	
	if ((pk=EVP_PKEY_new()) == NULL)
		goto err;

	if ((x=X509_REQ_new()) == NULL)
		goto err;

	rsa=RSA_generate_key(sCertReq->KeyLen,RSA_F4,callback,NULL);
	if (!EVP_PKEY_assign_RSA(pk,rsa))
		goto err;

	rsa=NULL;

	X509_REQ_set_pubkey(x,pk);

	name=X509_REQ_get_subject_name(x);

	/* This function creates and adds the entry, working out the
	 * correct string type and performing checks on its length.
	 * Normally we'd check the return value for errors...
	 */
	X509_NAME_add_entry_by_txt(name,"C",
				MBSTRING_ASC, sCertReq->Country, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"ST",
				MBSTRING_ASC, sCertReq->Province, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"L",
				MBSTRING_ASC, sCertReq->Locality, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"O",
				MBSTRING_ASC, sCertReq->Organization, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"OU",
				MBSTRING_ASC, sCertReq->Division, -1, -1, 0);
	X509_NAME_add_entry_by_txt(name,"CN",
				MBSTRING_ASC, sCertReq->CommonName, -1, -1, 0);

	if (!X509_REQ_sign(x,pk,EVP_md5()))
		goto err;

	*req=x;
	*pkeyp=pk;
	return 1;
err:
	return 0;
}

int Decode_certificate_pemfile(char *filePath, char *Properties) {
	BIO *in;
	X509 *x=NULL;

	in = BIO_new (BIO_s_file_internal ());
	if (in == NULL) {
		printf("<<<<<<<Err bio>>>>>>>>\n");
		return 0;
	}

	if (BIO_read_filename(in,filePath) <= 0) {
		printf("<<<<<<<Err file>>>>>>>>\n");
		return 0;
	}
	
	x = PEM_read_bio_X509 (in, NULL, 0, NULL);

	if (x == NULL) {
		printf("<<<<<<<Err Decode>>>>>>>>\n");
		return 0;
	}

	if (print_Properties (x, Properties)) 	return 1;
	else 					return 0;
}

int Decode_certreq_pemfile (char *filePath, char *Properties) {
	int j;
	BIO *in;
	X509_REQ *x=NULL;
	FILE *pStream;

	in=BIO_new(BIO_s_file_internal());
	if (in == NULL)
	{
		printf("<<<<<<<Err bio>>>>>>>>\n");
		return 0;
	}

	if (BIO_read_filename(in,filePath) <= 0)
	{
		printf("<<<<<<<Err file>>>>>>>>\n");
		return 0;
	}
	
	x=PEM_read_bio_X509_REQ(in,NULL,0,NULL);

	if (x == NULL) {
		printf("<<<<<<<Err Decode>>>>>>>>\n");
		return 0;
	}

	if (print_csr_Properties (x, Properties)) 	return 1;
	else 						return 0;
}

int print_Properties(X509 *x, char *buf) {
	long l;
	int ret=0,i;
	char *m=NULL,mlch = ' ';
	int nmindent = 0;
	X509_CINF *ci;
	ASN1_INTEGER *bs;
	EVP_PKEY *pkey=NULL;
	const char *neg;
	ASN1_STRING *str=NULL;
	char *pLeft;

	mlch = '\n';
	nmindent = 16;

	//Certificate:
	ci=x->cert_info;
	buf[0] = '\0';
	
	//Version
	l=X509_get_version(x);
	pLeft = strchr(buf, '\0');
	sprintf (pLeft, "Version: %lu\n", l+1);

	//Serial Number
	strcat (pLeft, "Serial Number:");
	bs=X509_get_serialNumber(x);
	if (bs->length <= 4)
		{
		l=ASN1_INTEGER_get(bs);
		if (l < 0)
			{
			l= -l;
			neg="-";
			}
		else
			neg="";
		pLeft = strchr(pLeft, '\0');
		sprintf (pLeft, " %s%lu\n",neg,l);
		}
	else
		{
		neg=(bs->type == V_ASN1_NEG_INTEGER)?" (Negative)":"";
		pLeft = strchr(pLeft, '\0');
		sprintf (pLeft, "\n%12s%s","",neg);

		for (i=0; i<bs->length; i++)
			{
			pLeft = strchr(pLeft, '\0');
			sprintf (pLeft, "%02x%c",bs->data[i], ((i+1 == bs->length)?'\n':':'));
			}
		}

	//Signature Algorithm
	pLeft = strchr(pLeft, '\0');
	sprintf (pLeft, "Signature Algorithm: ");
	if ((ci->signature->algorithm == NULL) || (ci->signature->algorithm->data == NULL))
		strcat(buf, "NULL");
	pLeft = strchr(pLeft, '\0');
	i2t_ASN1_OBJECT(pLeft,80,ci->signature->algorithm);

	//Issuer
	pLeft = strchr(pLeft, '\0');
	sprintf (pLeft, "\nIssuer: ");
	NAME_print(buf, X509_get_issuer_name(x), nmindent);
	
	//Validity
	strcat(buf, "\nValidity\n");
	strcat(buf, "    Not Before: ");
	pLeft = strchr(pLeft, '\0');
	if (!TIME_print(pLeft,X509_get_notBefore(x))) goto err;
	strcat(buf, "\n");
	strcat(buf, "    Not After:  ");
	pLeft = strchr(pLeft, '\0');
	if (!TIME_print(pLeft,X509_get_notAfter(x))) goto err;

	//Subject
	pLeft = strchr(pLeft, '\0');
	sprintf (pLeft, "\nSubject: ");
	NAME_print(buf, X509_get_subject_name(x), nmindent);

	//Subject Public Key Info
	pLeft = strchr(pLeft, '\0');
	sprintf(pLeft, "\nSubject Public Key Info:\n");
	pLeft = strchr(pLeft, '\0');
	sprintf(pLeft, "%4sPublic Key Algorithm: ","");

        if ((ci->key->algor->algorithm == NULL) || (ci->key->algor->algorithm->data == NULL))
                strcat(buf,"NULL");
	pLeft = strchr(pLeft, '\0');
        i2t_ASN1_OBJECT(pLeft,80,ci->key->algor->algorithm);

	//RSA Public Key: 
	pLeft = strchr(pLeft, '\0');
	pkey=X509_get_pubkey(x);
	if (pkey == NULL)
		sprintf(pLeft,"\n%4sUnable to load Public Key\n","");
	else
#ifndef OPENSSL_NO_RSA
	if (pkey->type == EVP_PKEY_RSA)
		{
		sprintf(pLeft,"\n%4sRSA Public Key: (%d bit)\n","", BN_num_bits(pkey->pkey.rsa->n));
		pLeft = strchr(pLeft, '\0');
		RSA_print_n(pLeft,pkey->pkey.rsa,4);
		}
	else
#endif
		sprintf(pLeft,"%4sUnknown Public Key:\n","");

	EVP_PKEY_free(pkey);

	//signature
	pLeft = strchr(pLeft, '\0');
	if(signature_print(pLeft, x->sig_alg, x->signature) <= 0) goto err;

	//Return
	ret=1;
err:
	if (str != NULL) ASN1_STRING_free(str);
	if (m != NULL) OPENSSL_free(m);
	return(ret);
}

int print_csr_Properties(X509_REQ *x, char *buf) {
	long l;
	int ret=0,i;
	char *m=NULL,mlch = ' ';
	int nmindent = 0;
	X509_REQ_INFO *ci;
	ASN1_INTEGER *bs;
	EVP_PKEY *pkey=NULL;
	const char *neg;
	ASN1_STRING *str=NULL;
	char *pLeft;

	mlch = '\n';
	nmindent = 16;

	//CSR
	ci=x->req_info;
	buf[0] = '\0';
	pLeft = buf;
	
	//Subject
	sprintf (pLeft, "Subject: ");
	NAME_print(buf, X509_REQ_get_subject_name(x), nmindent);

	//Subject Public Key Info
	pLeft = strchr(pLeft, '\0');
	sprintf(pLeft, "\nSubject Public Key Info:\n");
	pLeft = strchr(pLeft, '\0');
	sprintf(pLeft, "%4sPublic Key Algorithm: ","");

        if ((ci->pubkey->algor->algorithm == NULL) || (ci->pubkey->algor->algorithm->data == NULL))
                strcat(buf,"NULL");
	pLeft = strchr(pLeft, '\0');
        i2t_ASN1_OBJECT(pLeft,80,ci->pubkey->algor->algorithm);

	//RSA Public Key: 
	pLeft = strchr(pLeft, '\0');
	pkey=X509_REQ_get_pubkey(x);
	if (pkey == NULL)
		sprintf(pLeft,"\n%4sUnable to load Public Key\n","");
	else
#ifndef OPENSSL_NO_RSA
	if (pkey->type == EVP_PKEY_RSA)
		{
		sprintf(pLeft,"\n%4sRSA Public Key: (%d bit)\n","", BN_num_bits(pkey->pkey.rsa->n));
		pLeft = strchr(pLeft, '\0');
		RSA_print_n(pLeft,pkey->pkey.rsa,4);
		}
	else
#endif
		sprintf(pLeft,"%4sUnknown Public Key:\n","");

	EVP_PKEY_free(pkey);

	//signature
	pLeft = strchr(pLeft, '\0');
	if(signature_print(pLeft, x->sig_alg, x->signature) <= 0) goto err;

	ret=1;
err:
	if (str != NULL) ASN1_STRING_free(str);
	if (m != NULL) OPENSSL_free(m);
	return(ret);
}

int NAME_print(char *buf, X509_NAME *name, int obase) {
	char *s,*c,*b;
	int ret=0,l,ll,i,first=1;

	ll=80-2-obase;

	b=s=X509_NAME_oneline(name,NULL,0);
	if (!*s)
		{
		OPENSSL_free(b);
		return 1;
		}
	s++; /* skip the first slash */

	l=ll;
	c=s;
	for (;;)
		{
#ifndef CHARSET_EBCDIC
		if (	((*s == '/') &&
				((s[1] >= 'A') && (s[1] <= 'Z') && (
					(s[2] == '=') ||
					((s[2] >= 'A') && (s[2] <= 'Z') &&
					(s[3] == '='))
				 ))) ||
			(*s == '\0'))
#else
		if (	((*s == '/') &&
				(isupper(s[1]) && (
					(s[2] == '=') ||
					(isupper(s[2]) &&
					(s[3] == '='))
				 ))) ||
			(*s == '\0'))
#endif
			{
			if ((l <= 0) && !first)
				{
				first=0;
				strcat(buf, "\n");
				for (i=0; i<obase; i++)
					{
					strcat(buf, " ");
					}
				l=ll;
				}
			i=s-c;
			strncat(buf, c, i);
			c+=i;
			c++;
			if (*s != '\0')
				{
				strcat(buf, ", ");
				}
			l--;
			}
		if (*s == '\0') break;
		s++;
		l--;
		}
	
	ret=1;
	if (0)
		{
err:
		X509err(X509_F_X509_NAME_PRINT,ERR_R_BUF_LIB);
		}
	OPENSSL_free(b);
	return(ret);
}

int TIME_print(char *buf, ASN1_TIME *tm) {
	if(tm->type == V_ASN1_UTCTIME) return UTCTIME_print(buf, tm);
	if(tm->type == V_ASN1_GENERALIZEDTIME)
				return GENERALIZEDTIME_print(buf, tm);
	strcat(buf,"Bad time value");
	return(0);
}

static const char *mon[12]=
    {
    "Jan","Feb","Mar","Apr","May","Jun",
    "Jul","Aug","Sep","Oct","Nov","Dec"
    };

int GENERALIZEDTIME_print(char *buf, ASN1_GENERALIZEDTIME *tm)
	{
	char *v;
	int gmt=0;
	int i;
	int y=0,M=0,d=0,h=0,m=0,s=0;

	i=tm->length;
	v=(char *)tm->data;

	if (i < 12) goto err;
	if (v[i-1] == 'Z') gmt=1;
	for (i=0; i<12; i++)
		if ((v[i] > '9') || (v[i] < '0')) goto err;
	y= (v[0]-'0')*1000+(v[1]-'0')*100 + (v[2]-'0')*10+(v[3]-'0');
	M= (v[4]-'0')*10+(v[5]-'0');
	if ((M > 12) || (M < 1)) goto err;
	d= (v[6]-'0')*10+(v[7]-'0');
	h= (v[8]-'0')*10+(v[9]-'0');
	m=  (v[10]-'0')*10+(v[11]-'0');
	if (	(v[12] >= '0') && (v[12] <= '9') &&
		(v[13] >= '0') && (v[13] <= '9'))
		s=  (v[12]-'0')*10+(v[13]-'0');

	if (sprintf(buf,"%s %2d %02d:%02d:%02d %d%s",
		mon[M-1],d,h,m,s,y,(gmt)?" GMT":"") <= 0)
		return(0);
	else
		return(1);
err:
	return(0);
	}

int UTCTIME_print(char *buf, ASN1_UTCTIME *tm)
	{
	char *v;
	int gmt=0;
	int i;
	int y=0,M=0,d=0,h=0,m=0,s=0;

	i=tm->length;
	v=(char *)tm->data;

	if (i < 10) goto err;
	if (v[i-1] == 'Z') gmt=1;
	for (i=0; i<10; i++)
		if ((v[i] > '9') || (v[i] < '0')) goto err;
	y= (v[0]-'0')*10+(v[1]-'0');
	if (y < 50) y+=100;
	M= (v[2]-'0')*10+(v[3]-'0');
	if ((M > 12) || (M < 1)) goto err;
	d= (v[4]-'0')*10+(v[5]-'0');
	h= (v[6]-'0')*10+(v[7]-'0');
	m=  (v[8]-'0')*10+(v[9]-'0');
	if (	(v[10] >= '0') && (v[10] <= '9') &&
		(v[11] >= '0') && (v[11] <= '9'))
		s=  (v[10]-'0')*10+(v[11]-'0');

	if (sprintf(buf,"%s %2d %02d:%02d:%02d %d%s",
		mon[M-1],d,h,m,s,y+1900,(gmt)?" GMT":"") <= 0)
		return(0);
	else
		return(1);
err:
	return(0);
	}

int RSA_print_n(char *buf, const RSA *x, int off)
	{
	char str[128];
	const char *s;
	unsigned char *m=NULL;
	int ret=0;
	size_t buf_len=0, i;

	if (x->n)
		buf_len = (size_t)BN_num_bytes(x->n);
	if (x->e)
		if (buf_len < (i = (size_t)BN_num_bytes(x->e)))
			buf_len = i;
	if (x->d)
		if (buf_len < (i = (size_t)BN_num_bytes(x->d)))
			buf_len = i;
	if (x->p)
		if (buf_len < (i = (size_t)BN_num_bytes(x->p)))
			buf_len = i;
	if (x->q)
		if (buf_len < (i = (size_t)BN_num_bytes(x->q)))
			buf_len = i;
	if (x->dmp1)
		if (buf_len < (i = (size_t)BN_num_bytes(x->dmp1)))
			buf_len = i;
	if (x->dmq1)
		if (buf_len < (i = (size_t)BN_num_bytes(x->dmq1)))
			buf_len = i;
	if (x->iqmp)
		if (buf_len < (i = (size_t)BN_num_bytes(x->iqmp)))
			buf_len = i;

	m=(unsigned char *)OPENSSL_malloc(buf_len+10);
	if (m == NULL)
		{
		RSAerr(RSA_F_RSA_PRINT,ERR_R_MALLOC_FAILURE);
		goto err;
		}

	if (x->d != NULL)
		{
		if(!indent_n(buf,off,128))
		   goto err;
		sprintf(buf,"Private-Key: (%d bit)\n",BN_num_bits(x->n));
		}

	if (x->d == NULL)
		sprintf(str,"Modulus (%d bit):",BN_num_bits(x->n));
	else
		strcpy(str,"modulus:");
	if (!print_n(buf,str,x->n,m,off)) goto err;
	s=(x->d == NULL)?"Exponent:":"publicExponent:";
	if (!print_n(buf,s,x->e,m,off+4)) goto err;
	if (!print_n(buf,"privateExponent:",x->d,m,off)) goto err;
	if (!print_n(buf,"prime1:",x->p,m,off)) goto err;
	if (!print_n(buf,"prime2:",x->q,m,off)) goto err;
	if (!print_n(buf,"exponent1:",x->dmp1,m,off)) goto err;
	if (!print_n(buf,"exponent2:",x->dmq1,m,off)) goto err;
	if (!print_n(buf,"coefficient:",x->iqmp,m,off)) goto err;
	ret=1;
err:
	if (m != NULL) OPENSSL_free(m);
	return(ret);
	}

int indent_n(char *buf, int indent, int max)
	{
	if(indent < 0)
		indent=0;
	if(indent > max)
		indent=max;
	while(indent--)
		strcat(buf," ");
	return 1;
	}

static int print_n(char *output, const char *number, BIGNUM *num, unsigned char *buf, int off)
	{
	int n,i;
	const char *neg;
	char *pLeft;

	if (num == NULL) return(1);
	neg=(num->neg)?"-":"";
	if(!indent_n(output,off,128))
		return 0;

	if (BN_num_bytes(num) <= BN_BYTES)
		{
		pLeft = strchr(output, '\0');
		sprintf(pLeft,"%s %s%lu (%s0x%lx)\n",number,neg,
			(unsigned long)num->d[0],neg,(unsigned long)num->d[0]);
		}
	else
		{
		buf[0]=0;
		pLeft = strchr(output, '\0');
		sprintf(pLeft,"%s%s",number,(neg[0] == '-')?" (Negative)":"");
		n=BN_bn2bin(num,&buf[1]);
	
		if (buf[1] & 0x80)
			n++;
		else	buf++;

		for (i=0; i<n; i++)
			{
			if ((i%15) == 0)
				{
				if(strcat(output,"\n") <= 0 || !indent_n(output,off+4,128))
				    return 0;
				}
			pLeft = strchr(pLeft, '\0');
			sprintf(pLeft,"%02x%s",buf[i],((i+1) == n)?"":":");
			}
		strcat(output,"\n");
		}
	return(1);
	}

int signature_print(char *buf, X509_ALGOR *sigalg, ASN1_STRING *sig)
{
	unsigned char *s;
	int i, n;
	char *pLeft;

	strcat(buf,"Signature\n");
	strcat(buf,"    Algorithm: ");

        if ((sigalg->algorithm == NULL) || (sigalg->algorithm->data == NULL))
                strcat(buf,"NULL");
	pLeft = strchr(buf, '\0');
        i2t_ASN1_OBJECT(pLeft,80,sigalg->algorithm);

	strcat(buf,"\n    Signature:");

	n=sig->length;
	s=sig->data;
	for (i=0; i<n; i++)
	{
		if ((i%18) == 0)
			strcat(buf,"\n        ");
		pLeft = strchr(pLeft, '\0');
		sprintf(pLeft,"%02x%s",s[i],((i+1) == n)?"":":");
	}
	strcat(buf,"\n");
	return 1;
}

int check_cert_value(int type, char *input, tCertInfo *out ) /* 0: Failure,  1: Success */
{
	char *pToken = NULL;
	char *temp_input = (char*)malloc( strlen(input) + 1 );
	char *tempstring = NULL;
	char comma[] = ",";
	char *temp_pointer = NULL;
	unsigned int nLength = 0, i;
	int dataLen = 0, 
	    count = 0,
	    idx = 0; 

        dataLen = strlen (input);
        for (idx = 0; idx < dataLen; idx++) {
                if (input [idx] == ',') {
                        if (input [idx+1] == ',') 	return 0;
                        count++;
                }
        }
	if (type == HTTPS_CERTTYPE_CERT) {
        	if (count != 7 || input [dataLen-1] == ',') 	return 0;
	} else if (type == HTTPS_CERTTYPE_CSR) {
        	if (count != 6 || input [dataLen-1] == ',') 	return 0;
	} else 							return 0; 
	
    	strcpy( temp_input, input );

// Country[3]
    	pToken = strtok( temp_input, comma );
    	if( pToken == NULL || strlen(pToken) != HTTPS_COUNTRY_SIZE || !isalpha(pToken[0]) || !isalpha(pToken[1]) ) {
        	free( temp_input );
        	return 0;
    	}
    	strncpy( out->Country, pToken, 3 );
    	nLength += strlen( pToken );

// Province[41]
    	pToken = strtok( NULL, comma );
    	if ((strlen(pToken) > HTTPS_PROVINCE_SIZE) || !is_Distinguish_Name(pToken)) {
        	free( temp_input );
        	return 0;
    	}
    	strcpy( out->Province, pToken );
    	nLength += 1 + strlen( pToken );

// Locality[41]
    	pToken = strtok( NULL, comma );
    	if ((strlen(pToken) > HTTPS_LOCALITY_SIZE) || !is_Distinguish_Name(pToken)) {
        	free( temp_input );
        	return 0;
    	}
    	strcpy( out->Locality, pToken );
    	nLength += 1 + strlen( pToken );

// Org[41]
    	pToken = strtok( NULL, comma );
    	if ((strlen(pToken) > HTTPS_ORGANIZATION_SIZE) || !is_Distinguish_Name(pToken) ) {
        	free( temp_input );
        	return 0;
    	}
    	strcpy( out->Organization, pToken );
    	nLength += 1 + strlen( pToken );

// Division[41]
	pToken = strtok( NULL, comma );
    	if((strlen(pToken) > HTTPS_DIVISION_SIZE) || !is_Distinguish_Name(pToken) ) {
        	free( temp_input );
        	return 0;
    	}
    	strcpy( out->Division, pToken );
    	nLength += 1 + strlen( pToken );

// CommonName[65]
    	pToken = strtok( NULL, comma );
    	if ( pToken == NULL ) {
        	free( temp_input );
        	return 0;
    	}

	temp_pointer = pToken + strlen( pToken ) + 1;
    	if ((strlen(pToken) > HTTPS_COMMONANME_SIZE) || !is_Common_Name(pToken) ) {
        	free( temp_input );
        	return 0;
    	}
    	strcpy( out->CommonName, pToken );
    	nLength += 1 + strlen( pToken );

// KeyLen = 512, 1024 or 2048
    	pToken = strtok( temp_pointer, comma );
    	out->KeyLen = is_KeyLen(pToken);
    	if( out->KeyLen == 0 ) {
        	free( temp_input );
        	return 0;
    	}
    	nLength += 1 + strlen( pToken );

    	if (type == HTTPS_CERTTYPE_CERT) {
// Validity = 1~9999
    		pToken = strtok( NULL, comma );
   		out->Validity = is_Validity(pToken);
    		if( out->Validity == 0 ) {
        		free( temp_input );
        		return 0;
    		}
    		nLength += 1 + strlen( pToken );
	}

    	free( temp_input );
    	if( nLength != strlen( input ) ) 		return 0;
    	else 						return 1;
}

int is_Distinguish( char ch )
{
    if( isalpha( ch ) || isdigit( ch ) ||
        ch == '-' || ch == '_' || ch == '.' || ch == ' ' )
        return 1;
    else
        return 0;
}

int is_Distinguish_Name( char *name )
{
    int i=0; 
    int nLength = strlen( name );
    if( name == NULL )
        return 0;
    if( nLength == 0 || nLength > 40 ||
        !( isalpha(name[0]) || isdigit(name[0])) || 
        !( isalpha(name[nLength-1]) || isdigit(name[nLength-1]) || name[nLength-1] == '.') )
        return 0;
    for( i=1 ; i<nLength-1 ; i++ )
        if( !is_Distinguish(name[i]) )
            return 0;
    return 1;
}

int is_Common( char ch )
{
    if( isalpha( ch ) || isdigit( ch ) ||
        ch == '-' || ch == '_' || ch == '.' || ch == '*' )
        return 1;
    else
        return 0;
}

int is_IPv4_Address( char* pSourceIP )
{
    char* pProcess = (char*)malloc( strlen(pSourceIP) + 1 );

    char *pToken,
         dot[] = ".";
    unsigned int nSection = 0,
                 nSectionLength = 0,
                 nSumLength = 0,
                 i,
                 j;

	long lSection = strtol( pSourceIP, NULL, 10 ); // check the value range

    strcpy( pProcess, pSourceIP );
    if( pSourceIP == NULL || pSourceIP[0] == '.' || strstr( pSourceIP, "..") != NULL )
        return 0;

    for( j=0 ; j<4 ; j++ )
    {
        if( j!=0 )
            pToken = strtok( NULL, dot );
        else
            pToken = strtok( pProcess, dot );
        if( pToken == NULL )
        {
            free(pProcess);
            return 0;
        }

        nSectionLength = strlen( pToken );
        if( nSectionLength == 0 || ( nSectionLength > 1 && pToken[0] == '0' ) )
        {
            free(pProcess); // 01 003 is not allow
            return 0;
        }
        nSumLength += 1+nSectionLength;  

        switch( nSection )
        {
        case 0: if( lSection<=0 || lSection>=224 ){ free(pProcess); return 0; } break;
        case 1: if( lSection< 0 || lSection> 225 ){ free(pProcess); return 0; } break;
        case 2: if( lSection< 0 || lSection> 255 ){ free(pProcess); return 0; } break;
        case 3: if( lSection<=0 || lSection>=255 ){ free(pProcess); return 0; } break;
        default: free(pProcess); return 0;
        }
        nSection++;

        for( i=0 ; i<nSectionLength ; i++ )
            if( !isdigit( pToken[i] ) )
            {
                free(pProcess); // between in 0 ~ 9
                return 0;
            }
    }

    free(pProcess);

    nSumLength--; 
    if( nSumLength != strlen(pSourceIP) )
        return 0;
    else
        return 1;
}

int is_Host_Address( char *pName )                  // if not, return 0
{
    int nNameLength = strlen(pName),
        nLength,
        nSumLength,
        i;
    char* pProcess = (char*)malloc( nNameLength + 1 ),
         *pToken,
         dot[] = ".";

    if( pName == NULL || strstr( pName, "..") != NULL || strchr( pName+1, '*' ) != NULL )
        return 0;

    if( pName[nNameLength-1] == '.' )     // period can not be the end char
        return 0;

    strcpy( pProcess, pName );

    pToken = strtok( pProcess, dot );
    nSumLength = nLength = strlen( pToken );

    if( pToken[0] == '*' )   
    {
        if( nLength != 1 )
        {
            free(pProcess);
            return 0;
        }
    }
    else if( !( isalpha(pToken[0]) || isdigit(pToken[0]) ) || !( isalpha(pToken[nLength-1]) || isdigit(pToken[nLength-1]) ) )
    {
        free(pProcess);
        return 0;
    }
    for( i=1 ; i<nLength-1 ; i++ )
        if( !is_Common(pToken[i]) )
        {
            free(pProcess);
            return 0;
        }

    while( nSumLength < nNameLength )
    {
        pToken = strtok( NULL, dot );
        nLength = strlen( pToken );
        nSumLength += 1 + nLength;   // Add period & string 
        if( !( isalpha(pToken[0]) || isdigit(pToken[0]) ) || !( isalpha(pToken[nLength-1]) || isdigit(pToken[nLength-1] ) ) )
        {
            free(pProcess);
            return 0;
        }
        for( i=1 ; i<nLength-1 ; i++ )
            if( !is_Common(pToken[i]) )
            {
                free(pProcess);
                return 0;
            }
    }

    free(pProcess);
    if( nSumLength != nNameLength )
        return 0;
    else
        return 1;
}

int is_Common_Name( char *pAddress )                  // if not, return 0
{
    int judge_only_digit = 1; 

    int nLength = strlen( pAddress ),
        i;

    if( pAddress == NULL )
        return 0;
    for( i=0 ; i<nLength ; i++ )
    {
        if( !is_Common( pAddress[i] ) )
            return 0;
        if( !isdigit( pAddress[i] ) && pAddress[i] != '.' )
            judge_only_digit = 0;
    }

    if( judge_only_digit )
    {
        if( is_IPv4_Address( pAddress ) )
            return 1;
        else
            return 0;
    }
    else if( is_Host_Address( pAddress ) )
        return 1;
    else
        return 0;
}

int is_KeyLen( char *pNum )
{
    int nLength = strlen(pNum);
    if( pNum == NULL )
        return 0;
    if( nLength == 3 && !strncmp( pNum, "512\0", 4 ) )
        return 512;
    else if( nLength == 4 )
    {
        if( !strncmp( pNum, "1024\0", 5 ) )
            return 1024;
        else if( !strncmp( pNum, "2048\0", 5 ) )
            return 2048;
        else
            return 0;
    }
    else
        return 0;
}

int is_Validity( char *pNum )
{
    int nLength = strlen(pNum),
        i;
    if( pNum == NULL )
        return 0;
        
    if( nLength == 0 || nLength > 4 || pNum[0] == '0' )
        return 0;

    for( i=0 ; i<nLength ; i++ )
        if( !isdigit( pNum[i] ) )
            return 0;

    return strtol( pNum, NULL, 10 );
}

